/*
 * qstat 1.5
 * by Steve Jankowski
 * steve@activesw.com
 * http://www.activesw.com/people/steve/qstat.html
 *
 * Thanks to Per Hammer for the OS/2 patches (per@mindbend.demon.co.uk)
 *
 * Inspired by QuakePing by Len Norton
 *
 * Copyright 1996 by Steve Jankowski
 *
 *      Permission granted to use this software for any purpose you
 *      desire provided that existing copywrite notices are retained
 *      verbatim in all copies and you derive no monetary benefit from
 *      use of the source code or resulting program, files, or executable
 *      programs except as noted.
 *
 *      Specific rights reserved:
 *      o Resale of the programs, source code, files, or program
 *      resulting from compiling the source code is reserved without
 *      written permission of the author, Steve Jankowski.
 *      o Inclusion of the programs, source code or program resulting
 *      from compiling the source code within another program or
 *      system for resale is reserved without the written permission
 *      of the author, Steve Jankowski.
 *      o Inclusion of the programs, source code or programs resulting
 *      from compiling the source code within a "compilation" software
 *      product is reserved without the written permission of the
 *      author, Steve Jankowski.
 *      o Redistribution of a modified version of the archive or its
 *      files is reserved without the written permission of the author,
 *      Steve Jankowski.
 *
 *      Specific rights granted:
 *      o Permission is granted to use the programs and source code to
 *      generate information from which the user derives monetary
 *      benefit.
 */

#define VERSION "1.5"

/* OS/2 defines */
#ifdef __OS2__
#define BSD_SELECT
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <string.h>
#include <ctype.h>

#ifdef unix
#include <unistd.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>

#define INVALID_SOCKET -1
#define SOCKET_ERROR -1
#ifndef INADDR_NONE
#define INADDR_NONE ~0
#endif
#endif /* unix */

#ifdef _WIN32
#include <windows.h>
#include <sys/timeb.h>
#define close(a) closesocket(a)
int gettimeofday(struct timeval *now, void *blah)
{
    struct timeb timeb;
    ftime( &timeb);
    now->tv_sec= timeb.time;
    now->tv_usec= (unsigned int)timeb.millitm * 1000;
}
#endif

#ifdef __OS2__
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>
#include <utils.h>
 
#define INVALID_SOCKET -1
#define SOCKET_ERROR -1
#define close(a)        soclose(a)
#endif

/* Various magic numbers.
 */

#define MAXFD			20
#define DEFAULT_PORT		26000
#define DEFAULT_RETRIES		3
#define DEFAULT_RETRY_INTERVAL	500		/* milli-seconds */
#define MAXIP 4

/* Structures for keeping information about Quake servers, server
 * rules, and players.
 */

struct player;

struct qserver {
    char *arg;
    char *host_name;
    struct in_addr ip[MAXIP];
    int fd;
    int port;
    int retry1;
    int retry2;
    int n_retries;
    struct timeval packet_time1;
    struct timeval packet_time2;
    int ping_total;		/* average is ping_total / n_requests */
    int n_requests;
    int n_packets;

    char *server_name;
    char *address;
    char *map_name;
    int max_players;
    int num_players;
    int protocol_version;

    int next_player_info;
    int n_player_info;
    struct player *players;

    char *next_rule;
    int n_rules;
    struct rule *rules;
    int missing_rules;

    struct qserver *next;
};

struct player  {
    int number;
    char *name;
    int frags;
    int connect_time;
    int shirt_color;
    int pants_color;
    char *address;
    struct player *next;
};

struct rule  {
    char *name;
    char *value;
    struct rule *next;
};

/* Values set by command-line arguments
 */

int hostname_lookup= 0;		/* set if -H was specified */
int new_style= 1;		/* unset if -old was specified */
int n_retries= DEFAULT_RETRIES;
int retry_interval= DEFAULT_RETRY_INTERVAL;
int get_player_info= 0;
int get_server_rules= 0;
int up_servers_only= 0;
int no_full_servers= 0;
int raw_display= 0;
char *raw_delimiter= "\t";
int player_address= 0;
int hex_player_names= 0;

#define DEFAULT_COLOR_NAMES_RAW		0
#define DEFAULT_COLOR_NAMES_DISPLAY	1
int color_names= -1;

#define SECONDS 0
#define CLOCK_TIME 1
#define STOPWATCH_TIME 2
#define DEFAULT_TIME_FMT_RAW		SECONDS
#define DEFAULT_TIME_FMT_DISPLAY	CLOCK_TIME
int time_format= -1;

struct qserver *servers= NULL;
int connected= 0;

char *DOWN= "DOWN";
char *SYSERROR= "ERROR";
char *TIMEOUT= "TIMEOUT";

unsigned int sample= 0x0f000000;
unsigned char *byteswap= (unsigned char *) &sample;
#define BIG_END (*byteswap)
#define LITTLE_END (!(*byteswap))

/* Output and formatting functions
 */

void display_server( struct qserver *server);
void display_server_rules( struct qserver *server);
void display_player_info( struct qserver *server);

void raw_display_server( struct qserver *server);
void raw_display_server_rules( struct qserver *server);
void raw_display_player_info( struct qserver *server);

int is_default_rule( struct rule *rule);
char *escape(unsigned char*);
char *quake_color( int color);
char *play_time( int seconds);

/* MODIFY HERE
 * Change this routine to display stats however you like.
 */
void
display_server(
    struct qserver *server
)
{
    char name[100];

    if ( raw_display)  {
	raw_display_server( server);
	return;
    }

    if ( server->server_name == DOWN)  {
	printf( "%-16s %10s\n", server->arg, DOWN);
	return;
    }
    if ( server->server_name == TIMEOUT)  {
	if ( ! up_servers_only)
	    printf( "%-16s no response\n",
			(hostname_lookup) ? server->host_name : server->arg);
	return;
    }

    if ( no_full_servers && server->num_players == server->max_players)
	return;

    if ( new_style)  {
	printf( "%-17s %2d/%2d %11s %6d / %1d        %s\n",
	    (hostname_lookup) ? server->host_name : server->arg,
	    server->num_players, server->max_players,
	    server->map_name,
	    server->ping_total/server->n_requests,
	    server->n_retries,
	    server->server_name);
	if ( get_server_rules)
	    display_server_rules( server);
	if ( get_player_info)
	    display_player_info( server);
    }
    else  {
	sprintf( name, "\"%s\"", server->server_name);
	printf( "%-16s %10s map %s at %22s %d/%d players %d ms\n", 
	    (hostname_lookup) ? server->host_name : server->arg,
	    name, server->map_name,
	    server->address, server->num_players, server->max_players,
	    server->ping_total/server->n_requests);
    }
}

void
display_header()
{
    printf( "%-16s %8s %8s %15s    %s\n", "ADDRESS", "PLAYERS", "MAP",
	"RESPONSE TIME", "NAME");
}

void
display_server_rules( struct qserver *server)
{
    struct rule *rule;
    int printed= 0;
    rule= server->rules;
    for ( ; rule != NULL; rule= rule->next)  {
	if ( ! is_default_rule( rule))  {
	    printf( "%c%s=%s", (printed)?',':'\t', rule->name, rule->value);
	    printed++;
	}
    }
    if ( printed)
	puts("");
}

void
display_player_info( struct qserver *server)
{
    char fmt[128];
    struct player *player;

    strcpy( fmt, "\t#%-2d %3d frags %9s ");
    if ( color_names)
	strcat( fmt, "%9s:%-9s ");
    else
	strcat( fmt, "%2d:%-2d ");
    if ( player_address)
	strcat( fmt, "%22s ");
    else
	strcat( fmt, "%s");
    strcat( fmt, "%s\n");

    player= server->players;
    for ( ; player != NULL; player= player->next)  {
	printf( fmt,
		player->number,
		player->frags,
		play_time(player->connect_time),
		quake_color(player->shirt_color),
		quake_color(player->pants_color),
		(player_address)?player->address:"",
		escape( (unsigned char *) player->name));
    }
}


/* Raw output for web master types
 */

#define RD raw_delimiter

void
raw_display_server( struct qserver *server)
{
    if ( server->server_name == DOWN)  {
	if ( ! up_servers_only)
	    printf( "%s%s%s\n\n", server->arg, RD, DOWN);
	return;
    }
    if ( server->server_name == TIMEOUT)  {
	if ( ! up_servers_only)
	    printf( "%s%s%s\n\n", server->arg, RD, TIMEOUT);
	return;
    }

    printf( "%s" "%s%s" "%s%s" "%s%d" "%s%s" "%s%d" "%s%d" "%s%d" "%s%d",
	server->arg,
	RD, escape((unsigned char *) server->server_name),
	RD, server->address,
	RD, server->protocol_version,
	RD, server->map_name,
	RD, server->max_players,
	RD, server->num_players,
	RD, server->ping_total/server->n_requests,
	RD, server->n_retries
    );
    puts("");

    if ( get_server_rules)
	raw_display_server_rules( server);
    if ( get_player_info)
	raw_display_player_info( server);
    puts("");
}

void
raw_display_server_rules( struct qserver *server)
{
    struct rule *rule;
    int printed= 0;
    rule= server->rules;
    for ( ; rule != NULL; rule= rule->next)  {
	printf( "%s%s=%s", (printed)?RD:"", rule->name, rule->value);
	printed++;
    }
    if ( server->missing_rules)
	printf( "%s?", (printed)?RD:"");
    puts("");
}

void
raw_display_player_info( struct qserver *server)
{
    char fmt[128];
    struct player *player;

    strcpy( fmt, "%d" "%s%s" "%s%s" "%s%d" "%s%s");
    if ( color_names)
	strcat( fmt, "%s%s" "%s%s");
    else
	strcat( fmt, "%s%d" "%s%d");

    player= server->players;
    for ( ; player != NULL; player= player->next)  {
	printf( fmt,
		player->number,
		RD, escape( (unsigned char *) player->name),
		RD, player->address,
		RD, player->frags,
		RD, play_time(player->connect_time),
		RD, quake_color(player->shirt_color),
		RD, quake_color(player->pants_color)
	);
	puts("");
    }
}


/* Definitions for the Quake network protocol.
 */

#define PACKET_LEN 1600

/* Quake packet formats and magic numbers
 */
struct qheader  {
    unsigned char flag1;
    unsigned char flag2;
    unsigned short length;
    unsigned char op_code;
};

struct qpacket  {
    unsigned char flag1;
    unsigned char flag2;
    unsigned short length;
    unsigned char op_code;
    unsigned char data[1500];
};

#define Q_FLAG1			0x80
#define Q_FLAG2			0x00
#define Q_NET_PROTOCOL_VERSION	3
#define Q_HEADER_SIZE		5

#define Q_CCREQ_CONNECT		0x01
#define Q_CCREP_ACCEPT		0x81
#define Q_CCREP_REJECT		0x82

#define Q_CCREQ_SERVER_INFO	0x02
#define Q_CCREP_SERVER_INFO	0x83

#define Q_CCREQ_PLAYER_INFO	0x03
#define Q_CCREP_PLAYER_INFO	0x84

#define Q_CCREQ_RULE_INFO	0x04
#define Q_CCREP_RULE_INFO	0x85

#define Q_DEFAULT_SV_MAXSPEED	"320"
#define Q_DEFAULT_SV_FRICTION	"4"
#define Q_DEFAULT_SV_GRAVITY	"800"
#define Q_DEFAULT_NOEXIT	"0"
#define Q_DEFAULT_TEAMPLAY	"0"
#define Q_DEFAULT_TIMELIMIT	"0"
#define Q_DEFAULT_FRAGLIMIT	"0"

/* These packets are fixed size, so don't bother rebuilding them each time.
 * This only works because the packets are an even number in size.
 */
struct {
    unsigned char flag1;
    unsigned char flag2;
    unsigned short length;
    unsigned char op_code;
    char name[6];
    unsigned char version;
} qserverinfo =
	{ Q_FLAG1, Q_FLAG2, sizeof(qserverinfo), Q_CCREQ_SERVER_INFO,
	"QUAKE", Q_NET_PROTOCOL_VERSION };

struct {
    unsigned char flag1;
    unsigned char flag2;
    unsigned short length;
    unsigned char op_code;
    unsigned char player_number;
} qplayerinfo =
	{ Q_FLAG1, Q_FLAG2, sizeof(qplayerinfo), Q_CCREQ_PLAYER_INFO, 0 };


/* Misc flags
 */

char * NO_SERVER_RULES= NULL;
int NO_PLAYER_INFO= 0xffff;
struct timeval packet_recv_time;
#define FORCE 1

int cleanup_qserver( struct qserver *server, int force);

void deal_with_packet( struct qserver *server, char *pkt, int pktlen);
int server_info_packet( struct qserver *server, struct qpacket *pkt,
	int datalen);
int player_info_packet( struct qserver *server, struct qpacket *pkt,
	int datalen);
int rule_info_packet( struct qserver *server, struct qpacket *pkt, int datalen);

int time_delta( struct timeval *later, struct timeval *past);
char * strherror( int h_err);
int connection_refused();

void add_file( char *filename);
int add_qserver( char *arg);
int bind_qserver( struct qserver *server);
void bind_sockets();
void send_packets();
int send_server_request_packet( struct qserver *server);
int send_player_request_packet( struct qserver *server);
int send_rule_request_packet( struct qserver *server);

void set_fds( fd_set *fds);
void get_next_timeout( struct timeval *timeout);

/* Print an error message and the program usage notes
 */

void
usage( char *msg, char **argv)
{
    if ( msg)
	fprintf( stderr, msg);

    printf( "Usage: %s [options ...] [-retry retries] [-interval interval] [-raw delimiter] [-f file] host[:port] ...\n", argv[0]);
    printf( "\tWhere host is an IP address or host name\n");
    printf( "\tPort defaults to %d if not specified\n", DEFAULT_PORT);
    printf( "-H\t\tresolve host names\n");
    printf( "-R\t\tserver rules\n");
    printf( "-P\t\tplayer info\n");
    printf( "-u\t\tonly display servers that are up\n");
    printf( "-nf\t\tonly display servers that are not full\n");
    printf( "-cn\t\tdisplay color names instead of numbers\n");
    printf( "-ncn\t\tdisplay color numbers instead of names\n");
    printf( "-tc\t\tdisplay time in clock format (DhDDmDDs)\n");
    printf( "-tsw\t\tdisplay time in stop-watch format (DD:DD:DD)\n");
    printf( "-ts\t\tdisplay time in seconds\n");
    printf( "-pa\t\tdisplay player address\n");
    printf( "-hpn\t\tdisplay player names in hex\n");
    printf( "-old\t\told style display\n");
    printf( "-retry\t\tnumber of retries, default is %d\n", DEFAULT_RETRIES);
    printf( "-interval\tinterval between retries, default is %.2lf seconds\n",
	DEFAULT_RETRY_INTERVAL / 1000.0);
    printf( "-raw\t\toutput in raw format using delimiter\n");
    printf( "-f\t\tread hosts from file\n");
    printf( "qstat version %s\n", VERSION);
    exit(0);
}

main( int argc, char *argv[])
{
    int pktlen, rc, diff;
    char pkt[PACKET_LEN], *p;
    fd_set read_fds;
    struct timeval timeout;
    int arg, timeout_milli;
    struct qserver *server;
 
#ifdef _WIN32
    WORD version= MAKEWORD(1,1);
    WSADATA wsa_data;
    if ( WSAStartup(version,&wsa_data) != 0)  {
	fprintf( stderr, "Could not open winsock\n");
	exit(1);
    }
#endif

    if ( argc == 1)
	usage(NULL,argv);

    for ( arg= 1; arg < argc; arg++)  {
	if ( argv[arg][0] != '-')
	    break;
	if ( strcmp( argv[arg], "-f") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -f\n", argv);
	    add_file( argv[arg]);
	}
	else if ( strcmp( argv[arg], "-retry") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -r\n", argv);
	    n_retries= atoi( argv[arg]);
	    if ( n_retries <= 0)  {
		fprintf( stderr, "retries must be greater than zero\n");
		exit(1);
	    }
	}
	else if ( strcmp( argv[arg], "-interval") == 0)  {
	    double value= 0.0;
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -i\n", argv);
	    sscanf( argv[arg], "%lf", &value);
	    if ( value < 0.1)  {
		fprintf( stderr, "retry interval must be greater than 0.1\n");
		exit(1);
	    }
	    retry_interval= (int)(value * 1000);
	}
	else if ( strcmp( argv[arg], "-H") == 0)
	    hostname_lookup= 1;
	else if ( strcmp( argv[arg], "-u") == 0)
	    up_servers_only= 1;
	else if ( strcmp( argv[arg], "-nf") == 0)
	    no_full_servers= 1;
	else if ( strcmp( argv[arg], "-old") == 0)
	    new_style= 0;
	else if ( strcmp( argv[arg], "-P") == 0)
	    get_player_info= 1;
	else if ( strcmp( argv[arg], "-R") == 0)
	    get_server_rules= 1;
	else if ( strcmp( argv[arg], "-raw") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -r\n", argv);
	    raw_delimiter= argv[arg];
	    raw_display= 1;
	}
	else if ( strcmp( argv[arg], "-ncn") == 0)  {
	    color_names= 0;
 	}
	else if ( strcmp( argv[arg], "-cn") == 0)  {
	    color_names= 1;
 	}
	else if ( strcmp( argv[arg], "-tc") == 0)  {
	    time_format= CLOCK_TIME;
	}
	else if ( strcmp( argv[arg], "-tsw") == 0)  {
	    time_format= STOPWATCH_TIME;
	}
	else if ( strcmp( argv[arg], "-ts") == 0)  {
	    time_format= SECONDS;
	}
	else if ( strcmp( argv[arg], "-pa") == 0)  {
	    player_address= 1;
	}
	else if ( strcmp( argv[arg], "-hpn") == 0)  {
	    hex_player_names= 1;
	}
	else  {
	    fprintf( stderr, "unknown flag \"%s\"\n", argv[arg]);
	    usage(NULL,argv);
	}
    }

    if ( color_names == -1)
	color_names= ( raw_display) ? DEFAULT_COLOR_NAMES_RAW :
		DEFAULT_COLOR_NAMES_DISPLAY;

    if ( time_format == -1)
	time_format= ( raw_display) ? DEFAULT_TIME_FMT_RAW :
		DEFAULT_TIME_FMT_DISPLAY;

    while ( arg < argc)  {
	add_qserver( argv[arg]);
	arg++;
    }
    if ( servers == NULL)
	exit(1);

    if ( new_style && ! raw_display)
	display_header();

    qserverinfo.length= htons( qserverinfo.length);
    qplayerinfo.length= htons( qplayerinfo.length);

    bind_sockets();

    while (connected)  {
	FD_ZERO( &read_fds);
	set_fds( &read_fds);

	get_next_timeout( &timeout);
	rc= select( 64, &read_fds, NULL, NULL, &timeout);

	if (rc == 0)  {
	    send_packets();
	    bind_sockets();
	    continue;
	}
	if (rc == SOCKET_ERROR)  {
	    perror("select");
	    return -1;
	}

	gettimeofday( &packet_recv_time, NULL);
	for ( server= servers; server != NULL; server= server->next)  {
	    if ( server->fd == -1)
	        continue;
	    if ( ! FD_ISSET( server->fd, &read_fds))
		continue;
	    if ((pktlen= recv( server->fd, pkt, sizeof(pkt), 0)) ==
			SOCKET_ERROR)  {
		if ( connection_refused())  {
		    server->server_name= DOWN;
		    close(server->fd);
		    server->fd= -1;
		    if ( ! up_servers_only)
			display_server( server);
		    connected--;
		}
		continue;
	    }

	    deal_with_packet( server, pkt, pktlen);
	}
    }

    return 0;
}


void
add_file( char *filename)
{
    FILE *file;
    char name[200];
    if ( strcmp( filename, "-") == 0)
	file= stdin;
    else
	file= fopen( filename, "r");

    if ( file == NULL)  {
	perror( filename);
	return;
    }
    while ( fscanf( file, "%s", name) == 1)
	add_qserver( name);

    if ( file != stdin)
	fclose(file);
}

int
add_qserver( char *arg)
{
    struct sockaddr_in addr;
    struct hostent *ent= NULL;
    struct hostent temp_ent;
    struct hostent *name_ent= NULL;
    struct qserver *server;
    int i, port= DEFAULT_PORT;
    char **a, *colon, *arg_copy;
    char *h_addr_list[MAXIP];
    unsigned long ipaddr;
 
    arg_copy= strdup(arg);
    colon= strchr( arg, ':');
    if ( colon != NULL)  {
	if ( sscanf( colon+1, "%d", &port) != 1)  {
	    fprintf( stderr, "Could not parse port from \"%s\", using %d\n",
		arg, DEFAULT_PORT);
	    port= DEFAULT_PORT;
	}
	*colon= '\0';
    }

    ipaddr= inet_addr(arg);
    if ( ipaddr == INADDR_NONE)
	ent= gethostbyname(arg);

    if ( hostname_lookup && ipaddr != INADDR_NONE)
	name_ent= gethostbyaddr( (char*)&ipaddr, sizeof(ipaddr), AF_INET);

    if ( ent == NULL && ipaddr != INADDR_NONE)  {
	/* Maybe gethostbyname() doesn't parse dotted IP addresses,
	 * try building a hostent by hand.
	 */
	if ( ipaddr == INADDR_NONE)  {
	     fprintf( stderr, "%s: %s\n", arg, strherror(h_errno));
	     return -1;
	}
	ent= &temp_ent;
	ent->h_name= arg;
	ent->h_aliases= NULL;
	ent->h_addrtype= 2;
	ent->h_length= 4;
	h_addr_list[0]= (char *) &ipaddr;
	h_addr_list[1]= NULL;
	ent->h_addr_list= h_addr_list;
    }
    else if ( ent == NULL)  {
        fprintf( stderr, "%s: %s\n", arg, strherror(h_errno));
        return -1;
    }

    server= (struct qserver *) malloc( sizeof( struct qserver));
    server->arg= arg_copy;
    server->host_name= strdup((name_ent)?name_ent->h_name:ent->h_name);
    for ( a= ent->h_addr_list, i= 0; *a != NULL && i < MAXIP; a++, i++)  {
	memcpy( &server->ip[i].s_addr, *a, sizeof(server->ip[i].s_addr));
    }
    server->server_name= NULL;
    server->map_name= NULL;
    server->num_players= 0;
    server->port= port;
    server->fd= -1;
    server->retry1= n_retries;
    server->retry2= n_retries;
    server->n_retries= 0;
    server->ping_total= 0;
    server->n_packets= 0;
    server->n_requests= 0;

    server->next_rule= (get_server_rules) ? "" : NO_SERVER_RULES;
    server->next_player_info= (get_player_info) ? 0 : NO_PLAYER_INFO;

    server->n_player_info= 0;
    server->players= NULL;
    server->n_rules= 0;
    server->rules= NULL;
    server->missing_rules= 0;

    server->next= servers;
    servers= server;
    return 0;
}


/* Functions for binding sockets to Quake servers
 */

int
bind_qserver( struct qserver *server)
{
    struct sockaddr_in addr;

    if ((server->fd= socket( AF_INET, SOCK_DGRAM, 0)) == INVALID_SOCKET) {
        perror( "socket" );
	server->server_name= SYSERROR;
        return -1;
    }

    addr.sin_family = AF_INET;
    addr.sin_port = htons(0);
    addr.sin_addr.s_addr = htonl(INADDR_ANY);
    memset( &(addr.sin_zero), 0, sizeof(addr.sin_zero) );

    if ( bind( server->fd, (struct sockaddr *)&addr,
		sizeof(struct sockaddr)) == SOCKET_ERROR) {
        perror( "bind" );
	server->server_name= SYSERROR;
        return -1;
    }

    addr.sin_family = AF_INET;
    addr.sin_port = htons((short)server->port);
    addr.sin_addr = server->ip[0];
    memset( &(addr.sin_zero), 0, sizeof(addr.sin_zero) );

    if ( connect( server->fd, (struct sockaddr *)&addr, sizeof(addr)) ==
		SOCKET_ERROR)  {
	perror( "connect");
	server->server_name= SYSERROR;
	return -1;
    }
}

void
bind_sockets()
{
    struct qserver *server= servers;
    while ( server != NULL)  {
	if ( server->server_name == NULL && server->fd == -1 &&
		connected < MAXFD)  {
	    bind_qserver( server);
	    send_server_request_packet( server);
	    connected++;
	}
	server= server->next;
    }
}


/* Functions for sending packets
 */

void
send_packets()
{
    struct qserver *server= servers;
    struct timeval now;
    int rc;

    gettimeofday( &now, NULL);
    for ( ; server != NULL; server= server->next)  {
	if ( server->fd == -1)
	    continue;
	if ( server->server_name == NULL)  {
	    if ( server->retry1 != n_retries &&
		    time_delta( &now, &server->packet_time1) <
			(retry_interval*(n_retries-server->retry1+1)))
		continue;
	    if ( ! server->retry1)  {
		cleanup_qserver( server, 0);
		continue;
	    }
	    send_server_request_packet( server);
	    continue;
	}
	if ( server->next_rule != NO_SERVER_RULES)  {
	    if ( server->retry1 != n_retries &&
		    time_delta( &now, &server->packet_time1) <
			(retry_interval*(n_retries-server->retry1+1)))
		continue;
	    if ( ! server->retry1)  {
		server->next_rule= NULL;
		server->missing_rules= 1;
		cleanup_qserver( server, 0);
		continue;
	    }
	    send_rule_request_packet( server);
	}
	if ( server->next_player_info < server->num_players)  {
	    if ( server->retry2 != n_retries &&
		    time_delta( &now, &server->packet_time2) <
			(retry_interval*(n_retries-server->retry2+1)))
		continue;
	    if ( ! server->retry2)  {
		server->next_player_info++;
		if ( server->next_player_info >= server->num_players)  {
		    cleanup_qserver( server, 0);
		    continue;
		}
		server->retry2= n_retries;
	    }
	    send_player_request_packet( server);
	}
    }
}

int
send_server_request_packet( struct qserver *server)
{
    int rc;
    rc= send( server->fd, (const char *)&qserverinfo,
	sizeof(qserverinfo), 0);
    if ( rc == SOCKET_ERROR)
	perror( "send");
    if ( server->retry1 == n_retries)  {
	gettimeofday( &server->packet_time1, NULL);
	server->n_requests++;
    }
    else
	server->n_retries++;
    server->retry1--;
    server->n_packets++;
}

int
send_rule_request_packet( struct qserver *server)
{
    struct qpacket request= {Q_FLAG1,Q_FLAG2,0,Q_CCREQ_RULE_INFO,""};
    int rc, len;

    strcpy( (char*)request.data, server->next_rule);
    len= sizeof(struct qheader) + strlen((char*)request.data) + 1;
    request.length= htons( (short)len);
    rc= send( server->fd, (const char *)&request, len, 0);
    if ( rc == SOCKET_ERROR)
	perror( "send");
    if ( server->retry1 == n_retries)  {
	gettimeofday( &server->packet_time1, NULL);
	server->n_requests++;
    }
    else
	server->n_retries++;
    server->retry1--;
    server->n_packets++;
}

int
send_player_request_packet( struct qserver *server)
{
    int rc;

    qplayerinfo.player_number= server->next_player_info;
    rc= send( server->fd, (const char *)&qplayerinfo, sizeof(qplayerinfo), 0);
    if ( rc == SOCKET_ERROR)
	perror( "send");
    if ( server->retry2 == n_retries)  {
	gettimeofday( &server->packet_time2, NULL);
	server->n_requests++;
    }
    else
	server->n_retries++;
    server->retry2--;
    server->n_packets++;
}


/* Functions for figuring timeouts and when to give up
 */

int
cleanup_qserver( struct qserver *server, int force)
{
    int close_it= force;
    if ( server->server_name == NULL)  {
	server->server_name= TIMEOUT;
	close_it= 1;
    }
    else if ( server->next_rule == NO_SERVER_RULES &&
		server->next_player_info >= server->num_players)
	close_it= 1;

    if ( close_it)  {
	close( server->fd);
	server->fd= -1;
	connected--;
	display_server( server);
    }
}

void
get_next_timeout( struct timeval *timeout)
{
    struct qserver *server= servers;
    struct timeval now;
    int diff1, diff2, diff, smallest= retry_interval;
    gettimeofday( &now, NULL);
    for ( ; server != NULL; server= server->next)  {
	if ( server->fd == -1)
	    continue;
	diff2= 0xffff;
	diff1= 0xffff;
	if ( server->server_name == NULL)
	    diff1= retry_interval*(n_retries-server->retry1+1) -
		time_delta( &now, &server->packet_time1);
	else  {
	    if ( server->next_rule != NULL)
		diff1= retry_interval*(n_retries-server->retry1+1) -
			time_delta( &now, &server->packet_time1);
	    if ( server->next_player_info < server->num_players)
		diff2= retry_interval*(n_retries-server->retry2+1) -
			time_delta( &now, &server->packet_time2);
	}
	diff= (diff1<diff2)?diff1:diff2;
	if ( diff < smallest)
	    smallest= diff;
    }
    if ( smallest < 10)
	smallest= 10;
    timeout->tv_sec= smallest / 1000;
    timeout->tv_usec= (smallest % 1000) * 1000;
}

void
set_fds( fd_set *fds)
{
    struct qserver *server= servers;
    while ( server != NULL)  {
	if ( server->fd != -1)
	    FD_SET( server->fd, fds);
	server= server->next;
    }
}


/* Functions for handling response packets
 */

void
deal_with_packet( struct qserver *server, char *rawpkt, int pktlen)
{
    struct qpacket *pkt= (struct qpacket *)rawpkt;
    int rc;

    if ( ntohs( pkt->length) != pktlen)  {
	fprintf( stderr, "%s Ignoring bogus packet; length %d != %d\n",
		server->arg, ntohs( pkt->length), pktlen);
	cleanup_qserver(server,FORCE);
	return;
    }

    rawpkt[pktlen]= '\0';

    switch ( pkt->op_code)  {
    case Q_CCREP_ACCEPT:
    case Q_CCREP_REJECT:
	return;
    case Q_CCREP_SERVER_INFO:
	server->ping_total+= time_delta( &packet_recv_time,
		&server->packet_time1);
	rc= server_info_packet( server, pkt, pktlen-Q_HEADER_SIZE);
	break;
    case Q_CCREP_PLAYER_INFO:
	server->ping_total+= time_delta( &packet_recv_time,
		&server->packet_time2);
	rc= player_info_packet( server, pkt, pktlen-Q_HEADER_SIZE);
	break;
    case Q_CCREP_RULE_INFO:
	server->ping_total+= time_delta( &packet_recv_time,
		&server->packet_time1);
	rc= rule_info_packet( server, pkt, pktlen-Q_HEADER_SIZE);
	break;
    case Q_CCREQ_CONNECT:
    case Q_CCREQ_SERVER_INFO:
    case Q_CCREQ_PLAYER_INFO:
    case Q_CCREQ_RULE_INFO:
    default:
	return;
    }

    if ( rc == -1)
	fprintf( stderr, "%s error on packet opcode %x\n", server->arg,
		(int)pkt->op_code);

    cleanup_qserver( server, (rc == -1) ? FORCE : 0);
}

int
server_info_packet( struct qserver *server, struct qpacket *pkt, int datalen)
{
    int off= 0;

    /* ignore duplicate packets */
    if ( server->server_name != NULL)
	return 0;

    server->address= strdup((char*)&pkt->data[off]);
    off+= strlen(server->address) + 1;
    if ( off >= datalen)
	return -1;

    server->server_name= strdup((char*)&pkt->data[off]);
    off+= strlen(server->server_name) + 1;
    if ( off >= datalen)
	return -1;

    server->map_name= strdup((char*)&pkt->data[off]);
    off+= strlen(server->map_name) + 1;
    if ( off > datalen)
	return -1;

    server->num_players= pkt->data[off++];
    server->max_players= pkt->data[off++];
    server->protocol_version= pkt->data[off++];

    server->retry1= n_retries;

    if ( get_server_rules)
	send_rule_request_packet( server);
    if ( get_player_info)
	send_player_request_packet( server);

    return 0;
}

int
player_info_packet( struct qserver *server, struct qpacket *pkt, int datalen)
{
    char *name, *address;
    int off, colors, frags, connect_time, player_number;
    struct player *player, *last;

    off= 0;
    player_number= pkt->data[off++];
    name= (char*) &pkt->data[off];
    off+= strlen(name)+1;
    if ( off >= datalen)
	return -1;

    colors= pkt->data[off+3];
    colors= (colors<<8) + pkt->data[off+2];
    colors= (colors<<8) + pkt->data[off+1];
    colors= (colors<<8) + pkt->data[off];
    off+= sizeof(colors);

    frags= pkt->data[off+3];
    frags= (frags<<8) + pkt->data[off+2];
    frags= (frags<<8) + pkt->data[off+1];
    frags= (frags<<8) + pkt->data[off];
    off+= sizeof(frags);

    connect_time= pkt->data[off+3];
    connect_time= (connect_time<<8) + pkt->data[off+2];
    connect_time= (connect_time<<8) + pkt->data[off+1];
    connect_time= (connect_time<<8) + pkt->data[off];
    off+= sizeof(connect_time);

    address= (char*) &pkt->data[off];
    off+= strlen(address)+1;
    if ( off > datalen)
	return -1;

    last= server->players;
    while ( last != NULL && last->next != NULL)  {
	if ( last->number == player_number)
	     return 0;
	last= last->next;
    }
    if ( last != NULL && last->number == player_number)
	return 0;

    player= (struct player *) malloc( sizeof(struct player));
    player->number= player_number;
    player->name= strdup( name);
    player->address= strdup( address);
    player->connect_time= connect_time;
    player->frags= frags;
    player->shirt_color= colors>>4;
    player->pants_color= colors&0xf;
    player->next= NULL;

    if ( last == NULL)
	server->players= player;
    else
	last->next= player;

    server->next_player_info++;
    server->retry2= n_retries;
    if ( server->next_player_info < server->num_players)
	send_player_request_packet( server);

    return 0;
}

int
rule_info_packet( struct qserver *server, struct qpacket *pkt, int datalen)
{
    int off= 0;
    struct rule *rule, *last;
    char *name, *value;

    /* Straggler packet after we've already given up fetching rules */
    if ( server->next_rule == NULL)
	return 0;

    if ( ntohs(pkt->length) == Q_HEADER_SIZE)  {
	server->next_rule= NULL;
	return 0;
    }

    name= (char*)&pkt->data[off];
    off+= strlen( name)+1;
    if ( off >= datalen)
	return -1;

    value= (char*)&pkt->data[off];
    off+= strlen( value)+1;
    if ( off > datalen)
	return -1;

    last= server->rules;
    while ( last != NULL && last->next != NULL)  {
	if ( strcmp( last->name, name) == 0)
	     return 0;
	last= last->next;
    }
    if ( last != NULL && strcmp( last->name, name) == 0)
	return 0;

    rule= (struct rule *) malloc( sizeof( struct rule));
    rule->name= strdup( name);
    rule->value= strdup( value);
    rule->next= NULL;

    if ( last == NULL)
	server->rules= rule;
    else
	last->next= rule;

    server->n_rules++;
    server->next_rule= rule->name;
    server->retry1= n_retries;
    send_rule_request_packet( server);

    return 0;
}


/* Misc utility functions
 */

char *
escape( unsigned char *string)
{
    static char _q[256];

    unsigned char *s= string;
    char *q= _q;

    if ( hex_player_names)  {
	for ( ; *string; string++, q+= 2)
	    sprintf( q, "%02x", *string);
	*q= '\0';
	return _q;
    }

    for ( ; *string; string++)  {
	if ( isprint(*string))  {
	    *q++= *string;
	    continue;
	}

	if ( *string >= 0xa0)
	    *q++= *string & 0x7f;
	else if ( *string >= 0x92 && *string < 0x9c)
	    *q++= '0' + (*string - 0x92);
	else if ( *string >= 0x12 && *string < 0x1c)
	    *q++= '0' + (*string - 0x12);
	else if ( *string == 0x90 || *string == 0x10)
	    *q++= '[';
	else if ( *string == 0x91 || *string == 0x11)
	    *q++= ']';
	else if ( *string == 0xa || *string == 0xc || *string == 0xd)
	    *q++= ']';
    }
    *q= '\0';
    return _q;
}

int
is_default_rule( struct rule *rule)
{
    if ( strcmp( rule->name, "sv_maxspeed") == 0)
	return strcmp( rule->value, Q_DEFAULT_SV_MAXSPEED) == 0;
    if ( strcmp( rule->name, "sv_friction") == 0)
	return strcmp( rule->value, Q_DEFAULT_SV_FRICTION) == 0;
    if ( strcmp( rule->name, "sv_gravity") == 0)
	return strcmp( rule->value, Q_DEFAULT_SV_GRAVITY) == 0;
    if ( strcmp( rule->name, "noexit") == 0)
	return strcmp( rule->value, Q_DEFAULT_NOEXIT) == 0;
    if ( strcmp( rule->name, "teamplay") == 0)
	return strcmp( rule->value, Q_DEFAULT_TEAMPLAY) == 0;
    if ( strcmp( rule->name, "timelimit") == 0)
	return strcmp( rule->value, Q_DEFAULT_TIMELIMIT) == 0;
    if ( strcmp( rule->name, "fraglimit") == 0)
	return strcmp( rule->value, Q_DEFAULT_FRAGLIMIT) == 0;
    return 0;
}

char *
strherror( int h_err)
{
    static char msg[100];
    switch (h_err)  {
    case HOST_NOT_FOUND:	return "host not found";
    case TRY_AGAIN:		return "try again";
    case NO_RECOVERY:		return "no recovery";
    case NO_ADDRESS:		return "no address";
    default:	sprintf( msg, "%d", h_err); return msg;
    }
}

int
time_delta( struct timeval *later, struct timeval *past)
{
    if ( later->tv_usec < past->tv_usec)  {
	later->tv_sec--;
	later->tv_usec+= 1000000;
    }
    return (later->tv_sec - past->tv_sec) * 1000 +
	(later->tv_usec - past->tv_usec) / 1000;
}

int
connection_refused()
{
#ifdef unix
    return errno == ECONNREFUSED;
#endif

#ifdef _WIN32
    return WSAGetLastError() == WSAECONNABORTED;
#endif
}

int
print_packet( char *buf, int buflen)
{
    unsigned int *intbuf= (unsigned int *) buf;
    int i;
    for ( i= 0; i < buflen; i++)  {
	if ( isprint( buf[i])) printf( "%c", buf[i]);
	else printf( " %02x ", (unsigned int)buf[i]);
    }
    puts("");
}

char *
quake_color( int color)
{
    static char *colors[] = {
	"White",	/* 0 */
	"Brown",	/* 1 */
	"Lavender",	/* 2 */
	"Khaki",	/* 3 */
	"Red",		/* 4 */
	"Lt Brown",	/* 5 */
	"Peach",	/* 6 */
	"Lt Peach",	/* 7 */
	"Purple",	/* 8 */
	"Dk Purple",	/* 9 */
	"Tan",		/* 10 */
	"Green",	/* 11 */
	"Yellow",	/* 12 */
	"Blue"		/* 13 */
    };

    if ( color_names)
        return colors[color%14];
    else
	return (char*)color;
}

char *
play_time( int seconds)
{
    static char time_string[24];
    if ( time_format == CLOCK_TIME)  {
	strcpy( time_string, "         ");	/* nine spaces */
	if ( seconds/3600)
	    sprintf( time_string, "%2dh", seconds/3600);
	if ( (seconds%3600)/60 || seconds/3600)
	    sprintf( &time_string[3], "%2dm", (seconds%3600)/60);
	sprintf( &time_string[6], "%2ds", seconds%60);
    }
    else if ( time_format == STOPWATCH_TIME)
	sprintf( time_string, "%02d:%02d:%02d", seconds/3600, (seconds%3600)/60,
		seconds % 60);
    else
	sprintf( time_string, "%d", seconds);

    return time_string;
}

