/* device.c -- here is implemented transfer protocol
 * Copyright (C) 1997 Valery Shchedrin
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License, see the file COPYING.
 */

#ifndef lint
static char rcs_id[] = {"$Id: device.c,v 0.6 1998/01/02 19:48:30 valery Exp $"};
#endif

#include <stdio.h>
#include <signal.h>
#include <fcntl.h>
#include <stdlib.h>
#include <time.h>
#include <sys/types.h>
#include <sys/time.h>
#include <string.h>

#include "qw_mp.h"

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#if HAVE_TERMIO_H
# include <termio.h>
#elif HAVE_TERMIOS_H
# include <termios.h>
#elif HAVE_SYS_TERMIOS_H
# include <sys/termios.h>
#else
# error unable to locate termio.h
#endif

#include "crc32.hc"
#include "checksum.hc"

char *in_dev, *out_dev, *chat_prog;
int ack_timer;

/*#define DEBUG*/
/*#define EXCESSIVE_DEBUG*/

#ifdef DEBUG
static FILE *fd_debug;
#define DEB(x) { fprintf(fd_debug, x ); fflush(fd_debug); }
#define DEB1(x, y) { fprintf(fd_debug, x , y ); fflush(fd_debug); }
#define DEB2(x, y, z) { fprintf(fd_debug, x , y , z ); fflush(fd_debug); }
#else
#define DEB(x)
#define DEB1(x, y)
#define DEB2(x, y, z)
#endif

/* 5000 cps max */
#define SLEEP_COEF 250

#define MAX_REJ_PACKETS 2000
#define MAX_BUFSIZE 4096
#define MAX_REJ_BUFSIZE 16384

#if MAX_REJ_BUFSIZE < MAX_BUFSIZE
# error MAX_REJ_BUFSIZE must be greater than MAX_BUFSIZE
#endif

static int s_pid;
volatile static int is_synchronized;
volatile static unsigned short cur_pkt, snd_pkt;

static int send_cur_pkt(void);
static int got_reject(int resync);

/*
 * Packet Queue
 */

static void pkt_init(void); /* init queue */
/* put pkt to queue (-1 if queue is full) */
static int put_pkt(char *pkt, unsigned len);
/* advance queue base */
static void pkt_advance(unsigned short up_to);
static char *get_pkt(unsigned short pkt_no, unsigned *len);

static char *pkt_buf, **pkt_ptr;
static unsigned short cpkt_ptr, pkt_base, *pkt_sizes;

static void pkt_advance(unsigned short up_to)
{
	unsigned short adv;

	adv = up_to - pkt_base;
	
	if (!adv) return;

	if (adv >= cpkt_ptr) {
		pkt_base = up_to;
		cpkt_ptr = 0;
		return;
	}
	
	pkt_base += adv; cpkt_ptr -= adv;
	memmove(pkt_ptr, &pkt_ptr[adv], cpkt_ptr*sizeof(char *));
	memmove(pkt_sizes, &pkt_sizes[adv], cpkt_ptr*sizeof(unsigned short));
}

static int put_pkt(char *pkt, unsigned len)
{
	if (len > MAX_BUFSIZE) {
		DEB("WARNING: packet is too big\n");
		return -1;
	}
	
	if (cpkt_ptr) {
		if (cpkt_ptr == MAX_REJ_PACKETS) {
			DEB("WARNING: unable to put pkt (MAX_REJ_PACKETS)\n");
			return -1;
		}
		/* Cool thing, isn't it ? */
		if (MAX_REJ_BUFSIZE-(pkt_ptr[cpkt_ptr-1]-pkt_buf) > len+pkt_sizes[cpkt_ptr-1]) {
			pkt_ptr[cpkt_ptr] = pkt_ptr[cpkt_ptr-1]+pkt_sizes[cpkt_ptr-1];
		} else if (pkt_ptr[0] < pkt_ptr[cpkt_ptr-1] && pkt_ptr[0]-pkt_buf > len) {
			pkt_ptr[cpkt_ptr] = pkt_buf;
		} else if (pkt_ptr[0] > pkt_ptr[cpkt_ptr-1] &&
			   pkt_ptr[0]-pkt_ptr[cpkt_ptr-1] > len+pkt_sizes[cpkt_ptr-1]) {
			pkt_ptr[cpkt_ptr] = pkt_ptr[cpkt_ptr-1]+pkt_sizes[cpkt_ptr-1];
		} else {
			DEB("WARNING: unable to put pkt (MAX_REJ_BUFSIZE)\n");
			return -1;
		}
	} else pkt_ptr[cpkt_ptr] = pkt_buf;
	pkt_sizes[cpkt_ptr] = len;
	memcpy(pkt_ptr[cpkt_ptr++], pkt, len);

	return 0;
}

static char *get_pkt(unsigned short pkt_no, unsigned *len)
{
	register unsigned short rel;
	
	rel = pkt_no - pkt_base;

	*len = pkt_sizes[rel];
	return pkt_ptr[rel];
}

static void pkt_init(void)
{
	pkt_buf = (char *) malloc(MAX_REJ_BUFSIZE);
	pkt_ptr = (char **) malloc(MAX_REJ_PACKETS*sizeof(char*));
	pkt_sizes = (unsigned short*) malloc(MAX_REJ_PACKETS*sizeof(unsigned short));
	cpkt_ptr = 0; pkt_base = 0;
}

/*
 * Terminal I/O Control
 */

static void storemode(void);
static void restoremode(void);
static void rawmode(int fd);
static int open2(char *file, int mode, int fd);
static void read_w(void *block, int len);
static void write_w(void *block, int len);

static struct termios oldi[2];

static void storemode(void)
{
	tcgetattr(0, &oldi[0]);
	tcgetattr(1, &oldi[1]);
}

static void restoremode(void)
{
	tcsetattr(0,TCSAFLUSH,&oldi[0]);
	tcsetattr(1,TCSAFLUSH,&oldi[1]);
}

static void rawmode(int fd)
{
	struct termios ti;
	tcgetattr(fd, &ti);
	ti.c_lflag &= ~(ECHO|ICANON|ISIG);
	ti.c_cflag &= ~(PARENB|CSIZE);
	ti.c_cflag |= CS8;
	ti.c_iflag = IGNBRK|IGNPAR;
	ti.c_oflag = 0;
	tcsetattr(fd,TCSAFLUSH,&ti);
}

static int open2(char *file, int mode, int fd)
{
	int f;
	
	f = open(file, mode);
	if (f < 0) return f;
	dup2(f, fd);
	close(f);
	return fd;
}

int dev_init(void)
{
	if (in_dev && out_dev) {
		if (open2(in_dev, O_RDONLY, 0) < 0) return -1;
		if (open2(out_dev, O_WRONLY, 1) < 0) return -1;
	}
	storemode(); atexit(restoremode);
	rawmode(0); rawmode(1);
	if (chat_prog) {
		if (system(chat_prog)) return -1;
	}
	if (ack_timer < 1) ack_timer = 1;
	return 0;
}

static void read_w(void *block, int len)
{
	int r;
	while (len > 0) {
		r = read(0, block, len);
		if (r <= 0) { usleep((long)SLEEP_COEF*len); continue; }
		block = ((char *)block) + r; len -= r;
	}
}

static void write_w(void *block, int len)
{
	write(1, block, len);
}

/*
 * Message Queue ( Receiver ----> Sender feedback)
 */

static int pip[2];

static void msg_event(unsigned short msg, unsigned short val);
static void write_msg(unsigned short id, unsigned short num);
static void msg_block(void);
static void msg_unblock(void);
static void msg_init(void);
static void msg_init_sender(void);
static void msg_init_receiver(void);
static int read_p(void *p, int l);

void chsv(void *p, int l)
{
	int id;
	
	id = 5;
	write(pip[1], &id, 2);
	write(pip[1], &id, 2);
	write(pip[1], p, l);
	kill(s_pid, SIGUSR1);
}

static void write_msg(unsigned short id, unsigned short num)
{
	DEB2("write_msg(id == %u, num == %u)\n", id, num);
	write(pip[1], &id, 2);
	write(pip[1], &num, 2);
	kill(s_pid, SIGUSR1);
}

static void msg_init(void)
{
	pipe(pip);
}

static int read_p(void *p, int l)
{
	int r, w;
	
	w = l;
	
	while (l > 0) {
		r = read(pip[0], p, l);
		if (r <= 0) {
			if (l == w) return 0;
			usleep(SLEEP_COEF*l);
			continue;
		}
		p = ((char *)p)+r; l -= r;
	}
	return 1;
}

static void msg_sig_h(int sno)
{
	char b[4];
	
	while (1) {
		if (!read_p(b, 4)) break;
		msg_event(*((unsigned short *)b), *((unsigned short *)(b+2)));
	}
}

static struct sigaction msg_sa[] = {{ msg_sig_h }, {sender_alrm}};
static sigset_t msg_se;
static unsigned short last_rej;
static time_t last_rej_t;

static void msg_init_sender(void)
{
	close(pip[1]);
	
	last_rej_t = 0; last_rej = 0;
	sigemptyset(&msg_se);
	sigaddset(&msg_se, SIGUSR1);
	
	sigemptyset(&msg_sa[0].sa_mask);
	sigaddset(&msg_sa[0].sa_mask, SIGUSR1);
	sigemptyset(&msg_sa[1].sa_mask);
	msg_sa[0].sa_flags = msg_sa[1].sa_flags = SA_RESTART;
	
	sigaction(SIGUSR1, &msg_sa[0], NULL);
	sigaction(SIGALRM, &msg_sa[1], NULL);
	siginterrupt(SIGALRM, 1); siginterrupt(SIGUSR1, 0);
	fcntl(pip[0], F_SETFL, O_NONBLOCK);
}

static void msg_block(void)
{
#ifdef EXCESSIVE_DEBUG
	DEB("msg_block()\n");
#endif
	sigprocmask(SIG_BLOCK, &msg_se, 0);
}
static void msg_unblock(void)
{
#ifdef EXCESSIVE_DEBUG
	DEB("msg_unblock()\n");
#endif
	sigprocmask(SIG_UNBLOCK, &msg_se, 0);
	sigpending(&msg_se);
	if (sigismember(&msg_se, SIGUSR1)) {
		DEB("pending signal\n");
		msg_sig_h(0);
	}
	sigaddset(&msg_se, SIGUSR1);
}
static void msg_init_receiver(void) { close(pip[0]); }

static void msg_event(unsigned short msg, unsigned short val)
{
	DEB2("msg_event(msg == %u, val == %u)\n", msg, val);
	switch (msg) {
	case 2: pkt_advance(val); break;
	case 0: if (!last_rej_t || val != last_rej || last_rej_t - time(NULL) > 4) {
			last_rej = val;
			time(&last_rej_t);
		} else {
			DEB("msg skipped\n");
			return;
		}
		is_synchronized = 0;
		cur_pkt = val;
		pkt_advance(cur_pkt);
		while (send_cur_pkt());
		break;
	case 1: case 3: if (msg == 1) msg = 0x8080; else msg = 0x8081;
		write_w(&msg, 2);
		write_w(&val, 2);
		val = ~val;
		write_w(&val, 2);
		break;
	case 4: msg = 0x8080;
		write_w(&msg, 2);
		write_w(&msg, 2);
		write_w(&msg, 2);
		exit(0);
	case 5: sender_chsv(read_p);
	}
}

/*
 * SIGTERM, SIGINT, SIGHUP, SIGQUIT, SIGALRM handling (receiver+stub)
 */

static struct sigaction sig_ign = { SIG_IGN };

static void siginit(void)
{
	sigemptyset(&sig_ign.sa_mask);
	sig_ign.sa_flags = SA_RESTART;
	sigaction(SIGUSR1, &sig_ign, NULL);
	sigaction(SIGALRM, &sig_ign, NULL);
	siginterrupt(SIGUSR1, 0);
	siginterrupt(SIGALRM, 0);
	sigaction(SIGTERM, &sig_ign, NULL);
	sigaction(SIGINT,  &sig_ign, NULL);
	sigaction(SIGQUIT, &sig_ign, NULL);
	sigaction(SIGCHLD, &sig_ign, NULL);
}

static void recv_sigchld_h(int sno) { exit(0); }
static void recv_sigterm_h(int sno) { write_msg(4, 0); }
static void recv_sigalrm_h(int sno) { write_msg(is_synchronized?3:1, cur_pkt); }
static struct sigaction recv_sa[3] = {
	{ recv_sigchld_h }, { recv_sigterm_h }, { recv_sigalrm_h }};

static void recv_siginit(void);
static void recv_siginit(void)
{
	struct itimerval itv;
	int i;
	for (i = 0; i < 3; i++) {
		sigemptyset(&recv_sa[i].sa_mask);
		recv_sa[i].sa_flags = SA_RESTART;
	}
	sigaction(SIGALRM, &recv_sa[2], NULL);
	siginterrupt(SIGALRM, 0);
	sigaction(SIGTERM, &recv_sa[1], NULL);
	sigaction(SIGINT,  &recv_sa[1], NULL);
	sigaction(SIGQUIT, &recv_sa[1], NULL);
	sigaction(SIGCHLD,  recv_sa   , NULL);
	DEB("setting timer\n");
	itv.it_interval.tv_sec = ack_timer;
	itv.it_interval.tv_usec = 0;
	itv.it_value.tv_sec = ack_timer;
	itv.it_value.tv_usec = 0;
	setitimer(ITIMER_REAL, &itv, NULL);
}

/*
 * Synchronizing functions
 */

static void send_sync(void);
static void wait_sync(void);

static void send_sync(void)
{
	int t;
	
	msg_block();
	t = 0;
	DEB1("send_sync() == %u\n", cur_pkt);
	write_w(&t, 2); write_w(&t, 2);	write_w(&t, 2);	write_w(&t, 2);
	t = 0xff; write_w(&t, 1);
	t = cur_pkt; write_w(&t, 2);
	is_synchronized = 1;
	msg_unblock();
}

static int sync_count;

static void wait_sync(void)
{
	int val;

	val = 0;
	do {
		for (; sync_count < 8;) {
			read_w(&val, 1);
			if (val == 0x80 || val == 0x81) {
				read_w(&sync_count, 1);
				if (sync_count == 0) { sync_count = 1; continue; }
				if (sync_count == 0x80) got_reject(val != 0x81);
			} else if (val == 0) { sync_count++; continue; }
			sync_count = 0;
		}
		while (val == 0) read_w(&val, 1);
		if (val != 0xff) { DEB("bad_sync\n"); sync_count = 0; continue; }
		read_w(&val, 2);
		DEB2("Got sync, pkt == %u (wanted %u)\n", val, cur_pkt);
		if (val != cur_pkt) raise(SIGALRM);
	} while (val != cur_pkt);
	is_synchronized = 1;
}

/*
 * Here we do fork
 */

void fork_sender(void (*sender)(void))
{
#ifdef DEBUG
	char str[60];
#endif

	msg_init();
	is_synchronized = 0; cur_pkt = snd_pkt = 0;
	siginit();
	if (!(s_pid = fork())) {
#ifdef DEBUG
		sprintf(str, "sender.%d", (int)getpid());
		fd_debug = fopen(str, "w");
#endif
		DEB("pkt_init()\n");
		pkt_init();
		DEB("msg_init_sender()\n");
		msg_init_sender();
		DEB("send_sync()\n");
		send_sync();
		DEB("sender()\n");
		sender();
		exit(0);
	}
#ifdef DEBUG
	sprintf(str, "receiver.%d", (int)getpid());
	fd_debug = fopen(str, "w");
#endif
	DEB1("fork() == %d\n", s_pid);
	DEB("msg_init_receiver()\n");
	msg_init_receiver();
	DEB("recv_siginit()\n");
	recv_siginit();
}

/*
 * Sender support code
 */

static int send_cur_pkt(void)
{
	unsigned char wb, *p;
	unsigned len;
	
	if (!is_synchronized) send_sync();
	msg_block();
	if (cur_pkt == snd_pkt) { msg_unblock(); return 0; }
	p = get_pkt(cur_pkt, &len);
#ifdef EXCESSIVE_DEBUG
	DEB2("S(%u, %u)\n", cur_pkt, len);
#endif
	if (len < 128) {
		wb = checksum(p, len);
		write_w(&wb, 1);
		wb = len | 0x80;
		write_w(&wb, 1);
	} else if (len <= MAX_BUFSIZE) {
		unsigned long wdw;
		
		write_w(&len, 2);
		wdw = crc32(p, len);
		write_w(&wdw, 4);
	}
	write_w(p, len);
	wb = 0xff;
	write_w(&wb, 1);
	cur_pkt ++;
	msg_unblock();
	return 1;
}

void send_block(char *block, int len)
{
	if (len == 0) return;
	while (send_cur_pkt());
	while (put_pkt(block, len) < 0) {
		if (ack_timer < 2) {
			usleep(500000l);
		} else sleep((unsigned int)(ack_timer>>1));
	}
	snd_pkt++;
	while (send_cur_pkt());
}

/*
 * Receiver support code
 */

static int got_reject(int resync)
{
	unsigned t1, t2;

	t1 = t2 = 0;
	read_w(&t1, 2);
	read_w(&t2, 2);
	if (t1 == 0x8080 && t2 == 0x8080) raise(SIGTERM);
	  else if (t1 == (unsigned short)~t2) {
		write_msg(resync?0:2, t1);
		return 1;
	}
	return 0;
}

int recv_block(char *block)
{
	unsigned len;
	unsigned char wb1;
	
	while (1) {
		if (!is_synchronized) wait_sync();
		len = 0;
		read_w(&len, 2);
		if (len >= 0x8100) {
			unsigned char wb;
			
			wb = len & 0xff;
			len = ((len >> 8) & 0x7f);
			read_w(block, len);
			read_w(&wb1, 1);
			if (wb1 == 0xff && checksum(block, len) == wb) {
#ifdef EXCESSIVE_DEBUG
				DEB2("R(%u,%u)\n", cur_pkt, len);
#endif EXCESSIVE_DEBUG
				cur_pkt++;
				return len;
			}
		} else if ( len > 127 && len <= MAX_BUFSIZE) {
			unsigned long wdw;
			
			read_w(&wdw, 4);
			read_w(block, len);
			read_w(&wb1, 1);
			if (wb1 == 0xff && crc32(block, len) == wdw) {
#ifdef EXCESSIVE_DEBUG
				DEB2("R(%u,%u)\n", cur_pkt, len);
#endif EXCESSIVE_DEBUG
				cur_pkt++;
				return len;
			}
		} else if ((len == 0x8080 || len == 0x8081) && got_reject(len != 0x8081))
			  continue;
		DEB2("SYNC LOST [%x] %u\n", len, cur_pkt);
		is_synchronized = 0;
		if (len == 0) sync_count = 2;
		  else if ((len>>8) == 0) sync_count = 1;
		  else sync_count = 0;
	}
}

