/*

MINDEX.CPP

Mip indexing functions and classes for Mipindex

by Ed Kiser (edkiser@jaxnet.com) Copyright (c) 1996

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

History:
Version 1.1
Tue Sept 30 1996 :
 * Created by moving a bunch of stuff out of mipindex.cpp

Wed Oct 9 1996:
 * Modified functions several times to use directory classes.

Thu Oct 10 1996:
 * Decided to move history of all files to MIPINDEX.CPP.

*/

#ifndef __MINDEX_H
#include "mindex.h"
#endif

/***** The filelist *****/

const int filelist::increment = 0x10;

filelist::filelist(): files(0), storage(increment)
{ fullpath = new string[increment];
};

bool filelist::add(char * name, int & index_out)
{ char fullname[MAXPATH];
  get_full_name(name,fullname);
  for (int j=0; j<files; j++) if (strcmp(fullpath[j],fullname)==0)
  { index_out=j;
    return false;
  }
  if (files==storage)
  { string * i=new string[storage+increment];
    for (int j=0; j<files; j++) i[j]=fullpath[j];
    delete[] fullpath;
    fullpath=i;
    storage+=increment;
  }
  fullpath[files]=new char[strlen(fullname)+1];
  strcpy(fullpath[files],fullname);
  index_out=files;
  files++;
  return true;
};

void filelist::drop()
{ for (int j=0; j<files; j++) delete[] fullpath[j];
  delete[] fullpath;
  fullpath=0;
  storage=0;
  files=0;
};

void filelist::becopy(filelist const & f)
{ fullpath=new string[f.storage];
  storage=f.storage; files=f.files;
  for (int j=0; j<files; j++)
  { fullpath[j]=new char[strlen(f.fullpath[j])+1];
    strcpy(fullpath[j],f.fullpath[j]);
  }
};

/***** IDX information *****/

void miptex_index::read_idx(istream & i, ostream & cout)
{ /* read file list and construct a mapping between file numbers in the
     input file and file numbers in self */
  int num_miptextures = 0;
  int * file_mapping = 0;
  { int mIDX;
    i.read((char*)&mIDX,sizeof(int));
    if (mIDX!=0x5844496d)
    { cout << "IDX file does not begin with \"mIDX\".\n";
      return;
    }
    int num_files;
    i.read((char*)&num_files,sizeof(int));
    cout << howmany("Opening IDX file which indexes % file(s)",num_files);
    file_mapping=new int[num_files];
    for (int j=0; j<num_files; j++)
    { char buf[260];
      char * p=buf;
      do
      { i.read(p,1);
        p++;
      } while (*(p-1)!=0);
      int index;
      flist.add(buf,index);
      file_mapping[j]=index;
    }
  }
  i.read((char*)&num_miptextures,sizeof(int));
  cout << howmany(" and % miptexture(s).\n",num_miptextures);
  /* read all miptextures and locations, using the mapping to convert
     the file numbers in the locations */
  vector<location> where_to_visit;
  where_to_visit.reserve(num_miptextures);
  location palette_at(-1,0);
  for (int j=0; j<num_miptextures; j++)
  { miptex_id m;
    location l;
    i.read(&m,sizeof(miptex_id));
    i.read(&l,sizeof(location));
    l.file=file_mapping[l.file];
    bool succeeded=holder.insert(texlist::value_type(m,l)).second;
    if (succeeded && mv!=0) where_to_visit.push_back(l);
    if (succeeded && m==miptex_id::palette_id) palette_at=l;
  }
  delete[] file_mapping;
  /* try to visit all locations in index */
  if (mv!=0)
  { sort(where_to_visit.begin(),where_to_visit.end());
    int curfile=-1;
    ifstream input;
    for (vector<location>::const_iterator i=where_to_visit.begin();
         i!=where_to_visit.end(); i++)
    { location l=*i;
      if (l.file!=curfile)
      { if (input) input.close();
        curfile=l.file;
        cout << "\nOpening indexed file: " << flist[curfile];
        input.open(flist[curfile],ios::in | ios::binary, filebuf::openprot);
        input.seekg(0,ios::end);
        cout << " (" << input.tellg() << " bytes).\n";
      }
      if (input.fail()) throw "Error in index -- bad location";
      cout << "Visiting " << l.offset << ": ";
      input.seekg(l.offset);
      miptex mt(input,l==palette_at);
      miptex_id mtid=mt;
      cout << "Found " << mtid << "\n";
      if (!mt.is_valid()) throw "Error in index -- bad miptexture read";
      (*mv)(mt);
    }
  }
};

void miptex_index::write_idx(ostream & o, ostream &)
{ zap_unused_files();
  int mIDX=0x5844496d;
  o.write(&mIDX,sizeof(int));
  /* write file list */
  int numfiles=flist.count();
  o.write(&numfiles,sizeof(int));
  for (int i=0; i<numfiles; i++)
  { const char * b=flist[i];
    do
    { o.write(b,1);
      b++;
    } while (*(b-1)!=0);
  }
  /* write all miptextures and locations */
  int num_miptextures=holder.size();
  o.write(&num_miptextures,sizeof(int));
  for (texlist::const_iterator p=holder.begin(); p!=holder.end(); p++)
  { const pair< const miptex_id, location> & v=*p;
    miptex_id x=v.first;
    x.clean();
    write_image(x,o);
    write_image(v.second,o);
  }
};

/***** BSP information *****/

void miptex_index::read_bsp(istream & i, int f, ostream & cout)
/* asserts that i's file pointer is at the beginning of the BSP file. */
{ long items=holder.size();
  long start=i.tellg();
  bsp_header head(i);
  if (!head.is_valid())
  { cout << "Bad BSP file header (possibly not version 29).\n";
    return;
  }
  i.seekg(start+head.entry[bsp_header::miptextures].offset);
  assert(!i.fail());
  long numtextures; read_image(numtextures,i);
  cout << howmany("Opening BSP file with % miptexture(s).\n",numtextures);
  long * buffer=new long[numtextures];
  i.read((char*)buffer,numtextures*sizeof(long));
  if (i.fail()) throw "Failed to read BSP miptexture directory.\n";
  bsp_miptex_dir array(buffer,buffer+numtextures);
  delete[] buffer;
  for (bsp_miptex_dir::iterator q=array.begin(); q!=array.end(); q++)
  { if ((*q)==-1) continue;
    location l(f,start+head.entry[bsp_header::miptextures].offset+(*q));
    i.seekg(l.offset);
    if (i.fail()) throw "Bad offset in BSP file.";
    miptex mt(i);
    if (i.fail()) throw "Could not read miptexture.";
    if (mt.is_valid())
    { bool succeeded=
       holder.insert(texlist::value_type((miptex_id)mt,l)).second;
      if ((mv!=0) && succeeded)
      { (*mv)(mt);
      }
    } else
    { cout << "Invalid miptexture " << (miptex_id)mt
           << " at 0x" << hex << l.offset
           << " (base+0x" << *q << ")\n" << dec;
    }
  }
  cout << howmany("Found % new miptexture(s).\n",holder.size()-items);
};

/***** PAK information *****/

void miptex_index::read_pak(istream & i, int f, ostream & cout)
/* We should honor the PAK. After all, the PAK built the Ringworld. */
{ long start=i.tellg();
  pak_header head(i);
  if (!head.is_valid())
  { cout << "Invalid PAK file header.\n";
    return;
  }
  int direntries=head.dirsizebytes/64;
  cout << howmany("Opening PAK file with % file(s).\n",direntries);
  i.seekg(start+head.diroffset);
  pak_direntry * buffer=new pak_direntry[direntries];
  i.read((char*)buffer,direntries*sizeof(pak_direntry));
  if (i.fail()) throw "Failed to read PAK directory.\n";
  pak_dir array(buffer,buffer+direntries);
  delete[] buffer;
  for (unsigned item=0; item<array.size(); item++)
  { if (has_extension(array[item].filename,".bsp"))
    { cout << "Found BSP file " << array[item].filename
           << " (" << array[item].size
           << " bytes) at " << start+array[item].offset
           << ".\n";
      i.seekg(start+array[item].offset);
      assert(!i.fail());

      read_bsp(i,f,cout);
    } else if (has_extension(array[item].filename,".pak"))
    { cout << "Found rare nested PAK file " << array[item].filename
           << " (" << array[item].size
           << " bytes) at " << start+array[item].offset
           << ".\n";
      i.seekg(start+array[item].offset);
      assert(!i.fail());
      read_pak(i,f,cout);
    } else if (has_extension(array[item].filename,".wad"))
    { cout << "Found WAD2 file " << array[item].filename
           << " (" << array[item].size
           << " bytes) at " << start+array[item].offset
           << ".\n";
      i.seekg(start+array[item].offset);
      assert(!i.fail());
      read_wad2(i,f,cout);
    } else if (ends_with(array[item].filename,"palette.lmp") &&
               array[item].size==768)
    { cout << "Found " << array[item].filename
           << " (768 bytes) at " << start+array[item].offset
           << ".\n";
      location l(f,start+array[item].offset);
      i.seekg(l.offset);
      assert(!i.fail());
      miptex pal(i,true);
      if (pal.is_valid())
      { bool succeeded=
          holder.insert(texlist::value_type((miptex_id)pal,l)).second;
        if (succeeded)
        { cout << "Found 1 new palette.\n";
          if (mv)
          { (*mv)(pal);
          }
        } else cout << "Found 0 new palettes.\n";
      }
    };
/*
    else
    { cout << "Found other file " << array[item].filename
           << " (" << array[item].size
           << " bytes) at " << start+array[item].offset
           << ".\n";
    }
*/
  }
};

/***** WAD2 information *****/

void miptex_index::read_wad2(istream & i, int file, ostream & cout)
{ long start=i.tellg();
  long items=holder.size();
  wad2_header head(i);
  if (!head.is_valid())
  { cout << "Invalid WAD2 file header.\n";
    return;
  }
  cout << howmany("Opening WAD2 file with % lump(s).\n",head.numentries);
  i.seekg(start+head.diroffset);
  wad2_direntry * buffer=new wad2_direntry[head.numentries];
  i.read((char*)buffer,head.numentries*sizeof(wad2_direntry));
  wad2_dir array(buffer,buffer+head.numentries);
  delete[] buffer;
  assert(!i.fail());
  long type_miptexture=wad2_direntry::build_type(wad2_direntry::miptexture);
  long type_palette=wad2_direntry::build_type(wad2_direntry::palette);
  for (unsigned item=0; item<array.size(); item++)
  { if (array[item].get_type() == type_miptexture ||
        array[item].get_type() == type_palette)
    { location l(file,start+array[item].offset);
      i.seekg(l.offset);
      if (i.fail()) throw "Seek failed: bad offset in WAD2 file?";
      miptex mt(i,array[item].get_type() == type_palette);
      if (mt.is_valid())
      { bool succeeded=
         holder.insert(texlist::value_type((miptex_id)mt,l)).second;
        if ((mv!=0) && succeeded)
        { (*mv)(mt);
        }
      }
    }
  }
  cout << howmany("Found % new miptexture(s).\n",holder.size()-items);
};

/***** WAD2 writer *****/

const int wad2_writer::dirbuffersize = 0x20000;   // 1/8 MB

const int wad2_writer::filebuffersize = 0x200000; //  2  MB

wad2_writer::wad2_writer(char * name)
{ dirbuffer = new char[dirbuffersize];
  filebuffer = new char[filebuffersize];
  strcpy(filebuffer,name); // good a place to put it as any
  ofstream test(filebuffer,ios::out | ios::binary | ios::noreplace);
  if (test.fail()) throw "WAD2 output file already exists.";
  dirstream = new ostrstream(dirbuffer,dirbuffersize);
  is_written=false;
};

void wad2_writer::data_setup()
{ if (!is_written)
  { data= new ofstream(filebuffer,ios::out | ios::binary);
    data->rdbuf()->setbuf(filebuffer,filebuffersize);
    write_image(wad2_header(),*data);
    is_written=true;
  }
};

wad2_writer::~wad2_writer()
{ if (is_written)
  { int diroffset=data->tellp();
    int dirbytes=dirstream->pcount();
    data->write(dirbuffer,dirbytes);
    dirbytes/=sizeof(wad2_direntry);
    data->seekp(0);
    write_image(wad2_header(dirbytes,diroffset),*data);
    delete data;
  }
  delete dirstream;
  delete dirbuffer;
  delete filebuffer;
};

void wad2_writer::operator()(miptex m)
{ data_setup();
  int pbegin=data->tellp();
  if (m.is_valid()) {
    m.write_bytes(*data);
    int pend=data->tellp();
    wad2_direntry t(((miptex_id)m).name,pend-pbegin,pbegin);
    if (m.is_palette()) t.set_type(wad2_direntry::palette);
    write_image(t,*dirstream);
  }
};

/***** Printing as Text *****/

void miptex_index::write_text(ostream & cout)
{ zap_unused_files();
  if (flist.count()==0) cout << "No files indexed.\n";
  else
  { cout << howmany("Indexed % file(s):\n",flist.count());
    for (int i=0; i<flist.count(); i++)
    cout << "#" << i+1 << ": " << flist[i] << "\n";
  }
  if (holder.empty()) cout << "\nNo miptextures found.\n";
  else
  { cout << howmany("\nFound % miptexture(s) in all:\n\n",holder.size());
    for (texlist::const_iterator c=holder.begin(); c!=holder.end(); c++)
    { texlist::value_type const & p=*c;
      cout << p.first << " at "
           << p.second.offset << " in file #" << p.second.file+1 << "\n";

    }
  }
};

/***** Reading Files *****/

void miptex_index::read_file(char * name, ostream & comment)
{ ifstream i(name,ios::in | ios:: nocreate | ios::binary);
  if (i.fail()) throw "Could not open";
  int fileno;
  if (!flist.add(name,fileno))
  { comment << flist[fileno] << " already visited.\n"; return;
  }
  comment << "Analyzing file " << flist[fileno] << " :\n";
  if (has_extension(flist[fileno],".bsp")) read_bsp(i,fileno,comment);
  else if (has_extension(flist[fileno],".pak")) read_pak(i,fileno,comment);
  else if (has_extension(flist[fileno],".wad")) read_wad2(i,fileno,comment);
// else if (has_extension(flist[fileno],".dst")) read_dst(i,fileno,comment);
  else if (has_extension(flist[fileno],".idx"))
  { flist.scratch();
    read_idx(i,comment);
  }
  else if (ends_with(flist[fileno],"palette.lmp"))
  { i.seekg(0,ios::end);
    if (i.tellg()!=768) throw "Palette file is not 768 bytes";
    i.seekg(0);
    miptex pal(i,true);
    location l(fileno,0);
    bool succeeded=
     holder.insert(texlist::value_type((miptex_id)pal,l)).second;
    if (succeeded) cout << "Added palette.\n\n";
    else cout << "Palette already in index.\n\n";
    if ((mv!=0) && succeeded)
    { (*mv)(pal);
    }
  }
  else
  { comment << "Can't identify file type for " << name;
    flist.scratch();
  }
  cout << "\n";
};

/***** Zap unused files *****/

void miptex_index::zap_unused_files()
{ int ct=flist.count();
  int files_used=0;
  bool * is_used=new bool[ct];
  for (int i=0; i<ct; i++) is_used[i]=false;

  for (texlist::const_iterator c=holder.begin(); c!=holder.end(); c++)
  { texlist::value_type const & p=*c;
    int q=p.second.file;
    if (is_used[q]==false)
    { is_used[q]=true;
      files_used++;
    }
  };

  if (files_used!=ct)
  { int * mapping=new int[ct];
    { int j=0, removed=0;
      //cout << "ct=" << ct << " files_used=" << files_used << "\n";
      for (int i=0; i<ct; i++)
      { //cout << "i=" << i << " j=" << j << "\n";
        mapping[i]=j;
        if (is_used[i]) j++;
        else
        { flist.scratch(i-removed);
          removed++;
        }
      }
    }
    for (texlist::iterator c=holder.begin(); c!=holder.end(); c++)
    { texlist::value_type & p=*c;
      p.second.file=mapping[p.second.file];
    }
    delete[] mapping;
  }
  delete[] is_used;
};

/***** has / get *****/

bool miptex_index::has(miptex_id const & id) const
{ texlist::const_iterator i=holder.find(id);
  return (i!=holder.end());
};

bool miptex_index::get(miptex_id const & id, miptex & out) const
{ texlist::const_iterator i=holder.find(id);
  if (i==holder.end()) return false;
  location l=(*i).second;
  ifstream reader(flist[l.file],ios::in | ios::nocreate | ios::binary);
  reader.seekg(l.offset);
  bool success=false;
  try
  { miptex mt(reader);
    success=mt.is_valid();
    if (success) out=mt;
  }
  catch (char * i)
  { ++i; /* eat it */
  }
  return success;
};
