/*
 * Copyright (C) 1996 by Raphael Quinet.  All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation.  If more than a few
 * lines of this code are used in a program which displays a copyright
 * notice or credit notice, the following acknowledgment must also be
 * displayed on the same screen: "This product includes software
 * developed by Raphael Quinet for use in the Quake Editing Utilities
 * project."  THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR
 * IMPLIED WARRANTY.
 *
 * More information about the QEU project can be found on the WWW:
 * "http://www.montefiore.ulg.ac.be/~quinet/games/editing.html" or by
 * mail: Raphael Quinet, 9 rue des Martyrs, B-4550 Nandrin, Belgium.
 */

/*
 * F_PACK.C - Read and write Quake PACK files.
 */

#include "qeu.h"
#include "q_misc.h"
#include "q_files.h"
#include "f_pack.h"

/*
 * Read the PACK directory into memory.  The optional offset to the
 * start of the PACK file is given in "offset".  The number of files in
 * the directory is returned in *dirsize_r.
 */
PACKDirPtr ReadPACKDirectory(FILE *packfile, UInt32 offset, UInt16 *dirsize_r)
{
  PACKDirPtr dir;
  UInt32     pos, size;
  UInt16     max, i;

  *dirsize_r = 0;
  if (packfile == NULL)
    return NULL;
  if ((fseek(packfile, offset, SEEK_SET) < 0)
      || (ReadMagic(packfile) != FTYPE_PACK)
      || (ReadInt32(packfile, &pos) == FALSE)
      || (ReadInt32(packfile, &size) == FALSE)
      || (size == 0L)
      || (size / sizeof(struct PACKDirectory) > 65535L)
      || (fseek(packfile, offset + pos, SEEK_SET) < 0))
    return NULL;
  dir = (PACKDirPtr)QMalloc(size);
  max = (UInt16)(size / sizeof(struct PACKDirectory));
  for (i = 0; i < max; i++)
    {
      if (ReadBytes(packfile, &dir[i], sizeof(struct PACKDirectory)) == FALSE)
	{
	  QFree(dir);
	  return NULL;
	}
      ConvertFilePath(dir[i].name);
      dir[i].offset = SwapInt32(dir[i].offset);
      dir[i].size = SwapInt32(dir[i].size);
    }
  *dirsize_r = max;
  return dir;
}


/*
 * Return the index number of the first entry matching "entryname".
 * If "entryname" contains no directory separators ("/" or "\"), then
 * try also to match it with the base name of all entries.  A number
 * greater or equal to dirsize is returned if no match is found.
 */
UInt16 FindPACKEntry(PACKDirPtr dir, UInt16 dirsize, char *entryname)
{
  UInt16 i;
  char   *p;

  if (dir == NULL)
    ProgError("BUG: Cannot find entry in NULL directory");
  /* first, look for the full name */
  for (i = 0; i < dirsize; i++)
    if (!strnicmp(dir[i].name, entryname, 56))
      return i;
  /* then look for the base name (if entryname contains no "/" or "\") */
  for (p = entryname; *p; p++)
    if (*p == '/' || *p == '\\')
      return i;
  for (i = 0; i < dirsize; i++)
    if (!strnicmp(GetBaseName(dir[i].name), entryname, 56))
      return i;
  return i;
}


/*
 * Print the contents of the PACK directory in "outf".
 */
void DumpPACKDirectory(FILE *outf, PACKDirPtr dir, UInt16 dirsize)
{
  UInt16 i;
  UInt32 sum;
  char   buf[57];

  if (outf == NULL || dir == NULL || dirsize == 0)
    return;
  fprintf(outf, "num    offset     size    file name\n");
  fprintf(outf, "       (hex)      (dec)\n");
  sum = 0L;
  for (i = 0; i < dirsize; i++)
    {
      strncpy(buf, dir[i].name, 56);
      buf[56] = '\0';
      fprintf(outf, "%3u  0x%08lx  %6ld   %s\n",
	      i, dir[i].offset, dir[i].size, buf);
      sum += dir[i].size;
    }
  fprintf(outf, "\nTotal size for %3u entries:  %7lu bytes.\n", dirsize, sum);
  fprintf(outf, "Size of the PACK directory:  %7lu bytes.\n",
	  (UInt32)dirsize * (UInt32)sizeof(struct PACKDirectory));
  fprintf(outf, "Total (header + data + dir): %7lu bytes.\n",
	  12L + sum + (UInt32)dirsize * (UInt32)sizeof(struct PACKDirectory));
}


/*
 * If "entrynum" is smaller than dirsize, extract the corresponding
 * entry from a PACK file.  Otherwise, extract all entries and save
 * them in separate files.  The files will be saved in the directory
 * "prefixpath".  If "outf" is not null, progress information will be
 * printed in it.
 */
Bool UnPACKFile(FILE *outf, FILE *packfile, UInt32 offset, PACKDirPtr dir,
		UInt16 dirsize, UInt16 entrynum, char *prefixpath)
{
  char   *newname;
  char   *p;
  FILE   *newfile;
  FILE   *indexfile;
  UInt16  i;
  UInt32  sum;

  if (packfile == NULL || dir == NULL || dirsize == 0)
    return FALSE;
  if (prefixpath == NULL)
    prefixpath = ".";
  newname = (char *)QMalloc(strlen(prefixpath) + 56 + 2);
  strcpy(newname, prefixpath);
  p = &newname[strlen(newname) - 1];
#ifdef QEU_DOS
  if (*p != '\\')
    {
      p++;
      *p = '\\';
    }
#else
  if (*p != '/')
    {
      p++;
      *p = '/';
    }
#endif
  p++;
  strcpy(p, QEU_INDEX_FILE);
  if (outf != NULL)
    fprintf(outf, "Creating index file %s\n", newname);
  CreatePathToFile(newname);
  indexfile = fopen(newname, "a");
  fprintf(indexfile, "BEGIN PACK\n");
  sum = 0L;
  for (i = 0; i < dirsize; i++)
    {
      if (entrynum < dirsize && i != entrynum)
	continue; /* horrible trick... */
      strcpy(p, dir[i].name);
      if (outf != NULL)
	fprintf(outf, "Saving %6ld bytes to %s\n", dir[i].size, newname);
      CreatePathToFile(newname);
      newfile = fopen(newname, "wb");
      if (newfile == NULL)
	{
	  fclose(indexfile);
	  QFree(newname);
	  return FALSE;
	}
      fprintf(indexfile, "+ %s = %s\n", p, p);
      if ((fseek(packfile, offset + dir[i].offset, SEEK_SET) < 0)
	  || (CopyBytes(newfile, packfile, dir[i].size) == FALSE))
	{
	  fclose(newfile);
	  fclose(indexfile);
	  QFree(newname);
	  return FALSE;
	}
      fclose(newfile);
      sum += dir[i].size;
    }
  if (outf != NULL && entrynum >= dirsize)
    fprintf(outf, "Saved %lu bytes in %u files.\n", sum, dirsize);
  fprintf(indexfile, "END PACK\n");
  fclose(indexfile);
  QFree(newname);
  return TRUE;
}


/* ----------------------------------------------------------------------------
 * NOTE: How to save a pack file:
 *
 * UInt32     count;  - used by the saving routines
 * PACKDirPtr dir;    - PACK directory structure (created step by step)
 * UInt16     n;      - number of entries in PACK directory (idem)
 *
 * WritePACKHeader(f, &count, &dir, &n);            - write the header
 * size = WriteSomething(f, ...);                   - save one entry
 * AddPackEntry(f, &count, &dir, &n, name, size);   - add this entry to dir.
 * size = WriteSomethingElse(f, ...);               - save another entry
 * AddPackEntry(f, &count, &dir, &n, othername, size); - add it to dir. too
 * totalsize = WritePACKDirectory(f, &count, dir, n);  - write the directory
 */

/*
 * Write the PACK header to the file.  The header will be modified later,
 * when the directory is written to the file.
 */
Bool WritePACKHeader(FILE *packfile, UInt32 *count_r, PACKDirPtr *dir_r,
		     UInt16 *dirsize_r)
{
  char buf[100];

  sprintf(buf, "PACK********\r\nQEU %s\r\n", QEU_VERSION);
  if ((packfile == NULL)
      || (WriteBytes(packfile, buf, (UInt32)strlen(buf)) == FALSE))
    return FALSE;
  *dir_r = NULL;
  *count_r = (UInt32)strlen(buf);
  *dirsize_r = 0;
  return TRUE;
}


/*
 * Add a new entry to the PACK directory.  This entry should have been
 * saved previously and be "entrysize" bytes long.  It will be stored
 * in the PACK directory under the name "entryname".
 * All object saving routines in this package return the number of bytes
 * written, so that number can be passed directly to this routine.
 */
Bool AddPACKEntry(FILE *packfile, UInt32 *count_r, PACKDirPtr *dir_r,
		  UInt16 *dirsize_r, char *entryname, UInt32 entrysize)
{
  UInt16 n;

  if (packfile == NULL || *count_r == 0L)
    return FALSE;
  n = *dirsize_r;
  if (n == 0)
    *dir_r = (PACKDirPtr)QMalloc((UInt32)sizeof(struct PACKDirectory));
  else
    *dir_r = (PACKDirPtr)QRealloc(*dir_r, (UInt32)(n + 1)
				  * (UInt32)sizeof(struct PACKDirectory));
  QStrNCpy((*dir_r)[n].name, entryname, 56);
  (*dir_r)[n].offset = *count_r;
  (*dir_r)[n].size = entrysize;
  *count_r = *count_r + entrysize;
  *dirsize_r = n + 1;
  return TRUE;
}


/*
 * Write the PACK directory to the file.  This should only be done
 * after all entries have been saved (using the appropriate
 * Write... routine) and registered (using AddPACKEntry).  The PACK
 * header is updated so that it points to the directory.
 *
 * This routine returns the total number of bytes taken by the PACK
 * file (header + all entries + directory).  It is thus possible to
 * include a PACK file in another PACK file or in a WAD file.
 */
UInt32 WritePACKDirectory(FILE *packfile, UInt32 *count_r, PACKDirPtr dir,
			  UInt16 dirsize)
{
  UInt32 size, pos;
  UInt16 i;

  pos = *count_r;
  if (packfile == NULL || pos == 0L)
    return 0L;
  *count_r = 0L; /* invalidate the counter */
  size = (UInt32)dirsize * (UInt32)sizeof(struct PACKDirectory);
  if ((fseek(packfile, 4L - (Int32)(pos), SEEK_CUR) < 0)
      || (WriteInt32(packfile, &pos) == FALSE)
      || (WriteInt32(packfile, &size) == FALSE)
      || (fseek(packfile, (Int32)(pos) - 12L, SEEK_CUR) < 0))
    return 0L;
  for (i = 0; i < dirsize; i++)
    if ((WriteBytes(packfile, &dir[i], 56) == FALSE)
	|| (WriteInt32(packfile, &(dir[i].offset)) == FALSE)
	|| (WriteInt32(packfile, &(dir[i].size)) == FALSE))
      return 0L;
  return size + pos;
}

/* end of file */
