/*
	Source code is copyright 1997 by Viperware(tm)
	Permission to re-use code is granted
	
	Note: this code is not intended to compile- there are too many interdependancies 
		  to make this feasible.  However, this code is being used in real products
		  being sold by Viperware, and it should furthur illustrate the concepts
		  introduced in the "Tagged Data Storage Architecture" article.
 */

#ifndef __LB_HANDLECHUNK__
#include "LB_HandleChunk.h"
#endif
#ifndef __LB_CHUNKDATA__
#include "LB_ChunkData.h"
#endif
#ifndef __MM_MEMORYMANAGER__
#include "MM_MemoryManager.h"
#endif
#ifndef __SN_EXCEPTION__
#include "SN_Exception.h"
#endif
#ifndef __EX_EXTERMINATOR__
#include "EX_Exterminator.h"
#endif

LB_HandleChunk::LB_HandleChunk()
:	lbh_oldState(0)
{
}

LB_HandleChunk::LB_HandleChunk(Handle hData)
:	lbh_oldState(0)
{
	// Use the data handle.
	UseData((LB_DataReference) hData);
}

LB_HandleChunk::~LB_HandleChunk()
{	
	// This must be here because by the time we get to the LB_DataChunk destructor,
	// we can no longer access the subclass's methods.
	DestroyData();
}

LB_HandleChunk::LB_HandleChunk(LB_DataSize dataSize)
{
	// Create the data handle.
	Handle			hData;
	MM_AllocateRelocatableData(dataSize, hData);
	UseData((LB_DataReference) hData);

	if (HasValidData())
	{
		HUnlock(GetHandle());
		HNoPurge(GetHandle());
	}
}

SN_Error LB_HandleChunk::DestroyData()
{
	SN_Error			result;
	
	EX_TRY
	{
		if (GetHandle())
		{
			// If the Handle exists, release it from memory.
			Handle			constHandle = GetHandle();
			EX_THROW_ERROR(MM_ReleaseRelocatableData(constHandle));
		}
		
		UseData(nil);
	}
	catch (SN_Exception& rException)
	{
		result = rException;
	}
	
	return result;
}

SN_Error LB_HandleChunk::ReleaseData(LB_DataReference dataReference)
{
	SN_Error			result;
	
	EX_TRY
	{
		EX_THROW_ERROR(LB_DataChunk::ReleaseData(dataReference));
		EX_THROW_FALSE(HasValidData());
		
		if (CountDependancies() == 0)
			// This handle does not have any dependancies, so it can be relocatable.
			HSetState(GetHandle(), lbh_oldState);
	}
	catch (SN_Exception& rException)
	{
		result = rException;
	}
	
	return result;
}

SN_Error LB_HandleChunk::RetrieveData(LB_ChunkData& rChunkData)
{
	SN_Error			result;
	
	EX_TRY
	{
		rChunkData.UseData(nil);
		EX_THROW_NIL(HasValidData());
		EX_THROW_ERROR(AddDependancy());

		// Save the original state of the handle.
		lbh_oldState = HGetState(GetHandle());
		
		// Make sure the handle doesn't get relocated or purged so that
		// its dependant's data values are always correct.
		MoveHHi(GetHandle());
		HLock(GetHandle());
		HNoPurge(GetHandle());
		
		rChunkData.UseData(GetRealData());
		rChunkData.UseChunk(this);
	}
	catch (SN_Exception& rException)
	{
		result = rException;
	}

	return result;
}

Boolean LB_HandleChunk::IsValidData(LB_DataReference dataReference)
{
	Boolean			result = false;
	
	if ((Handle) dataReference)
		if (*((Handle) dataReference))
			// This is a valid, non-purged handle.
			result = true;
			
	return result;
}

Boolean LB_HandleChunk::MyDependancy(LB_DataReference dataReference)
{	
	Boolean			result = false;
	
	// Because the dependancy is a pointer and not a handle, we need to compare this
	// handle's pointer to the specified data reference.
	if (HasValidData())
		if (dataReference == (GetRealData()) && CountDependancies() > 0)
			result = true;
			
	return result;
}

LB_DataReference LB_HandleChunk::GetRealData()
{
	LB_DataReference		result = nil;
	
	if (HasValidData())
		// Return the pointer, not the handle.
		result =  (LB_DataReference) (*(GetHandle()));
	
	return result;
}

LB_DataSize LB_HandleChunk::GetDataSize()
{
	LB_DataSize			result = 0;
	
	if (HasValidData())
		// The data size is the size of the handle.
		result = GetHandleSize(GetHandle());
		
	return result;
}

LB_DataSize LB_HandleChunk::SetDataSize(LB_DataSize newSize)
{
	if (HasValidData())
		// Resize the data.
		SetHandleSize(GetHandle(), newSize);
		
	return GetDataSize();
}
