/*
	Source code is copyright 1997 by Viperware(tm)
	Permission to re-use code is granted
	
	Note: this code is not intended to compile- there are too many interdependancies 
		  to make this feasible.  However, this code is being used in real products
		  being sold by Viperware, and it should furthur illustrate the concepts
		  introduced in the "Tagged Data Storage Architecture" article.
 */

#ifndef __FT_TAGGABLEOBJECT__
#include "FT_TaggableObject.h"
#endif
#ifndef __FT_TYPES__
#include "FT_Types.h"
#endif
#ifndef __FT_TAGMANAGER__
#include "FT_TagManager.h"
#endif
#ifndef __MM_MEMORYMANAGER__
#include "MM_MemoryManager.h"
#endif
#ifndef __EX_EXTERMINATOR__
#include "EX_Exterminator.h"
#endif

FT_TaggableObject::FT_TaggableObject()
{
	// Create the tag collection.
	CreateCollection();
}

FT_TaggableObject::~FT_TaggableObject()
{
	Dispose(false, false);
}

SN_Error FT_TaggableObject::Dispose(Boolean destroy, Boolean stopOnError)
{
	SN_Error			result;
	
	EX_TRY
	{
		// DON'T remove the LL_Collectable collection, even if we are told
		// to destroy the object.  Only destroy members that are recoverable.
		
		if (stopOnError)
		{
			EX_THROW_ERROR(RemoveAllTags());
		}
		else
		{
			RemoveAllTags();
		}
	}
	catch (SN_Exception& rException)
	{
		result = rException;
	}
	
	return result;
}

SN_Error FT_TaggableObject::AllocateTagData(FT_TagData& rTagData, FT_TagDataSize tagDataSize)
{
	SN_Error			result;
	
	// All data for tags should be allocated as handles so that memory doesn't get fragmented
	// and so tag data can be resized.
	result = MM_AllocateRelocatableData(tagDataSize, rTagData);
	
	if (result.NoError() && rTagData)
	{
		// Setup the tag data so it can be re-located, but not purged.
		HNoPurge(rTagData);
		HUnlock(rTagData);
	}
	
	return result;
}

SN_Error FT_TaggableObject::DisposeTagData(FT_TagData& rTagData)
{
	// Release the tag data from memory.
	return MM_ReleaseRelocatableData(rTagData);
}

SN_Error FT_TaggableObject::ResizeTagData(FT_TagData& rTagData, FT_TagDataSize tagDataSize)
{
	SN_Error			result;
	
	if (rTagData == nil)
	{
		// Create a new handle for the data.
		MM_AllocateRelocatableData(tagDataSize, rTagData);
		
		if (rTagData)
		{
			HNoPurge(rTagData);
			HUnlock(rTagData);
		}
	}
	else
		// Resize the pointer to the tag data.
		SetHandleSize(rTagData, tagDataSize);
		
	result.SetType(MemError());
	
	return result;
}

FT_TagData FT_TaggableObject::GetAllocatedTagData(FT_TagDataSize tagDataSize)
{
	FT_TagData			result;
	
	// Allocate the data, but we don't care about the error.
	AllocateTagData(result, tagDataSize);
	
	return result;
}

SN_Error FT_TaggableObject::AttachDataTag(FT_TagName tagName, FT_TagType tagType, FT_TagData tagData)
{
	return AttachTag(tagName, tagType, tagData, FT_GetTagDataSize(tagData));
}

SN_Error FT_TaggableObject::AttachDataTagWithValue(FT_TagName tagName, FT_TagType tagType, FT_TagData tagData, Ptr pDefaultData)
{
	if (pDefaultData && FT_GetTagDataReference(tagData))
	{
		FT_UseTagData			useTagData(tagData);
		
		// Copy the default data into the tag data.
		BlockMove(pDefaultData, FT_GetTagDataReference(tagData), FT_GetTagDataSize(tagData));
	}
		
	return AttachDataTag(tagName, tagType, tagData);
}

SN_Error FT_TaggableObject::AttachTagWithValue(FT_TagName tagName, FT_TagType tagType, FT_TagData tagData, FT_TagDataSize tagDataSize, Ptr pDefaultData)
{
	if (pDefaultData && FT_GetTagDataReference(tagData))
	{
		FT_UseTagData			useTagData(tagData);
		
		// Copy the default data into the tag data.
		BlockMove(pDefaultData, FT_GetTagDataReference(tagData), FT_GetTagDataSize(tagData));
	}
		
	return AttachTag(tagName, tagType, tagData, tagDataSize);
}

SN_Error FT_TaggableObject::AttachCreatedTag(FT_TagName tagName, FT_TagType tagType, FT_TagData tagData, FT_TagDataSize tagDataSize, FT_Tag*& rpTag)
{
	SN_Error			result(noErr);
	
	EX_TRY
	{
		EX_THROW_NIL(rpTag);

		// Enter the information for the tag.
		rpTag->SetName(tagName);
		rpTag->SetType(tagType);
		rpTag->SetData(tagData, tagDataSize);
		
		// Add the tag to the list of tags.
		AddTag(*rpTag);
	}
	catch (SN_Exception& error)
	{
		// This is the error.
		result = error;
		
		// Release the tag from memory.
		delete rpTag;
		rpTag = nil;
	}
	
	return result;
}

SN_Error FT_TaggableObject::AttachExtendedTag(FT_TagName tagName, FT_TagType tagType, FT_Tag* pTag)
{
	SN_Error			result(ft_kBadTagReferenceError);
	
	if (pTag)
	{
		result = AttachCreatedTag(tagName, tagType, pTag->GetData(), pTag->GetDataSizeWithHeader(0), pTag);
	}
	
	return result;
}

SN_Error FT_TaggableObject::AttachTag(FT_TagName tagName, FT_TagType tagType, FT_TagData tagData, FT_TagDataSize tagDataSize)
{
	SN_Error			result(noErr);
	FT_Tag*				pTag = nil;
	
	EX_TRY
	{
		// Create the tag.
		pTag = new FT_Tag;
		EX_THROW_NIL(pTag);
		
		// Attach the created tag.
		EX_THROW_ERROR(AttachCreatedTag(tagName, tagType, tagData, tagDataSize, pTag));
	}
	catch (SN_Exception& error)
	{
		// This is the error.
		result = error;
		
		// Release the tag from memory.
		delete pTag;
	}
	
	return result;
}

SN_Error FT_TaggableObject::AddTag(FT_Tag& rTag)
{
	SN_Error			result;
	
	if (HasCollection())
	{
		// Add the tag to the list of tags.
		GetCollection()->AddElement(&rTag);
	}
	else
		result.SetType(ll_kBadCollectionError);

	return result;
}

SN_Error FT_TaggableObject::RemoveTag(FT_TagName tagName)
{
	FY_TCollection<FT_Tag>::FY_CollectionElement*	pElement = nil;
	SN_Error										result = RetrieveTagElement(tagName, pElement);
	
	if (result.GetType() == noErr && pElement)
	{
		// Remove the element from the collection.
		GetCollection()->RemoveElement(pElement);
	}
	
	return result;
}

SN_Error FT_TaggableObject::RemoveAllTags()
{
	SN_Error			result;
	
	if (HasCollection())
	{
		// Remove all the tags.
		while (GetCollection()->GetFirstElement())
			GetCollection()->RemoveElement(GetCollection()->GetFirstElement());
	}
	else
		// A collection must exist for tags.
		result.SetType(ll_kBadCollectionError);
		
	return result;
}

SN_Error FT_TaggableObject::RetrieveTagElement(FT_TagName tagName, FY_TCollection<FT_Tag>::FY_CollectionElement*& rpElement)
{
	SN_Error			result;
	FT_TagName			searchName;
	
	if (HasCollection())
	{
		// Search for the tag.
		for (FY_TCollection<FT_Tag>::FY_CollectionElement*	pTagItem = GetCollection()->GetFirstElement();
			 pTagItem; pTagItem = pTagItem->GetNextElement())
		{
			FT_Tag*			pTag = pTagItem->GetData();
			
			if (pTag)
			{
				searchName = pTag->GetName();
				
				if (searchName == tagName)
				{
					// This is the tag we are looking for.
					rpElement = pTagItem;
					goto breakRetrieve;
				}	
			}
		}
	
		// We didn't find the tag we wanted.
		result.SetType(ft_kTagDoesntExistError);
	}
	else
		// A collection must exist for tags.
		result.SetType(ll_kBadCollectionError);
	
breakRetrieve:
	return result;
}

SN_Error FT_TaggableObject::RetrieveTagNeedsDataAndType(FT_TagName tagName, FT_TagType tagType, FT_Tag*& rpTag)
{
	SN_Error			result;
	
	EX_TRY
	{
		// Retrieve the tag.
		EX_THROW_ERROR(RetrieveTag(tagName, rpTag));	
		EX_THROW_NIL(rpTag);
		
		// Make sure the tag has data.
		if (rpTag->GetData() == nil)
			EX_THROW_OSERR(ft_kNoTagDataError);
			
		// Make sure the tag is the correct type.
		if (! rpTag->GetType() == tagType)
			EX_THROW_OSERR(ft_kBadTagDataTypeError);
	}
	catch (SN_Exception& rException)
	{
		result = rException;
	}
	
	return result;
}

SN_Error FT_TaggableObject::RetrieveAndStuffTagNeedsDataAndType(FT_TagName tagName, FT_TagType tagType, FT_Tag*& rpFoundTag, Ptr pStuffedData)
{
	SN_Error			result;
	
	EX_TRY
	{
		// Retrieve the tag.
		EX_THROW_ERROR(RetrieveTagNeedsDataAndType(tagName, tagType, rpFoundTag));
		EX_THROW_NIL(rpFoundTag);
		
		if (pStuffedData && rpFoundTag->GetData())
		{
			FT_UseTagData			useTagData(rpFoundTag->GetData());
			
			// Stuff the tag's data into the variable.
			// Be careful!  You are responsible for making sure
			// the destination pointer is big enough to hold the tag data!
			BlockMove(FT_GetTagDataReference(rpFoundTag->GetData()), pStuffedData, FT_GetTagDataSize(rpFoundTag->GetData()));
		}
	}
	catch (SN_Exception& rException)
	{
		result = rException;
	}
		
	return result;
}

SN_Error FT_TaggableObject::RetrieveTag(FT_TagName tagName, FT_Tag*& rpTag)
{
	FY_TCollection<FT_Tag>::FY_CollectionElement*	pElement = nil;
	SN_Error										result = RetrieveTagElement(tagName, pElement);
	
	if (result.NoError() && pElement)
	{
		// Copy the found tag reference.
		rpTag = pElement->GetData();
	}
	
	return result;
}

SN_Error FT_TaggableObject::RetrieveTagWithIndex(FT_TagIndex tagIndex, FT_Tag*& rpTag)
{
	return RetrieveTagWithTypeAndIndex(kAnyTag, tagIndex, rpTag);
}

SN_Error FT_TaggableObject::RetrieveTagWithTypeAndIndex(FT_TagType tagType, FT_TagIndex tagIndex, FT_Tag*& rpTag)
{
	SN_Error			result;
	FT_TagIndex			searchIndex = 1;
	
	if (HasCollection())
	{
		// Search for the tag.
		for (FY_TCollection<FT_Tag>::FY_CollectionElement*	pTagItem = GetCollection()->GetFirstElement();
			 pTagItem; pTagItem = pTagItem->GetNextElement())
		{
			FT_Tag*			pTag = pTagItem->GetData();
			
			if (pTag)
			{	
				if (searchIndex == tagIndex)
				{
					if (pTag->GetType() == tagType || tagType == kAnyTag)
					{
						// This is the tag we are looking for.
						rpTag = pTag;
						goto breakRetrieve;
					}
				}
				
				if (pTag->GetType() == tagType || tagType == kAnyTag)
					searchIndex += 1;
			}
		}
	
		// We didn't find the tag we wanted.
		result.SetType(ft_kTagDoesntExistError);
	}
	else
		result.SetType(ll_kBadCollectionError);
	
breakRetrieve:
	return result;
}

SN_Error FT_TaggableObject::CountTags(FT_TagIndex& rTagCount)
{
	SN_Error			result;
	
	if (HasCollection())
	{
		// Retrieve the number of tags attached to this object.
		rTagCount = GetCollection()->CountElements();
	}
	else
		// A collection must exist for tags.
		result.SetType(ll_kBadCollectionError);
	
	return result;
}

SN_Error FT_TaggableObject::CalculateTotalDataSize(FT_TagDataSize& rTotalDataSize)
{
	return CalculateTotalDataSizeWithHeader(rTotalDataSize, 0);
}

SN_Error FT_TaggableObject::CalculateTotalDataSizeWithHeader(FT_TagDataSize& rTotalDataSize, FT_TagDataSize headerSize)
{
	SN_Error			result;	
	FT_TagIndex			tagIndex = 1;
	FT_Tag*				pFoundTag = nil;
	Boolean				done = false;
	SN_Error			error;
	
	// Reset the total data size.
	rTotalDataSize = 0;
	
	// Calculate the size of all of the tags combined.
	while (! done)
	{
		// Retrieve the current tag.		
		error = RetrieveTagWithIndex(tagIndex, pFoundTag);
		tagIndex += 1;		

		if (error.NoError() && pFoundTag)
		{
			// Add the size of this tag to the total object data size.
			rTotalDataSize += pFoundTag->GetDataSizeWithHeader(headerSize);
		}
		else
			done = true;
	}
	
	return result;
}

SN_Error FT_TaggableObject::RetrieveTagData(FT_TagName tagName, FT_TagData& rTagData)
{
	SN_Error			result;
	FT_Tag*				pFoundTag;
	
	rTagData = nil;			// In case there is no tag.
	
	// Retrieve the tag.
	result = RetrieveTag(tagName, pFoundTag);	
	
	if (result.NoError() && pFoundTag)
	{
		rTagData = pFoundTag->GetData();
		
		// Make sure the tag has data.
		if (pFoundTag->GetData() == nil)
			result.SetType(ft_kNoTagDataError);
	}
		
	return result;
}

SN_Error FT_TaggableObject::RetrieveTagDataReference(FT_TagName tagName, Ptr& rpTagData)
{
	FT_TagData			tagData;
	SN_Error			result;
	
	result = RetrieveTagData(tagName, tagData);
	
	// Access the data's reference.
	rpTagData = FT_GetTagDataReference(tagData);
	
	return result;
}
