package bmsi.tui;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ItemEvent;
import java.awt.peer.*;
import java.io.IOException;

/** MenuComponents are fairly mindless implementations of RemotePeer */

abstract class TUIMenuComponent implements MenuComponentPeer, RemotePeer {
  protected int winID;
  protected MenuComponent target;
  protected TUIKit toolkit;
  protected abstract void initialize();
  protected abstract void create(TUIMenuComponent parent);
  TUIMenuComponent(MenuComponent m,TUIKit toolkit) {
    target = m;
    this.toolkit = toolkit;
    MenuContainer parent = m.getParent();
    TUIMenuComponent parentpeer = null;
    if (parent != null)
      if (parent instanceof MenuBar)
	parentpeer = (TUIMenuComponent)((MenuBar)parent).getPeer();
      else if (parent instanceof Menu)
	parentpeer = (TUIMenuComponent)((Menu)parent).getPeer();
    create(parentpeer);
    initialize();
  }
  public void handleEvent(AWTEvent e) { }

  public int getID() { return winID; }
  public Object getTarget() { return target; }

  protected void finalize() { dispose(); }
  public void dispose() {
    toolkit.removePeer(winID);
    winID = -1;
    target = null;
  }
  public void remoteMethod(int cmd) throws IOException { }
}

class TUIMenuItem extends TUIMenuComponent implements MenuItemPeer {
  private boolean enabled = true;	// track state locally
  TUIMenuItem(MenuItem m,TUIKit toolkit) { super(m,toolkit); }
  protected void initialize() {
    MenuItem m = (MenuItem)target;
    setLabel(m.getLabel());
    if (!m.isEnabled())
      setEnabled(false);
  }
  protected void create(TUIMenuComponent parent) {
    winID = toolkit.createRemotePeer(parent,NEWMENUITEM,this);
  }
  public void setEnabled(boolean a) {
    if (a == enabled) return;
    enabled = a;
    if (a)
      toolkit.writeCmd(this,ENABLE);
    else
      toolkit.writeCmd(this,DISABLE);
  }
  public void setLabel(String lab) { toolkit.writeCmd(this,SETLABEL,lab); }
  public void remoteMethod(int cmd) throws IOException {
    MenuItem item = (MenuItem)target;
    /*JDK1.0: item.postEvent(new Event(target,Event.ACTION_EVENT,
	item.getLabel())); */
    toolkit.theQueue.postEvent(
      new ActionEvent(item,ActionEvent.ACTION_PERFORMED,
      item.getActionCommand())
    );
  }
  /* jdk 1.0 compatibility */
  public final void disable() { setEnabled(false); }
  public final void enable() { setEnabled(true); }
}

class TUICheckboxMenuItem extends TUIMenuItem implements CheckboxMenuItemPeer {
  TUICheckboxMenuItem(CheckboxMenuItem m,TUIKit toolkit) { super(m,toolkit); }
  protected void initialize() {
    CheckboxMenuItem m = (CheckboxMenuItem)target;
    if (m.getState())
      setState(true);
    super.initialize();
  }
  protected void create(TUIMenuComponent parent) {
    winID = toolkit.createRemotePeer(parent,NEWCHECKMENUITEM,this);
  }
  public void setState(boolean t) {
    toolkit.writeCmd(this,SETSTATE,t ? 1 : 0);
  }
  public void remoteMethod(int cmd) throws IOException {
    CheckboxMenuItem item = (CheckboxMenuItem)target;
    /*JDK1.0: item.postEvent(new Event(target,Event.ACTION_EVENT,
	item.getLabel())); */
    boolean t = !item.getState();
    item.setState(t);
    String label = item.getActionCommand();
    toolkit.theQueue.postEvent(
      new ItemEvent(item,ItemEvent.ITEM_STATE_CHANGED,label,
	t ? ItemEvent.SELECTED : ItemEvent.DESELECTED)
    );
    toolkit.theQueue.postEvent(
      new ActionEvent(item,ActionEvent.ACTION_PERFORMED,label)
    );
  }
}

class TUIMenu extends TUIMenuItem implements MenuPeer {
  /* needed in JDK1.0
  protected void initialize() {
    Menu m = (Menu)target;
    int n = m.countItems();
    for (int i = 0; i < n; ++i) {
      MenuItem item = m.getItem(i);
      if (item.getPeer() != null) {
	if (item.getLabel().equals("-"))
	  addSeparator();
	else
	  addItem(item);
      }
    }
    super.initialize();
  }
  */

  protected void create(TUIMenuComponent parent) {
    winID = toolkit.createRemotePeer(parent,NEWMENU,this);
  }

  TUIMenu(Menu m,TUIKit toolkit) { super(m,toolkit); }
  public void addItem(MenuItem i) {
    TUIMenuItem item = (TUIMenuItem)i.getPeer();
    toolkit.writeCmd(this,ADDITEM,item.winID);
  }
  public void addSeparator() {
    toolkit.writeCmd(this,ADDSEP);
  }
  public void delItem(int pos) {
    toolkit.writeCmd(this,DELITEM,pos);
  }
}

class TUIMenuBar extends TUIMenuComponent implements MenuBarPeer {
  /** Although JDK 1.1 will add the MenuItems to a MenuPeer, it will
    not add Menus to a MenuBarPeer, so we have to do it ourselves.   Wierd. */
  protected void initialize() {
    MenuBar m = (MenuBar)target;
    // java.awt in JDK 1.0 will addNotify a parent menu before it's submenus!
    // We have to addNotify() ourselves for the child Menus in this case.
    Menu menu = m.getHelpMenu();
    if (menu != null) {
      //if (menu.getPeer() == null)
      //  menu.addNotify();
      if (menu.getPeer() != null)
	addHelpMenu(menu);
    }
    int n = m.countMenus();
    for (int i = 0; i < n; ++i) {
      menu = m.getMenu(i);
      //if (menu.getPeer() == null)
      //  menu.addNotify();
      if (menu.getPeer() != null)
	addMenu(menu);
    }
  }

  protected void create(TUIMenuComponent parent) {
    winID = toolkit.createRemotePeer(parent,NEWMENUBAR,this);
  }

  TUIMenuBar(MenuBar m,TUIKit toolkit) { super(m,toolkit); }

  public void addHelpMenu(Menu m) {
    TUIMenu menu = (TUIMenu)m.getPeer();
    toolkit.writeCmd(this,ADDHELPMENU,menu.winID);
  }

  public void addMenu(Menu m) {
    TUIMenu menu = (TUIMenu)m.getPeer();
    toolkit.writeCmd(this,ADDMENU,menu.winID);
  }

  public void delMenu(int pos) {
    toolkit.writeCmd(this,DELMENU,pos);
  }
}
