package bmsi.tui;
import java.awt.*;
import java.awt.peer.*;
import java.awt.event.WindowEvent;
import java.io.IOException;

abstract class TUIContainer extends TUIComponent {
  protected TUIContainer(Container c,TUIKit toolkit) { super(c,toolkit); }
  /* needed in JDK1.0
  protected void initialize() {
    super.initialize();
    Container c = (Container)target;
    Component[] comp = c.getComponents();
    for (int i = 0; i < comp.length; ++i) {
      TUIComponent cp = (TUIComponent)comp[i].getPeer();
      if (cp != null)
	add(cp);
    }
  }
  */
  public Insets getInsets() { return new Insets(0,0,0,0); }
  public void add(TUIComponent c) {
    toolkit.writeCmd(this,ADDCOMPONENT,c.winID);
  }

  /** The purpose of beginValidate is not documented, but I *think*
   we are supposed to stop sending reshape commands for contained
   components until the Container finishes fiddling around and calls
   endValidate.  We'll ignore it for now since this is just an optimization.
   */
  public void beginValidate() { }
  public void endValidate() { }

  /* jdk 1.0 compatibility */
  public final Insets insets() { return getInsets(); }
}

class TUIPanel extends TUIContainer implements PanelPeer {
  protected TUIPanel(Panel c,TUIKit toolkit) { super(c,toolkit); }
  protected void initialize() {
    if (target.getBackground() == null)
      setBackground(Color.white);
    super.initialize();
  }
  protected void create(TUIContainer parent) {
    winID = toolkit.createRemotePeer(parent,NEWPANEL,this);
  }
}

class TUIWindow extends TUIContainer implements WindowPeer {
  private boolean defaultlocation = true;
  TUIWindow(Window w,TUIKit toolkit) { super(w,toolkit); }
  protected void create(TUIContainer parent) {
    winID = toolkit.createRemotePeer(parent,NEWWINDOW,this);
    toolkit.theQueue.postEvent(
      new WindowEvent((Window)target,WindowEvent.WINDOW_OPENED));
  }
  public void toBack() { toolkit.writeCmd(this,TOBACK); }
  public void toFront() { toolkit.writeCmd(this,TOFRONT); }
  public void remoteMethod(int cmd) throws IOException {
    switch (cmd) {
    case TOFRONT:
      if (toolkit.active != this) {
	if (toolkit.active != null)
	  toolkit.theQueue.postEvent(new WindowEvent(
	    (Window)toolkit.active.target,WindowEvent.WINDOW_DEACTIVATED));
	toolkit.active = this;
      }
      toolkit.theQueue.postEvent(
	new WindowEvent((Window)target,WindowEvent.WINDOW_ACTIVATED));
      return;
    case RESHAPE:
      super.remoteMethod(cmd);
      target.validate();
      return;
    case DISPOSE:
      //JDK1.0: target.postEvent(new Event(this,Event.WINDOW_DESTROY,null));
      toolkit.theQueue.postEvent(
	new WindowEvent((Window)target,WindowEvent.WINDOW_CLOSING));
      //target.removeNotify();
      return;
    }
    super.remoteMethod(cmd);
  }

  /** For top level windows, location on screen is just component location. */
  public Point getLocationOnScreen() { return target.getLocation(); }
  public synchronized void setVisible(boolean vis) {
    super.setVisible(vis);
    if (vis)
      toolkit.sync();
    else if (toolkit.active == this) {
      toolkit.theQueue.postEvent(new WindowEvent(
	(Window)toolkit.active.target,WindowEvent.WINDOW_DEACTIVATED));
      toolkit.active = null;
    }
  }
  public void setBounds(int x,int y,int w,int h) {
    /* Most apps put top level windows at top left of screen.  This
       heuristic tries to fix this ugliness by centering when
       the first location set is 0,0. */
    if (defaultlocation && x == 0 && y == 0) {
      defaultlocation = false;
      Container parent = target.getParent();
      Dimension sz = (parent != null)
		     ? parent.getSize() : toolkit.getScreenSize();
      if (sz.width > w && sz.height > h) {
	x = (sz.width - w)/2;
	y = (sz.height - h)/2;
      }
    }
    Insets i = getInsets();
    // Our TUI system sets and reports client area size, not total window size.
    super.setBounds(x+i.left,y+i.top,w-i.left-i.right,h-i.top-i.bottom);
  }
}

/** TUI Frames and Dialogs provide their own frames. */
class TUIFramedWindow extends TUIWindow {
  private Insets insets;
  TUIFramedWindow(Window w,TUIKit toolkit) {
    super(w,toolkit);
  }
  public void setTitle(String s) { toolkit.writeCmd(this,SETTITLE,s); }
  public void setResizable(boolean flg) {
    toolkit.writeCmd(this,SETRESIZABLE,flg ? 1 : 0);
  }
  // Our TUI system uses a 1 char frame on all sides (no shadowing).
  public Insets getInsets() {
    if (insets == null) {
      insets = new Insets(1,1,1,1);
      Dimension scale = toolkit.scale(new Dimension(1,1));
      insets.left *= scale.width;
      insets.right *= scale.width;
      insets.top *= scale.height;
      insets.bottom *= scale.height;
    }
    return insets;
  }
}

class TUIDialog extends TUIFramedWindow implements DialogPeer {
  TUIDialog(Dialog d,TUIKit toolkit) { super(d,toolkit); }
  protected void initialize() {
    Dialog dlg = (Dialog)target;
    String title = dlg.getTitle();
    if (title != null)
      setTitle(title);
    if (!dlg.isResizable())
      setResizable(false);
    super.initialize();
  }
  protected void create(TUIContainer parent) {
    winID = toolkit.createRemotePeer(parent,NEWDIALOG,this);
  }

  private boolean visible = false;

  public synchronized void setVisible(boolean vis) {
    if (vis == visible) return;
    super.setVisible(vis);
    visible = vis;
    Dialog dlg = (Dialog)target;
    if (!vis)
      notify();		// notify thread waiting for HIDE
    else if (dlg.isModal())
      try {
	while (dlg.isVisible())
	  wait();
      }
      catch (InterruptedException e) {
	throw new AWTError("Interrupted waiting for Dialog close");
      }
  }
}

class TUIFrame extends TUIFramedWindow implements FramePeer {
  TUIFrame(Frame f,TUIKit toolkit) { super(f,toolkit); }
  protected void initialize() {
    Frame frame = (Frame)target;
    String title = frame.getTitle();
    if (title != null)
      setTitle(title);
    if (!frame.isResizable())
      setResizable(false);
    MenuBar menu = frame.getMenuBar();
    if (menu != null) {
      if (menu.getPeer() == null)
	menu.addNotify();
      setMenuBar(menu);
    }
    super.initialize();
  }
  protected void create(TUIContainer parent) {
    winID = toolkit.createRemotePeer(parent,NEWFRAME,this);
  }
  /** While most text terminals support icons via font download, we
   have no time for such foolishness :-) */
  public void setIconImage(Image img) { }
  public void setMenuBar(MenuBar mnu) {
    int id = -1;
    if (mnu != null) {
      TUIMenuBar peer = (TUIMenuBar)mnu.getPeer();
      id = peer.getID();
    }
    toolkit.writeCmd(this,SETMENU,id);
  }
}
