unit SplashScreen;

{ ==============================================================
  SplashScreen 1.0.0
{ ==============================================================

  SUMMARY

  Inheritable Splashscreen and About box.
  Includes transparent display of company and product logo.
  Extracts company and application name and version information
  from the VersionInfo resource of the executable. Displays
  registration information or trial period.

  Author:       1996, Andy Schmidt
  Email:       Andy_Schmidt@CompuServe.com
  Compiler:    Delphi 2.01
  Runtime:     Win32


{ --------------------------------------------------------------


  INSTALL

  1. Open the SplashScreen form. Right click form and add the
     form the object repository.

  USAGE

  1. Open a project.

  2. Select 'New' from the File menu.

  3. Select the Splash Screen form from the repository.

  4. Choose 'Inherit'.

  5. Select 'Options' from the Project menu.

  6. Remove your inherited SplashScreen from the 'auto created' forms.

  7. Insert this function call behind 'Application.Initialize' in
     your project file:

     ShowSplashScreen(TYourSplashScreen, YourSplashScreen, Splash, clWhite);


{ --------------------------------------------------------------

  PROPERTIES

  DialogType       'Splash' = Displays form as a splash screen
                              with a timer.
                   'About'  = Displays form as an About window.

  Transparent      Color of the Company and Application logo bitmap
                   that should be considered transparent.
                   Defaults:  White.


  FUNCTIONS

  ShowSplashScreen - creates the SplashScreen object if necessary
                     and shows the form.


{ --------------------------------------------------------------

  LICENSE

  The Author hereby grants to you a nonexclusive license to use
  this software and the accompanying Instructions, only as
  authorized in this License.

  You agree that you will not assign, sublicense, transfer,
  pledge, lease, rent, or share your rights under this License
  in return for compensation of any kind. You may include this
  object in executable form with your own product, but before
  you use this software in source code format for commercial
  purposes, you are required to pay a license fee of $20.00
  to the Author.

  You acknowledge and agree that the Software and the
  accompanying Instructions are intellectual property of
  the Author, protected under U.S. copyright law. You further
  acknowledge and agree that all right, title and interest in
  and to the Software, are and shall remain with the Author.
  This License does not convey to you an interest in or to the
  Software, but only a limited and revocable right of use.

  THIS SOFTWARE IS LICENSED "AS IS," AND LICENSOR DISCLAIMS ANY
  AND ALL WARRANTIES, WHETHER EXPRESS OR IMPLIED, INCLUDING,
  WITHOUT LIMITATION, ANY IMPLIED WARRANTIES OF MERCHANTABILITY
  OR FITNESS FOR A PARTICULAR PURPOSE.

  Author's cumulative liability to you or any other party for
  any loss or damages resulting from any claims, demands, or
  actions arising out of or relating to this License shall not
  exceed the license fee paid (if any) to Author for the use of
  the Software. In no event shall Author be liable for any
  indirect, incidental, consequential, special, or exemplary
  damages or lost profits, even if Author has been advised of
  the possibility of such damages.

  This software and accompanying instructions are provided with
  restricted rights. Use, duplication or disclosure by the
  Government is subject to restrictions as set forth in
  subparagraph (c)(1)(ii) of The Rights in Technical Data and
  Computer Software clause at DFARS 252.227-7013 or
  subparagraphs (c)(1) and (2) of the Commercial Computer
  Software - Restricted Rights 48 CFR 52.227-19, as applicable.

{ --------------------------------------------------------------

  CHANGE HISTORY

  1.0.0 09-Feb-97 (AS)  Initial Development

  -------------------------------------------------------------- }


interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, StdCtrls,
  ProjectInfo, LabelEllipsis;

type
  TDialogType = (Splash, About);
  TSplashClass = class of TSplashScreen;
  TSplashScreen = class(TForm)
    FCompanyLogo: TImage;
    FTitle: TLabel;
    FLogoPanel: TPanel;
    FApplicationLogo: TImage;
    FApplicationPath: TLabelEllipsis;
    FDividerLine1: TBevel;
    FWindowFrame: TBevel;
    FVersionInfo: TLabelEllipsis;
    FCopyright: TLabel;
    Proj: TProjectInfo;
    FUserName: TLabel;
    FTimer: TTimer;
    LProgram: TLabel;
    LVersion: TLabel;
    procedure FormCreate(Sender: TObject);
    procedure FTimerTimer(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormPaint(Sender: TObject);
  private
    { Private declarations }
    FDialogType: TDialogType;
    FTransparentColor: TColor;
    FCompanyLogoList: TImageList;
    FApplicationLogoList: TImageList;
    procedure SetDialogType(const Value: TDialogType);
  public
    { Public declarations }
    property DialogType: TDialogType read FDialogType write SetDialogType;
    property Transparent: TColor read FTransparentColor write FTransparentColor default clWhite;
  end;

procedure ShowSplashScreen(const MyClass: TSplashClass; var MyObject; const MyDialogType: TDialogType; const MyTransparent: TColor);

implementation


{$R *.DFM}

{------------------------------------------------------------------------------
  Prepare information for the Splash Screen
------------------------------------------------------------------------------}
procedure TSplashScreen.FormCreate(Sender: TObject);

var
    TempReleaseStatus: string;

begin
    { Application name }
    FTitle.Caption := Proj.Title + ' ' + IntToStr(Proj.Version.vv);

    { Application path }
    FApplicationPath.Caption := Proj.ExeFileInfo.NamePath;

    { Product version, build status, and .EXE file timestamp }
    TempReleaseStatus := Proj.ReleaseStatus;
    if TempReleaseStatus <> '' then
        TempReleaseStatus := '(' + TempReleaseStatus + ') ';

    FVersionInfo.Caption := Proj.VersionString + ' ' +
                            TempReleaseStatus + '- ' +
                            DateTimeToStr(Proj.ExeFileInfo.Date);

    { Copyright info }
    FCopyright.Caption := Proj.Copyright;

    { Registered users name and license status }
    FUserName.Caption := 'This software may only be used by ' +
                         Proj.UserName + '.'#13 +
                         '(' + Proj.LicenseStatus  + ')';

end;

procedure TSplashScreen.FormDestroy(Sender: TObject);
begin
    FCompanyLogoList.Free;
    FApplicationLogoList.Free;
end;


{------------------------------------------------------------------------------
  Automatically close splash screen after time expired
  and prepare it so it can be used as an About box.
------------------------------------------------------------------------------}
procedure TSplashScreen.FTimerTimer(Sender: TObject);
begin
    FTimer.Enabled := false;                 // Prevent recurrent timer events
    Close;                                   // Close this form off the screen
end;

{------------------------------------------------------------------------------
  Define how this dialog will be used:
  Splash: borderless, without title bar and timed
  About: with border, title bar and without timer
------------------------------------------------------------------------------}
procedure TSplashScreen.SetDialogType(const Value: TDialogType);

var
    TempClientHeight: Integer;
    TempClientWidth: Integer;

begin
    if Value = Splash then
        begin
        TempClientHeight := ClientHeight;      // Remember available area before changes
        TempClientWidth := ClientWidth;
        BorderStyle := bsNone;                 // Remove title bar
        Height := Height - (ClientHeight - TempClientHeight);   // Reduce form for title bar
        Width := Width - (ClientWidth - TempClientWidth);
        FTimer.Enabled := true;                // SplashScreen will automatically close
        end
    else
        begin
        FTimer.Enabled := false;               // About box needs no timer
        TempClientHeight := ClientHeight;      // Remember available area before changes
        TempClientWidth := ClientWidth;
        BorderStyle := bsSingle;               // Provide title bar and "close" icon
        Height := Height + (TempClientHeight - ClientHeight);   // Enlarge form for title bar
        Width := Width + (TempClientWidth - ClientWidth);
        end
end;

{------------------------------------------------------------------------------
  Draw the transparent bitmaps for application and logo
------------------------------------------------------------------------------}
procedure TSplashScreen.FormPaint(Sender: TObject);
begin
    { Move company and product logos into an image list,
      and create transparent masks. }
    if (FCompanyLogoList = nil) and not FCompanyLogo.Picture.Bitmap.Empty then
        begin
        FCompanyLogoList :=  TImageList.CreateSize(FCompanyLogo.Width, FCompanyLogo.Height);
        FCompanyLogoList.AddMasked(FCompanyLogo.Picture.Bitmap, FTransparentColor);
        FCompanyLogoList.BkColor := Color;
        FCompanyLogo.Picture := nil;
        end;

    if (FApplicationLogoList = nil) and not FApplicationLogo.Picture.Bitmap.Empty then
        begin
        FApplicationLogoList := TImagelist.CreateSize(FApplicationLogo.Width, FApplicationLogo.Height);
        FApplicationLogoList.AddMasked(FApplicationLogo.Picture.Bitmap, FTransparentColor);
        FApplicationLogoList.BkColor := Color;
        FApplicationLogo.Picture := nil;
        end;

    { Draw the imagelist into the space provided by the former image }
    if FCompanyLogoList <> nil then FCompanyLogoList.Draw(FCompanyLogo.Canvas, 0, 0, 0);
    if FApplicationLogoList <> nil then FApplicationLogoList.Draw(FApplicationLogo.Canvas, 0, 0, 0);
end;

{------------------------------------------------------------------------------
  Procedure to create SplashScreen from Project File during Application Start
------------------------------------------------------------------------------}
procedure ShowSplashScreen(const MyClass: TSplashClass; var MyObject ; const MyDialogType: TDialogType; const MyTransparent: TColor);
begin
    if TSplashScreen(MyObject) = nil then
        TSplashScreen(MyObject) := MyClass.Create(Application);
    TSplashScreen(MyObject).Transparent := MyTransparent;
    TSplashScreen(MyObject).DialogType := MyDialogType;
    TSplashScreen(MyObject).Show;
end;

end.
