unit E_StgStr;
{-------------} Interface {--------------------}
Uses
     Ole2, SysUtils, Windows;
(*******************************************************************
                           E_StgStor
    Ole Structured Storage Helper Routines.
*******************************************************************)
Type
      TStructuredStorage = Class
       Private
         FFileName : String;
         Destructor Destroy; Override;
       Public
         hr          : HResult;
         StorageFile : IStorage;
         Constructor Create ; Virtual;

         Function IsStorageFile(Const FileName : String) : Boolean;

         Function CreateStorageFile(Const FileName : String;
                                          grfMode  : LongInt) : Boolean;

         Function OpenStorageFile(Const FileName : String;
                                        grfMode  : LongInt) : Boolean;

         Function CreateSubStorage(Const FileName    : String;
                                         grfMode     : LongInt;
                                   Var   Storage     : IStorage;
                                   Var   SubStorage  : IStorage) : Boolean;

         Function OpenSubStorage(Const FileName   : String;
                                       grfMode    : LongInt;
                                 Var   Storage    : IStorage;
                                 Var   SubStorage : IStorage) : Boolean;

         Procedure DoneStorage(Var Storage : IStorage);

         Function CreateStream(Var   Storage     : IStorage;
                               Const StreamName  : String;
                                     grfMode     : LongInt;
                               Var   Stream      : IStream) : Boolean;

         Function OpenStream(Var   Storage     : IStorage;
                             Const StreamName  : String;
                                   grfMode     : LongInt;
                             Var   Stream      : IStream) : Boolean;

         Procedure DoneStream(Var Stream : IStream);

         Function WriteString(Var   Stream : IStream;
                              Const S      : String) : Boolean;
         Function ReadString(Var Stream    : IStream;
                             Var S         : String) : Boolean;

         Function WriteInt(Var Stream : IStream;
                               TInt   : Integer) : Boolean;

         Function ReadInt(Var Stream    : IStream;
                          Var TInt      : Integer) : Boolean;
      end;
{-------------} Implementation {---------------}
(*
  STGM_DIRECT           = $00000000;
  STGM_TRANSACTED       = $00010000;
  STGM_SIMPLE           = $08000000;

  STGM_READ             = $00000000;
  STGM_WRITE            = $00000001;
  STGM_READWRITE        = $00000002;

  STGM_SHARE_DENY_NONE  = $00000040;
  STGM_SHARE_DENY_READ  = $00000030;
  STGM_SHARE_DENY_WRITE = $00000020;
  STGM_SHARE_EXCLUSIVE  = $00000010;

  STGM_PRIORITY         = $00040000;
  STGM_DELETEONRELEASE  = $04000000;

  STGM_CREATE           = $00001000;
  STGM_CONVERT          = $00020000;
  STGM_FAILIFTHERE      = $00000000;
*)
(*******************************************************************
                           StrToUniCode
 Converts a string to unicode and  returns the size of the allocated
 string in NewSize.  Y0u have to free up this memory yourself.
*******************************************************************)
function StrToUniCode(    S       : string;
                       Var NewSize : Integer): PWideChar;
Var
    Size : Integer;
    P    : PWideChar;
begin
  Size := Length(S);
  NewSize := Size * 2;
  P := VirtualAlloc(nil, Size, Mem_Commit, Page_ReadWrite);
  MultiByteToWideChar(CP_ACP, 0, PChar(S), Size, P, NewSize);
  Result := P;
end;
(*******************************************************************
 Structured Storage Object to help relieve some of pain when dealing
 with some of it's peculiarities.
*******************************************************************)
Constructor TStructuredStorage.Create;
begin
  Inherited Create;
  StorageFile:=Nil;
end;
Destructor TStructuredStorage.Destroy;
begin
  If Assigned(StorageFile) then
    StorageFile.Release;
  Inherited Destroy;
end;
(*******************************************************************
                           CreateStorageFile
 Create an Ole Structured Storage File
*******************************************************************)
Function TStructuredStorage.CreateStorageFile(Const FileName : String;
                                                    grfMode  : LongInt) : Boolean;
Var
    PSize : Integer;
    P     : PWideChar;
begin
  Result:=False;
  try
    P:=StrToUniCode(FileName,PSize);
    Hr:=StgCreateDocFile(P,grfMode,0,StorageFile);
    Result:=Hr=S_OK;
  finally
    VirtualFree(P,PSize,MEM_RELEASE);
  end;
end;
(*******************************************************************
                           CreateSubStorage
 Create Sub Storage within a Storage.
*******************************************************************)
Function TStructuredStorage.CreateSubStorage(Const FileName    : String;
                                                   grfMode     : LongInt;
                                             Var   Storage     : IStorage;
                                             Var   SubStorage  : IStorage) : Boolean;
Var
    PSize : Integer;
    P     : PWideChar;
begin
  Result:=False;
  Try
     P:=StrToUniCode(FileName,PSize);
     hr:=Storage.CreateStorage(P,grfMode,0,0,SubStorage);
     If Ole2.Failed(hr) then
       {Raise}
     else
       Result:=True;
  Finally
    VirtualFree(P,PSize,MEM_RELEASE);
  end;
end;
(*******************************************************************
                           IsStorageFile
 See if FileName is indeed a structured storage file.
*******************************************************************)
Function TStructuredStorage.IsStorageFile(Const FileName : String) : Boolean;
Var
    PSize : Integer;
    P     : PWideChar;
begin
  Result:=False;
  try
    P:=StrToUniCode(FileName,PSize);
    hr:=StgIsStorageFile(p);
    Result:=NOT Ole2.Failed(hr);
  finally
    VirtualFree(P,PSize,MEM_RELEASE);
  end;
end;
(*******************************************************************
                           OpenStorageFile
 Open an Ole Structured Storage File
*******************************************************************)
Function TStructuredStorage.OpenStorageFile(Const FileName : String;
                                                  grfMode  : LongInt) : Boolean;
Var
    PSize : Integer;
    P     : PWideChar;
begin
  Result:=False;
  try
    P:=StrToUniCode(FileName,PSize);
    hr:=StgIsStorageFile(p);
    If Ole2.Failed(hr) then
      {Raise}
    else
    begin
      Hr:=StgOpenStorage(P,Nil,grfMode,Nil,LongInt(Nil),StorageFile);
      Result:=Hr=S_OK;
    end;
  finally
    VirtualFree(P,PSize,MEM_RELEASE);
  end;
end;
(*******************************************************************
                           OpenSubStorage
 Open an Ole Structured sub-Storage within a storage
*******************************************************************)
Function TStructuredStorage.OpenSubStorage(Const FileName   : String;
                                                 grfMode    : LongInt;
                                           Var   Storage    : IStorage;
                                           Var   SubStorage : IStorage) : Boolean;
Var
    PSize : Integer;
    P     : PWideChar;
begin
  Result:=False;
  try
    P:=StrToUniCode(FileName,PSize);
    hr:=Storage.OpenStorage(P,Nil,grfMode,Nil,LongInt(Nil),SubStorage);
    If Ole2.Failed(hr) then
      {Raise}
    else
      Result:=True;
  finally
    VirtualFree(P,PSize,MEM_RELEASE);
  end;
end;
(*******************************************************************
 Free up an OLE storage.
*******************************************************************)
Procedure TStructuredStorage.DoneStorage(Var Storage : IStorage);
begin
  If Assigned(Storage) then
    Storage.Release;
  Storage:=Nil;
end;
(*******************************************************************
 Create an Ole Stream within the current storage
*******************************************************************)
Function TStructuredStorage.CreateStream(Var   Storage     : IStorage;
                                         Const StreamName  : String;
                                               grfMode     : LongInt;
                                         Var   Stream      : IStream) : Boolean;
Var
    PSize : Integer;
    P     : PWideChar;
begin
  Result:=False;
  try
    P:=StrToUniCode(StreamName,PSize);
    Hr:=Storage.CreateStream(P,grfMode,0,0,Stream);
    If Ole2.Failed(hr) then
      {Raise}
    else
      Result:=True;
  finally
    VirtualFree(P,PSize,MEM_RELEASE);
  end;
end;
(*******************************************************************
 Create an Ole Stream within the current storage
*******************************************************************)
Function TStructuredStorage.OpenStream(Var   Storage     : IStorage;
                                       Const StreamName  : String;
                                             grfMode     : LongInt;
                                       Var   Stream      : IStream) : Boolean;
Var
    PSize : Integer;
    P     : PWideChar;
begin
  Result:=False;
  try
    P:=StrToUniCode(StreamName,PSize);
    Hr:=Storage.OpenStream(P,Nil,grfMode,0,Stream);
    If Ole2.Failed(hr) then
      {Raise}
    else
      Result:=True;
  finally
    VirtualFree(P,PSize,MEM_RELEASE);
  end;
end;
(*******************************************************************
 Free up an OLE stream.
*******************************************************************)
Procedure TStructuredStorage.DoneStream(Var Stream : IStream);
begin
  If Assigned(Stream) then
    Stream.Release;
  Stream:=Nil;
end;
(*******************************************************************
                           WriteString
 Write a string to an opened storage stream.
*******************************************************************)
Function TStructuredStorage.WriteString(Var   Stream : IStream;
                                        Const S      : String) : Boolean;
Var
    Size : LongInt;
begin
  Result:=False;
  If WriteInt(Stream,Length(S)) then
  begin
    Size:=0;
    hr:=Stream.Write(PChar(S),Length(S),@Size);
    If Ole2.Failed(hr) then
      {Raise}
    else
      Result:=Size=Length(S);
  end;
end;
(*******************************************************************
                           ReadString
 Read a string from an opened storage stream.
*******************************************************************)
Function TStructuredStorage.ReadString(Var Stream    : IStream;
                                       Var S         : String) : Boolean;
Var
    StrSize : Integer;
    Size    : LongInt;
    Ps      : PChar;
begin
  Result:=False;
  S:='';
  If ReadInt(Stream,StrSize) then
  begin
    Size:=0;
    try
       GetMem(Ps,StrSize+1);
       hr:=Stream.Read(Ps,StrSize,@Size);
       If Ole2.Failed(hr) then
        {Raise}
       else If Size>0 then
       begin
         S:=String(Ps);
         SetLength(S,Size);
         Result:=Size=StrSize;
       end;
    finally
      FreeMem(Ps,StrSize+1);
    end;
  end;
end;
(*******************************************************************
                           WriteInt
 Write an Integer to the stream.
*******************************************************************)
Function TStructuredStorage.WriteInt(Var Stream : IStream;
                                         TInt   : Integer) : Boolean;
Var
    Size : LongInt;
begin
  Result:=False;
  Size:=0;
  hr:=Stream.Write(@TInt,SizeOf(Integer),@Size);
  If Ole2.Failed(hr) then
    {Raise}
  else
    Result:=Size=SizeOf(Integer);
end;
(*******************************************************************
                           ReadInt
 Read a string from an opened storage stream.
*******************************************************************)
Function TStructuredStorage.ReadInt(Var Stream    : IStream;
                                    Var TInt      : Integer) : Boolean;
Var
    Size : LongInt;
begin
  Result:=False;
  Size:=0;
  hr:=Stream.Read(@Tint,SizeOf(Integer),@Size);
  If Ole2.Failed(hr) then
    {Raise}
  else
    Result:=Size=SizeOf(Integer);
end;
{--------------------------END OF FILE--------------------------------}
end.
