/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "_termint.h"
#include "_termctl.h"
#include "parsedef.h"
#include "parsevar.h"
#include "stdmac.h"

#ifndef PB_SDK
	#include <string.h>
#else
	#include "pblibc.h"
#endif

/*
 * ^ f   a n d   ^ k   p a r s e r   c l a s s
 * 
 * implementation of the ^f anf ^k parser class (with additional features)
*/
Boolean
textcode_interp::init(uchar aSOS)
{
	Boolean retval = False;

	if( aSOS == 0x06 || aSOS == 0x0b )
	{
		term_interp::init(aSOS);
		m_kCode       = False;              // ^K code with parameters
		m_action      = PDEF_ACTION_ERROR;  // no default action
		m_maxLen      = 0;                  // max len is preset to 0
		m_justifyChar = '@';                // default to left justification
		retval        = True;
	}

	return retval;

}

void
textcode_interp::handle(uchar aChar)
{
	char     result[512];
	Boolean  fc;
	int      cc       = PDEF_ERROR;
	int      justCode = PDEF_JUST_LEFT;

	// check if this is the second input character (if it is, and
	// the first one is ^K, check for the special parameter codes)
	if( 1 == m_buff->getCount() && 0x0b == m_buff->bp()[0] )
	{
		switch( aChar )
		{
			case ']': m_action = PDEF_ACTION_PROMPT;  break;
			case '!': m_action = PDEF_ACTION_DISPLAY; break;
			case '~': m_action = PDEF_ACTION_PEXEC;   break;
			case '$': m_action = PDEF_ACTION_EXEC;    break;
			default : goto ProcessNormal;  // skip the whole charade
		}
		// we only get here if we have recognized the char
		m_kCode = True;
		m_buff->put(aChar);
		return;
	}

	// are we processing a special ^K cc with parameters
	if( m_kCode )
	{
		if( '|' == aChar )
		{  // this is the end of sequence
			m_buff->put(EOS);
			if( parse_action(m_action, &m_buff->bp()[2]) )
			{
				// we're fine here, signal end of processing
				m_eos = True;
			}
			else
			{  // action failed, signal this too
				m_fail = True;
				m_buff->put(aChar);
			}
		}
		else
		{
			m_buff->put(aChar);
			if( aChar < 32 ) m_fail = True; // reasonable
		}
		return; // we are either ok or failed here, return in any case
	}

ProcessNormal:
	// we are accumulating justification characters
	if( '@' == aChar || '#' == aChar || '%' == aChar )
	{
		m_buff->put(aChar);
		m_maxLen++;
		m_justifyChar = aChar;
		return;
	}

	// add 2 for the bounding chars
	if( 0 != m_maxLen ) m_maxLen += 2;
	fc = (6 == m_buff->bp()[0]) ? True : False;

	// ok, we have a terminating character, interpret it
	switch( aChar )
	{
		// codes are dependent on the first character (^F = 6, ^K = 11)
		case 'A': cc = fc ? PDEF_USR_NAME     : PDEF_SYS_TOTALCALLS; break;
		case 'B': cc = fc ? PDEF_USR_CITY     : PDEF_SYS_LASTCALLER; break;
		case 'C': cc = fc ? PDEF_USR_PASSWD   : PDEF_MSG_HMBTOTAL;   break;
		case 'D': cc = fc ? PDEF_USR_DATA     : PDEF_MSG_HMBLOW;     break;
		case 'E': cc = fc ? PDEF_USR_VOICE    : PDEF_MSG_LAST;       break;
		case 'F': cc = fc ? PDEF_USR_LASTDATE : PDEF_SYS_PAGES;      break;
		case 'G': cc = fc ? PDEF_USR_LASTTIME : PDEF_SYS_WEEKDAY;    break;
		case 'H': cc = fc ? PDEF_USR_FLAGSA   : PDEF_SYS_USERS;      break;
		case 'I': cc = fc ? PDEF_USR_FLAGSB   : PDEF_SYS_TIME;       break;
		case 'J': cc = fc ? PDEF_USR_FLAGSC   : PDEF_SYS_DATE;       break;
		case 'K': cc = fc ? PDEF_USR_FLAGSD   : PDEF_USR_MINONLINE;  break;
		case 'L': cc = fc ? PDEF_USR_CREDIT   : PDEF_USR_SECONLINE;  break;
		case 'M': cc = fc ? PDEF_USR_MWRITTEN : PDEF_USR_TONLINE;    break;
		case 'N': cc = fc ? PDEF_USR_MLASTREAD: PDEF_USR_SECTODAY;   break;
		case 'O': cc = fc ? PDEF_USR_LEVEL    : PDEF_USR_TLEFT;      break;
		case 'P': cc = fc ? PDEF_USR_CALLS    : PDEF_SYS_VERSION;    break;
		case 'Q': cc = fc ? PDEF_USR_UPLOADS  : PDEF_USR_TLIMIT;     break;
		case 'R': cc = fc ? PDEF_USR_KBUP     : PDEF_SYS_BAUD;       break;
		case 'S': cc = fc ? PDEF_USR_DOWNS    : PDEF_SYS_SHORTDOW;   break;
		case 'T': cc = fc ? PDEF_USR_KBDOWN   : PDEF_USR_KBLIMIT;    break;
		case 'U': cc = fc ? PDEF_USR_TTOTAL   : PDEF_SYS_EVENTMIN;   break;
		case 'V': cc = fc ? PDEF_USR_NUMLINES : PDEF_USR_FORWARD;    break;
		case 'W': cc = fc ? PDEF_USR_FIRST    : PDEF_SYS_NODE;       break;
		case 'Y': cc = fc ? PDEF_USR_PAUSE    : PDEF_USR_MAREA;      break;
		case 'Z': cc = fc ? PDEF_USR_CLRSCR   : PDEF_USR_FAREA;      break;
		case '0': cc = fc ? PDEF_USR_FSED     : PDEF_MSG_TOTAL;      break;
		case '1': cc = fc ? PDEF_USR_ALIAS    : PDEF_USR_MAREANO;    break;
		case '2': cc = fc ? PDEF_USR_STACKING : PDEF_USR_FAREANO;    break;
		case '(': cc = fc ? PDEF_USR_FGROUP   : PDEF_USR_LANGUAGE;   break;
		case '3': cc = fc ? PDEF_USR_IBM      : PDEF_USR_LASTPWD;    break;
		case '4': cc = fc ? PDEF_USR_STATE    : PDEF_USR_LASTFILES;  break;
		case '5': cc = fc ? PDEF_USR_DOB      : PDEF_USR_CHECKMAIL;  break;
		case '6': cc = fc ? PDEF_USR_EXPDATE  : PDEF_USR_CHECKFILES; break;
		case '7': cc = fc ? PDEF_USR_EXPDAYS  : PDEF_TB_TDIFF;       break;
		case '8': cc = fc ? PDEF_USR_AVATAR   : PDEF_TB_KBDIFF;      break;
		case '9': cc = fc ? PDEF_USR_AVTPLUS  : PDEF_TB_TGET;        break;
		case '\"':cc = fc ? PDEF_USR_COUNTRY  : PDEF_TB_KBGET;       break;
		case ':': cc = fc ? PDEF_USR_FIRSTCALL: PDEF_TB_TPUT;        break;
		case '\'':cc = fc ? PDEF_USR_ADDRESS2 : PDEF_TB_KBPUT;       break;
		case '&': cc = fc ? PDEF_USR_ADDRESS3 : PDEF_TB_TLOAN;       break;
		case '`': cc = fc ? PDEF_USR_SEX      : PDEF_TB_KBLOAN;      break;
		case ')': cc = fc ? PDEF_USR_MGROUP   : PDEF_TB_LAST;        break;
		case '+': cc = fc ? PDEF_USR_MGROUPNO : PDEF_TB_TPAY;        break;
		case '=': cc = fc ? PDEF_USR_FAX      : PDEF_TB_KBPAY;       break;
		// here we have a duplicate cc (we can add another one!)
		case '^': cc = fc ? PDEF_SYS_ACTIVE: PDEF_SYS_ACTIVE; break;
		// this one is special, ^K[xx is a color cc, make sure it's not
		// processed (as it won't be if left like this)
		case '[': if( fc ) cc = PDEF_USR_KBLEFT; break;
		// these are only valid for ^F sos (the ^K specifies action codes,
		// which have already been parsed (never gets to this point!), duh)
		case ']': if( fc ) cc = PDEF_USR_COMMENT;   break;
		case '~': if( fc ) cc = PDEF_USR_DELAY;     break;
		case '!': if( fc ) cc = PDEF_USR_DELAYLEFT; break;
		case '$': if( fc ) cc = PDEF_USR_ADDRESS1;  break;
		// the ^K codes of the following are actions, not display text
		case '/':
			if( fc ) cc = PDEF_USR_FLAGS;
			else m_action = PDEF_ACTION_CLREOL;
			break;
		case 'X':
			if( fc ) cc = PDEF_USR_ANSI;
			else m_action = PDEF_ACTION_HANGUP;
			break;
		case '*':
			if( fc ) cc = PDEF_USR_FGROUPNO;
			else m_action = PDEF_ACTION_PAUSE;
			break;
		default: // don't know how to handle this, pass thru
			m_buff->put(aChar);
			m_fail = True;
			return;
	}

	// see if we are performing an action sequence
	if( PDEF_ACTION_ERROR != m_action )
	{
		if( !parse_action(m_action, 0) )
		{
			m_buff->put(aChar);
			m_fail = True;
		}
		else
		{
			m_eos = True;
		}
		return;
	}

	// note that the function will return 0 on parser_error
	switch( m_justifyChar )
	{
		case '#': justCode = PDEF_JUST_RIGHT; break;
		case '%': justCode = PDEF_JUST_CENTER; break;
		default : justCode = PDEF_JUST_LEFT; break;
	}

	if( 0 != parse_token(result, cc, justCode, m_maxLen) )
	{
		// write the string and signal everything is ok
		m_ctrl->put(result, strlen(result));
		m_eos = True;
	}
	else
	{ // something is wrong, get out
		m_buff->put(aChar);
		m_fail = True;
	}
}
