/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "profile.h"
#include "file.h"
#include "str.h"

#ifndef PB_SDK
	#include <assert.h>
	#include <string.h>
	#include <ctype.h>
	#include <stdlib.h>
#else
	#include "pblibc.h"
#endif

/*
 * the entry section must always be present when reading and writing.
 *
 * when reading, the section name parameter can be NULL. in that case,
 * sections are ignored and the entry will be the first one matching the
 * request.
 *
 * when writing, the section name parameter can be NULL. in that case,
 * sections will be ignored and the entry writing algorithm is standard.
 *
 * when writing, the current file is processed and if the entry is found,
 * the new value replaces the old. if not found, a new entry is created.
*/

zProfile::zProfile():
	m_foundSection(False),
	m_mustCopy(False),
	m_useSection(True),
	m_fp(0),
	m_fpCopy(0)
{
	m_name[0] = EOS;
}

zProfile::~zProfile()
{
	Close();
}

/*
 * assigns an .INI file to the object. this must be called before any other
 * processing is attempted. note that this only sets the name. the file is
 * not actually opened or created at this point. the fname argument is the
 * file name. if 'fext' is not NULL, it must contain the default extension
 * to use (defaults to ".ini"). note that the string must include the leading
 * period. in that case, the extension will be forced on the 'fname' param.
 * this means that you can use Assign(argv[0]) to assign the .INI file name
 * to be the same as your program name, with the ".INI" extension.
*/
void
zProfile::Assign( const char *fname, const char *fext )
{
	if( fext ) file_chext(m_name, fname, fext);
	else strcpy(m_name, fname);
}
/*
 * closes the open .INI file (if any). also removes it if length is 0 bytes
*/
void
zProfile::Close()
{
	if( m_fp )
	{
		fclose(m_fp);
		m_fp = 0;
		file_killempty(m_name);
	}
}

/*
 * finds a section in the file. it will return the section name. if the
 * 'm_mustCopy' flag is set and 'm_fpCopy' is not NULL, it will also copy
 * all lines read up to the (but not including) the section name to the
 * new file. note that the string matching is not sensitive and that the
 * parameter 'szSection' must not contain the enclosing brackets. if the
 * required section is not found, NULL is returned, otherwise the well-
 * formed name "[Section]" - which will be in a static buffer. all lines
 * that begin with ';' or '#' are considered comments and will not be parsed.
 * this routine will not rewind() the file, so to start the search from the
 * the calling routine must perform that.
*/
char*
zProfile::FindSection(char *szSection, char *buf, size_t buflen)
{
	char    section[80];
	char   *p;
	size_t  slen;

	assert( 0 != m_fp );
	assert( 0 != szSection );
	assert( m_mustCopy ? 0 != m_fpCopy : 1 );

	sprintf(section, "[%s]", szSection);
	slen = strlen(section);

	fgets(buf, buflen, m_fp);
	while( !feof(m_fp) )
	{
		p = strskpw(buf);
		if( '[' == *p && MEMIEQU(p, section, slen) ) return buf;
		if( m_mustCopy ) fputs(buf, m_fpCopy);
		fgets(buf, buflen, m_fp);
	}
	return NULL;
}

/*
 * finds an entry. the search starts from the current position in the .INI
 * file. if the 'm_useSection' flag is set, a new section found will terminate
 * the seatch. otherwise, the file will be searched ignoring sections. if
 * the 'm_mustCopy' flag is set, each line is copied to the 'm_fpCopy' file as
 * it is read. comparison are not case-sensitive. on success, the whole
 * line (entry with the possible value) are returned and they will not be
 * copied to the output. on failure to find the entry, NULL is returned.
 * note that if a new section is found, it will not be copied either. instead
 * it will be placed in the buffer for later use and m_foundSection will be
 * set to True. lines that start with ';' or '#' will be ignored completely.
*/
char*
zProfile::FindEntry(char *szEntry, char *buf, size_t buflen)
{
	size_t  slen = strlen(szEntry);
	char   *p;

	assert( 0 != m_fp );
	assert( m_mustCopy ? 0 != m_fpCopy : 1 );

	m_foundSection = False;
	fgets(buf, buflen, m_fp);
	while( !feof(m_fp) )
	{
		p = strskpw(buf);
		if( m_useSection && '[' == *p )
		{
			m_foundSection = True;
			break;
		}
		if( ';' != *p && '#' != *p && MEMIEQU(p, szEntry, slen) ) return buf;
		if( m_mustCopy ) fputs(buf, m_fpCopy);
		fgets(buf, buflen, m_fp);
	}
	return NULL;
}

/*
 * opens the .INI file and prepares it for reading. you must use Assign()
 * before calling this routine. if the 'bCreate' parameter is False (default),
 * attempting to open a non-existant file will fail. otherwise, the file
 * will be created. note that if a file is not open, all Read/Write functions
 * will return the default values (not errors).
*/
Boolean
zProfile::Open( Boolean bCreate )
{
	char *mode = "r+t";

	assert( m_name[0] != EOS );  // we have a file name
	assert( m_fp == 0 );  // we don't have an open file

	if( !file_exist(m_name) )
	{
		if( bCreate ) mode = "w+t";
		else return False;
	}

	return Boolean( 0 != (m_fp = fopen(m_name, mode)) );
}


/*
 * reads a boolean value from the .INI file. the supported strings are:
 *
 *              true result    |    false result
 *          ------------------------------------------
 *                  Y(es)      |       N(o)
 *                  T(rue)     |       F(alse)
 *                  1          |        0
 *           (keyword present) |       n/a
*/
Boolean
zProfile::ReadBool(char *section, char *entry, Boolean *dest, Boolean def)
{
	char buf[255];

	assert(dest != 0);

	*dest = def;
	if( ReadString(section, entry, buf, 0) )
	{
		strlwr(buf);
		if( STREQU(buf, "no") || STREQU(buf, "false") || STREQU(buf, "0") )
		{
			*dest = False;
		}
		else if( EOS == buf[0] || STREQU(buf, "yes")
				|| STREQU(buf, "true") || STREQU(buf, "1") )
		{
			*dest = True;
		}
	}
	return *dest;
}

/*
 * reads a short integer (if signed wanted, cast the result to signed)
*/
short
zProfile::ReadShort(char *section, char *entry, short *dest, short def)
{
	long value;

	assert(dest != 0);
	*dest = (short)ReadLong(section, entry, &value, (long)def);
	return *dest;
}


/*
 * reads a long integer from the .INI file, supports hex and octal values
 * (see the docs for the strtol() routine)
*/
long
zProfile::ReadLong(char *section, char *entry, long *dest, long def)
{
	char buf[255];

	assert(dest != 0);
	if( ReadString(section, entry, buf, 0) ) *dest = strtol(buf, 0, 0);
	else *dest = def;

	return *dest;
}

/*
 * read a string from the .INI file. 'szSection' can be NULL, in which case
 * the sections will be ignored. if a match is found, the value of the entry
 * is returned ("", if just the keyword and nothing else). any failure or
 * error condition will result in 'szDefault' being returned.
*/
char*
zProfile::ReadString(char *section, char *entry, char *dest, char *def)
{
	Boolean  found = False;
	char    *p, buf[BUFSIZE];

	assert( 0 != entry );
	assert( 0 != dest  );

	if( 0 != m_fp )
	{
		rewind(m_fp);
		m_mustCopy = False;
		m_useSection = Boolean( 0 != section );
		if( (!m_useSection || FindSection(section, buf, sizeof(buf)))
			&& FindEntry(entry, buf, sizeof(buf)) )
		{
			if( 0 != (p = strchr(buf, '=')) )
			{
				if( EOS != (p = strskpw(p+1)) )
				{
					if( '\n' == LAST_CHAR(p) ) LAST_CHAR(p) = EOS;
					memmove(buf, p, strlen(p) + 1);
				}
				else buf[0] = EOS;
			}
			else buf[0] = EOS;
			found = True;
		}
	}

	if( found ) strcpy(dest, buf);
	else strcpy(dest, def);

	return dest;
}

/*
 * writes a boolean value to the .INI file
*/
Boolean
zProfile::WriteBool(char *szSection, char *szEntry, Boolean bValue)
{
	char *bString = bValue ? "True" : "False";
	return WriteString(szSection, szEntry, bString);
}

/*
 * writes a short integer to the .INI file
*/
Boolean
zProfile::WriteShort(char *szSection, char *szEntry, short nValue)
{
	return WriteLong(szSection, szEntry, (long)nValue);
}

/*
 * writes a long integer to the .INI file
*/
Boolean
zProfile::WriteLong(char *szSection, char *szEntry, long nValue)
{
	char szBuf[20];

	sprintf(szBuf, "%ld", nValue);
	return WriteString(szSection, szEntry, szBuf);
}

/*
 * this writes an entry to the .INI file. if szSection is NULL, sections will
 * be ignored. if the szEntry is found, the new value will replace the old
 * one (if the new value is NULL, the entry will be deleted). to create
 * entries without a value (just keywords), use "" as the value. if the entry
 * does not exist, it will be created (at the end of the section, if enabled
 * or at the end of the file)
*/
Boolean
zProfile::WriteString(char *szSection, char *szEntry, char *szValue)
{
	Boolean retval = False;
	char    tempName[MAXPATH];
	char    new_buf[BUFSIZE], buf[BUFSIZE];
	Boolean foundEntry;

	assert( 0 != szEntry );

	if( m_fp )
	{
		rewind(m_fp);
		m_mustCopy = True;
		m_useSection = Boolean( 0 != szSection );
		file_chext(tempName, m_name, ".tmp");
		if( 0 != (m_fpCopy = fopen(tempName, "wt")) )
		{
			/*
			 * locate the section. if found, copy to new file and
			 * save the last char on the line (for later checks!)
			*/
			if( m_useSection )
			{
				if( FindSection(szSection, buf, sizeof(buf)) )
				{
					fputs(buf, m_fpCopy);
				}
				else
				{
					sprintf(buf, "[%s]\n", szSection);
					fputs(buf, m_fpCopy);
				}
			}
			/*
			 * find the entry. if found: if the new value is NULL,
			 * do nothing (this will delete the entry), otherwise
			 * is the new value is empty string, just copy the entry
			 * name (without the value), otherwise - replace the old
			 * value with the new one! if entry not found, just
			 * write the new one.
			*/
			foundEntry = Boolean(0 != FindEntry(szEntry, buf, sizeof(buf)));
			if( 0 != szValue )
			{
				if( EOS == szValue[0] ) sprintf(new_buf, "%s\n", szEntry);
				else sprintf(new_buf, "%s = %s\n", szEntry, szValue);
				fputs(new_buf, m_fpCopy);
				if( !foundEntry && m_foundSection ) fputs(buf, m_fpCopy);
			}
			/*
			 * copy the rest of the file and then cleanup the mess!
			*/
			fgets(buf, sizeof(buf), m_fp);
			while( !feof(m_fp) )
			{
				fputs(buf, m_fpCopy);
				fgets(buf, sizeof(buf), m_fp);
			}
			fclose(m_fpCopy);
			fclose(m_fp);
			file_remove(m_name);
			rename(tempName, m_name);
			m_fpCopy = 0;
			m_fp = 0;
			Open();
			retval = True;
		}
	}

	return retval;
}
