/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "edit.h"
#include "stdmac.h"
#include "kbcodes.h"
#include "utils.h"
#include "comdef.h"

#ifndef PB_SDK
	#include <string.h>
	#include <conio.h>
	#include <ctype.h>
	#include "compiler.h"
#else
	#include "pblibc.h"
    #include "pblsdk.h"
#endif

char zEditLine::leftArrow  = '';
char zEditLine::rightArrow = '';

zEditLine::zEditLine(const zRect &bounds, short aMaxLen):
	data(new char [aMaxLen + 1]),
    maxLen(aMaxLen),
    curPos(0),
    firstPos(0),
    startSel(0),
    endSel(0),
	focused(False),
	mustPatch(False)
{
	static const char *cpzEditLine =  //......................default palette
		"\x1E" //.....................active foreground color, yellow on blue
		"\x00" //....................................................not used
		"\x31" //.......................active background color, blue on cyan
		""    //.........................active background fill character, 
		"\x31" //........................active selection color, blue on cyan
		"\x00" //....................................................not used
		"\x13" //.....................inactive foreground color, cyan on blue
		"\x00" //....................................................not used
		"\x19" //...............inactive background color, light blue on blue
		"";   //.......................inactive background fill character, 

	origin.x = bounds.a.x;
	origin.y = bounds.a.y;
	size.x = bounds.b.x - bounds.a.x;
	size.y = bounds.b.y - bounds.b.y;
	*data = EOS;
	if( size.x < 4 ) size.x = 4;
    setPalette(cpzEditLine);
}

zEditLine::~zEditLine()
{
	if( data ) delete[] data;
}

// get the string
void
zEditLine::getData(void *ptr)
{
	strcpy((char *)ptr, data);
}

void
zEditLine::setData(void *ptr)
{
	strncpy(data, (char *)ptr, maxLen);
    data[maxLen] = EOS;
    selectAll();
}

void
zEditLine::drawView()
{
	Boolean state = focused;
	focused = True;
	draw();
	focused = state;
}

// True if the line can scroll in the direction
Boolean
zEditLine::canScroll(Boolean left)
{
	if( left ) return Boolean(firstPos > 0);
	else return Boolean(strlen(data) - firstPos > size.x - 2);
}

// redraws the current line with scrolling indicators if necessary
void
zEditLine::draw()
{
	char   buf[256];
    short  leftSel, rightSel;
    size_t slen;
    char   backChar  = (True == focused) ? getPalette(3) : getPalette(9);
    uchar  foreColor = (True == focused) ? getPalette(0) : getPalette(6);
    uchar  backColor = (True == focused) ? getPalette(2) : getPalette(8);

    // construct the string as it would appear (with indicators)
    slen = min(size.x - 2, (int)strlen(data+firstPos));
    memset(buf, backChar, size.x);
	for( size_t i = 0; i < slen; ++i ) buf[i] = OnDraw(*(data+firstPos+i));
	buf[size.x - 2] = EOS;

    // check to see if something is selected currently
    if( focused )
    {
		leftSel  = startSel - firstPos;
        rightSel = endSel - firstPos;
		leftSel  = max((short)0, leftSel);
		rightSel = min(short(size.x-2), rightSel);
	}

    // print the left leading character
	gotoxy(origin.x, origin.y);
	textattr(backColor);
	if( canScroll(True) ) putch( leftArrow );
	else putch(backChar);

	textattr(foreColor);
    buf[slen] = EOS;
    if( !focused || !(leftSel < rightSel) ) cprintf(buf);
    else
    { // we might have a selection to highlight
    	short pos = 0;

		while( pos < leftSel ) putch(buf[pos++]);
		textattr(getPalette(4));
		while( pos < rightSel) putch(buf[pos++]);
		textattr(foreColor);
		while( pos < slen ) putch(buf[pos++]);
	}

    if( slen < size.x - 2 )
    { // fill with the background if necessary here
		textattr(backColor);
        buf[slen] = backChar;
        cprintf(&buf[slen]);
	}

    // print the right trailing char (arrow if necessary)
	textattr(backColor);
	if( canScroll(False) ) putch( rightArrow );
	else putch(backChar);

    // sync the cursor
	gotoxy(origin.x + curPos - firstPos + 1, origin.y);
}

// handle keystrokes
void
zEditLine::handle(ushort aKeyCode)
{
	static Boolean extendKey   = False;
    static Boolean extendBlock = False;
    static Boolean mustKeep    = False;

	if( !focused )
	{
		return;
    }
	else if( kbCtlQ == aKeyCode )
	{	// extended key movements
		extendKey = True;
        return;
    }
	else if( kbCtlK == aKeyCode )
	{	// extended selection movements
    	extendBlock = True;
        return;
	}

	switch( CtrlToArrow(aKeyCode) )
	{
		case kbLeft     : if( curPos ) curPos--; break;
		case kbRight    : if( curPos < strlen(data) ) curPos++; break;
		case kbIns      : insertMode = Boolean(!insertMode); break;
		case kbCtlT     : zapWord(); break;
		case kbCtlLeft  : leftWord(); break;
		case kbCtlRight : rightWord();	break;
		case kbCtlY     : *data = EOS; curPos = 0; break;

        case 'S': case 's': if( !extendKey ) goto handleChar;
		case kbHome: curPos = 0; break;

        case 'D': case 'd': if( !extendKey ) goto handleChar;
		case kbEnd: curPos = strlen(data); break;

        case 'Y': case 'y':
			if( !extendKey ) goto handleChar;
			data[curPos] = EOS;
		    break;

        case 'L': case 'l':
			if( !extendBlock ) goto handleChar;
            startSel = 0;
            endSel = strlen(data);
            extendBlock = False;
            mustKeep = True;
            break;

        case 'H': case 'h':
        	if( !extendBlock ) goto handleChar;
            startSel = endSel = 0;
            extendBlock = False;
            break;

		case kbBkSp:
        	if( 0 < curPos )
        	{
            	strcpy(data+curPos-1, data+curPos);
                curPos--;
                if( 0 < firstPos ) firstPos--;
			}
            break;

		case kbDel:
        	if( endSel == startSel && curPos < strlen(data) )
        	{
                startSel = curPos;
                endSel = curPos + 1;
			}
			deleteSelect();
            break;

        default   :
        handleChar:
			if( ' ' <= aKeyCode && aKeyCode <= 255 )
			{
            	deleteSelect();
				if( !insertMode && curPos < strlen(data) )
                	strcpy(data+curPos, data+curPos+1);
                if( strlen(data) < maxLen )
                {
                    memmove(data+curPos+1, data+curPos, strlen(data+curPos)+1);
                    data[curPos++] = aKeyCode;
				}
			}
            else return;
	}

    extendKey = False;
    if( mustKeep )
    {
		mustKeep = False;
        if( startSel > endSel )
        {
        	short temp = startSel;
            startSel = endSel;
            endSel = temp;
		}
    }
	else startSel = endSel = 0;
	if( firstPos > curPos ) firstPos = curPos;
	firstPos = max(firstPos, short(curPos - size.x + 2));

    if( mustPatch )
    {	// make sure the first char is visible
		firstPos++;		// when doing the ctrl+right arrow skips
        mustPatch = False;
    }

	draw();
}

// return an element from the palette (cast to unsigned character)
uchar
zEditLine::getPalette(short pos)
{
	return uchar(palette[pos % sizeof(palette)]);
}

// set a new palette (overrides the default)
void
zEditLine::setPalette(const char *aPalette)
{
	memcpy(palette, aPalette, sizeof(palette));
}

// selects or deselects the entire string
void
zEditLine::selectAll()
{
	firstPos = curPos = startSel = 0;
	if( focused ) endSel = strlen(data);
	else endSel = 0;
}

// deletes the current selection
void
zEditLine::deleteSelect()
{
	if( endSel > startSel )
	{
    	strcpy(data+startSel, data+endSel);
        curPos = startSel;
	}
}

// sets the input focus to a new state (and redraws)
void
zEditLine::setState(Boolean ssFocused)
{
	focused = ssFocused;
	selectAll();
	draw();
}

Boolean
zEditLine::getState()
{
	return focused;
}

// goes to the start of the previous word (to the left)
void
zEditLine::leftWord()
{
	if( curPos ) curPos--;
	while( 0 < curPos && !isalnum(data[curPos])) curPos--;
	while( 0 < curPos && isalnum(data[curPos]) ) curPos--;
	if( curPos ) curPos++;
}

// goes to the start of the next word to the right
void
zEditLine::rightWord()
{
	size_t len = strlen(data);
	if( curPos < len ) curPos++;
	while( curPos < len && !isalnum(data[curPos])) curPos++;
	while( curPos < len && isalnum(data[curPos]) ) curPos++;
	while( curPos < len && !isalnum(data[curPos])) curPos++;
	if( curPos < len && curPos - firstPos > size.x - 1 ) mustPatch = True;
}

// delete current pos to end of word plus spaces
void
zEditLine::zapWord()
{
	short  pos = curPos;
    size_t len = strlen(data);

    while( pos < len && !isspace(data[pos])) pos++;
    while( pos < len && isspace(data[pos]) ) pos++;
    if( pos > curPos ) strcpy(data+curPos, data+pos);
}
