/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "cmdbar.h"
#include "stdmac.h"
#include "utils.h"
#include "kbcodes.h"

#ifndef PB_SDK
	#include <conio.h>
	#include <string.h>
	#include <ctype.h>
	#include "compiler.h"
#else
	#include "pblibc.h"
	#include "pblsdk.h"
#endif

/////////////////////////////////////////////////////////////////////////////
// the status line class
/////////////////////////////////////////////////////////////////////////////
zCommandBar::zCommandBar(const zRect &bounds, zCommandItem *aItems):
	items(aItems),
	current(items)
{
	origin.x = bounds.a.x;
	origin.y = bounds.a.y;
	size.x = bounds.b.x - bounds.a.x;
	size.y = bounds.b.y - bounds.a.y;
	if( size.x < 4 ) size.x = 4;	// make sure we have some space
	size.x %= 81;					// limit to 80 characters
	options = (focus | delim);		// focused and with special delimeters
	setPalette(0);					// default palette
#ifndef PB_SDK
	_wscroll = 0;					// macro to disable scrolling
#endif
}

zCommandBar::~zCommandBar()
{
	while( 0 != items )
	{
		zCommandItem *p = items;
		items = items->next;
		delete p;
	}
}

// first displayable item
zCommandItem*
zCommandBar::firstItem()
{
	for( zCommandItem *p = items; p; p = p->next)
	{
		if( p->text )
		{
			short pos = getItemPos(p);
			if( pos + cstrlen(p->text) < size.x ) return p;  // got it!
		}
	}
	return 0;
}

// redraw the whole status line
void
zCommandBar::draw()
{
	short width = 0;
	char  buf[81];
	uchar backColor = (options & focus) ? getColor(5) : getColor(10);
	uchar backChar  = (options & focus) ? getColor(6) : getColor(11);

	gotoxy(origin.x, origin.y);

	if( options & hide )
	{	// clear the rectangle in proper color
		memset(buf, backChar, size.x);
		buf[size.x] = EOS;
		textattr(backColor);
		cprintf(buf);
	}
	else
	{		// display the items
		zCommandItem *item = items;
		while( item )
		{
			if( item->text )
			{
				width += cstrlen(item->text) + 2;
				if( width <= size.x )
				{		// will it fit on the line
					textattr(backColor);
					putch(backChar);
					drawItem(item);
					textattr(backColor);
					putch(backChar);
				}
				else break;	// won't fit... we're thru
			}
			item = item->next;
		}
		if( width < size.x )
		{
			memset(buf, backChar, size.x - width + 1);
			buf[size.x - width + 1] = EOS;
			textattr(backColor);
			cprintf(buf);
		}
	}
}

// display an item at the current cursor position. the item is assumed to
// be visible and is also assumed to fit on the line (no checks done).
void
zCommandBar::drawItem(const zCommandItem *item)
{
	uchar       normColor, highColor, specColor;
	const char *p = item->text;
	Boolean     hilite = False;

	// no text data, forget it then
	if( !item || !item->text ) return;

	if( options & focus )
	{	// status line is focused
		if( item == current )
		{	// current item is highlighted
			normColor = (item->enabled) ? getColor(3) : getColor(4);
			highColor = normColor;
			specColor = getColor(13);
		}
		else
		{					// current item is not highlighted
			normColor = (item->enabled) ? getColor(0) : getColor(2);
			highColor = (item->enabled) ? getColor(1) : normColor;
			specColor = getColor(12);
		}
	}
	else
	{						// line is not focused (no highlight at all)
		normColor = (item->enabled) ? getColor(7) : getColor(9);
		highColor = (item->enabled) ? getColor(8) : normColor;
		specColor = getColor(14);
	}

	if( options & delim )
	{
		textattr(specColor);
		if( EOS != *p ) putch(*p++);
	}
	// we've got to have more characters
	if( EOS != *p )
	{
		textattr(normColor);
		for( ; EOS != p[1]; ++p )
		{
			if( '~' == *p )
			{	// overrides the special color if first
				hilite = Boolean( !hilite );
				textattr( hilite ? highColor : normColor );
			}
			else putch(*p);
		}
		// set to different color for last character (must not be '~')
		if( options & delim ) textattr(specColor);
		if( '~' != *p ) putch(*p);
	}
}

// sets or resets the enable status of a command
void
zCommandBar::enableCommand(ushort aCommand, Boolean enable)
{
	for( zCommandItem *p = items; p; p = p->next )
	{
		if( p->command == aCommand )
		{
			p->enabled = enable;
			break;
		}
	}
}

// moves the selection bar to the item parameter if it is displayable
// this will always remove the current selection bar (but may not write
// a new one if the new item has no text or can't fit on the line)
// this method also changes the currency pointer to the new item
void
zCommandBar::focusItem(const zCommandItem *item)
{
	zCommandItem *temp = current;
	short        pos;

	if( item )
	{
		pos = getItemPos(temp);
		current = (zCommandItem *)item; // reset currency (old redraws as normal)
		if( pos < size.x )
		{ // redraw the old item if necessary
			if( temp->text && pos + cstrlen(temp->text) < size.x )
			{
				gotoxy(origin.x + pos, origin.y);
				drawItem(temp);
			}
		}
		pos = getItemPos(item);	// find the location of the new item
		if( pos < size.x )
		{	// redraw if necessary with highlight bar
			if( item->text && pos + cstrlen(item->text) < size.x )
			{
				gotoxy(origin.x + pos, origin.y);
				drawItem(item);
			}
		}
	}
}

uchar
zCommandBar::getColor(short aColor)
{
	return uchar(palette[aColor % sizeof(palette)]);
}

// the keystroke handler is here, this will return the command for the
// appropriate selection (if any) or cmNoCommand when no selection made
short
zCommandBar::handle(ushort aKeyCode)
{
	zCommandItem *item;

	// not focused, no processing
	if( !(options & focus) ) return cmNoCommand;

	switch( CtrlToArrow(aKeyCode) )
	{
		case kbEsc  : return cmCancel;
		case kbEnter:
			if( current->enabled ) return current->command;
			break;

		case kbRight: focusItem( nextItem() ); break;
		case kbLeft : focusItem( prevItem() ); break;
		case kbHome : focusItem( firstItem()); break;
		case kbEnd  : focusItem( lastItem() ); break;

		default:
			// try locating by a hotkey
			if( isascii(aKeyCode) ) aKeyCode = toupper(aKeyCode);
			for( item = items; item; item = item->next )
			{
				if( aKeyCode == item->keyCode )
				{
					focusItem(item);
					if( item->enabled ) return item->command;
					else return cmNoCommand;
				}
			}
	}
	return cmNoCommand;
}

// return the x coordinate for the current item, this may exceed 80,
// in which case the item is not visible and should be skipped by the
// highlight bar, and should be ignored by the End function. this skips
// over items that do not have any text defined, also takes into account
// the one-character gaps before and after each visible item. does not
// care if the current item is displayable at all (it may not be).
short
zCommandBar::getItemPos(const zCommandItem *item)
{
	zCommandItem *p = items;
	short        pos = 1;	// one leading gap always before the item

	while( p && p != item )
	{
		if( p->text )
		{                   // we have a text to display
			pos += 2 + cstrlen(p->text); // count displayable chars and gaps
		}
		p = p->next;
	}
	return pos;
}

// last displayable item
zCommandItem*
zCommandBar::lastItem()
{
	zCommandItem *p;

	for( p = items; p->next; p = p->next );
	for( ; p; p = p->prev )
	{
		if( p->text )
		{
			short pos = getItemPos(p);
			if( pos + cstrlen(p->text) < size.x ) return p;
		}
	}
	return 0;
}

// finds next displayable item (wraps around if necessary), return 0 if none
zCommandItem*
zCommandBar::nextItem()
{
	for( zCommandItem *p = current->next; p != current; p = p->next)
	{
		if( !p ) p = items;		// wrap around
		if( p->text )
		{
			short pos = getItemPos(p);
			if( pos + cstrlen(p->text) < size.x ) return p;  // got it!
		}
	}
	return 0;
}

static const char *cpStatusLine = //......default palette for the status line
	"\x0F"  //...................................focused text, white on black
	"\x03"  //....................focused highlight characters, cyan on black
	"\x08"  //......................focused disabled text, dark grey on black
	"\x1E"  //..........................focused highlight bar, yellow on blue
	"\x13"  //...................focused disabled highlight bar, cyan on blue
	"\x00"  //.......................focused background color, black on black
	" "     //............................focused background character, space
	"\x70"  //................................non-focused text, black on grey
	"\x74"  //.......................non-focused highlight chars, red on grey
	"\x78"  //...................non-focused disabled text, dark grey on grey
	"\x70"  //....................non-focused background color, black on grey
	" "     //........................non-focused background character, space
	"\x09"  //.............focused first and last chars, bright blue on black
	"\x09"  //..........focused highlight first and last, bright blue on blue
	"\x7F"; //......................non-focused first and last, white on grey

void
zCommandBar::setPalette(const char *aPalette)
{
	if( 0 != aPalette ) memcpy(palette, aPalette, sizeof(palette));
	else memcpy(palette, cpStatusLine, sizeof(palette));
}

// previous displayable item
zCommandItem*
zCommandBar::prevItem()
{
	for( zCommandItem *p = current->prev; p != current; p = p->prev )
	{
		if( !p ) for( p = items; p->next; p = p->next );
		if( p->text )
		{
			short pos = getItemPos(p);
			if( pos + cstrlen(p->text) < size.x ) return p;
		}
	}
	return 0;
}

/////////////////////////////////////////////////////////////////////////////
// constructs an item for the status line
/////////////////////////////////////////////////////////////////////////////
zCommandItem::zCommandItem(const char *aText, ushort aKey, ushort aCommand,
						 zCommandItem *aNext):
	keyCode(isascii(aKey) ? toupper(aKey) : aKey),
	command(aCommand),
	text(aText),
	next(aNext),
	prev(0),
	enabled(True)
{
	if( aNext ) aNext->prev = this;
}

/////////////////////////////////////////////////////////////////////////////
// the global operator used for appending items to the list
/////////////////////////////////////////////////////////////////////////////
zCommandItem*
operator+(zCommandItem *item, zCommandItem &link)
{
	if( item )
	{
		zCommandItem *p;
		for( p = item; p->next; p = p->next )
			;
		p->next = &link;
		link.prev = p;
	}
	return item;
}
