/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#ifndef INCLUDED_GEOMETRY_H
#define INCLUDED_GEOMETRY_H
#include "typedef.h"

/*
 * p o i n t   a n d   r e c t a n g l e   o b j e c t s
 * 
 * these are the declarations of the basic point and rectangle classes.
*/

///////////////////////////////////////////////////////////////////////////
// point class
///////////////////////////////////////////////////////////////////////////
class zPoint
{
	friend Boolean operator==(const zPoint &p1, const zPoint &p2);
    friend Boolean operator!=(const zPoint &p1, const zPoint &p2);
    friend zPoint  operator+ (const zPoint &p1, const zPoint &p2);
    friend zPoint  operator- (const zPoint &p1, const zPoint &p2);

public:
	void operator+=(const zPoint &point);
    void operator-=(const zPoint &point);

// data members
    int x, y;
};

///////////////////////////////////////////////////////////////////////////
// the rectangle class
///////////////////////////////////////////////////////////////////////////
class zRect
{
public:
	zRect();
	zRect(int ax, int ay, int bx, int by);
	zRect(const zPoint &aPoint, const zPoint &bPoint);
    zRect(const zRect &rect);

	Boolean contains(const zPoint &point) const;
    Boolean empty() const;
    Boolean operator==(const zRect &rect) const;
    Boolean operator!=(const zRect &rect) const;

    void operator|=(const zRect &rect);
	void operator&=(const zRect &rect);
    void operator+=(const zRect &rect);
    void operator+=(const zPoint &delta);
    void operator-=(const zRect &rect);
    void operator-=(const zPoint &delta);

    int    Height() const;
    int    Width() const;
    zPoint Size() const;

    void grow(int dx, int dy);
    void move(int dx, int dy);

    void Intersect(const zRect &rect);
    void Union(const zRect &rect);

    void Normalize();

// data members
	zPoint a, b;
};

///////////////////////////////////////////////////////////////////////////
// inlined member functions for zPoint
///////////////////////////////////////////////////////////////////////////
// add a point to the current point
inline void
zPoint::operator+=(const zPoint &point)
{
	x += point.x;
    y += point.y;
}

// subtract a point from the current point
inline void
zPoint::operator-=(const zPoint &point)
{
	x -= point.x;
    y -= point.y;
}

///////////////////////////////////////////////////////////////////////////
// inlined member functions for zRect
///////////////////////////////////////////////////////////////////////////
// create a null rectangle
inline
zRect::zRect()
{
	a.x = a.y = b.x = b.y = 0;
}

// create rectangle with given values
inline
zRect::zRect(int ax, int ay, int bx, int by)
{
	a.x = ax; a.y = ay;
    b.x = bx; b.y = by;
}

// create rectangle from two points
inline
zRect::zRect(const zPoint &topLeft, const zPoint &bottomRight)
{
	a.x = topLeft.x;
	a.y = topLeft.y;
	b.x = bottomRight.x;
	b.y = bottomRight.y;
}

// crete a rectangle same as parameter (copy constructor)
inline
zRect::zRect(const zRect &rect)
{
	a.x = rect.a.x;
	a.y = rect.a.y;
	b.x = rect.b.x;
	b.y = rect.b.y;
}

// check if point is in a rectangle
inline Boolean
zRect::contains(const zPoint &p) const
{
	return Boolean( p.x >= a.x && p.x < b.x && p.y >= a.y && p.y < b.y );
}

// check if the rectangle is empy
inline Boolean
zRect::empty() const
{
	return Boolean( !(a.x < b.x && a.y < b.y) );
}

// check if rectangles are equal
inline Boolean
zRect::operator==(const zRect &rect) const
{
	return Boolean( a == rect.a && b == rect.b );
}

// check if two rectangles are not equal
inline Boolean
zRect::operator!=(const zRect &rect) const
{
	return Boolean( a != rect.a || b != rect.b );
}

// return the height of the rectangle
inline int
zRect::Height() const
{
	return b.y - a.y;
}

// return the width of the rectangle
inline int
zRect::Width() const
{
	return b.x - a.x;
}

// return the size of the rectangle as a zPoint
inline zPoint
zRect::Size() const
{
	zPoint size;

	size.x = b.x - a.x;
	size.y = b.y - a.y;
	return size;
}

// change the dimensions of the rectangle
inline void
zRect::grow(int dx, int dy)
{
	a.x -= dx; a.y -= dy;
    b.x += dx; b.y += dy;
}

// move the rectangle position
inline void
zRect::move(int dx, int dy)
{
	a.x += dx; a.y += dy;
    b.x += dx; b.y += dy;
}

// operator for union of two rectangles
inline void
zRect::operator|=(const zRect &rect)
{
	Union(rect);
}

// intersection of two rectangles
inline void
zRect::operator&=(const zRect &rect)
{
	Intersect(rect);
}

// adds to current rectangle
inline void
zRect::operator+=(const zRect &rect)
{
	a.x += rect.a.x;
	a.y += rect.a.y;
    b.x += rect.b.x;
    b.y += rect.b.y;
}

// subtracts from the rectangle
inline void
zRect::operator-=(const zRect &rect)
{
	a.x -= rect.a.x;
    a.y -= rect.a.y;
    b.x -= rect.b.x;
    b.y -= rect.b.y;
}

// adds delta values in point format
inline void
zRect::operator+=(const zPoint &delta)
{
	a.x += delta.x;
    a.y += delta.y;
    b.x += delta.x;
    b.y += delta.y;
}

// subtract delta values in point format
inline void
zRect::operator-=(const zPoint &delta)
{
	a.x -= delta.x;
    a.y -= delta.y;
    b.x -= delta.x;
    b.y -= delta.y;
}

#endif /* INCLUDED_GEOMETRY_H */
