//*****************************************************************************
//***  NAME              : iptest.c                                         ***
//***  PART OF           : IpSpy                                            ***
//***  SHORT DESCRIPTION : tests the ipspy dll                              ***
//--------------------------------------------------------------------------***
//***  AUTHOR            : E.Buerkle                                        ***
//***  CREATION DATE     : 15.01.97                                         ***
//***  COPYRIGHT         : (C) 1997 E.Buerkle                               ***
//*****************************************************************************

#define INCL_BASE
#include <os2.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <stddef.h>


#define  OS2
#ifdef __EMX__
#include <sys\types.h>
#else
#include <types.h>
#endif
#include <sys/socket.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <sys/select.h>
#include <sys/ioctl.h>

#include <net\if.h>
#include <net\route.h>
#include <netinet\in.h>
#include <netinet\in_systm.h>
#include <netinet\ip.h>
#include <netinet\ip_icmp.h>
#include <netinet\if_ether.h>
#include <netinet\tcp.h>
#ifdef __EMX__
#include <arpa\inet.h>
#endif

#include <ipspy.h>

ULONG  ulHandle;
UCHAR  auchInterface[IFNAMSIZ+1];
USHORT usOldMode;

//**************************************************************************
//  Name       :  viewICMP
//  Description:  view icmp packets
//  Parameters :  ip  - pointer to ip packet
//  Return     :  none
//**************************************************************************
VOID viewICMP(struct ip *ip)
{
   char               *type, line[96];
   struct icmp        *icmp;
   ULONG              headerLen,dataLen,len;

   headerLen = ip->ip_hl << 2;
   len = ntohs(ip->ip_len) - headerLen;
   if(len < ICMP_MINLEN)
   {
     printf("packet too short [%d bytes]\n", len);
     return;
   }
   dataLen =  len - sizeof(struct icmp);
   type = "";

   icmp = (struct icmp *)((UCHAR *) ip + headerLen);

   if (icmp->icmp_type == ICMP_UNREACH)
   {
     switch(icmp->icmp_code)
     {
       case ICMP_UNREACH_NET:
         type = "Network unreachable";
         break;
       case ICMP_UNREACH_HOST:
         type = "Host unreachable";
         break;
       case ICMP_UNREACH_PROTOCOL:
         type = "Invalid protocol";
         break;
       case ICMP_UNREACH_PORT:
         type = "Invalid port";
         break;
       case ICMP_UNREACH_NEEDFRAG:
         type = "Need fragmenting";
         break;
       case ICMP_UNREACH_SRCFAIL:
         type = "Source route failure";
         break;
       default:
         sprintf(line, "Unknown code #%i", icmp->icmp_code);
         type = line;
         break;
     }
     printf(" *** ICMP Type: unreach, Code: %s ***\n", type);
   }
   else
   {
     switch(icmp->icmp_type)
     {
       case ICMP_ECHOREPLY:
         type =  "echo reply";
         break;
       case ICMP_UNREACH:
         type =  "dest unreachable, codes:";
         break;
       case ICMP_SOURCEQUENCH :
         type =  "packet lost, slow down";
         break;
       case ICMP_REDIRECT:
         type =  "shorter route, codes: ";
         break;
       case ICMP_ECHO:
         type =     "echo service";
         break;
       case ICMP_TIMXCEED:
         type =     "time exceeded, code:";
         break;
       case ICMP_PARAMPROB:
         type =    "ip header bad";
         break;
       case ICMP_TSTAMP:
         type =   "timestamp request";
         break;
       case ICMP_TSTAMPREPLY:
         type =   "timestamp reply";
         break;
       case ICMP_IREQ:
         type =   "information request";
         break;
       case ICMP_IREQREPLY:
         type =  "information reply";
         break;
       case ICMP_MASKREQ:
         type = "address mask request";
         break;
       case ICMP_MASKREPLY:
         type = "address mask reply";
         break;
       default:
         sprintf(line, "Unknown type #%i", icmp->icmp_type);
         type = line;
         break;
     }

     printf("*** ICMP Type: %s, Code:%d ***\n", type, icmp->icmp_code);
   }

//   printf("Icmp length:%d(0x%x)\n", dataLen, dataLen);
//   if(dataLen > 0)
//     dump( (PVOID) (ip + 1), dataLen);

}


//**************************************************************************
//  Name       :  filterICMP
//  Description:  search icmp packets
//  Parameters :  ip  - pointer to ip packet
//  Return     :  TRUE if found
//**************************************************************************
BOOL filterICMP(struct ip *ip)
{
     BOOL fICMP = FALSE;

     printf("Protocol: ");
     switch(ip->ip_p)
     {
       case IPPROTO_IP:
         printf("IP\n");
         break;
       case IPPROTO_ICMP:
         printf("ICMP\n");
         fICMP = TRUE;         // hello Mr. Ping
         break;
       case IPPROTO_GGP:
         printf("GGP\n");
         break;
       case IPPROTO_TCP:
         printf("TCP\n");
         break;
       case IPPROTO_EGP:
         printf("EGP\n");
         break;
       case IPPROTO_PUP:
         printf("PUP\n");
         break;
       case IPPROTO_UDP:
         printf("UDP\n");
         break;
       case IPPROTO_IDP:
         printf("IDP\n");
         break;
       case IPPROTO_RAW:
         printf("RAW IP\n");
         break;
       default:
         printf("UNKOWN: #%d\n", ip->ip_p);
         break;
     }
/********************************
     printf("IP Header length %d\t\t Version %d\t\t Service %d\n",
            ip->ip_hl << 2, ip->ip_v, ip->ip_tos);
     printf("IP total Length %d\t\t ID %d\t\t Offset %d \n",
            ntohs(ip->ip_len), ntohs(ip->ip_id), ntohs(ip->ip_off));
     printf("Time to live %d\t\t checksum:%d\n",
            ip->ip_ttl, ntohs(ip->ip_sum));
     printf("service: %s\n", getservbyport(ip->ip_tos, &ip->ip_p));
********************************/
     printf("Packet-Destination %s\t\t",
            inet_ntoa(ip->ip_dst));
     printf("Packet-Source %s\n",
            inet_ntoa(ip->ip_src));

     return fICMP;
}

//**************************************************************************
//  Name       :  filterIP
//  Description:  search ip packets
//  Parameters :  type - packet type
//                data - raw data
//                ip   - return pointer to ip packet
//  Return     :  TRUE if found
//**************************************************************************
BOOL filterIP(USHORT usType, PVOID data, struct ip **ip)
{
   UCHAR   *ptr;
   USHORT  EtherType;

   printf("Packet Type: ");
   switch(usType)
   {
        case IP_PACKET_TYPE:
          printf("IP\n");
          *ip = data;
          return TRUE;
        case ETHERNET_PACKET_TYPE: // DIX
          printf("ETHERNET\n");
          ptr = (UCHAR *) data;
          EtherType=ntohs(((struct ether_header *)ptr)->ether_type);
          if(EtherType != ETHERTYPE_IP)
            return FALSE;
          *ip = (struct ip *)(ptr + sizeof(struct ether_header));
          return TRUE;
        case IEEE_802_3_PACKET_TYPE:
          printf("802.3\n");
          return FALSE;
        case IEEE_802_5_PACKET_TYPE:
          printf("802.5\n");
          ptr = (UCHAR *) data;
          while(!(*ptr == HEAD_802_END && *(ptr+1) == HEAD_802_END))
            ptr++;
          EtherType = ntohs(*((USHORT *) (ptr+6)));
          if(EtherType != ETHERTYPE_IP)
            return FALSE;
          *ip = (struct ip *) (ptr + 8);
          return TRUE;
        case SLIP_PACKET_TYPE:
          printf("SLIP\n");
          if(*((UCHAR *) data) == 0xC0 || *((UCHAR *) data) == 0xDB)
          {
            *ip = (struct ip *) ((UCHAR *)(data) + 1);
            return TRUE;
          }
          printf("unknown SLIP Packet\n");
          return FALSE;
        case SOFTWARE_LB_PACKET_TYPE:          // PPP !!!
        case PPP_PACKET_TYPE:
          printf("PPP\n");
          ptr = (UCHAR *) data;
          if(*ptr == 0x7e)
            ptr++;
          // exclude LCP's and NCP'S
          if(*ptr == 0x21)
            *ip = (struct ip *) (ptr + 1);
          else if(*ptr == 0xff &&
                  *(ptr + 1) == 0x03 &&
                  *(ptr + 2) <= 0x3f &&
                  *(ptr + 3) == 0x21)
            *ip = (struct ip *) (ptr + 4);
          else if(*ptr == 0xff &&
                  *(ptr + 1) == 0x7d &&
                  *(ptr + 2) == 0x23 &&
                  *(ptr + 3) <= 0x3f &&
                  *(ptr + 4) == 0x21)
            *ip = (struct ip *) (ptr + 5);
          else
            return FALSE;
          return TRUE;
        default:
          printf("UNKNOWN %d\n", usType);
          return FALSE;
   }
}

//**************************************************************************
//  Name       :  writePacket
//  Description:  write packet to local interface
//  Parameters :  data        - pointer to raw packet
//                usLength    - size of packet
//                usType      - packet type
//                ulTimeStamp - timestamp
//                usUnknown   - unknown
//  Return     :  VOID nothing
//**************************************************************************
VOID writePacket(VOID *data, USHORT usLength, USHORT usType, ULONG ulTimeStamp, USHORT usUnknown)
{
  ULONG  ulHandle;
  APIRET rc;

  if((rc=IpSpy_Init(&ulHandle, "lo")) != RC_IPSPY_NOERROR)
    printf("IpSpyInit Error: %d\n", rc);

  // throw the packet to local interface
  if((rc=IpSpy_WriteRaw(ulHandle, data, usLength, usType, ulTimeStamp, usUnknown)) != RC_IPSPY_NOERROR)
    printf("IpSpyWrite Error: %d\n", rc);

  if((rc=IpSpy_Exit(ulHandle)) != RC_IPSPY_NOERROR)
    printf("IpSpyExit Error: %d\n", rc);

}

//**************************************************************************
//  Name       :  IpExitProc
//  Description:  close interface and reset adapter to normal mode
//  Parameters :  usTermCode - termination info
//  Return     :  none
//**************************************************************************
VOID APIENTRY IpExitProc(ULONG usTermCode)
{
  APIRET rc;

  fflush(NULL);

  // end monitor
  if((rc=IpSpy_Exit(ulHandle)) != RC_IPSPY_NOERROR)
    printf("IpSpyExit Error: %d\n", rc);

  // ip stack relexation
  if((rc=IpSpy_SetReceiveMode(usOldMode,
                              auchInterface,
                              NULL)) != RC_IPSPY_NOERROR)
    printf("IpSpy_SetReceiveMode Error: %d\n", rc);

  printf("Spy ends\n");
}

//**************************************************************************
//  Name       :  main
//  Description:  read 10 raw inet packets and filter icmp
//                and then send the last packet to "lo"
//  Parameters :  argc argv env
//  Return     :  none
//**************************************************************************
VOID main(ULONG argc, UCHAR **argv, UCHAR **env)
{
  APIRET rc;
  UCHAR  auchData[2000], *pVersion, **pIFs;
  USHORT usLength, usType, i;
  ULONG  ulTimeStamp;
  USHORT usUnknown;
  struct ip *ip;
  UCHAR  *pSocketError;
  ULONG  ulSocketError;

  // query all available interfaces
  if((rc=IpSpy_QueryInterfaces(&pIFs)) != RC_IPSPY_NOERROR)
  {
    printf("IpSpyQueryIF Error: %d\n", rc);
    return;
  }

  // check arguments
  if(argc != 2 || IFNAMSIZ < strlen(argv[1]))
  {
    printf("usage: iptest [interface]\n");
    printf(" where interface is one of:\n");
    // show all available interfaces
    if(pIFs != NULL)
      for(i=0; pIFs[i] != 0; i++)
        printf(" %s\n", pIFs[i]);

    return;
  }

  strcpy(auchInterface, argv[1]);
  printf("Spy [%s], quit with Ctrl-C\n", auchInterface);

  // view the version
  if((rc=IpSpy_Version(&pVersion)) != RC_IPSPY_NOERROR)
    printf("IpSpyVersion Error: %d\n", rc);
  else
    printf("IpSpy Version: %s\n", pVersion);

  // save receive mode
  if((rc=IpSpy_QueryReceiveMode(&usOldMode, NULL)) != RC_IPSPY_NOERROR)
    printf("IpSpy_QueryReceiveMode Error: %d\n", rc);

  // we want all packets
  if((rc=IpSpy_SetReceiveMode(DIRECTED_MODE | BROADCAST_MODE | PROMISCUOUS_MODE,
                              auchInterface,
                              NULL)) != RC_IPSPY_NOERROR)
  {
    if(rc == RC_IPSPY_MODE_NOT_SUPPORTED)
      printf("Promiscuous mode not supported\n");
    else if(rc == RC_IPSPY_CANNOT_OPEN_DRIVER)
      printf("IpSpy_SetReceiveMode Error: Cannot open ipspy.os2\n");
    else
      printf("IpSpy_SetReceiveMode Error: %d\n", rc);
  }


  // init monitor
  if((rc=IpSpy_Init(&ulHandle, auchInterface)) != RC_IPSPY_NOERROR)
  {
    if(rc == RC_IPSPY_SOCKET_ERROR)
    {
      IpSpy_GetLastSocketError(&ulSocketError, &pSocketError);
      printf("IpSpyInit SocketError: [%d] %s\n", ulSocketError, pSocketError);
    }
    else
      printf("IpSpyInit Error: %d\n", rc);
  }

  // good housekeeping
  DosExitList(EXLST_ADD, IpExitProc);



  //
  // process 10 packets
  //

  for(i=1;i<=10;i++)
  {
    usLength = sizeof(auchData);
    if((rc=IpSpy_ReadRaw(ulHandle, auchData, &usLength, &usType, &ulTimeStamp, &usUnknown)) != RC_IPSPY_NOERROR)
      printf("IpSpyRead Error: %d\n", rc);
    else
    {
       printf("\nPacket No. %d\n", i);
       if(filterIP(usType, auchData, &ip ) == TRUE)
       {
         if(filterICMP(ip) == TRUE)
           viewICMP(ip);

         // route the last packet to the local interface
         if(i == 10)
         {
           ip->ip_dst = inet_makeaddr(127, 1);
           writePacket(auchData, usLength, usType, ulTimeStamp, usUnknown);
         }
       }
    }
  }

}
