unit Unit1;

{
Tests whether API menu calls that use MF_BYCOMMAND operate differently in
Windows 3.1, Windows 95, and Windows NT when using 16-bit and 32-bit apps.

The discrepancy concerns whether or not MF_BYCOMMAND calls affect only menu
items, or whether they also affect popups. (A menu item is a leaf on the menu
tree, corresponding to a command which the user can choose, whereas a popup
is a non-leaf node -- that is, a node which contains additional nodes.)

The problem arises because certain API functions may or may not affect popups,
depending on the environment in which the program is run. For instance,
EnableMenuItem never affects popups for 16-bit apps under 3.1, and in 32-bit
apps under 95 and NT, but does affect popups for 16-bit apps under 95 and NT.

Affect Popups in:     3.1(16)     95(16)     NT(16)     95(32)     NT(32)
EnableMenuItem        no          yes        yes        no         no
ModifyMenu*           no          yes        yes        no         no
InsertMenu            no          yes        yes        no         no
CheckMenuItem*        no          yes        yes        no         no
RemoveMenu            no          yes        yes        no         no
DeleteMenu            no          yes        yes        no         no
GetMenuState          no          yes        yes        no         no

*Notes:
In each "yes" case, when an item and a popup on the same menu share an identifier,
the item is used preferentially. But when an item on a later menu shares an
identifier with a popup on a previous menu, it's the popup that's affected.

CheckMenuItem never affects popups in the menu bar itself, but does affect
popups in menus (popups which contain cascading menus).

ModifyMenu also affects popups on earlier menus in preference to items on later
menus, but does not affect menus on the menu bar unless there are no items with
matching identifiers.
}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, ExtCtrls, Menus, Grids;

type
  TForm1 = class(TForm)
    MainMenu1: TMainMenu;
    Test1: TMenuItem;
    Popup1: TMenuItem;
    N1: TMenuItem;
    Exit1: TMenuItem;
    Subitem1: TMenuItem;
    Enable: TButton;
    Disable: TButton;
    Modify: TButton;
    Insert: TButton;
    Remove: TButton;
    Delete: TButton;
    CheckMenu: TButton;
    GetState: TButton;
    GetString: TButton;
    Hilite: TButton;
    Unhilite: TButton;
    Setbitmaps: TButton;
    Edit8: TMenuItem;
    EditItem11: TMenuItem;
    Panel2: TPanel;
    Grid: TStringGrid;
    GroupBox2: TGroupBox;
    TargetID: TEdit;
    Panel1: TPanel;
    TargetPresent: TRadioButton;
    TargetAbsent: TRadioButton;
    Panel3: TPanel;
    MatchPopupMenu: TRadioButton;
    MatchFileMenu: TRadioButton;
    Panel4: TPanel;
    Status: TEdit;
    EditName: TEdit;
    Label1: TLabel;
    Label10: TLabel;
    procedure EnableClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure Exit1Click(Sender: TObject);
    procedure DisableClick(Sender: TObject);
    procedure HiliteClick(Sender: TObject);
    procedure UnhiliteClick(Sender: TObject);
    procedure CheckMenuClick(Sender: TObject);
    procedure ModifyClick(Sender: TObject);
    procedure InsertClick(Sender: TObject);
    procedure RemoveClick(Sender: TObject);
    procedure DeleteClick(Sender: TObject);
    procedure GetStateClick(Sender: TObject);
    procedure GetStringClick(Sender: TObject);
    procedure SetbitmapsClick(Sender: TObject);
    procedure ModifyPopupClick(Sender: TObject);
    procedure ModifyFileClick(Sender: TObject);
    procedure EditNameChange(Sender: TObject);
    procedure TargetPresentClick(Sender: TObject);
  private
    { Private declarations }
    theMenuBar, fileID, editID, popupID, originalID, itemToModify: hMenu;
    procedure CheckIDs;
    procedure CheckOne (parent: hMenu; position: integer; col, row: integer);
    function CheckColumn (col: integer): word;
    function MakeText (value: word): string;
    procedure SetTestTarget (theID: hMenu);
    procedure CheckStatus (val: bool);
  public
    { Public declarations }
  end;

var
  Form1: TForm1;

implementation

{$R *.DFM}
{ Enabling/disabling finds the actual command if it's on the menu bar. But if it's been }
{ removed, enabling/disabling will affect any popup whose identifier matches the command. }
{ Hilite and Unhilite called with MF_BYCOMMAND work only on commands, not on popups. }
{ However, the actual hilighting is applied to the top-level menu containing the command. }

procedure TForm1.CheckOne (parent: hMenu; position: integer; col, row: integer);
var id: word;
    s: array [0..79] of char;
    pSt, valSt: string;
begin
if (GetMenuState (parent, position, MF_BYPOSITION) and MF_SEPARATOR) <> 0
  then pSt := '----------'
else
  begin
  GetMenuString (parent, position, s, sizeOf (s) - 1, MF_BYPOSITION);
  pSt := StrPas (s);
  id := GetSubMenu (parent, position);
  if id = 0
    then id := GetMenuItemID (parent, position)
    else pSt := Concat (pSt, '>');
  Str (id, valSt);
  pSt := Concat (pSt, ': ', valSt);
  end;
Grid.Cells [col, row] := pSt;
end;

function TForm1.CheckColumn (col: integer): word;
var i, theID: word;
begin
theID := GetSubMenu (theMenuBar, col);
CheckOne (theMenuBar, col, col, 0);
if theID <> 0 then
  for i := 0 to 4 do
    if i < GetMenuItemCount (theID)
      then CheckOne (theID, i, col, i+1)
      else Grid.Cells [col, i+1] := '';
CheckColumn := theID;
end;

procedure TForm1.CheckIDs;
begin
DrawMenuBar (handle);
theMenuBar := GetMenu (handle);
fileID := CheckColumn (0);
editID := CheckColumn (1);
end;

procedure TForm1.SetTestTarget (theID: hMenu);
var flag: word;
    name: array [0..79] of char;
    s: string;
begin
itemToModify := theID;
Str (theID, s);
TargetID.Text := s;
if TargetAbsent.Checked
  then theID := originalID;
flag := GetMenuState (editID, 0, MF_BYPOSITION);
GetMenuString (editID, 0, @name, sizeOf (name) - 1, MF_STRING OR MF_BYPOSITION);
ModifyMenu (editID, 0, flag or MF_BYPOSITION, theID, @name);
CheckIDs;
end;

procedure TForm1.FormCreate(Sender: TObject);
var flag: word;
    name: array [0..79] of char;
begin
CheckIDs;
popupID := GetSubMenu (fileID, 0);
originalID := GetMenuItemID (editID, 0);
SetTestTarget (popupID);
end;

procedure TForm1.CheckStatus (val: bool);
var s: string;
begin
Str (word (val), s);
Status.Text := s;
end;

procedure TForm1.Exit1Click(Sender: TObject);
begin
Close;
end;

function TForm1.MakeText (value: word): string;
var s: string;
begin
Str (value, s);
MakeText := s;
end;

procedure TForm1.EnableClick(Sender: TObject);
begin
CheckStatus (EnableMenuItem (theMenuBar, itemToModify, MF_BYCOMMAND OR MF_ENABLED));
DrawMenuBar (handle);
end;

procedure TForm1.DisableClick(Sender: TObject);
begin
CheckStatus (EnableMenuItem (theMenuBar, itemToModify, MF_BYCOMMAND OR MF_GRAYED));
CheckIDs;
end;

procedure TForm1.HiliteClick(Sender: TObject);
begin
CheckStatus (HiliteMenuItem(handle, theMenuBar, itemToModify, MF_HILITE OR MF_BYCOMMAND));
end;

procedure TForm1.UnhiliteClick(Sender: TObject);
begin
CheckStatus (HiliteMenuItem(handle, theMenuBar, itemToModify, MF_UNHILITE OR MF_BYCOMMAND));
end;
                    
procedure TForm1.CheckMenuClick(Sender: TObject);
var theItem, flag: word;
begin
theItem := itemToModify;
flag := GetMenuState (theMenuBar, theItem, MF_BYCOMMAND);
if (flag and MF_CHECKED) <> 0
  then flag := MF_UNCHECKED
  else flag := MF_CHECKED;
CheckStatus (CheckMenuItem (theMenuBar, theItem, MF_BYCOMMAND or flag));
CheckIDs;
end;

procedure TForm1.ModifyClick(Sender: TObject);
var newName: array [0..79] of char;
    itm: word;
begin
StrPCopy (newName, EditName.Text);
itm := itemToModify;
CheckStatus (ModifyMenu (theMenuBar, itm, MF_BYCOMMAND OR MF_STRING, itm, newName));
CheckIDs;
end;

procedure TForm1.InsertClick(Sender: TObject);
var newName: array [0..79] of char;
begin
StrPCopy (newName, EditName.Text);
CheckStatus (InsertMenu (theMenuBar, itemToModify, MF_BYCOMMAND, 100, newName));
CheckIDs;
end;

procedure TForm1.RemoveClick(Sender: TObject);
begin
CheckStatus (RemoveMenu (theMenuBar, itemToModify, MF_BYCOMMAND));
CheckIDs;
end;

procedure TForm1.DeleteClick(Sender: TObject);
begin
CheckStatus (DeleteMenu (theMenuBar, itemToModify, MF_BYCOMMAND));
CheckIDs;
end;

procedure TForm1.GetStateClick(Sender: TObject);
var state: word;
    s: string;
begin
state := GetMenuState (theMenuBar, itemToModify, MF_BYCOMMAND);
CheckStatus (state <> $FFFF);
if (state and MF_POPUP) <> 0
  then s := 'popup '
  else s := '';
if (state and MF_CHECKED) <> 0
  then s := Concat (s, 'checked ');
if (state and MF_DISABLED) <> 0
  then s := Concat (s, 'disabled ');
if (state and MF_GRAYED) <> 0
  then s := Concat (s, 'gray ');
EditName.Text := s;
end;

procedure TForm1.GetStringClick(Sender: TObject);
var name: array [0..79] of char;
begin
CheckStatus (-1 <> GetMenuString (theMenuBar, itemToModify, name, sizeOf (name) - 1, MF_BYCOMMAND));
EditName.Text := StrPas (name);
end;

procedure TForm1.SetbitmapsClick(Sender: TObject);
begin
CheckStatus (SetMenuItemBitmaps (theMenuBar, itemToModify, MF_BYCOMMAND,
      LoadBitmap (0, pChar (OBM_OLD_DNARROW)), LoadBitmap (0, pChar (OBM_OLD_UPARROW))));
end;

procedure TForm1.ModifyPopupClick(Sender: TObject);
begin
SetTestTarget (popupID);
end;

procedure TForm1.ModifyFileClick(Sender: TObject);
begin
SetTestTarget (fileID);
end;

procedure TForm1.EditNameChange(Sender: TObject);
begin
Modify.Enabled := length (EditName.Text) > 0;
end;

procedure TForm1.TargetPresentClick(Sender: TObject);
begin
SetTestTarget (itemToModify);
end;

end.
