/*
  Snmpping.cpp --- SNMPPING.EXE utility.
  
  Copyright 1995, 1996 by MG-SOFT d.o.o.
  All rights reserved.
    E-mail: info@mg-soft.si
    Web URL: http://www.mg-soft.si/

  This source file is for demonstrational purpose only.
  Check LICENSE.TXT for licensing information.
*/
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <conio.h>
#include <windows.h>
#include <winsock.h>

#include "..\..\include\mgtypes.h"
#include "..\..\include\winsnmp.h"
#include "..\..\include\wsnmpmib.h"

// SnmpPing version
#define SNMPPING_VERSION "V1.0 Beta 5"
// WinSNMP messages
#define WM_SNMP_MSG        WM_USER + 1
#define WM_SNMP_MSG_STOP   WM_USER + 2
// Default community
#define PUBLIC_COMMUNITY   "public"
// Output string flags
#define OUTPUT_OID		0x01
#define OUTPUT_OID_NAME		0x02
#define OUTPUT_VALUE		0x04
#define OUTPUT_SEPARATOR	0x08
#define OUTPUT_VALUE_TYPE	0x10
#define OUTPUT_NEWLINE		0x20

// Function definitions
class SnmpQuery;
void setDefaultParameters(void);
void printParameters(void);
void getArgs(int argc, char *argv[]);
void usage(void);
SNMPAPI_STATUS convertString2smiValue(smiLPVALUE value, char *buffer);
WINAPI_STATUS allocateNotificationHandle(void);
WINAPI_STATUS freeNotificationHandle(void);
WINAPI_STATUS startupWinsock(void);
WINAPI_STATUS shutdownWinsock(void);
LONG APIENTRY notificationSnmpWndProc( HWND hWnd, UINT message, UINT wParam, LONG lParam);

// Snmp return values to string conversion fot translate mode
static char *sTranslateMode[] = {
   "Translated",
   "Untranslated V1",
   "Untranslated V2",
   ""};

#define MAX_SNMP_ERRORS SNMP_ERROR_INCONSISTENTNAME
static char *sSnmpErrors[] = {
   "no error",
   "too big",
   "no such name",
   "bad value",
   "read-only",
   "generic error",
   "no access",
   "wrong type",
   "wrong length",
   "wrong encoding",
   "wrong value",
   "no creation",
   "inconsistent value",
   "resource unavailable",
   "commit failed",
   "undo failed",
   "authorization error",
   "not writable",
   "inconsistent name",
   ""
};

// Snmp return values to string conversion fot retransmit mode
static char *sRetransmitMode[] = {
   "Not executing the retransmission policy.",
   "Executing the retransmission policy.",
   ""};

// WinSNMP notificaiton window class and handle
static WNDCLASS  wc;
static HWND      hWinSnmpNotification = 0;

// parameters structure
struct parameters {
  char sdip[20];             // destination IP 
  char LocalHostName[100];   // local host name
  char LocalHostIp[20];      // local IP
  int Retransmit;
  int Timeout;
  BOOL RetransmitMode;
  BOOL BrowseWholeTree;
  BOOL BrowseSystemInfo;
  BOOL BrowseDepth;
  BOOL OidToStringConversion;
  BOOL CustomStartStopConditions;
  BOOL pingVersion;					 // get version 
  BOOL ResolveAgentIp;
  BOOL PrintValueType;
  BOOL PrintLineNumber;
  BOOL PrintParameters;
  BOOL SetSnmpValue;
  BOOL PingUntilInterrupted;
  int  PingInterval;

  unsigned int CurrentLineNumber;
  char *sCommunity;
  char *startOid;
  char *stopOid;
  char *sValueToSet;
  smiUINT32 ValueType;
} sp;

// SnmpQuery class
class SnmpQuery {
  private:
    // General WinSnmp data
    static smiUINT32 nMajorVersion;
    static smiUINT32 nMinorVersion;
    static smiUINT32 nLevel;
    static smiUINT32 nTranslateMode;
    static smiUINT32 nRetransmitMode;
    static BOOL g_SnmpInitialized;
    // Initialize depended data
    BOOL bInitialized;
    BOOL bWinSnmpStarted;
    // Session depended data
    char *psIPManager;
    char *psIPAgent;
    char sIPManager[80];
    char sIPAgent[80];
    char sCommunity[80];
    smiOCTETS smiCommunity;
    // Session values
    smiOID stopOid;
    smiUINT32 stopOidVal[MAXOBJIDSIZE];
    smiOID oid;
    smiUINT32 oidVal[MAXOBJIDSIZE];
    smiVALUE value;
    int requestId;
    // Session handles
    HSNMP_SESSION hSnmpSession;
    HSNMP_ENTITY hAgentEntity;
    HSNMP_ENTITY hManagerEntity;
    HSNMP_CONTEXT hViewContext;
    HSNMP_VBL hVbl;
    HSNMP_PDU hPdu;
    // WinSnmp notification
    HWND notifyHandle;
    UINT notifyMessage;
  public:
    // functions
    SnmpQuery(HANDLE notifyHandle, UINT notifyMessage);
    ~SnmpQuery(void);
    WINAPI_STATUS Startup(void);
    WINAPI_STATUS Cleanup(void);
    void NotificationCallback(void);
    WINAPI_STATUS doQuery(char *qName);
    WINAPI_STATUS SendGetRequest(void);
    WINAPI_STATUS SendGetNextRequest(void);
    WINAPI_STATUS SendSetRequest(void);
    WINAPI_STATUS SnmpOutputValue(smiOID oid, smiVALUE value, BYTE flags);
};
class SnmpQuery *sq;

// global class variables initialization
smiUINT32 SnmpQuery::nMajorVersion = 0;
smiUINT32 SnmpQuery::nMinorVersion = 0;
smiUINT32 SnmpQuery::nLevel = 0;
smiUINT32 SnmpQuery::nTranslateMode = 0;
smiUINT32 SnmpQuery::nRetransmitMode = 0;
BOOL SnmpQuery::g_SnmpInitialized = FALSE;

/*
  main

	Application entry function
*/
int main(int argc, char *argv[])
{
  int rv = 1;
  // Initialize parameters struct
  memset(&sp, 0, sizeof(struct parameters));

  fprintf(stdout, "\nSNMPPING Utility %s; C-1996 by MG-SOFT Corporation, Slovenia.", SNMPPING_VERSION);
  // set default parameters
  setDefaultParameters();
  // parse arguments and override default parameters
  getArgs(argc, argv);

  // Allocate the hidden notification window handle
  if (allocateNotificationHandle() == WINAPI_FAILURE) {
    rv = 0;
    goto doExit;
  }

  // Startup winsock (for local host resolving)
  startupWinsock();

  // Initialize class and WInSNMP
  sq = new SnmpQuery(hWinSnmpNotification, WM_SNMP_MSG);
  // Startup winsnmp
  sq->Startup();
  // print parameters (after snmp startup)
  if (sp.PrintParameters)
    printParameters();

  // Check if destination IP is assigned
  if (sp.sdip[0] == 0) {
    rv = 0;
    goto doExit;
  }

  // Get remote host name
  struct hostent *he;
  unsigned long ia;
  ia = htonl(inet_addr(&sp.sdip[0]));
  he = (sp.ResolveAgentIp) ? gethostbyaddr((char *) &ia, 4, PF_INET) : NULL;
  if (he)
    fprintf(stdout, "\n\nPinging %s [%s] with SNMP protocol", he->h_name, sp.sdip);
  else
    fprintf(stdout, "\n\nPinging %s with SNMP protocol", sp.sdip);
  
  fprintf(stdout, "\nCommunity: %s", sp.sCommunity);

  // Create first query
  if (sq->doQuery(NULL) == SNMPAPI_FAILURE) {
   // Something is wrong, terminate application
   rv = 0;
   goto terminate;
  }

  // Loop until STOP message from SnmpQuery class
  MSG msg;
  while (GetMessage(&msg, NULL, 0, 0)) {
    if (msg.message == WM_SNMP_MSG_STOP)
      break;
    TranslateMessage(&msg);    
    DispatchMessage(&msg);     
  }
  
  // Cleanup WinSNMP and destroy class 
terminate:
  fprintf(stdout, "\n");
  sq->Cleanup();
  delete sq;

  // Free the notification window handle
  freeNotificationHandle();

doExit:
  return rv;
}

/*
  printParameters

  Prints parameters
*/
void printParameters(void)
{
  smiOID tOid;
  smiVALUE tValue;
	
  fprintf(stdout, "\n\nCurrent parameters:\n");
  fprintf(stdout, "\nRemote agent IP address: %s", sp.sdip);
  fprintf(stdout, "\nPrint WinSNMP module version: %s",  sp.pingVersion ? "TRUE" : "FALSE");
  fprintf(stdout, "\nRetransmits: %d", sp.Retransmit);
  fprintf(stdout, "\nTimeout (ms): %d", sp.Timeout);
  fprintf(stdout, "\nRetransmit mode: %s", (sp.RetransmitMode == SNMPAPI_ON) ? "ON" : "OFF");
  fprintf(stdout, "\nCommunity: %s", sp.sCommunity);
  fprintf(stdout, "\nBrowse whole tree: %s", sp.BrowseWholeTree ? "TRUE" : "FALSE");
  fprintf(stdout, "\nBrowse system info: %s", sp.BrowseSystemInfo ? "TRUE" : "FALSE");
  fprintf(stdout, "\nBrowse subtrees: %s", sp.BrowseDepth ? "TRUE" : "FALSE");
  fprintf(stdout, "\nOid to string conversion: %s", sp.OidToStringConversion ? "TRUE" : "FALSE");
  if (sp.startOid) {
    fprintf(stdout, "\nStart OID: %s", sp.startOid);
    if (SnmpStrToOid(sp.startOid, &tOid) == SNMPAPI_SUCCESS) {
      fprintf(stdout, " [");
      sq->SnmpOutputValue(tOid, tValue, OUTPUT_OID_NAME);
      SnmpFreeDescriptor(SNMP_SYNTAX_OID, (smiLPOPAQUE) &tOid);
      fprintf(stdout, "]");
    }
  }
  if (sp.stopOid) {
    fprintf(stdout, "\nStop OID:  %s", sp.stopOid);
    if (SnmpStrToOid(sp.stopOid, &tOid) == SNMPAPI_SUCCESS) {
      fprintf(stdout, " [");
      sq->SnmpOutputValue(tOid, tValue, OUTPUT_OID_NAME);
      SnmpFreeDescriptor(SNMP_SYNTAX_OID, (smiLPOPAQUE) &tOid);
      fprintf(stdout, "]");
    }
  }
  fprintf(stdout, "\nSet SNMP value: %s", sp.SetSnmpValue ? "TRUE" : "FALSE");
  if (sp.sValueToSet) {
    tValue.syntax = sp.ValueType;
    fprintf(stdout, ", "); 
    if (sp.startOid) {
      fprintf(stdout, " %s", sp.startOid);
      if (SnmpStrToOid(sp.startOid, &tOid) == SNMPAPI_SUCCESS) {
       fprintf(stdout, " [");
       sq->SnmpOutputValue(tOid, tValue, OUTPUT_OID_NAME);
       SnmpFreeDescriptor(SNMP_SYNTAX_OID, (smiLPOPAQUE) &tOid);
       fprintf(stdout, "]");
      }
    }
    fprintf(stdout, ", "); 
    sq->SnmpOutputValue(tOid, tValue, OUTPUT_VALUE_TYPE);
    fprintf(stdout, "'%s'", sp.sValueToSet);
  }
  fprintf(stdout, "\nResolve SNMP agent IP address: %s", sp.ResolveAgentIp ? "TRUE" : "FALSE");
  fprintf(stdout, "\nPrint value types: %s", sp.PrintValueType ? "TRUE" : "FALSE");
  fprintf(stdout, "\nPrint line numbers: %s", sp.PrintLineNumber ? "TRUE" : "FALSE");
  fprintf(stdout, "\nPrint parameters: %s", sp.PrintParameters ? "TRUE" : "FALSE");
  fprintf(stdout, "\nPing specified agent until interrupted: %s", sp.PingUntilInterrupted ? "TRUE" : "FALSE");
  fprintf(stdout, "\n");
}


/*
  setDefaultParameters

  Sets default parameters which are efficient when
	no arguments are set
*/
void setDefaultParameters(void)
{
  sp.sdip[0] = 0;
  sp.pingVersion = FALSE;             // Write a SNMP version?
  sp.Retransmit = 1;                  // times
  sp.Timeout = 5000;                  // (5s) miliseconds
  sp.RetransmitMode = SNMPAPI_ON;     // Do a retransmitting
  sp.sCommunity = PUBLIC_COMMUNITY;
  sp.BrowseWholeTree = FALSE;
  sp.BrowseSystemInfo = FALSE;
  sp.BrowseDepth = FALSE;
  sp.OidToStringConversion = TRUE;
  sp.CustomStartStopConditions = FALSE;
  sp.startOid = NULL;
  sp.stopOid = NULL;
  sp.ResolveAgentIp = FALSE;
  sp.PrintValueType = FALSE;
  sp.PrintLineNumber = FALSE;
  sp.CurrentLineNumber = 1;
  sp.PrintParameters = FALSE;
  sp.SetSnmpValue = FALSE;
  sp.sValueToSet = NULL;
  sp.ValueType = SNMP_SYNTAX_NULL;
  sp.PingUntilInterrupted = FALSE;
  sp.PingInterval = 1000;
}

/*
  getArgs

  Parses arguments and overrides default parameters.
	No space between switch and parameter is allowed.
*/
void getArgs(int argc, char *argv[])
{
  int i, j;
  int dip[4];

  if (argc <= 1)
    usage();
  while (--argc) {
    register char *p = *++argv;
    // first argument byte should be the '-' or '/'
    if ((p[0] == '-') || (p[0] == '/')) {
      // second argument byte should be the switch name
      switch (tolower(p[1])) {
        case 'v':
          sp.pingVersion = TRUE;
          break;
        case 'c':
          sp.sCommunity = &p[2];
          break;
        case 'f':
          sp.BrowseWholeTree = TRUE;
          break;
        case 'i':
          sp.BrowseSystemInfo = TRUE;
          break;
        case 'b':
          sp.BrowseDepth = TRUE;
          break;
	case 's':
          sp.startOid = &p[2];
          sp.CustomStartStopConditions = TRUE;
   	  break;
	case 'e':
          sp.stopOid = &p[2];
          sp.CustomStartStopConditions = TRUE;
	  break;
	case 'm':
	  switch (p[2]) {
	    case 'o':
              sp.ValueType = SNMP_SYNTAX_OCTETS;
   	      break;
  	    default:
              break;
	  }
          sp.sValueToSet = &p[3];
          sp.SetSnmpValue = TRUE;
 	  break;
	case 'a':
          sp.ResolveAgentIp = TRUE;
	  break;
        case 'o':
          sp.OidToStringConversion = FALSE;
          break;
        case 't':
          sp.PrintValueType = TRUE;
          break;
        case 'l':
          sp.PrintLineNumber = TRUE;
 	  break;
        case 'u':
          sp.Timeout = atoi(&p[2]);
	  break;
        case 'r':
          sp.Retransmit = atoi(&p[2]);
	  break;
        case 'p':
          sp.PrintParameters = TRUE;
 	  break;
        case 'd':
          sp.PingUntilInterrupted = TRUE;
 	  if (p[2] != '\0') 
	    sp.PingInterval = atoi(&p[2]);
	  break;
        default :
 	  // print usage if switch is unknown
          usage();
      }
    }
    else {
      // Get destination IP address
      for (i = 0; i < 4; i++)
        dip[i] = -1;
      j = sscanf(p, "%d.%d.%d.%d", &dip[0], &dip[1], &dip[2], &dip[3]);
      if (j != 4)
        usage();
      for (i = 0; i < 4; i++) {
        if ((dip[i] < 0) || (dip[i] > 255))
          usage();
      }
      sprintf(&sp.sdip[0], "%d.%d.%d.%d", dip[0], dip[1], dip[2], dip[3]);
    }
  }
}

/*
  usage

  Prints command line usage
*/
void usage(void)
{
  static char *msg[] = {
		"\n",
    "\nUsage: snmpping [-v] [-c(community)] [-r] [-u] [-f] [-b] [-i] [-o]",
		"\n                [-a] [-s(start OID)] [-e(end OID)] [-m(type)(value)]",
		"\n                [-l] [-t] [-p] [-d(interval)] ip.ip.ip.ip",
		"\n",
    "\nOptions (No space between switch and parameter):",
		"\n",
    "\n   -v              Display WinSNMP module version.",
    "\n   -c community    Community [default = public].",
    "\n   -r              Number of retransmits [default = 1].",
    "\n   -u              Timeout per retransmit in miliseconds [default = 5000].",
    "\n   -f              Browse whole MIB tree.",
    "\n   -b              Browse subtrees (-s must be specified)."
    "\n   -i              Browse system info.",
    "\n   -o              Don't resolve object ID names.",
    "\n   -a              Resolve remote SNMP agent IP address.",
    "\n   -s start-Oid    Start OID [default = 1.3.6.1.2.1.1.3 (sysUpTime)].",
    "\n   -e end-Oid      End OID [default = 1.3.6.1.2.1.1.4 (sysContact)].",
    "\n   -m type value   Set value (-s must be specified).",
    "\n                   Types: o octets, (other types currently not supported).",
    "\n   -l              Print line numbers.",
    "\n   -t              Print value types.",
    "\n   -p              Print parameters.",
		"\n   -d interval-ms  Ping util interrupted [default interval = 1000 miliseconds].",
    "\n",
		"\nExamples: \n",
		"\nPing: snmpping ip.ip.ip.ip",
		"\nPing: snmpping -d ip.ip.ip.ip",
		"\nInfo: snmpping -i -t ip.ip.ip.ip",
		"\nWalk: snmpping -f -t ip.ip.ip.ip",
		"\nWalk: snmpping -f -s1.3.6.1 ip.ip.ip.ip",
		"\nWalk: snmpping -b -s1.3.6.1.2.1.1.1 ip.ip.ip.ip",
		"\nWalk: snmpping -s1.3.6.1 -e1.3.6.1.2.1.2 ip.ip.ip.ip",
		"\nSet:  snmpping -mo\"Contact person name\" -s1.3.6.1.2.1.1.4.0 ip.ip.ip.ip",
		"\nPerformance: snmpping -d0 -l ip.ip.ip.ip",
		"\n",
    "\nUse 'snmpping > file' or 'snmpping | more' when output is larger than screen.",
    "\n",
    ""
  };
  register char **pmsg = msg;

  while (**pmsg)
    fprintf(stdout, *pmsg++);
  exit (0);
}

/*
  allocateNotificationHandle

  Registers windows class and and creates a
	window for WinSNMP messages.
*/
WINAPI_STATUS allocateNotificationHandle(void)
{
  // Create new window class
  // Register the window class for main window.                                                           */
  wc.style = 0;                       // Class style.
  wc.lpfnWndProc = (WNDPROC) notificationSnmpWndProc; // Window procedure for this class.
  wc.cbClsExtra = 0;                  // No per-class extra data.
  wc.cbWndExtra = 0;                  // No per-window extra data.
  wc.hInstance = 0;//hInstance;           // Application that owns the class.
  wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
  wc.hCursor = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = GetStockObject(GRAY_BRUSH);
  wc.lpszMenuName =  "";   // Name of menu resource in .RC file.
  wc.lpszClassName = "MG-SOFT_SNMPPING_NOTIFICATION_CLASS"; // Name used in call to CreateWindow.

  if (RegisterClass(&wc) == 0)
    return WINAPI_FAILURE;

  // Create a main window for this application instance.
  hWinSnmpNotification = CreateWindow (
      "MG-SOFT_SNMPPING_NOTIFICATION_CLASS",
      "MG-SOFT WinSNMP Ping Notification Window C-1996",
      WS_OVERLAPPEDWINDOW,            // Window style.
      CW_USEDEFAULT,                  // Default horizontal position.
      CW_USEDEFAULT,                  // Default vertical position.
      CW_USEDEFAULT,                  // Default width.
      CW_USEDEFAULT,                  // Default height.
      NULL,                           // Overlapped windows have no parent.
      NULL,                           // Use the window class menu.
      NULL,                           // This instance owns this window.
      NULL                            // Pointer not needed.
  );
  return (hWinSnmpNotification == NULL) ? WINAPI_FAILURE : WINAPI_SUCCESS;
}

/*
  freeNotificationHandle

  Unregisters windows class and and destroyes a
	window for WinSNMP messages.
*/
WINAPI_STATUS freeNotificationHandle(void)
{
  // Destroy hidden window
  DestroyWindow(hWinSnmpNotification);
  // Unregister class
  UnregisterClass((LPCTSTR) &wc.lpszClassName, NULL);
  return WINAPI_SUCCESS;
}

/*
  startupWinsock

	Starts winsock (needed for name resolving)
*/
WINAPI_STATUS startupWinsock(void)
{
  WORD wVersionRequested;
  WSAData  wsaData;
  int err;
  struct hostent *he;
  in_addr *(*heLocalIP);
  in_addr localHostIp;

  // Get local IP
  wVersionRequested = MAKEWORD(1, 1);
  err = WSAStartup(wVersionRequested, &wsaData);
  if (err == 0) {
    if ((LOBYTE(wsaData.wVersion) != 1) ||
        (HIBYTE(wsaData.wVersion) != 1)) {
      WSACleanup();
    }
    else {
      // Retreive local host IP address
      if (0 == gethostname(&sp.LocalHostName[0], 200)) {
        he = gethostbyname(&sp.LocalHostName[0]);
        if (he != NULL) {
          heLocalIP = (struct in_addr **) he->h_addr_list;
          while (heLocalIP != NULL) {
            localHostIp.S_un.S_addr = (*heLocalIP)[0].S_un.S_addr;
            strncpy(&sp.LocalHostIp[0], inet_ntoa(localHostIp), 20);
            heLocalIP++;
            break;
          }
        }
      }
    }
  }
  return WINAPI_SUCCESS;
}

/*
  shutdownWinsock

  Shutdowns winsock.
*/
WINAPI_STATUS shutdownWinsock(void)
{
  WSACleanup();
  return WINAPI_SUCCESS;
}



/*
  notificationSnmpWndProc

	This function is called by the system when
	application has an message in the input message
	que.
*/
LONG APIENTRY notificationSnmpWndProc( HWND hWnd, UINT message, UINT wParam, LONG lParam)
{
  LONG rv = 0;

  // fprintf(stdout, "\nmessage %d", message);
  switch (message) {
    case WM_CREATE:
      break;
    case WM_DESTROY:                  
      break;
    case WM_SNMP_MSG:
      if (sq)
       sq->NotificationCallback();
      break;

    default:
      return (DefWindowProc(hWnd, message, wParam, lParam));
  }
  return rv;
}

/*
  SnmpQuery::SnmpQuery

  SnmpQuery class constructor
*/
SnmpQuery::SnmpQuery(HANDLE notifyHandle, UINT notifyMessage)
{
  this->notifyHandle = notifyHandle;
  this->notifyMessage = notifyMessage;
  hSnmpSession = 0;
  bInitialized = FALSE;
  bWinSnmpStarted = FALSE;
}

/*
  SnmpQuery::~SnmpQuery

  SnmpQuery class destructor
*/
SnmpQuery::~SnmpQuery(void)
{
}

/*
  SnmpQuery::Startup

  Startup WinSNMP module
*/
WINAPI_STATUS SnmpQuery::Startup(void)
{
  SNMPAPI_STATUS s;
  WINAPI_STATUS rv = WINAPI_SUCCESS;

  if (!g_SnmpInitialized) {
    s = SnmpStartup(&nMajorVersion, &nMinorVersion,
                    &nLevel, &nTranslateMode,
                    &nRetransmitMode);
    rv = (s == SNMPAPI_SUCCESS) ? WINAPI_SUCCESS : WINAPI_FAILURE;
  }

  if (sp.pingVersion) {
    fprintf(stdout, "\n\nWinSNMP module version:\n");
    fprintf(stdout, "\nMajor version: %d", nMajorVersion);
    fprintf(stdout, "\nMinor version: %d", nMinorVersion);
    fprintf(stdout, "\nLevel: %d", nLevel);
    fprintf(stdout, "\nTranslate mode: %s", sTranslateMode[(nTranslateMode < 3) ? nTranslateMode : 3]);
    fprintf(stdout, "\nRetransmit mode: %s", sRetransmitMode[(nRetransmitMode < 2) ? nRetransmitMode : 2]);
		fprintf(stdout, "\n");
  }
  return rv;
}

/*
  SnmpQuery::Cleanup

  Cleanup WinSNMP module
*/
WINAPI_STATUS SnmpQuery::Cleanup(void)
{
  SNMPAPI_STATUS s;
  WINAPI_STATUS rv;

  // Free descriptors
  if (oid.ptr)
    s = SnmpFreeDescriptor(SNMP_SYNTAX_OID, (smiLPOPAQUE) &oid);
  if (stopOid.ptr)
    s = SnmpFreeDescriptor(SNMP_SYNTAX_OID, (smiLPOPAQUE) &stopOid);

  s = SnmpCleanup();
  rv = (s == SNMPAPI_SUCCESS) ? WINAPI_SUCCESS : WINAPI_FAILURE;
  return rv;
}

/*
  SnmpQuery::doQuery

  Creates a first query. All other queries will be 
	created in WinSNMP message handler function.
*/
WINAPI_STATUS SnmpQuery::doQuery(char *qName)
{
  SNMPAPI_STATUS s;
  smiVALUE tValue;

  if (hSnmpSession == 0) {
    hSnmpSession = SnmpOpen(notifyHandle, notifyMessage);
  }
  if (hSnmpSession == SNMPAPI_FAILURE) {
    return WINAPI_FAILURE;
  }
  // Set agent address
  strcpy(&sIPAgent[0], sp.sdip);
  // Set manager address
  strcpy(&sIPManager[0], "0.0.0.0");
  // Set community
  smiCommunity.len = strlen(sp.sCommunity);
  smiCommunity.ptr = (smiLPBYTE) sp.sCommunity;
  // Set the WinSNMP internals
  hAgentEntity = SnmpStrToEntity(hSnmpSession, &sIPAgent[0]);
  hManagerEntity = SnmpStrToEntity(hSnmpSession, &sIPManager[0]);
  hViewContext = SnmpStrToContext(hSnmpSession, &smiCommunity);
  s = SnmpSetRetry(hManagerEntity, sp.Retransmit);
  s = SnmpSetTimeout(hManagerEntity, sp.Timeout / 10);
  s = SnmpSetRetry(hAgentEntity, sp.Retransmit);
  s = SnmpSetTimeout(hAgentEntity, sp.Timeout / 10);
  s = SnmpSetRetransmitMode(sp.RetransmitMode ? SNMPAPI_ON : SNMPAPI_OFF);
  bInitialized = TRUE;

  // Initialize start OID and stop OID  
  // first: highest priority, last: lowest priority
  oid.len = 0;
  stopOid.len = 0;
  if (sp.PingUntilInterrupted) {
    // Start conditions set to sysUpTime
    SnmpStrToOid("1.3.6.1.2.1.1.3.0", &oid);
    // Stop conditions set to sysContact
    SnmpStrToOid("1.3.6.1.2.1.1.4", &stopOid);
  }
  else if (sp.SetSnmpValue) {
    // Start condition set by arguments
    if (sp.startOid) {
      s = SnmpStrToOid(sp.startOid, &oid);
      s = SnmpStrToOid(sp.startOid, &stopOid);
      sp.startOid = NULL;
    }
  }
  else if (sp.CustomStartStopConditions) {
    // Start condition set by arguments
    if (sp.startOid) 
      s = SnmpStrToOid(sp.startOid, &oid);
    // Stop conditions set by arguments
    if (!sp.BrowseWholeTree) {
      if (sp.stopOid)
        s = SnmpStrToOid(sp.stopOid, &stopOid);
      else {
        s = SnmpStrToOid(sp.startOid, &stopOid);
        (*(stopOid.ptr + stopOid.len - 1))++; 
      }
    }
    else {
      // no stop oid condition
      stopOid.len = 0;
      stopOid.ptr = NULL;
    }
  } 
  else if (sp.BrowseWholeTree) {
    SnmpStrToOid("1.3.6.1.2.1.1.1", &oid);
    // no stop oid condition
    stopOid.len = 0;
    stopOid.ptr = NULL;
  }
  else if (sp.BrowseSystemInfo) {
    // System info starts with sysDescriptor
    SnmpStrToOid("1.3.6.1.2.1.1.1", &oid);
    // Stop conditions set to interfaces
    SnmpStrToOid("1.3.6.1.2.1.2", &stopOid);
  }
  else {
    // Start conditions set to sysUpTime
    SnmpStrToOid("1.3.6.1.2.1.1.3", &oid);
    // Stop conditions set to sysContact
    SnmpStrToOid("1.3.6.1.2.1.1.4", &stopOid);
  }

  if (sp.BrowseDepth) {
    // System info starts with sysDescriptor
    // Start condition set by arguments
    if (sp.startOid) {
      s = SnmpStrToOid(sp.startOid, &oid);
      s = SnmpStrToOid(sp.startOid, &stopOid);
      stopOid.len--;
      (*(stopOid.ptr + stopOid.len - 1))++; 
    }
  }

  // Print start and stop OID
  if (oid.ptr) {
    fprintf(stdout, "\nStart OID: ");
    SnmpOutputValue(oid, tValue, OUTPUT_OID);
    fprintf(stdout, " [");
    SnmpOutputValue(oid, tValue, OUTPUT_OID_NAME);
    fprintf(stdout, "]");
  }
  if (stopOid.ptr) {
    fprintf(stdout, "\nStop OID:  ");
    SnmpOutputValue(stopOid, tValue, OUTPUT_OID);
    fprintf(stdout, " [");
    SnmpOutputValue(stopOid, tValue, OUTPUT_OID_NAME);
    fprintf(stdout, "]");
  }
  fprintf(stdout, "\n");

  // Check start and stop OIDs before query
  if ((oid.len > 0) && (stopOid.len > 0)) {
    // check if stop condition matches current oid
    smiINT oidCompareResult;
    if (SNMPAPI_SUCCESS == SnmpOidCompare(&stopOid, &oid, 0, &oidCompareResult)) {
      if (oidCompareResult == -1) {
         fprintf(stdout, "\nEnd OID is less than start OID.");
         fprintf(stdout, "\nNothing to do.");
	 return WINAPI_FAILURE;
      }
    }
  }

  if (sp.SetSnmpValue)
   SendSetRequest();
  else if (sp.PingUntilInterrupted)
    SendGetRequest();
  else
    SendGetNextRequest();
  return SNMPAPI_SUCCESS;
}

/*
  SnmpQuery::SendGetRequest

  Creates a first query. All other queries will be 
	created in WinSNMP message handler function.
*/
WINAPI_STATUS SnmpQuery::SendGetRequest(void)
{
  SNMPAPI_STATUS s;

  // Create a variable binding list
  hVbl = SnmpCreateVbl(hSnmpSession, &oid, NULL);
  // Request ID should be increased every time a new message is sent
  requestId = requestId + 1;
  // Create a PDU
  hPdu = SnmpCreatePdu(hSnmpSession, SNMP_PDU_GET, requestId, 0, 0, hVbl);
  // Send SNMP message
  s = SnmpSendMsg(hSnmpSession, hManagerEntity, hAgentEntity, hViewContext, hPdu);
  // Free resources
  s = SnmpFreeVbl(hVbl);
  s = SnmpFreePdu(hPdu);
  return SNMPAPI_SUCCESS;
}

/*
  SnmpQuery::SendGetNextRequest

  Creates a first query. All other queries will be 
	created in WinSNMP message handler function.
*/
WINAPI_STATUS SnmpQuery::SendGetNextRequest(void)
{
  SNMPAPI_STATUS s;

  // Create a variable binding list
  hVbl = SnmpCreateVbl(hSnmpSession, &oid, NULL);
  // Request ID should be increased every time a new message is sent
  requestId = requestId + 1;
  // Create a PDU
  hPdu = SnmpCreatePdu(hSnmpSession, SNMP_PDU_GETNEXT, requestId, 0, 0, hVbl);
  // Send SNMP message
  s = SnmpSendMsg(hSnmpSession, hManagerEntity, hAgentEntity, hViewContext, hPdu);
  // Free resources
  s = SnmpFreeVbl(hVbl);
  s = SnmpFreePdu(hPdu);
  return SNMPAPI_SUCCESS;
}

/*
  SnmpQuery::SendSetRequest

  Creates a SNMP set request.
*/
WINAPI_STATUS SnmpQuery::SendSetRequest(void)
{
  SNMPAPI_STATUS s;

  // Stop oid is same as start oid
  // Create a variable binding list
  s = SnmpOidCopy(&oid, &stopOid);
  // Create one VB with value set
  value.syntax = sp.ValueType;
  s = convertString2smiValue(&value, sp.sValueToSet);
  hVbl = SnmpCreateVbl(hSnmpSession, &oid, &value);
  requestId = requestId + 1;
  // Create a PDU
  hPdu = SnmpCreatePdu(hSnmpSession, SNMP_PDU_SET, requestId, 0, 0, hVbl);
  // Send SNMP message
  s = SnmpSendMsg(hSnmpSession, hManagerEntity, hAgentEntity, hViewContext, hPdu);
  // Free resources
  s = SnmpFreeVbl(hVbl);
  s = SnmpFreePdu(hPdu);
  return SNMPAPI_SUCCESS;
}


/*
  SnmpQuery::NotificationCallback

  If WinSNMP message is in the application input que
	this function ic called.

	At first it chechs the message reason, next it checks
	if stopOid condition is set. If stop condition is not
	set then it sends a new query. If stop condition is set
	the it sends a STOP message to the application. STOP 
	message is also sent if SnmpRecvMsg returns an error.
	Application will break the message loop and exit on STOP
	message.
*/
void SnmpQuery::NotificationCallback(void)
{
  SNMPAPI_STATUS s;
  HSNMP_ENTITY hRcvAgentEntity;
  HSNMP_ENTITY hRcvManagerEntity;
  HSNMP_CONTEXT hRcvViewContext;
  HSNMP_VBL hRcvVbl;
  HSNMP_PDU hRcvPdu;
  int outputFlags;

  smiINT32 requestId;
  smiINT pduType, errorStatus, errorIndex;

  int oidc;
  smiOID oid;
  smiVALUE value;

  int i;
  smiINT oidCompareResult;
  BOOL stopCondition = FALSE;
  BOOL setConfirmation = FALSE;

  // fprintf(stdout, "\n Got SNMP response");
  s = SnmpRecvMsg(hSnmpSession, &hRcvAgentEntity, &hRcvManagerEntity,
                                &hRcvViewContext, &hRcvPdu);
  if (s == SNMPAPI_FAILURE) {
    s = SnmpGetLastError(hSnmpSession);
    if (s == SNMPAPI_TL_TIMEOUT) {
      stopCondition = TRUE;
      fprintf(stdout, "\nRequest timed out.");
      while (!PostThreadMessage(GetCurrentThreadId(), WM_SNMP_MSG_STOP, 0, 0))
        ;
      return;
    }
    if (s == SNMPAPI_NOOP) {
      // MyWriteLn('NOOP');
      return;
    }
    // MyWriteLn('ERROR');
    // no more queries
    fprintf(stdout, "\nUnhandled WinSNMP error %d.", s);
    while (!PostThreadMessage(GetCurrentThreadId(), WM_SNMP_MSG_STOP, 0, 0))
      ;
    return;
  }

  s = SnmpGetPduData(hRcvPdu, &pduType, &requestId, &errorStatus, &errorIndex, &hRcvVbl);
  // Did we reach the end of agents' MIB tree?
  if (errorStatus != 0) {
    if (!sp.SetSnmpValue) {
      if (errorStatus == SNMP_ERROR_NOSUCHNAME) {
        stopCondition = TRUE;
        goto doExitCleanCallback;
      }
    }
    fprintf(stdout, "\nError returned by remote agent: %s (%d)", 
        	((errorStatus < MAX_SNMP_ERRORS) ? sSnmpErrors[errorStatus] : "unknown error"),
		errorStatus);
  }

  oidc = SnmpCountVbl(hRcvVbl);
  for (i = 1; i <= oidc; i++) {
    s = SnmpGetVb(hRcvVbl, i, &oid, &value);
    // remember first SNMP binding value for the next query
    if (i == 1) {
			// free oid descriptor from previous query
      if (oid.ptr) 
        SnmpFreeDescriptor(SNMP_SYNTAX_OID, (smiLPOPAQUE) &this->oid);
			// make oid for new query
      SnmpOidCopy(&oid, &this->oid);
      this->value.syntax = SNMP_SYNTAX_NULL;
    }
    // stop condition set ?
    if (this->stopOid.len > 0) {
      // check if stop condition matches current oid
      if (SNMPAPI_SUCCESS == SnmpOidCompare(&stopOid, &oid, 0, &oidCompareResult)) {
        if (oidCompareResult == -1) {
					// Stop condition is set. 
          stopCondition = TRUE;
        }
        else {
          outputFlags = OUTPUT_VALUE | OUTPUT_SEPARATOR | OUTPUT_NEWLINE;
					if (sp.OidToStringConversion)
            outputFlags |= OUTPUT_OID_NAME;
          else
            outputFlags |= OUTPUT_OID;
          if (sp.PrintValueType)
            outputFlags |= OUTPUT_VALUE_TYPE;
					if (sp.PrintLineNumber) {
						fprintf(stdout, "\n%d: ", sp.CurrentLineNumber++);
            outputFlags &= ~OUTPUT_NEWLINE;
					}
          SnmpOutputValue(oid, value, outputFlags);
				}
      }
    }
    else {
      outputFlags = OUTPUT_VALUE | OUTPUT_SEPARATOR | OUTPUT_NEWLINE;
      if (sp.OidToStringConversion)
        outputFlags |= OUTPUT_OID_NAME;
      else
        outputFlags |= OUTPUT_OID;
      if (sp.PrintValueType)
        outputFlags |= OUTPUT_VALUE_TYPE;
      if (sp.PrintLineNumber) {
	fprintf(stdout, "\n%d: ", sp.CurrentLineNumber++);
        outputFlags &= ~OUTPUT_NEWLINE;
      }
      SnmpOutputValue(oid, value, outputFlags);
    }
    //  Free receive descriptors
    s = SnmpFreeDescriptor(SNMP_SYNTAX_OID, (smiLPOCTETS) &oid);
    switch (value.syntax) {
      case SNMP_SYNTAX_OCTETS:
        s = SnmpFreeDescriptor(value.syntax, &value.value.string);
        break;
      case SNMP_SYNTAX_OID:
        s = SnmpFreeDescriptor(value.syntax, (smiLPOCTETS) &value.value.oid);
        break;
      case SNMP_SYNTAX_IPADDR:
        s = SnmpFreeDescriptor(value.syntax, &value.value.string);
        break;
    }
  }

 doExitCleanCallback:
  // Free WinSNMP internal data
  SnmpFreePdu(hRcvPdu);
  SnmpFreeVbl(hRcvVbl);
  SnmpFreeEntity(hRcvAgentEntity);
  SnmpFreeEntity(hRcvManagerEntity);
  SnmpFreeContext(hRcvViewContext);


  // Decide what to do next...
  // Ping until interrupted
  if (sp.PingUntilInterrupted) {
    stopCondition = TRUE;
    // Print remote SNMP agent IP address
    fprintf(stdout, " (%s)", sp.sdip);
    // Wait for interval
    Sleep(sp.PingInterval);
    // Send get-request to confirm set-request
    SnmpFreeVbl(hVbl);
    hVbl = SnmpCreateVbl(hSnmpSession, &this->oid, NULL);
    this->requestId++;
    SnmpFreePdu(hPdu);
    hPdu = SnmpCreatePdu(hSnmpSession, SNMP_PDU_GET, this->requestId, 0, 0, hVbl);
    s = SnmpSendMsg(hSnmpSession, hManagerEntity, hAgentEntity, hViewContext, hPdu);
    return;
  }
  // Set-value
  if (sp.SetSnmpValue) {
    stopCondition = TRUE;
    sp.SetSnmpValue = FALSE;
    fprintf(stdout, "\nSet confirmation...");
    // Send get-request to confirm set-request
    SnmpFreeVbl(hVbl);
    hVbl = SnmpCreateVbl(hSnmpSession, &this->oid, NULL);
    this->requestId++;
    SnmpFreePdu(hPdu);
    hPdu = SnmpCreatePdu(hSnmpSession, SNMP_PDU_GET, this->requestId, 0, 0, hVbl);
    s = SnmpSendMsg(hSnmpSession, hManagerEntity, hAgentEntity, hViewContext, hPdu);
    return;
  }
  // Mib-walk
  // Is mib-walk stop condition set?
  if (!stopCondition) {
    // yet another query
    SnmpFreeVbl(hPdu);
    hVbl = SnmpCreateVbl(hSnmpSession, &this->oid, NULL);
    this->requestId++;
    SnmpFreePdu(hPdu);
    hPdu = SnmpCreatePdu(hSnmpSession, SNMP_PDU_GETNEXT, this->requestId, 0, 0, hVbl);
    s = SnmpSendMsg(hSnmpSession, hManagerEntity, hAgentEntity, hViewContext, hPdu);
  }
  else {
    // Post finish message to thread main message handler
    fprintf(stdout, "\n\nFinished.");
    while (!PostThreadMessage(GetCurrentThreadId(), WM_SNMP_MSG_STOP, 0, 0))
      ;
  }
}

/*
  SnmpQuery::SnmpOutputValue

  Outputs the oid and value to stdout. 
	Flags define output style and contents.
*/
WINAPI_STATUS SnmpQuery::SnmpOutputValue(smiOID oid, smiVALUE value, BYTE flags)
{
  smiLPUINT32 oidp;
  long i;
  unsigned int ui;
  int j;
  int d,h,m,s,th;
  HWSM_NODE mibNode;
  int initialOidLen;
  char ts[2048];
  char ts1[2048];
  char ts2[5];
  smiBYTE *chp;
  BOOL printHex;

  ts[0] = '\0';
  ts1[0] = '\0';

  initialOidLen = oid.len;
  // append newline
  if (flags & OUTPUT_NEWLINE) {
   strcat(ts, "\n");
  }
  // append OID
  if (flags & OUTPUT_OID) {
    SnmpOidToStr(&oid, 500, &ts1[0]);
    strcat(ts, ts1);
  }
  // append OID string name
  if (flags & OUTPUT_OID_NAME) {
    for (j = oid.len; j > 0; j--) {
      oid.len = j;
      mibNode = WsmMapOidToNode(&oid);
      if (mibNode != 0) {
	i = 500;
	WsmGetNodeString(mibNode, WSM_INFO_NAME, (long *) &i, &ts1[0]);
	strcat(ts, ts1);
	// fprintf(stdout, "%s", ts);
	break;
      }
    }
    oid.len = initialOidLen;
    // append OID reminder to the string
    oidp = oid.ptr;
    oidp += j;
    for (ui = j; ui < oid.len; ui++) {
      if (ui != 0)
      strcat(ts, ".");
      sprintf(ts1, "%d", *oidp);
      strcat(ts, ts1);
      oidp++;
    }
  }
  // append separator
  if (flags & OUTPUT_SEPARATOR) {
    strcat(ts, " **** ");
  }
  if (flags & OUTPUT_VALUE_TYPE) {
    switch (value.syntax) {
      case SNMP_SYNTAX_INT:       strcat(ts, "(int,int32)"); break;
      //case SNMP_SYNTAX_INT32:     strcat(ts, "(int32)"); break;
      case SNMP_SYNTAX_UINT32:    strcat(ts, "(uint32)"); break;
      case SNMP_SYNTAX_CNTR32:    strcat(ts, "(cntr32)"); break;
      case SNMP_SYNTAX_GAUGE32:   strcat(ts, "(gauge32)"); break;
      case SNMP_SYNTAX_TIMETICKS: strcat(ts, "(timeticks)"); break;
      case SNMP_SYNTAX_CNTR64:    strcat(ts, "(cntr64)"); break;
      case SNMP_SYNTAX_OCTETS:    strcat(ts, "(octets)"); break;
      case SNMP_SYNTAX_BITS:      strcat(ts, "(bits)"); break;
      case SNMP_SYNTAX_OPAQUE:    strcat(ts, "(opaque)"); break;
      case SNMP_SYNTAX_IPADDR:    strcat(ts, "(ipaddr)"); break;
      case SNMP_SYNTAX_NSAPADDR:  strcat(ts, "(nsapaddr)"); break;
      case SNMP_SYNTAX_OID:       strcat(ts, "(oid)"); break;
      case SNMP_SYNTAX_NULL:      strcat(ts, "(null)"); break;
      case SNMP_SYNTAX_NOSUCHOBJECT:    strcat(ts, "(nosuchobject)"); break;
      case SNMP_SYNTAX_NOSUCHINSTANCE:  strcat(ts, "(nosuchinstance)"); break;
      case SNMP_SYNTAX_ENDOFMIBVIEW:    strcat(ts, "(endofmibview)"); break;
    }
    strcat(ts, " ");
  }
  // append value
  if (flags & OUTPUT_VALUE) {
    switch (value.syntax) {
      case SNMP_SYNTAX_INT:
        // case SNMP_SYNTAX_INT32:
        sprintf(ts1, "%d", value.value.sNumber);
        strcat(ts, ts1);
	break;
      case SNMP_SYNTAX_UINT32:
      case SNMP_SYNTAX_CNTR32:
      case SNMP_SYNTAX_GAUGE32:
	sprintf(ts1, "%d", value.value.uNumber);
	strcat(ts, ts1);
	break;
      case SNMP_SYNTAX_TIMETICKS:
	d = value.value.uNumber;
	th = d % 100; d = d / 100;
	s = d % 60;   d = d / 60;
	m = d % 60;   d = d / 60;
	h = d % 24;
	d = d / 24;
	if (d == 1)
          sprintf(ts1, "%d day %02.02dh:%02.02dm:%02.02ds.%02.02dth", d, h, m, s, th);
	else
          sprintf(ts1, "%d days %02.02dh:%02.02dm:%02.02ds.%02.02dth", d, h, m, s, th);
	strcat(ts, ts1);
	break;
      case SNMP_SYNTAX_CNTR64:
	sprintf(ts1, "%x.%x", value.value.hNumber.hipart, value.value.hNumber.lopart);
	strcat(ts, ts1);
	strcat(ts, " (hex)");
	break;
      case SNMP_SYNTAX_OCTETS:
        if (value.value.string.len < 1) {
          strcat(ts, "(zero-length)");
          break;
 	}
        printHex = FALSE;
        chp = value.value.string.ptr;
        strcpy(ts1, "");
        for (ui = 0; ui < value.value.string.len; ui++) {
 	  if (((isprint(*chp) == 0) && 
              (isspace(*chp) == 0)) &&
              ((*chp == '\0') &&
              (ui != (value.value.string.len - 1)))) {
            // to print string only last char can be '\0'
            printHex = TRUE;
          }
          if (ui != 0)
            strcat(ts1, ".");
	  sprintf(ts2, "%02x", *chp++);
          strcat(ts1, ts2);
	}
        if (printHex) {
          // print octets
          strcat(ts, ts1);
          strcat(ts, " (hex)");
          break;
        }
	// print string
	memcpy(ts1, value.value.string.ptr, value.value.string.len);
	ts1[value.value.string.len] = '\0';
	strcat(ts, ts1);
	break;
      case SNMP_SYNTAX_BITS:
	chp = value.value.string.ptr;
	for (ui = 0; ui < value.value.string.len; ui++) {
          if (ui != 0)
            strcat(ts, ".");
          sprintf(ts1, "%02x", *chp++);
          strcat(ts, ts1);
        }
	strcat(ts, " (hex)");
	break;
      case SNMP_SYNTAX_OPAQUE:
	chp = value.value.string.ptr;
	for (ui = 0; ui < value.value.string.len; ui++) {
          if (ui != 0)
            strcat(ts, ".");
          sprintf(ts1, "%02x", *chp++);
          strcat(ts, ts1);
	}
	strcat(ts, " (hex)");
	break;
      case SNMP_SYNTAX_IPADDR:
	chp = value.value.string.ptr;
	for (ui = 0; ui < value.value.string.len; ui++) {
          if (ui != 0)
            strcat(ts, ".");
          sprintf(ts1, "%d", *chp++);
            strcat(ts, ts1);
	}
        break;
      case SNMP_SYNTAX_NSAPADDR:
	chp = value.value.string.ptr;
	for (ui = 0; ui < value.value.string.len; ui++) {
          if (ui != 0)
            strcat(ts, ".");
          sprintf(ts1, "%02x", *chp++);
          strcat(ts, ts1);
	}
	strcat(ts, " (hex)");
	break;
      case SNMP_SYNTAX_OID:
        // check if OID is NULL-OID 0.0 
        if (value.value.oid.len == 2) {
          oidp = value.value.oid.ptr;
          if (*oidp++ == 0) 
            if (*oidp == 0)
              strcat(ts, "(null-oid) ");
        }
      	//  find a string alias for the OID
        initialOidLen = value.value.oid.len;
        for (j = value.value.oid.len; j > 0; j--) {
          value.value.oid.len = j;
          mibNode = WsmMapOidToNode(&value.value.oid);
          if (mibNode != 0) {
            i = 500;
            WsmGetNodeString(mibNode, WSM_INFO_NAME, (long *) &i, &ts1[0]);
            strcat(ts, ts1);
            break;
          }
	}
        value.value.oid.len = initialOidLen;
	// append OID reminder to the string
	oidp = value.value.oid.ptr;
	oidp += j;
	for (ui = j; ui < value.value.oid.len; ui++) {
          if (ui != 0)
            strcat(ts, ".");
          sprintf(ts1, "%d", *oidp);
          strcat(ts, ts1);
          oidp++;
        }
	break;
      case SNMP_SYNTAX_NOSUCHOBJECT:
	strcat(ts, "NOSUCHOBJECT");
	break;
      case SNMP_SYNTAX_NOSUCHINSTANCE:
	strcat(ts, "NOSUCHINSTANCE");
	break;
      case SNMP_SYNTAX_ENDOFMIBVIEW:
	strcat(ts, "ENDOFMIBVIEW");
	break;
      case SNMP_SYNTAX_NULL:
	strcat(ts, "NULL");
	break;
      default:
	return WINAPI_FAILURE;
    }
  }
  // Output result
  fprintf(stdout, "%s", ts);

  return WINAPI_SUCCESS;
}

SNMPAPI_STATUS convertString2smiValue(smiLPVALUE value, char *buffer)
{
  SNMPAPI_STATUS rv = SNMPAPI_SUCCESS;
  LPSTR ts;
  int i, n;
  smiOID tOid;
  smiUINT32 tOidVal[MAXOBJIDSTRSIZE];
  smiVALUE tValue;
  smiBYTE tValueVal[500];

  switch (value->syntax) {
    case SNMP_SYNTAX_INT:
    // case SNMP_SYNTAX_INT32:
      value->value.sNumber = atol(buffer);
      break;
    case SNMP_SYNTAX_UINT32:
    case SNMP_SYNTAX_CNTR32:
    case SNMP_SYNTAX_GAUGE32:
      value->value.uNumber = atol(buffer);
      break;
    case SNMP_SYNTAX_TIMETICKS:
      // To do!
      value->value.uNumber = 12345;
      break;
    case SNMP_SYNTAX_CNTR64:
      // To do!
      value->value.hNumber.hipart = 12345;
      value->value.hNumber.lopart = 12345;
      break;
    case SNMP_SYNTAX_OCTETS:
      value->value.string.len = strlen(buffer); 
      value->value.string.ptr = (smiBYTE *) malloc(value->value.string.len * sizeof(smiBYTE));
      if (value->value.string.ptr) {
        memcpy(value->value.string.ptr, buffer, value->value.string.len * sizeof(smiBYTE));
      }
      else {
        rv = SNMPAPI_FAILURE;
      }
			break;
    case SNMP_SYNTAX_BITS:
    case SNMP_SYNTAX_OPAQUE:
    case SNMP_SYNTAX_IPADDR:
      tValue.syntax = value->syntax;
      tValue.value.string.len = 0;
      tValue.value.string.ptr = tValueVal;
      ts = (LPSTR) buffer;
      for (n = 0; n < 4; n++) {
        if (sscanf(ts, "%d", &i) == 0) {
          rv = SNMPAPI_FAILURE;
          break;
        }
        tValue.value.string.len++;
        tValue.value.string.ptr[n] = i;
        while ((*ts != '.') && (*ts != '\0'))
          ts++;
        if (*ts == '\0') break;
        ts++;         // One more for the first digit
        if (*ts == '\0') break;
      }
      if (n != 3) {
        rv = SNMPAPI_FAILURE;
        break;
      }
      // Allocate memory for value.oid value and copy oid
      // leave syntax unchanged
      value->value.string.ptr = (smiBYTE *) malloc(tValue.value.string.len * sizeof(smiBYTE));
      if (value->value.string.ptr) {
        value->value.string.len = tValue.value.string.len;
        memcpy(value->value.string.ptr, tValue.value.string.ptr, tValue.value.string.len * sizeof(smiBYTE));
      }
      else {
        rv = SNMPAPI_FAILURE;
      }
      break;
    case SNMP_SYNTAX_NSAPADDR:
      break;
    case SNMP_SYNTAX_OID:
      // Use SnmpStrToOid ...
      tOid.len = 0;
      tOid.ptr = &tOidVal[0];
      value->value.oid.len = 0;
      n = 0;
      ts = (LPSTR) buffer;
      while (1) {
        if (sscanf(ts, "%d", &i) == 0) {
          rv = SNMPAPI_FAILURE;
          break;
        }
        tOid.len++;
        tOid.ptr[n++] = i;
        while ((*ts != '.') && (*ts != '\0'))
          ts++;
        if (*ts == '\0') break;
        ts++;         // One more for the first digit
        if (*ts == '\0') break;
      }
      if (rv != TRUE) 
        break;
      // Allocate memory for value.oid value and copy oid
      value->value.oid.ptr = (smiUINT32 *) malloc(tOid.len * sizeof(smiUINT32));
      if (value->value.oid.ptr) {
        value->value.oid.len = tOid.len;
        memcpy(value->value.oid.ptr, tOid.ptr, tOid.len * sizeof(smiUINT32));
      }
      else {
        rv = SNMPAPI_FAILURE;
      }
      break;
    case SNMP_SYNTAX_NULL:
    case SNMP_SYNTAX_NOSUCHOBJECT:
    case SNMP_SYNTAX_NOSUCHINSTANCE:
    case SNMP_SYNTAX_ENDOFMIBVIEW:
      value->value.empty = 0;
      break;
    default:
      rv = SNMPAPI_FAILURE;
  }
  return rv;
}

