/*
 * TemplatePortDriver.java
 * 
 * Copyright (c) 1996-1997 Central Data Corporation
 * All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software or its
 * documentation, or derivatives or compilations of them, is described
 * in the file "license.html".
 *
 * CENTRAL DATA CORPORATION IS MAKING THE SOFTWARE AND ITS DOCUMENTATION
 * AVAILABLE "AS IS" FOR NO FEE. CENTRAL DATA CORPORATION MAKES NO
 * REPRESENTATIONS OR WARRANTIES WITH REGARD TO THIS SOFTWARE OR ITS
 * DOCUMENTATION, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. CENTRAL DATA CORPORATION SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, COPYING,
 * MODIFYING, OR DISTRIBUTING THIS SOFTWARE OR ITS DOCUMENTATION, OR
 * DERIVATIVES OR COMPILATIONS OF THEM.
 */

package portio;

import java.util.*;
import java.io.*;
import java.net.*;

/**
 * This is a template serial/parallel driver.
 * This class must be instantiated before any
 * Port objects can be created.
 *
 * Note that Runnable only needs to be implemented if a thread
 * must be created to handle incoming data from the device.
 *
 * @version $Revision: 2.1 $ ($Date: 1997/02/19 18:00:00 $)
 */
public class TemplatePortDriver extends PortDriver implements Runnable {

    private Thread receiver;            // handles all incoming data

    /**
     * This creates a new instance of a PortDriver with specified options.
     * For a driver that supports native (built-in) ports for the machine
     * Java(TM) is running on, nothing more than options need to be passed.
     * If the driver supports external ports, there may need to be some sort
     * of identifier passed as well. Since EtherLite devices are identified
     * by their TCP/IP host name, the EtherLiteDriver must be passed that
     * host name at startup.
     *
     * @param options lowest three bits set messageLevel defined in PortDriver
     * @exception PortIOException if communication with device not possible
     */
    public TemplatePortDriver( int options )
        throws PortIOException
    {
        info.driverRev = "V1.1";
        messageLevel = options & 7;

        if( messageLevel >= DEVALL ) System.out.println( "Opening TemplatePort device" );

        // initialize the device and fill the info object
        initDevice();

        // make port array to hold private port info
        portPrivate = new PortPrivate[info.serialCount + info.parallelCount];
        portState = new int[info.serialCount + info.parallelCount];

        // sign on
        if( messageLevel >= DEV ) System.out.println( "Manufacturer: " + info.manufacturer + ", Device: " + info.deviceId + ", Revision: " + info.deviceRev );

        // start the receive thread
        receiver = new Thread( this );
        receiver.start();

        if( messageLevel >= DEVALL ) System.out.println( "TemplatePort device opened" );
    }

    /**
     * This creates a new instance of a PortDriver with default options (no messages).
     *
     * @exception PortIOException if communication with device not possible
     */
    public TemplatePortDriver( )
        throws PortIOException
    {
        this( 0 );
    }

    /**
     * Not called directly, used only indirectly through Port objects.
     * Opens a port of the appropriate type with specified open options.
     *
     * @param port Port object of desired type
     * @exception PortIOException if <i>port</i> is out of range
     */
    void open( Port port )
        throws PortIOException
    {
        int portNumber = port.portNumber;
        // see which port type they're opening
        if( port instanceof SerialPort ) {
            if( ( portNumber < 0 ) || ( portNumber >= info.serialCount ) )
                throw new PortIOException( "Serial port " + portNumber + " out of range" );
        }
        else {
            if( ( portNumber < 0 ) || ( portNumber >= info.parallelCount ) )
                throw new PortIOException( "Parallel port " + portNumber + " out of range" );

            // convert to index into portPrivate array
            portNumber += info.serialCount;
        }

        if( messageLevel >= PORTALL ) System.out.println( "Open for port " + portNumber );

        // make sure we exclusively have the port
        synchronized( this ) {
            while( portState[portNumber] != PORT_UNOPENED ) {
                if( portState[portNumber] == PORT_CLOSING )
                    try { wait(); } catch( InterruptedException e ) {}
                else throw new PortIOException ( "Port already in use" );
            }
            portState[portNumber] = PORT_OPENING;
        }
        // let the super class do most of the work
        super.open( port, (PortPrivate)new TemplatePortPrivate( portNumber, port ) );
    }

    /**
     * The receive thread; manages incoming data from the device. This data may
     * come as a stream which this thread can wait() on, or this thread may need to
     * poll to receive the data.
     */
    public void run()
    {
        int port = 0;
        int length = 0;
        byte data[] = new byte[256];

        if( messageLevel >= DEVALL ) System.out.println( "Receiver thread started" );

        try {
            while( true ) {
                // either call a routine or do something here to wait for data from
                // the device; the following values should be set appropriately:
                //
                //   port - the port number the data relates to
                //   length - the length of data from the port
                //   data[] - the actual data received

                if( messageLevel == ALL ) System.out.println( "Receive for port " + port );
                // if this port still has a PortPrivate object, then pass the data to it
                if( portPrivate[port] != null )
                    ((TemplatePortPrivate)portPrivate[port]).callback( data, length );
            }
        }
        catch( PortIOException e ) { System.out.println( "Receiver thread: " + e ); }
    }

    /*
     * Initializes the device for service and fills in info about device.
     */
    private void initDevice()
        throws PortIOException
    {
        // lots of device-specific code goes here to get the device started
        // from ground-zero

        // the following may be fixed for a particular PortDriver, or it may change
        // if the PortDriver supports multiple devices (as the EtherLiteDriver does)
        info.serialCount = 2;
        info.parallelCount = 1;
        info.manufacturer = "Great Company";
        info.deviceId = "Great Device";
        info.deviceRev = "1.0";
    }
}