/*
 * SerialPortParams.java
 * 
 * Copyright (c) 1996-1997 Central Data Corporation
 * All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software or its
 * documentation, or derivatives or compilations of them, is described
 * in the file "license.html".
 *
 * CENTRAL DATA CORPORATION IS MAKING THE SOFTWARE AND ITS DOCUMENTATION
 * AVAILABLE "AS IS" FOR NO FEE. CENTRAL DATA CORPORATION MAKES NO
 * REPRESENTATIONS OR WARRANTIES WITH REGARD TO THIS SOFTWARE OR ITS
 * DOCUMENTATION, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. CENTRAL DATA CORPORATION SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, COPYING,
 * MODIFYING, OR DISTRIBUTING THIS SOFTWARE OR ITS DOCUMENTATION, OR
 * DERIVATIVES OR COMPILATIONS OF THEM.
 */

package portio;

/**
 * Port parameters for standard RS-232 serial ports. This serves simply as a
 * parameter block and provides no associated methods. Use the SerialPort.setParams
 * to set parameters. Note that the PortParams class itself is empty.
 *
 * @version $Revision: 2.1 $ ($Date: 1997/02/19 18:00:00 $)
 * @see Port
 * @see SerialPort
 */
public class SerialPortParams extends PortParams implements Cloneable {

    /**
     * no parity
     */
    public static final int PARITY_NONE = 0;

    /**
     * odd parity
     */
    public static final int PARITY_ODD = 1;

    /**
     * even parity
     */
    public static final int PARITY_EVEN = 2;

    /**
     * mark parity
     */
    public static final int PARITY_MARK = 3;

    /**
     * space parity
     */
    public static final int PARITY_SPACE = 4;

    /**
     * no flow control mode
     */
    public static final int FLOW_NONE = 0x00;

    /**
     * software flow control mode
     */
    public static final int FLOW_SOFTWARE = 0x01;

    /**
     * IXANY output mode (any character restarts stopped output)
     */
    public static final int FLOW_IXANY = 0x02;

    /**
     * output hardware flow control mode, uses CTS to pace output (most common)
     */
    public static final int FLOW_OUT_CTS = 0x04;

    /**
     * output hardware flow control mode, uses DSR to pace output (optional)
     */
    public static final int FLOW_OUT_DSR = 0x08;

    /**
     * output hardware flow control mode, uses DCD to pace output (optional)
     */
    public static final int FLOW_OUT_DCD = 0x10;

    /**
     * input hardware flow control mode, uses RTS to pace input (most common)
     */
    public static final int FLOW_IN_RTS = 0x20;

    /**
     * input hardware flow control mode, uses DTR to pace input (used by some printers
     * but optionally supported--make a special cable to route RTS on
     * the port to DTR on such printers when not supported)
     */
    public static final int FLOW_IN_DTR = 0x40;

    /**
     * Output baud rate
     */
    public int outBaud;

    /**
     * Input baud rate
     */
    public int inBaud;

    /**
     * Character size; typically 5, 6, 7, or 8 bits/character.
     */
    public int charSize;

    /**
     * Parity bit code
     */
    public int parity;

    /**
     * Framing bits, typically 1, 1.5, and 2 are legal values.  Some
     * hardware support other fractional stop bit settings.  It's up to
     * the driver to determine what is supported and how to round values.
     */
    public double framing;

    /**
     * output flow control mode (mode values can be OR'd together)
     */
    public int outFlowControl;

    /**
     * input flow control mode (mode values can be OR'd together)
     */
    public int inFlowControl;

    /**
     * software flow control <i>start</i> character
     */
    public byte flowStart;

    /**
     * software flow control <i>stop</i> character
     */
    public byte flowStop;

    /**
     * causes eof (-1) to be returned on reads and writes upon loss of carrier (DCD)
     */
    public boolean carrierSense;

    /**
     * This causes an exception to be thrown on reads if a parity error
     * is encountered. Regardless of the setting
     * of this flag, error characters are placed on the input stream as read from the
     * hardware.
     */
    public boolean exceptionOnParityError;

    /**
     * This causes an exception to be thrown on reads if a
     * framing error is encountered. Regardless of the setting
     * of this flag, error characters are placed on the input stream as read from the
     * hardware.
     */
    public boolean exceptionOnFramingError;

    /**
     * This causes an exception to be thrown on reads if an
     * overrun error is encountered.
     */
    public boolean exceptionOnOverrunError;

    /**
     * This causes an exception to be generated on reads if a break condition
     * is received. Regardless of the setting of this flag, the break will appear
     * as one or more zero bytes on the input stream.
     */
    public boolean exceptionOnBreak;

    /**
     * optionally implemented, enables/disable receiver at hardware level
     */
    public boolean receiveEnable;

    /**
     * optionally implemented, sets hardware loopback for testing
     */
    public boolean loopback;

    /**
     * Instantiates a default serial port params object defaulting to
     * 9600, 8, N, 1 with no flow control, receive enabled, and
     * start/stop chars set to the usual Control-S and Control-Q.
     * This is the default parameter set that serial ports should be set to on open
     * for this package.
     */
    public SerialPortParams()
    {
        inBaud = outBaud = 9600;
        charSize = 8;
        parity = PARITY_NONE;
        framing = 1;
        outFlowControl = inFlowControl = FLOW_NONE;
        flowStart = (byte)17;
        flowStop = (byte)19;
        carrierSense = false;
        exceptionOnParityError = exceptionOnFramingError = false;
        exceptionOnOverrunError = exceptionOnBreak = false;
        receiveEnable = true;
        loopback = false;
    }

    /**
     * Converts the object to a string.
     */
    public String toString()
    {
        return "Input Baud=" + inBaud + ", Output Baud=" + outBaud +
              ", Character Size=" + charSize + ", Parity=" + parity +
              ", Framing Bits=" + framing + ", Output Flow Control=" + outFlowControl +
              ", Input Flow Control=" + inFlowControl + ", Flow Start=" + flowStart +
              ", Flow Stop=" + flowStop +
              ", Carrier Sense=" + (carrierSense ? "On" : "Off") +
              ", Catch Errors: Parity=" + (exceptionOnParityError ? "On" : "Off") +
              ", Framing=" + (exceptionOnFramingError ? "On" : "Off") +
              ", Overrun=" + (exceptionOnOverrunError ? "On" : "Off") +
              ", Catch Breaks=" + (exceptionOnBreak ? "On" : "Off") +
              ", Receiver=" + (receiveEnable ? "On" : "Off") +
              ", Loopback=" + (loopback ? "On" : "Off");
    }

    /**
     * Clones the port params object.
     */
    public Object clone()
    {
        try { return super.clone(); }
        catch( CloneNotSupportedException e ) { System.out.println( "SerialPortParams: " + e ); }
        return null;
    }
}
