/*
 * PortPrivate.java
 * 
 * Copyright (c) 1996-1997 Central Data Corporation
 * All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software or its
 * documentation, or derivatives or compilations of them, is described
 * in the file "license.html".
 *
 * CENTRAL DATA CORPORATION IS MAKING THE SOFTWARE AND ITS DOCUMENTATION
 * AVAILABLE "AS IS" FOR NO FEE. CENTRAL DATA CORPORATION MAKES NO
 * REPRESENTATIONS OR WARRANTIES WITH REGARD TO THIS SOFTWARE OR ITS
 * DOCUMENTATION, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. CENTRAL DATA CORPORATION SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, COPYING,
 * MODIFYING, OR DISTRIBUTING THIS SOFTWARE OR ITS DOCUMENTATION, OR
 * DERIVATIVES OR COMPILATIONS OF THEM.
 */

package portio;

/**
 * This abstract class specifies the functionality to be provided
 * by private port driver. Private port drivers hold port-specific data
 * and have methods to access that data.
 *
 * @version $Revision: 2.1 $ ($Date: 1997/02/19 18:00:00 $)
 */
abstract public class PortPrivate {

    byte portNumber;                            // our port number

    /**
     * Resets port and sets default params
     */
    abstract void open()
        throws PortIOException;

    /**
     * Resets port and shuts down input interrupts
     */
    abstract void close()
        throws PortIOException;

    /**
     * Reads data from the input buffer, or blocks if none is available
     *
     * @param b data buffer to be written
     * @param off offset into buffer
     * @param length length to write
     * @return length written or -1 if eof
     */
    abstract int read( byte buf[], int off, int length )
        throws PortIOException;

    /**
     * Handles writes to this port
     *
     * @param b data buffer to be written
     * @param off offset into buffer
     * @param length length to write
     * @return length written or -1 if eof
     */
    abstract int write( byte buf[], int off, int length )
        throws PortIOException;

    /**
     * Returns the number of bytes of input currently available to be read (without
     * blocking)
     *
     * @return int number of bytes available
     */
    abstract int available()
        throws PortIOException;

    /**
     * Waits for output to drain output completely both at local buffer and
     * at hardware.
     *
     * @param waitTime number of seconds to wait before giving up (zero is forever)
     */
    abstract void flush( int waitTime )
        throws PortIOException;

    /**
     * Dumps input/output both locally and at firmware
     *
     * @param write if true, purge output
     * @param read if true, purge input
     */
    abstract void purge( boolean write, boolean read )
        throws PortIOException;

    /**
     * Returns params for this port
     *
     * @return PortParams current parameters for this port
     */
    abstract PortParams getParams()
        throws PortIOException;

    /**
     * Check params for legal and supported values and then sets them
     *
     * @param PortParams new parameters for this port
     */
    abstract void setParams( SerialPortParams newParams )
        throws PortIOException;

    /**
     * Returns appropriate type status block for port
     *
     * @param wait if true, waits for something to change before returning
     * @return PortStatus current port status
     */
    abstract PortStatus getStatus( boolean wait )
        throws PortIOException;

    /**
     * Sets modem signal for serial ports
     */
    abstract void setStatus( SerialPortStatus status )
        throws PortIOException;

    /**
     * Sends a break indication
     */
    abstract void sendBreak( int duration )
        throws PortIOException;
}
