/*
 * PortOutputStream.java
 * 
 * Copyright (c) 1996-1997 Central Data Corporation
 * All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software or its
 * documentation, or derivatives or compilations of them, is described
 * in the file "license.html".
 *
 * CENTRAL DATA CORPORATION IS MAKING THE SOFTWARE AND ITS DOCUMENTATION
 * AVAILABLE "AS IS" FOR NO FEE. CENTRAL DATA CORPORATION MAKES NO
 * REPRESENTATIONS OR WARRANTIES WITH REGARD TO THIS SOFTWARE OR ITS
 * DOCUMENTATION, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. CENTRAL DATA CORPORATION SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, COPYING,
 * MODIFYING, OR DISTRIBUTING THIS SOFTWARE OR ITS DOCUMENTATION, OR
 * DERIVATIVES OR COMPILATIONS OF THEM.
 */

package portio;

import  java.io.*;

/**
 * Implements an output stream to port objects.  Note: not a public class.
 *
 * @version $Revision: 2.1 $ ($Date: 1997/02/19 18:00:00 $)
 */
class PortOutputStream  extends OutputStream {
    private Port port;
    private byte temp[] = new byte[1];

    /**
     * Never instantiated directly.  Port output streams are
     * only provided through <TT>Port's</TT> <i>getOutputStream</i>
     * method.
     *
     * @param port the associated Port object
     */
    PortOutputStream( Port port )
    {
        this.port = port;
    }

    /**
     * Writes a byte to the port.
     * Unlike standard OutputStream definition, this only queues to byte for output
     * and does not wait until they are fully written. Blocks if queue is full.
     * @param b the data to be written
     * @exception PortIOException If an I/O error/event has occurred.
     */
    public void write( int b )
        throws PortIOException
    {
    	temp[0] = (byte)b;
    	port.driverPrivate.write( temp, 0, 1 );
    }

    /**
     * Writes the contents of the buffer <i>b</i> to the port.
     * Unlike standard OutputStream definition, this only queues to bytes for output
     * and does not wait until they are fully written.  Blocks if queue is full.
     * @param b the data to be written
     * @exception PortIOException If an I/O error/event has occurred.
     */
    public void write( byte b[] )
        throws PortIOException
    {
        port.driverPrivate.write( b, 0, b.length );
    }

    /**
     * Writes <i>length</i> bytes from buffer <i>b</i> starting at
     * offset <i>len</i>.
     * Unlike standard OutputStream definition, this only queues to bytes for output
     * and does not wait until they are fully written.  Blocks if queue is full.
     * @param b the data to be written
     * @param off the start offset in the data
     * @param len the number of bytes that are written
     * @exception SocketException If an I/O error has occurred.
     */
    public void write( byte b[], int off, int len )
        throws PortIOException
    {
        port.driverPrivate.write( b, off, len );
    }

    /**
     * Waits for output to fully complete at port hardware.
     */
    public void flush()
        throws PortIOException
    {
        port.driverPrivate.flush( 0 );
    }
}
