/*
 * PortInputStream.java
 * 
 * Copyright (c) 1996-1997 Central Data Corporation
 * All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software or its
 * documentation, or derivatives or compilations of them, is described
 * in the file "license.html".
 *
 * CENTRAL DATA CORPORATION IS MAKING THE SOFTWARE AND ITS DOCUMENTATION
 * AVAILABLE "AS IS" FOR NO FEE. CENTRAL DATA CORPORATION MAKES NO
 * REPRESENTATIONS OR WARRANTIES WITH REGARD TO THIS SOFTWARE OR ITS
 * DOCUMENTATION, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. CENTRAL DATA CORPORATION SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, COPYING,
 * MODIFYING, OR DISTRIBUTING THIS SOFTWARE OR ITS DOCUMENTATION, OR
 * DERIVATIVES OR COMPILATIONS OF THEM.
 */

package portio;

import java.io.*;

/**
 * Implements an input stream for a port object.
 *
 * @version $Revision: 2.1 $ ($Date: 1997/02/19 18:00:00 $)
 */
class PortInputStream extends InputStream {
    private Port port;
    private boolean eof;
    private byte temp[] = new byte[1];

    /**
     * Never instantiated directly.  Port input streams are
     * only provided through <TT>Port's</TT> <i>getInputStream</i>
     * method.
     *
     * @param port the associated Port object
     */
    PortInputStream( Port port )
    {
        this.port = port;
    }

    /**
     * Reads into a byte array <i>b</i> at offset <i>off</i>,
     * <i>length</i> bytes of data.
     * @param b the buffer into which the data is read
     * @param off the start offset of the data
     * @param len the maximum number of bytes read
     * @return the actual number of bytes read, -1 is
     *          returned when the end of the stream is reached.
     * @exception PortIOException If an I/O error/event has occurred.
     */
    public int read( byte b[], int off, int length )
        throws PortIOException
    {
    	if ( eof ) return -1;
    	int n = port.driverPrivate.read( b, off, length );
    	if ( n <= 0 ) {
    	    eof = true;
    	    return -1;
    	}
    	return n;
    }

    /**
     * Reads into a byte array data from the port.
     * @param b the buffer into which the data is read
     * @return the actual number of bytes read, -1 is
     *          returned when the end of the stream is reached.
     * @exception PortIOException If an I/O error/event has occurred.
     */
    public int read( byte b[] )
        throws PortIOException
    {
    	return read( b, 0, b.length );
    }

    /**
     * Reads a single byte from the port.
     * @exception PortIOException If an I/O error/event has occurred.
     */
    public int read()
        throws PortIOException
    {
     	int n = read(temp, 0, 1);
    	if (n <= 0) return -1;
    	return temp[0] & 0xff;
    }

    /**
     * Returns the number of bytes that can be read without blocking.
     */
    public int available()
        throws PortIOException
    {
        return port.driverPrivate.available();
    }
}