/* lzo_test.c -- comprehensive test driver for the LZO library

   This file is part of the LZO real-time data compression library.

   Copyright (C) 1997 Markus Franz Xaver Johannes Oberhumer
   Copyright (C) 1996 Markus Franz Xaver Johannes Oberhumer

   The LZO library is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   The LZO library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with the LZO library; see the file COPYING.
   If not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   Markus F.X.J. Oberhumer
   markus.oberhumer@jk.uni-linz.ac.at
 */


#include <lzoconf.h>

#if defined(LZO_HAVE_CONFIG_AC)
#  include <config.ac>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <assert.h>
#if defined(__DJGPP__) || defined(__BORLANDC__)
#  include <dir.h>
#endif
#if defined(HAVE_UNISTD_H) || defined(__DJGPP__) || defined(__EMX__)
#  include <unistd.h>
#endif

#include <time.h>
#if !defined(CLOCKS_PER_SEC) && defined(CLK_TCK)
#  define CLOCKS_PER_SEC	CLK_TCK
#endif
#if defined(UCLOCKS_PER_SEC)
#  undef clock
#  undef clock_t
#  undef CLOCKS_PER_SEC
#  define clock()			uclock()
#  define clock_t			uclock_t
#  define CLOCKS_PER_SEC	UCLOCKS_PER_SEC
#endif


/*************************************************************************
//
**************************************************************************/

#define HAVE_LZO1_H
#define HAVE_LZO1A_H
#define HAVE_LZO1B_H
#define HAVE_LZO1C_H
#define HAVE_LZO1F_H
#define HAVE_LZO1X_H
#define HAVE_LZO2A_H
#if 1 && !defined(HAVE_ZLIB_H) && defined(MFX)
#define HAVE_ZLIB_H
#endif

#if defined(__LZO_MSDOS16)
#undef HAVE_LZO1_H
#undef HAVE_LZO1A_H
#undef HAVE_LZO1C_H
#undef HAVE_LZO2A_H
#undef HAVE_LZO2B_H
#undef HAVE_ZLIB_H
#endif


/*************************************************************************
// define all methods
**************************************************************************/

enum {
/* compression algorithms */
	M_LZO1B_1  =     1,
	M_LZO1B_2, M_LZO1B_3, M_LZO1B_4, M_LZO1B_5,
	M_LZO1B_6, M_LZO1B_7, M_LZO1B_8, M_LZO1B_9,

	M_LZO1C_1  =    11,
	M_LZO1C_2, M_LZO1C_3, M_LZO1C_4, M_LZO1C_5,
	M_LZO1C_6, M_LZO1C_7, M_LZO1C_8, M_LZO1C_9,

	M_LZO1     =    21,
	M_LZO1A    =    31,

	M_LZO1B_99 =   901,
	M_LZO1B_999=   902,
	M_LZO1C_99 =   911,
	M_LZO1C_999=   912,
	M_LZO1_99  =   921,
	M_LZO1A_99 =   931,

	M_LZO1F_1  =    61,
	M_LZO1F_999=   962,
	M_LZO1X_1  =    71,
	M_LZO1X_999=   972,

	M_LZO2A_999=   942,
	M_LZO2B_999=   952,

	M_LAST_LZO_COMPRESSOR = 998,

/* other compressors */
	M_ZLIB_8_1 =  1101,
	M_ZLIB_8_2, M_ZLIB_8_3, M_ZLIB_8_4, M_ZLIB_8_5,
	M_ZLIB_8_6, M_ZLIB_8_7, M_ZLIB_8_8, M_ZLIB_8_9,

/* dummy compressor */
	M_MEMCPY   =   999,

	M_LAST_COMPRESSOR = 4999,

/* dummy algorithms */
	M_MEMSET   =  5001,

/* checksum algorithms */
	M_ADLER32  =  6001,

	M_UNUSED
};


#if defined(LZO_99_UNSUPPORTED)
#define M_LZO1_99		(-1)
#define M_LZO1A_99		(-1)
#define M_LZO1B_99		(-1)
#define M_LZO1C_99		(-1)
#endif
#if defined(LZO_999_UNSUPPORTED)
#define M_LZO1B_999		(-1)
#define M_LZO1C_999		(-1)
#define M_LZO1F_999		(-1)
#define M_LZO1X_999		(-1)
#define M_LZO2A_999		(-1)
#define M_LZO2B_999		(-1)
#endif


/* LZO algorithms */
#if defined(HAVE_LZO1_H)
#  include <lzo1.h>
#endif
#if defined(HAVE_LZO1A_H)
#  include <lzo1a.h>
#endif
#if defined(HAVE_LZO1B_H)
#  include <lzo1b.h>
#endif
#if defined(HAVE_LZO1C_H)
#  include <lzo1c.h>
#endif
#if defined(HAVE_LZO1F_H)
#  include <lzo1f.h>
#endif
#if defined(HAVE_LZO1X_H)
#  include <lzo1x.h>
#endif
#if defined(HAVE_LZO2A_H)
#  include <lzo2a.h>
#endif
#if defined(HAVE_LZO2B_H)
#  include <lzo2b.h>
#endif

/* other compressors */
#if defined(HAVE_ZLIB_H)
#  include <zlib.h>
#endif
#if defined(MFX)
#  include "contrib/t_config.ch"
#endif


/*************************************************************************
//
**************************************************************************/

#if 1
   /* always use the fast decompressors */
#  define lzo1b_decompress_safe		lzo1b_decompress
#  define lzo1c_decompress_safe		lzo1c_decompress
#  define lzo1f_decompress_safe		lzo1f_decompress
#  define lzo1x_decompress_safe		lzo1x_decompress
#  define lzo2a_decompress_safe		lzo2a_decompress
#  define lzo2b_decompress_safe		lzo2b_decompress
#endif


#if defined(HAVE_LZO1X_H)
int method = M_LZO1X_1;
#elif defined(HAVE_LZO1B_H)
int method = M_LZO1B_1;
#elif defined(HAVE_LZO1C_H)
int method = M_LZO1C_1;
#elif defined(HAVE_LZO1F_H)
int method = M_LZO1F_1;
#else
int method = 0;
#endif

int verbose = 2;

lzo_bool use_assembler_decompressor = 0;
lzo_bool try_to_compress_0_bytes = 1;

/* set these to 1 to measure the speed impact of a checksum (option '-a') */
lzo_bool compute_adler32 = 0;
lzo_bool compute_crc32 = 0;

static lzo_uint32 adler_in, adler_out;


static const short all_methods[] = {
	M_LZO1,
	M_LZO1A,
	M_LZO1B_1, M_LZO1B_2, M_LZO1B_3, M_LZO1B_4, M_LZO1B_5,
	M_LZO1B_6, M_LZO1B_7, M_LZO1B_8, M_LZO1B_9,
	M_LZO1C_1, M_LZO1C_2, M_LZO1C_3, M_LZO1C_4, M_LZO1C_5,
	M_LZO1C_6, M_LZO1C_7, M_LZO1C_8, M_LZO1C_9,
	M_LZO1F_1,
	M_LZO1X_1,
	M_LZO1_99, M_LZO1A_99, M_LZO1B_99, M_LZO1C_99,
	M_LZO1B_999, M_LZO1C_999, M_LZO1F_999, M_LZO1X_999,
	M_LZO2A_999,
	0
};

static const short benchmark_methods[] = {
	M_LZO1B_1, M_LZO1B_9,
	M_LZO1C_1, M_LZO1C_9,
	M_LZO1F_1,
	M_LZO1X_1,
	0
};

static const short x1_methods[] = {
	M_LZO1, M_LZO1A, M_LZO1B_1, M_LZO1C_1, M_LZO1F_1, M_LZO1X_1,
	0
};

static const short x99_methods[] = {
	M_LZO1_99, M_LZO1A_99, M_LZO1B_99, M_LZO1C_99,
	0
};

static const short x999_methods[] = {
	M_LZO1B_999, M_LZO1C_999, M_LZO1F_999, M_LZO1X_999,
	M_LZO2A_999,
	0
};

static const short zlib_methods[] = {
	M_ZLIB_8_1, M_ZLIB_8_2, M_ZLIB_8_3, M_ZLIB_8_4, M_ZLIB_8_5,
	M_ZLIB_8_6, M_ZLIB_8_7, M_ZLIB_8_8, M_ZLIB_8_9,
	0
};

static const short avail_methods[] = {
	0
};


/* exit codes of this test program */
#define EXIT_OK			0
#define EXIT_USAGE		1
#define EXIT_FILE		2
#define EXIT_MEM		3
#define EXIT_ADLER		4
#define EXIT_LZO_ERROR	5
#define EXIT_LZO_INIT	6


/*************************************************************************
// work memory
**************************************************************************/

#if (UINT_MAX >= 0xffffffffL)		/* 32 bit or more */

#define BLOCK_SIZE		(256*1024l)
#define MAX_BLOCK_SIZE	(4*BLOCK_SIZE)
#define DATA_LEN		(1024*1024l)
#define WORK_LEN		(768*1024l)

#elif defined(__LZO_ATARI16)

/* adjust memory so that it works on a 4 meg Atari ST */
#define BLOCK_SIZE		(256*1024l)
#define MAX_BLOCK_SIZE	(1*BLOCK_SIZE)
#define DATA_LEN		(1024*1024l)
#define WORK_LEN		(768*1024l)

#else								/* 16 bit MSDOS */

/* have to squeeze everything into ~600 kB  */
#if 0
#define BLOCK_SIZE		(256*1024l)
#define MAX_BLOCK_SIZE	(256*1024l)
#define DATA_LEN		(128*1024l)
#else
#define BLOCK_SIZE		32000u
#define MAX_BLOCK_SIZE	32000u
#define DATA_LEN		(60*1024l)
#endif
#define WORK_LEN		(64*1024l)

#endif


static lzo_uint block_size = BLOCK_SIZE;

#ifndef LZO_MEMALLOC_T
#define LZO_MEMALLOC_T	lzo_byte
#endif
static LZO_MEMALLOC_T	_block1[MAX_BLOCK_SIZE * 9L / 8 + 256 + 1024];
static LZO_MEMALLOC_T	_block2[MAX_BLOCK_SIZE * 9L / 8 + 256 + 1024];
static LZO_MEMALLOC_T	_data[DATA_LEN + 256];
static LZO_MEMALLOC_T	_wrkmem[WORK_LEN + 256];

/* align memory blocks (cache issues) */
static lzo_byte *block1 = NULL;
static lzo_byte *block2 = NULL;
static lzo_byte *data = NULL;
static lzo_byte *wrkmem = NULL;

static void align_mem(void)
{
	block1 = (lzo_byte *) LZO_ALIGN(_block1,256);
	block2 = (lzo_byte *) LZO_ALIGN(_block2,256);
	data   = (lzo_byte *) LZO_ALIGN(_data,256);
	wrkmem = (lzo_byte *) LZO_ALIGN(_wrkmem,256);
}


/***********************************************************************
// read from a file
************************************************************************/

lzo_uint my_fread(FILE *f, lzo_voidp s, lzo_uint len)
{
#if (LZO_UINT_MAX <= UINT_MAX)
	return fread(s,1,len,f);
#else
	lzo_byte *p = (lzo_byte *) s;
	lzo_uint l = 0;
	size_t k;
	unsigned char buf[512], *b;

	while (l < len)
	{
		k = len - l > sizeof(buf) ? sizeof(buf) : (size_t) (len - l);
		k = fread(buf,1,k,f);
		if (k <= 0)
			break;
		l += k;
		b = buf;
		do *p++ = *b++; while (--k > 0);
	}
	return l;
#endif
}


/*************************************************************************
// compress a block
**************************************************************************/

static int
do_compress(const lzo_byte *in, lzo_byte *out,
                  lzo_uint in_len, lzo_uint *out_len)
{
	int r = -100;
	unsigned long o_len = ULONG_MAX;

	if (method <= 0)
		;
#if defined(HAVE_LZO1_H)
	else if (method == M_LZO1)
		r = lzo1_compress(in,in_len,out,out_len,wrkmem);
#if !defined(LZO_99_UNSUPPORTED)
	else if (method == M_LZO1_99 && (WORK_LEN >= LZO1_99_MEM_COMPRESS))
		r = lzo1_99_compress(in,in_len,out,out_len,wrkmem);
#endif
#endif
#if defined(HAVE_LZO1A_H)
	else if (method == M_LZO1A)
		r = lzo1a_compress(in,in_len,out,out_len,wrkmem);
#if !defined(LZO_99_UNSUPPORTED)
	else if (method == M_LZO1A_99 && (WORK_LEN >= LZO1A_99_MEM_COMPRESS))
		r = lzo1a_99_compress(in,in_len,out,out_len,wrkmem);
#endif
#endif
#if defined(HAVE_LZO1B_H)
	else if (method >= M_LZO1B_1 && method <= M_LZO1B_9)
	{
		int clevel = method - M_LZO1B_1 + 1;
		r = lzo1b_compress(in,in_len,out,out_len,wrkmem,clevel);
	}
#if !defined(LZO_99_UNSUPPORTED)
	else if (method == M_LZO1B_99 && (WORK_LEN >= LZO1B_99_MEM_COMPRESS))
		r = lzo1b_99_compress(in,in_len,out,out_len,wrkmem);
#endif
#if !defined(LZO_999_UNSUPPORTED)
	else if (method == M_LZO1B_999 && (WORK_LEN >= LZO1B_999_MEM_COMPRESS))
		r = lzo1b_999_compress(in,in_len,out,out_len,wrkmem);
#endif
#endif
#if defined(HAVE_LZO1C_H)
	else if (method >= M_LZO1C_1 && method <= M_LZO1C_9)
	{
		int clevel = method - M_LZO1C_1 + 1;
		r = lzo1c_compress(in,in_len,out,out_len,wrkmem,clevel);
	}
#if !defined(LZO_99_UNSUPPORTED)
	else if (method == M_LZO1C_99 && (WORK_LEN >= LZO1C_99_MEM_COMPRESS))
		r = lzo1c_99_compress(in,in_len,out,out_len,wrkmem);
#endif
#if !defined(LZO_999_UNSUPPORTED)
	else if (method == M_LZO1C_999 && (WORK_LEN >= LZO1C_999_MEM_COMPRESS))
		r = lzo1c_999_compress(in,in_len,out,out_len,wrkmem);
#endif
#endif
#if defined(HAVE_LZO1F_H)
	else if (method == M_LZO1F_1)
		r = lzo1f_1_compress(in,in_len,out,out_len,wrkmem);
#if !defined(LZO_999_UNSUPPORTED)
	else if (method == M_LZO1F_999 && (WORK_LEN >= LZO1F_999_MEM_COMPRESS))
		r = lzo1f_999_compress(in,in_len,out,out_len,wrkmem);
#endif
#endif
#if defined(HAVE_LZO1X_H)
	else if (method == M_LZO1X_1)
		r = lzo1x_1_compress(in,in_len,out,out_len,wrkmem);
#if !defined(LZO_999_UNSUPPORTED)
	else if (method == M_LZO1X_999 && (WORK_LEN >= LZO1F_999_MEM_COMPRESS))
		r = lzo1x_999_compress(in,in_len,out,out_len,wrkmem);
#endif
#endif
#if defined(HAVE_LZO2A_H)
#if !defined(LZO_999_UNSUPPORTED)
	else if (method == M_LZO2A_999 && (WORK_LEN >= LZO2A_999_MEM_COMPRESS))
		r = lzo2a_999_compress(in,in_len,out,out_len,wrkmem);
#endif
#endif
#if defined(HAVE_LZO2B_H)
#if !defined(LZO_999_UNSUPPORTED)
	else if (method == M_LZO2B_999 && (WORK_LEN >= LZO2B_999_MEM_COMPRESS))
		r = lzo2b_999_compress(in,in_len,out,out_len,wrkmem);
#endif
#endif
#if defined(HAVE_ZLIB_H)
	else if (method >= M_ZLIB_8_1 && method <= M_ZLIB_8_9)
	{
		/* use the undocumented feature to suppress the zlib header */
		z_stream stream;
		int err;
		int level = method - M_ZLIB_8_1 + 1;

		stream.next_in = (lzo_byte *) in;		/* UNCONST */
		stream.avail_in = in_len;
		stream.next_out = out;
		stream.avail_out = *out_len;

		stream.zalloc = (alloc_func)0;
		stream.zfree = (free_func)0;
		*out_len = 0;
#if 0
		err = deflateInit(&stream, level);
#else
		err = deflateInit2(&stream, level, Z_DEFLATED, -MAX_WBITS,
		                   MAX_MEM_LEVEL > 8 ? 8 : MAX_MEM_LEVEL,
		                   Z_DEFAULT_STRATEGY);
#endif
		if (err == Z_OK)
		{
			err = deflate(&stream, Z_FINISH);
			if (err != Z_STREAM_END)
			{
				deflateEnd(&stream);
				err = err == Z_OK ? Z_BUF_ERROR : err;
			}
			else
			{
				*out_len = (lzo_uint) stream.total_out;
				err = deflateEnd(&stream);
			}
		}
		r = err == Z_OK ? 0 : -1;
	}
#endif
#if defined(MFX)
#  include "contrib/t_comp.ch"
#endif
	else if (method == M_MEMCPY)
	{
		lzo_memcpy(out,in,in_len);
		*out_len = in_len; r = 0;
	}
	else if (method == M_MEMSET)
	{
		lzo_memset(out,0,in_len);
		*out_len = in_len; r = 0;
	}
	else if (method == M_ADLER32)
	{
		lzo_uint32 adler;
		adler = lzo_adler32(0, NULL, 0);
		adler = lzo_adler32(adler, out, in_len);
		*out_len = in_len; r = 0;
	}
	else
		*out_len = 0;

	if (o_len != ULONG_MAX)
		*out_len = (lzo_uint) o_len;

	if (r == 0 && compute_adler32)
	{
		lzo_uint32 adler;
		adler = lzo_adler32(0, NULL, 0);
		adler = lzo_adler32(adler, in, in_len);
		adler_in = adler;
	}
#if defined(HAVE_ZLIB_H)
	if (r == 0 && compute_crc32)
	{
		uLong crc;
		crc = crc32(0L, Z_NULL, 0);
		crc = crc32(crc, in, in_len);
	}
#endif

	return r;
}


/*************************************************************************
// decompress a block
**************************************************************************/

static int
do_decompress(const lzo_byte *in, lzo_byte *out,
              lzo_uint in_len, lzo_uint *out_len)
{
	int r = -100;
	unsigned long o_len = ULONG_MAX;

	if (method <= 0)
		;
#if defined(HAVE_LZO1_H)
	else if (method == M_LZO1 || method == M_LZO1_99)
		r = lzo1_decompress(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZO1A_H)
	else if (method == M_LZO1A || method == M_LZO1A_99)
		r = lzo1a_decompress(in,in_len,out,out_len,NULL);
#endif
#if defined(HAVE_LZO1B_H)
	else if (method >= M_LZO1B_1 && method <= M_LZO1B_9)
		r = lzo1b_decompress(in,in_len,out,out_len,wrkmem);
	else if (method == M_LZO1B_99 || method == M_LZO1B_999)
		r = lzo1b_decompress_safe(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZO1C_H)
#if defined(LZO1C_DECOMPRESS_ASM)
	else if (use_assembler_decompressor &&
		((method >= M_LZO1C_1 && method <= M_LZO1C_9) ||
		 (method == M_LZO1C_99 || method == M_LZO1C_999)))
		r = lzo1c_decompress_asm(in,in_len,out,out_len,wrkmem);
#endif
	else if (method >= M_LZO1C_1 && method <= M_LZO1C_9)
		r = lzo1c_decompress(in,in_len,out,out_len,wrkmem);
	else if (method == M_LZO1C_99 || method == M_LZO1C_999)
		r = lzo1c_decompress_safe(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZO1F_H)
#if defined(LZO1F_DECOMPRESS_ASM)
	else if (use_assembler_decompressor &&
		 (method == M_LZO1F_1 || method == M_LZO1F_999))
		r = lzo1f_decompress_asm(in,in_len,out,out_len,wrkmem);
#endif
	else if (method == M_LZO1F_1)
		r = lzo1f_decompress(in,in_len,out,out_len,wrkmem);
	else if (method == M_LZO1F_999)
		r = lzo1f_decompress_safe(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZO1X_H)
#if defined(LZO1X_DECOMPRESS_ASM)
	else if (use_assembler_decompressor &&
		 (method == M_LZO1X_1 || method == M_LZO1X_999))
		r = lzo1x_decompress_asm(in,in_len,out,out_len,wrkmem);
#endif
	else if (method == M_LZO1X_1)
		r = lzo1x_decompress(in,in_len,out,out_len,wrkmem);
	else if (method == M_LZO1X_999)
		r = lzo1x_decompress_safe(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZO2A_H)
	else if (method == M_LZO2A_999)
		r = lzo2a_decompress_safe(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZO2B_H)
	else if (method == M_LZO2B_999)
		r = lzo2b_decompress_safe(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_ZLIB_H)
	else if (method >= M_ZLIB_8_1 && method <= M_ZLIB_8_9)
	{
		/* use the undocumented feature to suppress the zlib header */
		z_stream stream;
		int err;

		stream.next_in = (lzo_byte *) in;		/* UNCONST */
		stream.avail_in = in_len;
		stream.next_out = out;
		stream.avail_out = *out_len;

		stream.zalloc = (alloc_func)0;
		stream.zfree = (free_func)0;
		*out_len = 0;
#if 0
		err = inflateInit(&stream);
#else
		err = inflateInit2(&stream, -MAX_WBITS);
#endif
		if (err == Z_OK)
		{
			err = inflate(&stream, Z_FINISH);
			if (err == Z_OK)		/* sometimes returns Z_OK !!! */
			{
				*out_len = (lzo_uint) stream.total_out;
				err = inflateEnd(&stream);
			}
			else if (err == Z_STREAM_END)
			{
				*out_len = (lzo_uint) stream.total_out;
				err = inflateEnd(&stream);
			}
			else
				inflateEnd(&stream);
		}
		r = err == Z_OK ? 0 : -1;
	}
#endif
#if defined(MFX)
#  include "contrib/t_decomp.ch"
#endif
	else if (method == M_MEMCPY)
	{
		lzo_memcpy(out,in,in_len);
		*out_len = in_len; r = 0;
	}
	else if (method == M_MEMSET)
	{
		lzo_memset(out,0,in_len);
		*out_len = in_len; r = 0;
	}
	else if (method == M_ADLER32)
	{
		lzo_uint32 adler;
		adler = lzo_adler32(0, NULL, 0);
		adler = lzo_adler32(adler, out, in_len);
		*out_len = in_len; r = 0;
	}
	else
		*out_len = 0;

	if (o_len != ULONG_MAX)
		*out_len = (lzo_uint) o_len;

	if (r == 0 && compute_adler32)
	{
		lzo_uint32 adler;
		adler = lzo_adler32(0, NULL, 0);
		adler = lzo_adler32(adler, out, *out_len);
		adler_out = adler;
	}
#if defined(HAVE_ZLIB_H)
	if (r == 0 && compute_crc32)
	{
		uLong crc;
		crc = crc32(0L, Z_NULL, 0);
		crc = crc32(crc, out, *out_len);
	}
#endif

	return r;
}


/*************************************************************************
// method info
**************************************************************************/

static char method_name[16+1];

static lzo_bool info(FILE *f)
{
	char *s = method_name;

	s[0] = 0;
	if (method <= 0)
		;
#if defined(HAVE_LZO1_H)
	else if (method == M_LZO1)
	{
		int rbits, clevel;
		lzo1_info(&rbits,&clevel);
		sprintf(s,"LZO1-%d",clevel);
	}
#if !defined(LZO_99_UNSUPPORTED)
	else if (method == M_LZO1_99 && (WORK_LEN >= LZO1_99_MEM_COMPRESS))
		sprintf(s,"LZO1-99");
#endif
#endif
#if defined(HAVE_LZO1A_H)
	else if (method == M_LZO1A)
	{
		int rbits, clevel;
		lzo1a_info(&rbits,&clevel);
		sprintf(s,"LZO1A-%d",clevel);
	}
#if !defined(LZO_99_UNSUPPORTED)
	else if (method == M_LZO1A_99 && (WORK_LEN >= LZO1A_99_MEM_COMPRESS))
		sprintf(s,"LZO1A-99");
#endif
#endif
#if defined(HAVE_LZO1B_H)
	else if (method >= M_LZO1B_1 && method <= M_LZO1B_9)
	{
		int clevel = method - M_LZO1B_1 + 1;
		sprintf(s,"LZO1B-%d",clevel);
	}
#if !defined(LZO_99_UNSUPPORTED)
	else if (method == M_LZO1B_99 && (WORK_LEN >= LZO1B_99_MEM_COMPRESS))
		sprintf(s,"LZO1B-99");
#endif
#if !defined(LZO_999_UNSUPPORTED)
	else if (method == M_LZO1B_999 && (WORK_LEN >= LZO1B_999_MEM_COMPRESS))
		sprintf(s,"LZO1B-999");
#endif
#endif
#if defined(HAVE_LZO1C_H)
	else if (method >= M_LZO1C_1 && method <= M_LZO1C_9)
	{
		int clevel = method - M_LZO1C_1 + 1;
		sprintf(s,"LZO1C-%d",clevel);
	}
#if !defined(LZO_99_UNSUPPORTED)
	else if (method == M_LZO1C_99 && (WORK_LEN >= LZO1C_99_MEM_COMPRESS))
		sprintf(s,"LZO1C-99");
#endif
#if !defined(LZO_999_UNSUPPORTED)
	else if (method == M_LZO1C_999 && (WORK_LEN >= LZO1C_999_MEM_COMPRESS))
		sprintf(s,"LZO1C-999");
#endif
#endif
#if defined(HAVE_LZO1F_H)
	else if (method == M_LZO1F_1)
		sprintf(s,"LZO1F-1");
#if !defined(LZO_999_UNSUPPORTED)
	else if (method == M_LZO1F_999 && (WORK_LEN >= LZO1F_999_MEM_COMPRESS))
		sprintf(s,"LZO1F-999");
#endif
#endif
#if defined(HAVE_LZO1X_H)
	else if (method == M_LZO1X_1)
		sprintf(s,"LZO1X-1");
#if !defined(LZO_999_UNSUPPORTED)
	else if (method == M_LZO1X_999 && (WORK_LEN >= LZO1X_999_MEM_COMPRESS))
		sprintf(s,"LZO1X-999");
#endif
#endif
#if defined(HAVE_LZO2A_H)
#if !defined(LZO_999_UNSUPPORTED)
	else if (method == M_LZO2A_999 && (WORK_LEN >= LZO2A_999_MEM_COMPRESS))
		sprintf(s,"LZO2A-999");
#endif
#endif
#if defined(HAVE_LZO2B_H)
#if !defined(LZO_999_UNSUPPORTED)
	else if (method == M_LZO2B_999 && (WORK_LEN >= LZO2B_999_MEM_COMPRESS))
		sprintf(s,"LZO2B-999");
#endif
#endif
#if defined(HAVE_ZLIB_H)
	else if (method >= M_ZLIB_8_1 && method <= M_ZLIB_8_9)
		sprintf(s,"zlib-%d/%d", Z_DEFLATED, method - M_ZLIB_8_1 + 1);
#endif
#if defined(MFX)
#  include "contrib/t_info.ch"
#endif
	else if (method == M_MEMCPY)
		sprintf(s,"memcpy()");
	else if (method == M_MEMSET)
		sprintf(s,"memset()");
	else if (method == M_ADLER32)
		sprintf(s,"adler32");

	if (s[0] == 0)
	{
		if (f != NULL)
		{
			fprintf(f,"invalid method %d !\n",method);
			exit(EXIT_USAGE);
		}
		return 0;
	}
	else
	{
		if (f != NULL)
			fprintf(f,"%s",s);
		return 1;
	}
}


/*************************************************************************
// compress and decompress a file
**************************************************************************/

static int do_file(const char *name, int c_loops, int d_loops, lzo_uint32 *ad)
{
	FILE *f;
	int i, r = 0;
	lzo_uint l = 0, t_len = 0;
	unsigned long ll;
	int blocks = 0;
	int t_loops = 1;
	clock_t t_time = 0, c_time = 0, d_time = 0;
	clock_t t_start, c_start, d_start;
	unsigned long t_bytes, c_bytes, d_bytes;
	double t_kbs, c_kbs, d_kbs;
	double t_secs, c_secs, d_secs;
	double perc, bits;
	lzo_uint32 adler;

	if (c_loops < 1)  c_loops = 1;
	if (d_loops < 1)  d_loops = 1;

	fflush(stdout);
	fflush(stderr);

	adler_in = adler_out = 0;
	if (ad)
		*ad = 0;

	/* read the whole file */
	f = fopen(name,"rb");
	if (f == NULL)
	{
		fprintf(stderr,"%s: ",name);
		perror("fopen");
		return 0;				/* do not raise an error */
	}
	l = my_fread(f,data,DATA_LEN);
	if (fclose(f) != 0)
	{
		fprintf(stderr,"%s: ",name);
		perror("fclose");
		return EXIT_FILE;
	}

	adler = lzo_adler32(0, NULL, 0);
	adler = lzo_adler32(adler, data, l);
	if (ad)
		*ad = adler;

	if (verbose >= 2)
	{
		printf("File %s: %lu bytes   adler32 = 0x%08lx\n",
		        name, (long) l, (long) adler);
		printf("  compressing %lu bytes (%d/%d/%d loops, %lu block-size)\n",
		        (long) l, t_loops, c_loops, d_loops, (long) block_size);
		printf("  "); info(stdout); printf("\n");
	}

	/* compress the file */

	t_start = clock();
	for (i = 0; i < t_loops; i++)
	{
		lzo_uint len, c_len, c_len_max, d_len;
		lzo_byte *d = data;

		len = l;
		t_len = 0;
		blocks = 0;
		if (len > 0 || try_to_compress_0_bytes) do
		{
			int j;
			lzo_byte *dd = d;
			lzo_byte *b1 = block1;
			lzo_byte *b2 = block2;
			const lzo_uint bl = len > block_size ? block_size : len;
			lzo_uint bl_overwrite = bl;
#if !defined(__BOUNDS_CHECKING_ON)
			unsigned char random_byte;
#endif

			if (use_assembler_decompressor)
				bl_overwrite += 3;		/* may overwrite 3 bytes past the end */

			blocks++;

#if defined(__BOUNDS_CHECKING_ON)
			/* malloc a block of the exact block size to detect
		 	* any overun.
		 	*/
			dd = malloc(bl_overwrite);
			if (dd == NULL)
				return EXIT_MEM;
			memcpy(dd,d,bl);
			b2 = dd;
#endif

		/* compress a block */
			c_len = c_len_max = 0;
			c_start = clock();
			for (j = r = 0; r == 0 && j < c_loops; j++)
			{
				c_len = sizeof(_block1) - 256;
				r = do_compress(dd,b1,bl,&c_len);
				if (r == 0 && c_len > c_len_max)
					c_len_max = c_len;
			}
			c_time += clock() - c_start;
			if (r != 0)
			{
				printf("  compression failed in block %d (%d) (%lu %lu)\n",
					blocks, r, (long)c_len, (long)bl);
				return EXIT_LZO_ERROR;
			}

		/* decompress the block and verify */
#if defined(__BOUNDS_CHECKING_ON)
			lzo_memset(b2,0,bl);
#else
			lzo_memset(b2,0,bl_overwrite);
			/* check that no data past the end gets overwritten */
			random_byte = (unsigned char) c_start;
			b2[bl_overwrite] = random_byte;
#endif
			d_start = clock();
			for (j = r = 0; r == 0 && j < d_loops; j++)
			{
				d_len = bl;
				r = do_decompress(b1,b2,c_len,&d_len);
				if (d_len != bl)
					break;
			}
			d_time += clock() - d_start;
			if (r != 0)
			{
				printf("  decompression failed in block %d (%d) "
					"(%lu %lu %lu)\n", blocks, r,
					(long)c_len, (long)d_len, (long)bl);
				return EXIT_LZO_ERROR;
			}
			if (d_len != bl)
			{
				printf("  decompression size error in block %d (%lu %lu %lu)\n",
					blocks, (long)c_len, (long)d_len, (long)bl);
				return EXIT_LZO_ERROR;
			}
			if (method <= M_LAST_COMPRESSOR)
			{
				if (lzo_memcmp(d,b2,bl) != 0)
				{
					lzo_uint x = 0;
					while (x < bl && b2[x] == d[x])
						x++;
					printf("  decompression data error in block %d at offset "
						"%lu (%lu %lu)\n", blocks, (long)x,
						(long)c_len, (long)d_len);
					if (compute_adler32)
						printf("      checksum: 0x%08lx 0x%08lx\n",
							(long)adler_in, (long)adler_out);
#if 0
					printf("Orig:  ");
					for (j = (x >= 10 ? -10 : 0); j <= 10 && x + j < bl; j++)
						printf(" %02x", (int)b2[x+j]);
					printf("\nDecomp:");
					for (j = (x >= 10 ? -10 : 0); j <= 10 && x + j < bl; j++)
						printf(" %02x", (int)d[x+j]);
					printf("\n");
#endif
					return EXIT_LZO_ERROR;
				}
				if (compute_adler32 && adler_in != adler_out)
				{
					printf("  checksum error in block %d (%lu %lu)\n",
						blocks, (long)c_len, (long)d_len);
					printf("      checksum: 0x%08lx 0x%08lx\n",
						(long)adler_in, (long)adler_out);
					return EXIT_LZO_ERROR;
				}
			}
#if !defined(__BOUNDS_CHECKING_ON)
			if (b2[bl_overwrite] != random_byte)
			{
				printf("  decompression overwrite error in block %d "
					"(%lu %lu %lu)\n",
					blocks, (long)c_len, (long)d_len, (long)bl);
				return EXIT_LZO_ERROR;
			}
#endif

#if defined(__BOUNDS_CHECKING_ON)
			if (dd != d)
				free(dd);
#endif
			d += bl;
			len -= bl;
			t_len += c_len_max;
		}
		while (len > 0);
	}
	t_time += clock() - t_start;


	/* print some statistics */

	ll = l;
	perc = (ll > 0) ? t_len * 100.0 / ll : 0;
	bits = perc * 0.08;
	t_secs = t_time / (double)(CLOCKS_PER_SEC);
	t_bytes = (ll * c_loops + ll * d_loops) * t_loops;
	t_kbs = (t_secs > 0.001) ? (t_bytes / t_secs) / 1000.0 : 0;
	c_secs = c_time / (double)(CLOCKS_PER_SEC);
	c_bytes = ll * c_loops * t_loops;
	c_kbs = (c_secs > 0.001) ? (c_bytes / c_secs) / 1000.0 : 0;
	d_secs = d_time / (double)(CLOCKS_PER_SEC);
	d_bytes = ll * d_loops * t_loops;
	d_kbs = (d_secs > 0.001) ? (d_bytes / d_secs) / 1000.0 : 0;

	if (verbose >= 2)
	{
		printf("  compressed into %lu bytes, %.2f%%   %.3f\n",
			(long) t_len, perc, bits);

		printf("%-15s %5d: ","overall", t_loops);
		printf("%10lu bytes, %8.2f secs, %8.2f KB/sec\n",
		        t_bytes, t_secs, t_kbs);
		printf("%-15s %5d: ","compress", c_loops);
		printf("%10lu bytes, %8.2f secs, %8.2f KB/sec\n",
		        c_bytes, c_secs, c_kbs);
		printf("%-15s %5d: ","decompress", d_loops);
		printf("%10lu bytes, %8.2f secs, %8.2f KB/sec\n",
		        d_bytes, d_secs, d_kbs);
		printf("\n");
	}

	/* create a line for util/table.pl */

	if (verbose >= 1)
	{
		/* get basename */
		const char *n, *nn;
		for (nn = n = name; *nn; nn++)
			if (*nn == '/' || *nn == '\\' || *nn == ':')
				n = nn + 1;

		info(NULL);
		printf("%-8s | %-14s %7ld %4d %7ld %6.1f %5.2f %8.2f %8.2f |\n",
			method_name,
			n, (long) l, blocks, (long) t_len, perc, bits, c_kbs, d_kbs);
	}

	if (verbose >= 2)
		printf("\n");

	return EXIT_OK;
}


/*************************************************************************
// Calgary Corpus test suite driver
**************************************************************************/

static const struct
{
	const char *name;
	int loops;
	lzo_uint32 adler;
}
corpus[] =
{
	{ "bib",       8,  0x4bd09e98L },
	{ "book1",     1,  0xd4d3613eL },
	{ "book2",     1,  0x6fe14cc3L },
	{ "geo",       6,  0xf3cc5be0L },
	{ "news",      2,  0x2ed405b8L },
	{ "obj1",     35,  0x3887dd2cL },
	{ "obj2",      4,  0xf89407c4L },
	{ "paper1",   17,  0xfe65ce62L },
	{ "paper2",   11,  0x1238b7c2L },
	{ "pic",       4,  0xf61a5702L },
	{ "progc",    25,  0x4c00ba45L },
	{ "progl",    20,  0x4cba738eL },
	{ "progp",    28,  0x7495b92bL },
	{ "trans",    15,  0x52a2cec8L }
};


static int do_corpus(const char *path, int c_loops, int d_loops)
{
	int i;
	size_t n;
	char name[256];

	if (path == NULL || strlen(path) >= sizeof(name) - 12)
		return EXIT_USAGE;

	strcpy(name,path);
	n = strlen(name);
	if (n > 0 && name[n-1] != '/' && name[n-1] != '\\')
	{
		strcat(name,"/");
		n++;
	}

	for (i = 0; i < (int) (sizeof(corpus)/sizeof(*(corpus))); i++)
	{
		lzo_uint32 adler;
		int c = c_loops * corpus[i].loops;
		int d = d_loops * corpus[i].loops;
		int r;

		strcpy(name+n,corpus[i].name);
		r = do_file(name, c, d, &adler);
		if (r != 0)
			return r;
		if (adler != corpus[i].adler)
			return EXIT_ADLER;
	}
	return EXIT_OK;
}


/*************************************************************************
//
**************************************************************************/

static void usage(const char *name, int show_methods)
{
	FILE *f;
	int i;

	f = stdout;

	fflush(stdout); fflush(stderr);

	fprintf(f,"Usage: %s [option..] file...\n", name);
	fprintf(f,"\n");
	fprintf(f,"Options:\n");
	fprintf(f,"  -m#     compression method\n");
	fprintf(f,"  -b#     set input block size (default %ld)\n",
		(long) block_size);
	fprintf(f,"  -n#     number of compression/decompression runs\n");
	fprintf(f,"  -c#     number of compression runs\n");
	fprintf(f,"  -d#     number of decompression runs\n");
	fprintf(f,"  -z      use assembler decompressor (if available)\n");
	fprintf(f,"  -s dir  process Calgary Corpus test suite in directory 'dir'\n");
	fprintf(f,"  -@      read list of files to compress from stdin\n");
	fprintf(f,"  -q      be quiet\n");
	fprintf(f,"  -Q      be very quiet\n");
	fprintf(f,"  -L      display software license\n");

	if (show_methods)
	{
		fprintf(f,"\n");
		fprintf(f,"The following compression methods are available:\n");
		fprintf(f,"\n");

		for (i = 0; i <= M_LAST_COMPRESSOR; i++)
		{
			method = i;
			if (info(NULL))
			{
				char n[16];
				sprintf(n,"-m%d",i);
				fprintf(f,"  %-6s  %s\n",n,method_name);
			}
		}
	}
	else
	{
		fprintf(f,"\n");
		fprintf(f,"Type '%s -m' to list all available methods.\n", name);
	}

	fflush(f);
	exit(EXIT_USAGE);
}


static void license(void)
{
	FILE *f;

	f = stdout;

	fflush(stdout); fflush(stderr);

fprintf(f,
"   The LZO library is free software; you can redistribute it and/or\n"
"   modify it under the terms of the GNU General Public License as\n"
"   published by the Free Software Foundation; either version 2 of\n"
"   the License, or (at your option) any later version.\n"
"\n"
"   The LZO library is distributed in the hope that it will be useful,\n"
"   but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
"   GNU General Public License for more details.\n"
"\n"
"   You should have received a copy of the GNU General Public License\n"
"   along with the LZO library; see the file COPYING.\n"
"   If not, write to the Free Software Foundation, Inc.,\n"
"   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.\n"
"\n"
"   Markus F.X.J. Oberhumer\n"
"   markus.oberhumer@jk.uni-linz.ac.at\n"
"\n"
	);

	fflush(f);
	exit(EXIT_OK);
}


/*************************************************************************
//
**************************************************************************/

int main(int argc, char *argv[])
{
	int i = 1, ii, j = 0;
	int c_loops = 0;
	int d_loops = 0;
	int r = EXIT_OK;
	const char *corpus_path = NULL;
	lzo_bool read_from_stdin = 0;
	const short *methods = NULL;
	int a_method = 0;
	const char *argv0 = "";
	time_t t_total;

#if defined(__EMX__)
	_response(&argc,&argv);
	_wildcard(&argc,&argv);
#endif

	argv0 = argv[0];
	align_mem();

	printf("\nLZO real-time data compression library (v%s, %s).\n",
			LZO_VERSION_STRING, LZO_VERSION_DATE);
	printf("Copyright (C) 1996, 1997 Markus Franz Xaver Johannes Oberhumer\n\n");

	if (lzo_init() != LZO_E_OK)
	{
		printf("lzo_init() failed !!!\n");
		exit(EXIT_LZO_INIT);
	}

	if (argc < 2)
		usage(argv0,0);

	if (i < argc && isdigit(argv[i][0]) && argv[i][0] > '0')
		method = atoi(&argv[i++][0]), methods = NULL;

	for ( ; i < argc && argv[i][0] == '-'; i++)
	{
		char *p = argv[i];

		if (isdigit(p[1]))
			method = atoi(p+1), methods = NULL;
		else if (strncmp(p,"-m",2) == 0)
		{
			if (isdigit(p[2]))
				method = atoi(p+2), methods = NULL;
			else if (strcmp(p,"-mall") == 0)
				methods = all_methods;
			else if (strcmp(p,"-mavail") == 0)
				methods = avail_methods;
			else if (strcmp(p,"-mbench") == 0)
				methods = benchmark_methods;
			else if (strcmp(p,"-mx1") == 0)
				methods = x1_methods;
			else if (strcmp(p,"-mx99") == 0)
				methods = x99_methods;
			else if (strcmp(p,"-mx999") == 0)
				methods = x999_methods;
			else if (strcmp(p,"-mzlib") == 0)
				methods = zlib_methods;
#if defined(MFX)
#  include "contrib/t_opt_m.ch"
#endif
			else
				usage(argv0,1);
		}
		else if (strcmp(p,"-a") == 0)
			compute_adler32 = 1;
		else if (strncmp(p,"-b",2) == 0)
		{
			if (isdigit(p[2]))
				block_size = atoi(p+2);
			else
				block_size = MAX_BLOCK_SIZE;
		}
		else if (strncmp(p,"-c",2) == 0 && isdigit(p[2]))
			c_loops = atoi(p+2);
		else if (strncmp(p,"-d",2) == 0 && isdigit(p[2]))
			d_loops = atoi(p+2);
		else if (strcmp(p,"-h") == 0 || strcmp(p,"-?") == 0)
			usage(argv0,0);
		else if (strcmp(p,"--help") == 0)
			usage(argv0,0);
		else if (strcmp(p,"-L") == 0 || strcmp(p,"--license") == 0)
			license();
		else if (strncmp(p,"-n",2) == 0 && isdigit(p[2]))
			c_loops = atoi(p+2), d_loops = c_loops;
		else if (strcmp(p,"-q") == 0)
			verbose = 1;
		else if (strcmp(p,"-Q") == 0)
			verbose = 0;
		else if (strncmp(p,"-s",2) == 0 && i + 1 < argc)
		{
			if (isdigit(p[2]))
				c_loops = atoi(p+2), d_loops = c_loops;
			corpus_path = argv[++i];
		}
		else if (strcmp(p,"--version") == 0)
			exit(r);
		else if (strcmp(p,"-z") == 0)
			use_assembler_decompressor = 1;
		else if (strcmp(p,"-@") == 0)
			read_from_stdin = 1;
		else
		{
			printf("%s: invalid option '%s'\n\n",argv0,p);
			usage(argv0,0);
		}
	}

	if (methods != NULL && read_from_stdin)
	{
		printf("%s: cannot use multiple methods and '-@'\n", argv0);
		exit(EXIT_USAGE);
	}

	if (block_size < 16)
		block_size = 16;
	if (block_size > MAX_BLOCK_SIZE)
		block_size = MAX_BLOCK_SIZE;

	t_total = time(NULL);
	ii = i;
	do {
		i = ii;
		if (i >= argc && corpus_path == NULL && !read_from_stdin)
			usage(argv0,0);
		if (methods != NULL && j == 0 && verbose >= 1)
			printf("%lu block-size\n\n", (long) block_size);

		if (methods == avail_methods)
		{
			lzo_bool found = 0;
			while (!found && a_method <= M_LAST_COMPRESSOR)
			{
				method = a_method++;
				found = info(NULL);
			}
			if (!found)
				break;
		}
		else if (methods != NULL)
		{
			while ((method = *methods++) < 0)
				;
			if (method == 0)
				break;
			if (!info(NULL))
				info(stdout);
		}
		else
		{
			if (!info(NULL))
				info(stdout);
			if (verbose >= 1)
			{
				info(stdout);
				printf(" algorithm   %lu block-size\n\n", (long) block_size);
			}
		}

		if (corpus_path != NULL)
			r = do_corpus(corpus_path,c_loops,d_loops);
		else
		{
			for ( ; i < argc && r == EXIT_OK; i++)
				r = do_file(argv[i],c_loops,d_loops,NULL);
			if (read_from_stdin)
			{
				char buf[512], *p;

				while (r == EXIT_OK && fgets(buf,sizeof(buf)-1,stdin) != NULL)
				{
					buf[sizeof(buf)-1] = 0;
					p = buf + strlen(buf);
					if (p > buf)
						while (--p > buf && isspace(((unsigned char) *p)))
							*p = 0;
					p = buf;
					while (*p && isspace(((unsigned char) *p)))
						p++;
					if (*p)
						r = do_file(p,c_loops,d_loops,NULL);
				}
				read_from_stdin = 0;
			}
		}
		j++;
	}
	while (r == EXIT_OK && methods != NULL);

	if (r == EXIT_OK && methods != NULL && verbose >= 1)
	{
		t_total = time(NULL) - t_total;
		printf("\nexecution time: %lu seconds\n", (long) t_total);
	}

	return r;
}

/*
vi:ts=4
*/

