//******************************************************************************
//
//  Name:  PipingSet.cpp
//
//  Description:  Implementation of PipingSet class.
//
//******************************************************************************
#include "pipeset.h"
#include "DAS1600.h"

// Mail messages:
#define START_CONTROL         1
#define STOP_CONTROL          2
#define CHANGE_SETPOINT_START 3
#define SETPOINT_COMPLETE     4

extern DAS1600Driver *Das1600;
//------------------------------------------------------------------------------
//
//  Name:  PipingSet::PipingSet()
//
//  Description:  Constructor for the piping set.
//
//------------------------------------------------------------------------------
PipingSet::PipingSet( double kP, double kI, double kD, double lI, double cD) :
           PidControl( kP, kI, kD, lI, cD )
{
	CurrentMode = PIPE_MEASURE;
   mbox = OSMboxCreate( (void*) 0 );
   OSTaskCreate( (void(*) (void*))PipingSet::StartControlTask, (void *)this,
                 &TaskStack[CONTROL_STACK_SIZE], PD_CONTROL_PRIO );
}
//------------------------------------------------------------------------------
//
//  Name:  PipingSet::SetServoVoltage()
//
//  Description:  Sets the voltage on the servo valve(s).
//
//------------------------------------------------------------------------------
int PipingSet::SetServoVoltage( float voltage )
{
   int status = PIPE_SET_SUCCESS;

   if( voltage <= 10.0 && voltage >= -10.0 ) {
      Das1600->DacOut( 0, voltage );
      }
   else {
      status = PIPE_SET_INVALID_SERVO_VOLTS;
      }
   return status;
}
//------------------------------------------------------------------------------
//
//  Name:  PipingSet::StartControl()
//
//  Description:  Sends a message to the control task to start controlling
//                pressure using the pressure sensor object passed in as
//                the controller reference.
//
//------------------------------------------------------------------------------
void PipingSet::StartControl( PressureSensor *sensor )
{
   controlSensor = sensor;
   OSMboxPost( mbox, (void*)START_CONTROL );
}
//------------------------------------------------------------------------------
//
//  Name:  PipingSet::StopControl()
//
//  Description:  Sends a message the to control task indicating it should stop.
//                Closes Control Valve, Pressure Supply Valve, and exhaust valve.
//
//------------------------------------------------------------------------------
void PipingSet::StopControl()
{
   OSMboxPost( mbox, (void*)STOP_CONTROL );

}
//------------------------------------------------------------------------------
//
//  Name:  PipingSet::SetSetpoint()
//
//  Description:  Stops the control task while the setpoint is changed.
//                SetPoint is converted from PSI to full-scale
//------------------------------------------------------------------------------
void PipingSet::SetSetpoint( double SetPoint )
{

   OSMboxPost( mbox, (void*)CHANGE_SETPOINT_START );
   PidControl::SetSetpoint( ( SetPoint / FullScale ) * 100 );
   OSMboxPost( mbox, (void*)SETPOINT_COMPLETE );
}

//------------------------------------------------------------------------------
//
//  Name:  PipingSet::StartControlTask()
//
//  Description:  Static member function used solely to kick-start the control
//                task.
//
//------------------------------------------------------------------------------
void PipingSet::StartControlTask( void *PipeSetInstance )
{
   ((PipingSet *)PipeSetInstance)->ControlTask();
}
//------------------------------------------------------------------------------
//
//  Name:  PipingSet::ControlTask()
//
//  Description:  Uses a PID control algorithm to control the pressure in the
//                piping set.
//
//------------------------------------------------------------------------------
void PipingSet::ControlTask( void )
{
   UBYTE err;
   double pressure;
   int    stop_control;
   double gain;
   float  volts;
   void   *msg;

   for(;;) {
      msg = OSMboxPend( mbox, 0, &err );
      if( (int ) msg == START_CONTROL ) {
      	CurrentMode = PIPE_CONTROL;
         stop_control = FALSE;
         while( !stop_control ) {
            pressure = controlSensor->get_pressure();
            gain = CalculateGain( ( pressure / FullScale ) * 100 );
            volts = -( gain / 100 );
            if( volts > 10 )
            	volts = 10;
            else if( volts < -10 )
            	volts = -10;
            SetServoVoltage( volts );
            msg = OSMboxPend( mbox, 1, &err );
            if( err == OS_NO_ERR ) {
               if( (int)msg == STOP_CONTROL ) {
                  SetServoVoltage( 0.0 );
                  stop_control = true;
                  }
               else if( (int)msg == CHANGE_SETPOINT_START ) {
                  do
                     msg = OSMboxPend( mbox, 0, &err );
                     while( (int)msg != SETPOINT_COMPLETE );
                  }
               }
            }
         CurrentMode = PIPE_MEASURE;
         }
      }
}
//------------------------------------------------------------------------------
//
//  Name:  PipingSet::SetFullScale()
//
//  Description:  Sets the full-scale data member of this class and sets the
//                High-pressure/Low-Pressure power solenoid.
//
//------------------------------------------------------------------------------
void PipingSet::SetFullScale( float full_scale, bool highVoltage )
{
   FullScale = full_scale;
}

