// BIN.H

// A technique to handle binary constants
// at compile-time. Use as follows:
// int n = BIN_BYTE( 1110 ) ;
// to generate at *compile-time* code like:
// int n = 14 ;
// Use BIN_WORD_LE( n, m ) to generate
// two-byte words in little endian order
// (that is, n * 256 + m).
// Uses a template metaprogramming technique
// to convert from decimal representation
// to binary numbers at compile time.
// Generates compile-time errors if number
// contains non-binary digits or exceeds
// the byte range.

// Author: Dr. Carlo Pescio
// Eptacom Consulting
// Via Bernardo Forte 2-3
// 17100 Savona - ITALY
// Fax +39-19-854761
// email pescio@programmers.net


#ifndef BIN_H

#define BIN_H

// Just to generate errors,
// see below
unsigned long NotABinaryDigit() 
  {
  return( 0 ) ;
  }

// Default is to generate a compile-time error,
// by using a non-constant expression.
// You may change the following code with some
// compiler-specific expression that generates
// a more readable error message.
template< unsigned long N > struct BinDigitOrError
  {
  enum { value = NotABinaryDigit() } ;
  } ;

// Specializations for 0 and 1,
// returns the value itself.
struct BinDigitOrError< 1 >
  {
  enum { value = 1 } ;
  } ;

struct BinDigitOrError< 0 >
  {
  enum { value = 0 } ;
  } ;

// Just to generate errors,
// see below
int ByteOverflow()
  {
  return( 0 ) ;
  }

// Check that N is at most 8 bits long. 
// Here I assume all digits are 0 or 1, 
// that's checked elsewhere.
template< unsigned long N > struct InRangeBinByte
  {
  enum { value = (N <= 11111111L) } ;
  } ;

// Convert a decimal number N, whose digits are
// all 1 or 0, into a binary number with the 
// same representation of N, using recursion.
// Generate a compile-time error if N is
// larger than a byte can hold, or if some
// of the digits in N are not 0 or 1.
template< unsigned long N > struct BinByte
  {
  enum { value = InRangeBinByte< N > :: value ? 
          BinDigitOrError< (N % 10) > :: value + 
          2 * BinByte< N / 10 > :: value :
          ByteOverflow() } ;
  } ;

// Base of recursion.
struct BinByte< 0 >
  {
  enum { value = 0 } ;
  } ;

// Just some syntactic sugar macros
#define BIN_BYTE( N ) \
  ((unsigned int)BinByte< (N) > :: value)
#define BIN_WORD_LE( N, M ) \
  ( 256 * BIN_BYTE( N ) + BIN_BYTE( M ) )

#endif // ifndef BIN_H
