// ****************************************************************************
//
// Module:  parser.C
// Author:  Dick Lam
//
// Purpose: C++ class source file for cgiParser
//
// Notes:  This is a base class. It handles the reading from an input stream
//         and subsequent decoding and parsing of CGI data. The result is
//         a list of cgiName and cgiValue strings.
//
// ****************************************************************************

#include <strstrea.h>
#include <cgi/parser.h>

// ****************************************************************************

// cgiParser - constructor

cgiParser::cgiParser(istream& input)
   : in(input)

{
   // init instance variables
}

// ----------------------------------------------------------------------------

// ~cgiParser - destructor

cgiParser::~cgiParser()

{
}

// ----------------------------------------------------------------------------

// parse - reads, decodes, and parses the input data into name, value pairs

int cgiParser::parse()

{
   // erase all entries in the data list and seek to the
   // start of the input stream
   clear();
   in.seekg( streampos(0) );

   // read all the name, value pairs - the format is:
   // "name1=value1&name2=value2..."
   while ( !in.eof() ) {
      // read the next name (which terminates with '=')
      cgiName name( getString('=') );
      if (name.size() == 0)
         break;

      // read its value (which terminates with '&') - note that a value may
      // or may not be present
      cgiValue value( getString('&') );

      // put the pair into the data list
      data[name] = value;
   }

   return 0;
}

// ----------------------------------------------------------------------------

// clear - erase all entries in the data list

void cgiParser::clear()

{
   if ( !data.empty() )
      data.erase( data.begin(), data.end() );
}

// ----------------------------------------------------------------------------

// getString - returns a string from the input stream, stopping at the
//             specified delimiter character

string cgiParser::getString(char delimiter)

{
   // form a string until end of file or delimiter is found
   char c;
   string s("");

   while ( !in.eof() ) {
      in >> c;    // skips whitespace
      if (c == delimiter)
        return decode(s);

      s += c;
   }

   return decode(s);
}

// ----------------------------------------------------------------------------

// decode - removes escaped characters and changes '+' characters back
//          into spaces

string cgiParser::decode(const string& encoded)

{
   // make sure encoded string has something in it
   if (encoded.size() == 0)
      return string("");

   // remove any escaped characters
   string decoded("");

   for (int i = 0; i < encoded.size(); i++) {
      if (encoded[i] == '%') {
         // convert escaped character
         decoded += unescaped(encoded[i+1], encoded[i+2]);
         i += 2;
      } else {
         // copy the regular character
         decoded += encoded[i];
      }
   }

   // change the '+' characters to spaces
   for (i = 0; i < decoded.size(); i++) {
      if (decoded[i] == '+')
         decoded[i] = ' ';
   }

   return decoded;
}

// ----------------------------------------------------------------------------

// unescaped - returns a character corresponding to the hex value h1h2

char cgiParser::unescaped(char h1, char h2)

{
   strstream ss;
   int i;

   ss << h1 << h2 << ends;
   ss >> hex >> i;

   return (char)i;
}

// ****************************************************************************

// end of parser.C
