/*
    ssdetect.cpp
    Written by Dan Shappir 1996.
*/
#ifndef STRICT
#define STRICT 1
#endif
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef WIN32_EXTRA_LEAN
#define WIN32_EXTRA_LEAN
#endif

#include <windows.h>
#include <tlhelp32.h>

const LPCTSTR SS_WINDOW_CLASS = TEXT("WindowsScreenSaverClass");
const LPCTSTR SS_EXT          = TEXT(".SCR");
const int     SS_EXT_LEN      = lstrlen(SS_EXT);
const LPTSTR  SS_DESK         = TEXT("screen-saver");
const LPCTSTR TH_DLL          = TEXT("KERNEL32.DLL");
const LPCTSTR TH_SNAPSHOT     = TEXT("CreateToolhelp32Snapshot");
const LPCTSTR TH_FIRST        = TEXT("Process32First");
const LPCTSTR TH_NEXT         = TEXT("Process32Next");

static DWORD GetOSVersion()
{
    OSVERSIONINFO verInfo;
    verInfo.dwOSVersionInfoSize = sizeof(verInfo);
    GetVersionEx(&verInfo);
    return verInfo.dwPlatformId;
}
const DWORD   OS_VERSION      = GetOSVersion();

static BOOL IsTopmost(HWND hWnd)
{
    return (GetWindowLong(hWnd, GWL_EXSTYLE) & WS_EX_TOPMOST) != 0;
}


static BOOL IsNotConfigureOrPreview(HWND hWnd)
{
extern BOOL IsFullScreen(HWND hWnd);

    return IsWindow(hWnd) &&
           ( GetParent(hWnd) == NULL || IsTopmost(hWnd) ||
             IsFullScreen(hWnd) );
}

static BOOL ScreenSaverClassRunning()
{
    HWND hWnd = FindWindow(SS_WINDOW_CLASS, NULL);
    return IsNotConfigureOrPreview(hWnd);
}

static BOOL ScreenSaverActiveNT()
{
    HDESK hDesktop = OpenDesktop(SS_DESK,0,FALSE,MAXIMUM_ALLOWED);
    if ( hDesktop ) {
        CloseDesktop(hDesktop);
        return TRUE;
    }
    return GetLastError() == ERROR_ACCESS_DENIED;
}

static BOOL HasScreenSaverExt(LPCTSTR lpszName)
{
    int length = lstrlen(lpszName);
    return length >= SS_EXT_LEN &&
           lstrcmpi(lpszName+length-SS_EXT_LEN, SS_EXT) == 0;
}

static BOOL ScreenSaverActiveW32s()
{
    HWND hWnd = GetActiveWindow();
    if ( IsNotConfigureOrPreview(hWnd) ) {
        HINSTANCE hInstance = reinterpret_cast<HINSTANCE>
            (GetWindowLong(hWnd, GWL_HINSTANCE));
        if ( hInstance ) {
            TCHAR szBuffer[MAX_PATH];
            GetModuleFileName(hInstance,szBuffer,sizeof(szBuffer));
            return HasScreenSaverExt(szBuffer);
        }
    }
    return FALSE;
}

class CToolHelp {
    typedef HANDLE (WINAPI *CREATE_SNAPSHOT)(DWORD, DWORD);
    typedef BOOL (WINAPI *FIND_PROCESS)(HANDLE, LPPROCESSENTRY32);

    HINSTANCE        m_library;
    CREATE_SNAPSHOT m_createSnapshot;
    FIND_PROCESS    m_firstProcess;
    FIND_PROCESS    m_nextProcess;

public:
    CToolHelp();
    ~CToolHelp();
    HANDLE FirstProcess(PROCESSENTRY32& pe) const;
    BOOL NextProcess(HANDLE hSnapshot, PROCESSENTRY32& pe) const
    {
        return m_nextProcess(hSnapshot, &pe);
    }
};

CToolHelp::CToolHelp() : m_library(LoadLibrary(TH_DLL))
    {
    if ( m_library ) {
        m_createSnapshot = reinterpret_cast<CREATE_SNAPSHOT>
            (GetProcAddress(m_library, TH_SNAPSHOT));
        if ( m_createSnapshot ) {
            m_firstProcess = reinterpret_cast<FIND_PROCESS>
                (GetProcAddress(m_library, TH_FIRST));
            m_nextProcess = reinterpret_cast<FIND_PROCESS>
                (GetProcAddress(m_library, TH_NEXT));
            }
        }
    else
        m_createSnapshot = NULL;
    }

CToolHelp::~CToolHelp()
    {
    if ( m_library )
        FreeLibrary(m_library);
    }
HANDLE CToolHelp::FirstProcess(PROCESSENTRY32& pe) const
    {
    if ( m_nextProcess ) {
        HANDLE hSnapshot =
            m_createSnapshot(TH32CS_SNAPPROCESS, 0);
        if ( hSnapshot ) {
            pe.dwSize = sizeof(PROCESSENTRY32);
            if ( m_firstProcess(hSnapshot, &pe) )
                return hSnapshot;
            CloseHandle(hSnapshot);
            }
        }
    return NULL;
    }

static BOOL ScreenSaverActiveW95()
{
    HWND hWnd = GetForegroundWindow();
    if ( IsNotConfigureOrPreview(hWnd) ) {
        static CToolHelp ToolHelp;
        PROCESSENTRY32 pe;
        HANDLE hSnapshot = ToolHelp.FirstProcess(pe);
        if ( hSnapshot ) {
            DWORD dwProcessId;
            GetWindowThreadProcessId(hWnd, &dwProcessId);
            do {
                if ( dwProcessId == pe.th32ProcessID ) {
                    CloseHandle(hSnapshot);
                    return HasScreenSaverExt(pe.szExeFile);
                }
            } while ( ToolHelp.NextProcess(hSnapshot, pe) );
            CloseHandle(hSnapshot);
        }
    }
    return FALSE;
}

struct CScr {
    const int H, W;
    CScr() : W(GetSystemMetrics(SM_CXSCREEN)),
             H(GetSystemMetrics(SM_CYSCREEN))  {}
};

BOOL IsFullScreen(HWND hWnd)
{
    if ( IsWindowVisible(hWnd) ) {
        WINDOWPLACEMENT wp;
        wp.length = sizeof(wp);
        if ( GetWindowPlacement(hWnd, &wp) &&
             wp.showCmd != SW_SHOWMINIMIZED ) {
            CScr scr;
            RECT r;
            return GetWindowRect(hWnd, &r) &&
                   r.left <= 0 && r.right >= scr.W &&
                   r.top <= 0 && r.bottom >= scr.H;
        }
    }
    return FALSE;
}

BOOL ScreenSaverActive()
{
    if ( ScreenSaverClassRunning() )
        return TRUE;
    switch ( OS_VERSION ) {
        case VER_PLATFORM_WIN32_NT:
            return ScreenSaverActiveNT();
        case VER_PLATFORM_WIN32s:
            return ScreenSaverActiveW32s();
        case VER_PLATFORM_WIN32_WINDOWS:
            return ScreenSaverActiveW95();
    }
    return FALSE;
}
// End Of File
