
// __________________________________________________________
//
//                          ComSpy.c
//               Serial Communication Spy V1.00
//                11-16-1996 Sven B. Schreiber
//                 100557.177@compuserve.com
// __________________________________________________________

#include <ntddk.h>     // nt ddk definitions
#include "ComSpy.h"    // device independent definitions
#include "ComSpy.dev"  // device dependent definitions

// =================================================================
// GLOBAL VARIABLES
// =================================================================

PDEVICE_OBJECT pFilterDevice, pControlDevice;

// =================================================================
// DISCARDABLE FUNCTIONS
// =================================================================

NTSTATUS CreateDevices (PDRIVER_OBJECT DriverObject,
                        PUNICODE_STRING RegistryPath);

NTSTATUS DriverEntry   (PDRIVER_OBJECT DriverObject,
                        PUNICODE_STRING RegistryPath);

// -----------------------------------------------------------------

#ifdef ALLOC_PRAGMA

#pragma alloc_text (INIT, CreateDevices)
#pragma alloc_text (INIT, DriverEntry)

#endif

// =================================================================
// PROTOCOL MANAGEMENT
// =================================================================

void ProtocolReset (PCONTROL_CONTEXT pControlContext)
    {
    pControlContext->dStatus = 0;
    pControlContext->dRead   = 0;
    pControlContext->dWrite  = 0;
    return;
    }

// -----------------------------------------------------------------

DWORD ProtocolGetStatus (PCONTROL_CONTEXT pControlContext)
    {
    return pControlContext->dStatus;
    }

// -----------------------------------------------------------------

DWORD ProtocolSetStatus (PCONTROL_CONTEXT pControlContext,
                         DWORD            dNewStatus)
    {
    DWORD dOldStatus;

    dOldStatus = pControlContext->dStatus;
    pControlContext->dStatus = dNewStatus;
    return dOldStatus;
    }

// -----------------------------------------------------------------

DWORD ProtocolRead (PCONTROL_CONTEXT pControlContext,
                    PBYTE            pbData,
                    DWORD            dSize)
    {
    DWORD i, n;

    i = pControlContext->dRead;
    n = 0;
    while ((n != dSize) && (i != pControlContext->dWrite))
        {
        pbData [n++] = pControlContext->Data [i++];
        if (i == SPY_BUFFER_SIZE) i = 0;
        }
    pControlContext->dRead = i;
    return n;
    }

// -----------------------------------------------------------------

DWORD ProtocolWrite (PCONTROL_CONTEXT pControlContext,
                     PBYTE            pbData,
                     DWORD            dSize)
    {
    BYTE  b;
    DWORD i, j, n;

    i = pControlContext->dRead;
    j = pControlContext->dWrite;
    n = 0;
    while (n != dSize)
        {
        pControlContext->Data [j++] = pbData [n++];
        if (j == SPY_BUFFER_SIZE) j = 0;
        if (j == i)
            {
            do  {
                b = pControlContext->Data [i++];
                if (i == SPY_BUFFER_SIZE) i = 0;
                }
            while ((b != '\n') && (i != j));
            }
        }
    pControlContext->dRead  = i;
    pControlContext->dWrite = j;
    return n;
    }

// =================================================================
// PROTOCOL OUTPUT
// =================================================================

DWORD OutAnsi (PCONTROL_CONTEXT pControlContext,
               PSTR             psData,
               DWORD            dMinSize,
               char             cFiller)
    {
    DWORD i, n;

    for (i = 0; psData [i]; i++);
    n = ProtocolWrite (pControlContext, psData, i);
    for (; i < dMinSize; i++)
        {
        n += ProtocolWrite (pControlContext, &cFiller, 1);
        }
    return n;
    }

// -----------------------------------------------------------------

DWORD OutWide (PCONTROL_CONTEXT pControlContext,
               PWSTR            puData,
               DWORD            dMinSize,
               char             cFiller)
    {
    char  c;
    DWORD i, n;

    n = 0;
    for (i = 0; puData [i]; i++)
        {
        c = (puData [i] < 0x100 ? puData [i] : '?');
        n += ProtocolWrite (pControlContext, &c, 1);
        }
    for (; i < dMinSize; i++)
        {
        n += ProtocolWrite (pControlContext, &cFiller, 1);
        }
    return n;
    }

// -----------------------------------------------------------------

DWORD OutChar (PCONTROL_CONTEXT pControlContext,
               char             cData,
               DWORD            dMinSize,
               char             cFiller)
    {
    DWORD i, n;

    n = ProtocolWrite (pControlContext, &cData, 1);
    for (i = 1; i < dMinSize; i++)
        {
        n += ProtocolWrite (pControlContext, &cFiller, 1);
        }
    return n;
    }

// -----------------------------------------------------------------

DWORD OutUnsigned (PCONTROL_CONTEXT pControlContext,
                   DWORD            dNumber,
                   DWORD            dMinSize,
                   char             cFiller)
    {
    char  sNumber [100+1];
    DWORD x, i;

    x = dNumber;
    i = 100;
    sNumber [i] = 0;
    do  {
        sNumber [--i] = (char) (x % 10) + '0';
        x /= 10;
        }
    while (x);
    while (i && ((100-i) < dMinSize))
        {
        sNumber [--i] = cFiller;
        }
    return OutAnsi (pControlContext, &sNumber [i], 0, ' ');
    }

// -----------------------------------------------------------------

DWORD OutSigned (PCONTROL_CONTEXT pControlContext,
                 DWORD            dNumber,
                 DWORD            dMinSize,
                 char             cFiller)
    {
    char  sNumber [100+1];
    DWORD x, i;

    x = (dNumber & 0x80000000 ? (~dNumber)+1 : dNumber);
    i = 100;
    sNumber [i] = 0;
    do  {
        sNumber [--i] = (char) (x % 10) + '0';
        x /= 10;
        }
    while (x);
    if (dNumber & 0x80000000)
        {
        sNumber [--i] = '-';
        }
    while (i && ((100-i) < dMinSize))
        {
        sNumber [--i] = cFiller;
        }
    return OutAnsi (pControlContext, &sNumber [i], 0, ' ');
    }

// -----------------------------------------------------------------

DWORD OutHex (PCONTROL_CONTEXT pControlContext,
              DWORD            dNumber,
              DWORD            dMinSize,
              char             cFiller)
    {
    static char sHex [] = "0123456789ABCDEF";
    char        sNumber [100+1];
    DWORD       x, i;

    x = dNumber;
    i = 100;
    sNumber [i] = 0;
    do  {
        sNumber [--i] = sHex [x & 0xF];
        x >>= 4;
        }
    while (x);
    while (i && ((100-i) < dMinSize))
        {
        sNumber [--i] = cFiller;
        }
    return OutAnsi (pControlContext, &sNumber [i], 0, ' ');
    }

// -----------------------------------------------------------------

DWORD OutText (PCONTROL_CONTEXT pControlContext,
               PSTR             psFormat,
               PVOID            pArg)
    {
    PSTR   psText;
    PDWORD pdNext;
    DWORD  i, n, dMinSize;
    char   cFiller;

    psText = psFormat;
    pdNext = pArg;

    n = 0;
    while (psText [0])
        {
        for (i = 0; psText [i] && (psText [i] != '%'); i++);
        n      += ProtocolWrite (pControlContext, psText, i);
        psText += i;

        i = 0;
        if (psText [i] == '%')
            {
            cFiller  = (psText [++i] == '0' ? '0' : ' ');
            dMinSize = 0;
            while (((DWORD) psText [i] - '0') < 10)
                {
                dMinSize *= 10;
                dMinSize += (psText [i++] - '0');
                }
            switch (psText [i++])
                {
                case 'a':

                    if (cFiller == '0') cFiller = '.';
                    n += OutAnsi
                             (pControlContext,
                              *(PSTR *) pdNext++,
                              dMinSize,
                              cFiller);
                    break;

                case 'w':

                    if (cFiller == '0') cFiller = '.';
                    n += OutWide
                             (pControlContext,
                              *(PWSTR *) pdNext++,
                              dMinSize,
                              cFiller);
                    break;

                case 'c':

                    if (cFiller == '0') cFiller = *(PCHAR) pdNext;
                    n += OutChar
                             (pControlContext,
                              *(PCHAR) pdNext++,
                              dMinSize,
                              cFiller);
                    break;

                case 'u':

                    n += OutUnsigned
                             (pControlContext,
                              *(PDWORD) pdNext++,
                              dMinSize,
                              cFiller);
                    break;

                case 's':

                    n += OutSigned
                             (pControlContext,
                              *(PDWORD) pdNext++,
                              dMinSize,
                              cFiller);
                    break;

                case 'h':

                    n += OutHex
                             (pControlContext,
                              *(PDWORD) pdNext++,
                              dMinSize,
                              cFiller);
                    break;

                case '!':

                    pdNext++;
                    break;

                case '%':

                    n += ProtocolWrite
                             (pControlContext,
                              psText+i-1,
                              1);
                    break;

                default:

                    n += ProtocolWrite
                             (pControlContext,
                              psText,
                              i);
                    break;
                }
            psText += i;
            }
        }
    return n;
    }

// -----------------------------------------------------------------

DWORD Out (PCONTROL_CONTEXT pControlContext,
           PSTR             psFormat,
           ...)
    {
    return OutText (pControlContext,
                    psFormat,
                    (PVOID) ((DWORD) &psFormat + sizeof (PSTR)));
    }

// -----------------------------------------------------------------

DWORD Dump (PCONTROL_CONTEXT pControlContext,
            PSTR             psCaption,
            PBYTE            pbData,
            DWORD            dSize,
            DWORD            dOffset)
    {
    DWORD i, j, k, l, n;
    BYTE  b;
    
    n = 0;
    if (dSize)
        {
        n += Out (pControlContext, psCaption);
        }
    i = dOffset;
    j = 0;
    while (j != dSize)
        {
        n += Out (pControlContext, "   %08h: ", i & 0xFFFFFFF0);
        k  = j;
        for (l = 0; l < 16; l++)
            {
            n += Out (pControlContext, "%c", (l != 8 ? ' ' : '-'));
            if ((l < (i & 0x0000000F)) || (j == dSize))
                {
                n += Out (pControlContext, "..");
                }
            else
                {
                b  = pbData [j++];
                n += Out (pControlContext, "%02h", b);
                }
            }
        n += Out (pControlContext, "  ");
        j  = k;
        for (l = 0; l < 16; l++)
            {
            if ((l < (i & 0x0000000F)) || (j == dSize))
                {
                n += Out (pControlContext, ".");
                }
            else
                {
                b = pbData [j++];
                if ((b < 0x20) || (b == 0x7F)) b = '.';
                n += Out (pControlContext, "%c", b);
                }
            }
        i += (16 - (i & 0x0000000F));
        n += Out (pControlContext, "\n");
        }
    return n;
    }

// =================================================================
// INFORMATION ROUTINES
// =================================================================

void GetFunctionName (DWORD dFunction,
                      DWORD dSubFunction,
                      PSTR  *ppsFunction,
                      PSTR  *ppsSubFunction)
    {
    *ppsFunction    = NULL;
    *ppsSubFunction = NULL;

    switch (dFunction)
        {
        case IRP_MJ_CREATE:
            *ppsFunction = "Create";
            break;

        case IRP_MJ_CREATE_NAMED_PIPE:
            *ppsFunction = "Create Named Pipe";
            break;

        case IRP_MJ_CLOSE:
            *ppsFunction = "Close";
            break;

        case IRP_MJ_READ:
            *ppsFunction = "Read";
            break;

        case IRP_MJ_WRITE:
            *ppsFunction = "Write";
            break;

        case IRP_MJ_QUERY_INFORMATION:
            *ppsFunction = "Query Information";
            break;

        case IRP_MJ_SET_INFORMATION:
            *ppsFunction = "Set Information";
            break;

        case IRP_MJ_QUERY_EA:
            *ppsFunction = "Query Extended Attribute";
            break;

        case IRP_MJ_SET_EA:
            *ppsFunction = "Set Extended Attribute";
            break;

        case IRP_MJ_FLUSH_BUFFERS:
            *ppsFunction = "Flush Buffers";
            break;

        case IRP_MJ_QUERY_VOLUME_INFORMATION:
            *ppsFunction = "Query Volume Information";
            break;

        case IRP_MJ_SET_VOLUME_INFORMATION:
            *ppsFunction = "Set Volume Information";
            break;

        case IRP_MJ_DIRECTORY_CONTROL:
            *ppsFunction = "Directory Control";
            switch (dSubFunction)
                {
                case IRP_MN_QUERY_DIRECTORY:
                    *ppsSubFunction = "Query Directory";
                    break;

                case IRP_MN_NOTIFY_CHANGE_DIRECTORY:
                    *ppsSubFunction = "Notify Change Directory";
                    break;
                }
            break;

        case IRP_MJ_FILE_SYSTEM_CONTROL:
            *ppsFunction = "File System Control";
            switch (dSubFunction)
                {
                case IRP_MN_USER_FS_REQUEST:
                    *ppsSubFunction = "User File System Request";
                    break;

                case IRP_MN_MOUNT_VOLUME:
                    *ppsSubFunction = "Mount Volume";
                    break;

                case IRP_MN_VERIFY_VOLUME:
                    *ppsSubFunction = "Verify Volume";
                    break;

                case IRP_MN_LOAD_FILE_SYSTEM:
                    *ppsSubFunction = "Load File System";
                    break;
                }
            break;

        case IRP_MJ_DEVICE_CONTROL:
            *ppsFunction = "Device Control";
            break;

        case IRP_MJ_INTERNAL_DEVICE_CONTROL:
            *ppsFunction = "Internal Device Control";
            break;

        case IRP_MJ_SHUTDOWN:
            *ppsFunction = "Shutdown";
            break;

        case IRP_MJ_LOCK_CONTROL:
            *ppsFunction = "Lock Control";
            switch (dSubFunction)
                {
                case IRP_MN_LOCK:
                    *ppsSubFunction = "Lock";
                    break;

                case IRP_MN_UNLOCK_SINGLE:
                    *ppsSubFunction = "Unlock Single";
                    break;

                case IRP_MN_UNLOCK_ALL:
                    *ppsSubFunction = "Unlock All";
                    break;

                case IRP_MN_UNLOCK_ALL_BY_KEY:
                    *ppsSubFunction = "Unlock All By Key";
                    break;
                }
            break;

        case IRP_MJ_CLEANUP:
            *ppsFunction = "Clean Up";
            break;

        case IRP_MJ_CREATE_MAILSLOT:
            *ppsFunction = "Create Mailslot";
            break;

        case IRP_MJ_QUERY_SECURITY:
            *ppsFunction = "Query Security";
            break;

        case IRP_MJ_SET_SECURITY:
            *ppsFunction = "Set Security";
            break;

        case IRP_MJ_SET_POWER:
            *ppsFunction = "Set Power";
            break;

        case IRP_MJ_QUERY_POWER:
            *ppsFunction = "Query Power";
            break;
        }
    return;
    }

// =================================================================
// DRIVER SERVICES
// =================================================================

NTSTATUS SpyControl (PDEVICE_OBJECT pDeviceObject,
                     PIRP           pIrp,
                     PVOID          pContext)
    {
    PCONTROL_CONTEXT   pControlContext;
    PIO_STACK_LOCATION pIrpThis;
    PVOID              pInput, pOutput;
    DWORD              dInput, dOutput, dFunction, dSubFunction;
    DWORD              Information = 0;
    NTSTATUS           Status = STATUS_SUCCESS;

    pControlContext = pControlDevice->DeviceExtension;
    pIrpThis        = IoGetCurrentIrpStackLocation (pIrp);
    dFunction       = pIrpThis->MajorFunction;
    dSubFunction    = pIrpThis->MinorFunction;

    switch (dFunction)
        {
        case IRP_MJ_CREATE:
        case IRP_MJ_CLEANUP:
        case IRP_MJ_CLOSE:
            
            break;

        case IRP_MJ_DEVICE_CONTROL:
            
            pInput  = pIrp->AssociatedIrp.SystemBuffer;
            dInput  = IRP_IO (InputBufferLength);

            pOutput = pIrp->AssociatedIrp.SystemBuffer;
            dOutput = IRP_IO (OutputBufferLength);

            switch (IRP_IO (IoControlCode))
                {
                case IOCTL_INFO:

                    if (dOutput >= sizeof (FILTER_INFO))
                        {
                        RtlCopyMemory (pOutput,
                                       &FilterInfo,
                                       sizeof (FILTER_INFO));
                        Information = sizeof (FILTER_INFO);
                        }
                    else
                        {
                        Status = STATUS_BUFFER_TOO_SMALL;
                        }
                    break;

                case IOCTL_RESET:

                    ProtocolReset (pControlContext);
                    ProtocolSetStatus (pControlContext, 0);
                    Information = Out (pControlContext,
                                       "\n+++ DEVICE RESET +++\n");
                    break;

                case IOCTL_READ:

                    Information = ProtocolRead (pControlContext,
                                                pOutput,
                                                dOutput);
                    break;

                case IOCTL_WRITE:

                    Information = ProtocolWrite (pControlContext,
                                                 pInput,
                                                 dInput);
                    break;

                default:

                    Status = STATUS_IO_DEVICE_ERROR;
                    Information = Out (pControlContext,
                                       "\n+++ DEVICE ERROR +++\n");
                    break;
                }
            break;

        default:

            Status = STATUS_NOT_IMPLEMENTED;
            break;
        }
    pIrp->IoStatus.Status      = Status;
    pIrp->IoStatus.Information = Information;
    IoCompleteRequest (pIrp, IO_NO_INCREMENT);
    return Status;
    }

// -----------------------------------------------------------------

NTSTATUS SpyFilterBegin (PDEVICE_OBJECT pDeviceObject,
                         PIRP           pIrp,
                         PVOID          pContext)
    {
    PCONTROL_CONTEXT   pControlContext;
    PIO_STACK_LOCATION pIrpThis;
    PVOID              pInput;
    DWORD              dInput, dFunction, dSubFunction;
    DWORD              dIoCode, dIoMethod, dIoFunction;
    PSTR               psFunction, psSubFunction, psIoFunction;
    PSTR               psIoMethod;

    pControlContext = pControlDevice->DeviceExtension;
    pIrpThis        = IoGetCurrentIrpStackLocation (pIrp);
    dFunction       = pIrpThis->MajorFunction;
    dSubFunction    = pIrpThis->MinorFunction;

    if (ProtocolSetStatus (pControlContext, 1))
        {
        Out (pControlContext,
             "\n"
             "----------------------------------------"
             "---------------------------------------\n");
        }
    GetFunctionName (dFunction,   dSubFunction,
                     &psFunction, &psSubFunction);

    Out (pControlContext, "\nFunction:   %4u", dFunction);
    if (psFunction != NULL)
        {
        Out (pControlContext, " = %a", psFunction);
        }
    Out (pControlContext, "\nSubfunction:%4u", dSubFunction);
    if (psSubFunction != NULL)
        {
        Out (pControlContext, " = %a", psSubFunction);
        }
    Out (pControlContext, "\n");

    pInput     = NULL;
    psIoMethod = NULL;

    switch (dFunction)
        {
        case IRP_MJ_CREATE:

            if (IRP_FILE (Length))
                {
                Out (pControlContext,
                     "\n   Open/create file object \"%w\"\n",
                     IRP_FILE (Buffer));
                }
            else
                {
                Out (pControlContext,
                     "\n   Open device object\n");
                }
            break;

        case IRP_MJ_WRITE:

            psIoMethod  = "   Unknown";
            dInput      = IRP_WRITE (Length);
            dIoMethod   = pFilterDevice->Flags &
                          (DO_BUFFERED_IO | DO_DIRECT_IO);

            switch (dIoMethod)
                {
                case DO_BUFFERED_IO:

                    pInput     = pIrp->AssociatedIrp.SystemBuffer;
                    psIoMethod = "  Buffered";
                    break;

                case DO_NEITHER_IO:

                    pInput     = pIrp->UserBuffer;
                    psIoMethod = "Unbuffered";
                    break;
                
                case DO_DIRECT_IO:

                    psIoMethod = "Direct Out";
                    break;
                }
            break;

        case IRP_MJ_DEVICE_CONTROL:
        case IRP_MJ_INTERNAL_DEVICE_CONTROL:
        case IRP_MJ_FILE_SYSTEM_CONTROL:

            psIoMethod  = "   Unknown";
            dInput      = IRP_IO (InputBufferLength);
            dIoCode     = IRP_IO (IoControlCode);
            dIoFunction = IoGetFunctionCodeFromCtlCode (dIoCode);
            dIoMethod   = IoGetMethodCodeFromCtlCode   (dIoCode);

            switch (dIoMethod)
                {
                case METHOD_BUFFERED:

                    pInput     = pIrp->AssociatedIrp.SystemBuffer;
                    psIoMethod = "  Buffered";
                    break;

                case METHOD_NEITHER:

                    pInput     = IRP_IO (Type3InputBuffer);
                    psIoMethod = "Unbuffered";
                    break;

                case METHOD_IN_DIRECT:

                    psIoMethod = " Direct In";
                    break;

                case METHOD_OUT_DIRECT:

                    psIoMethod = "Direct Out";
                    break;
                }
            Out (pControlContext,
                 "\n   I/O control code:    0x%08h",
                 dIoCode);

            Out (pControlContext,
                 "\n   I/O function code:        0x%03h",
                 dIoFunction);

            GetIoFunctionName (dIoFunction, &psIoFunction);
            if (psIoFunction != NULL)
                {
                Out (pControlContext, " = %a", psIoFunction);
                }
            Out (pControlContext, "\n");

            break;
        }
    if (psIoMethod != NULL)
        {
        Out (pControlContext,
             "\n   I/O transfer type:   %a",
             psIoMethod);

        Out (pControlContext,
             "\n   Input buffer size:   %10u\n",
             dInput);

        if ((pInput != NULL) && dInput)
            {
            Out (pControlContext,
                 "   Input buffer base:   0x%08h\n",
                 pInput);

            Dump (pControlContext, "\n", pInput, dInput, 0);
            }
        }
    return STATUS_SUCCESS;
    }

// -----------------------------------------------------------------

NTSTATUS SpyFilterEnd (PDEVICE_OBJECT pDeviceObject,
                       PIRP           pIrp,
                       PVOID          pContext)
    {
    PCONTROL_CONTEXT   pControlContext;
    PIO_STACK_LOCATION pIrpThis;
    PVOID              pOutput;
    DWORD              dOutput, dReturned, dFunction, dSubFunction;
    DWORD              dIoCode, dIoMethod;
    PSTR               psIoMethod;

    pControlContext = pControlDevice->DeviceExtension;
    pIrpThis        = IoGetCurrentIrpStackLocation (pIrp);
    dFunction       = pIrpThis->MajorFunction;
    dSubFunction    = pIrpThis->MinorFunction;

    pOutput    = NULL;
    psIoMethod = NULL;

    switch (dFunction)
        {
        case IRP_MJ_READ:

            psIoMethod  = "   Unknown";
            dOutput     = IRP_READ (Length);
            dIoMethod   = pFilterDevice->Flags &
                          (DO_BUFFERED_IO | DO_DIRECT_IO);

            switch (dIoMethod)
                {
                case DO_BUFFERED_IO:

                    pOutput    = pIrp->AssociatedIrp.SystemBuffer;
                    psIoMethod = "  Buffered";
                    break;

                case DO_NEITHER_IO:

                    pOutput    = pIrp->UserBuffer;
                    psIoMethod = "Unbuffered";
                    break;
                
                case DO_DIRECT_IO:

                    psIoMethod = "Direct Out";
                    break;
                }
            break;

        case IRP_MJ_DEVICE_CONTROL:
        case IRP_MJ_INTERNAL_DEVICE_CONTROL:
        case IRP_MJ_FILE_SYSTEM_CONTROL:

            psIoMethod = "   Unknown";
            dOutput    = IRP_IO (OutputBufferLength);
            dIoCode    = IRP_IO (IoControlCode);
            dIoMethod  = IoGetMethodCodeFromCtlCode (dIoCode);

            switch (dIoMethod)
                {
                case METHOD_BUFFERED:

                    pOutput    = pIrp->AssociatedIrp.SystemBuffer;
                    psIoMethod = "  Buffered";
                    break;

                case METHOD_NEITHER:

                    pOutput    = pIrp->UserBuffer;
                    psIoMethod = "Unbuffered";
                    break;

                case METHOD_IN_DIRECT:

                    psIoMethod = " Direct In";
                    break;

                case METHOD_OUT_DIRECT:

                    psIoMethod = "Direct Out";
                    break;
                }
            break;
        }
    if (psIoMethod != NULL)
        {
        dReturned = pIrp->IoStatus.Information;

        Out (pControlContext,
             "\n   I/O transfer type:   %a",
             psIoMethod);

        Out (pControlContext,
             "\n   Output buffer size:  %10u",
             dOutput);

        Out (pControlContext,
             "\n   Output packet size:  %10u\n",
             dReturned);

        if ((pOutput != NULL) && dOutput)
            {
            Out (pControlContext,
                 "   Output buffer base:  0x%08h\n",
                 pOutput);

            Dump (pControlContext, "\n", pOutput, dReturned, 0);
            }
        }
    Out (pControlContext,
         "\nRequest status = 0x%08h"
         "\nRequest info   = %10u\n",
         pIrp->IoStatus.Status,
         pIrp->IoStatus.Information);

    return STATUS_SUCCESS;
    }

// -----------------------------------------------------------------

NTSTATUS SpyFilterExit (PDEVICE_OBJECT pDeviceObject,
                        PIRP           pIrp,
                        PVOID          pContext)
    {
    if (pIrp->PendingReturned)
        {
        IoMarkIrpPending (pIrp);
        }
    else
        {
        SpyFilterEnd (pDeviceObject, pIrp, pContext);
        }
    return pIrp->IoStatus.Status;
    }

// -----------------------------------------------------------------

NTSTATUS SpyFilter (PDEVICE_OBJECT pDeviceObject,
                    PIRP           pIrp,
                    PVOID          pContext)
    {
    PFILTER_CONTEXT    pFilterContext;
    PIO_STACK_LOCATION pIrpThis, pIrpNext;

    pFilterContext = pDeviceObject->DeviceExtension;
    pIrpThis       = IoGetCurrentIrpStackLocation (pIrp);
    pIrpNext       = IoGetNextIrpStackLocation (pIrp);

    SpyFilterBegin (pDeviceObject, pIrp, pContext);

    *pIrpNext = *pIrpThis;

    IoSetCompletionRoutine (pIrp,
                            SpyFilterExit,
                            pContext,
                            TRUE,
                            TRUE,
                            TRUE);
    
    return IoCallDriver (pFilterContext->pTargetDevice, pIrp);
    }

// -----------------------------------------------------------------

NTSTATUS SpyHandler (PDEVICE_OBJECT pDeviceObject,
                     PIRP           pIrp)
    {
    NTSTATUS Status;

    if (pDeviceObject == pControlDevice)
        {
        Status = SpyControl (pDeviceObject, pIrp, NULL);
        }
    else
        {
        Status = SpyFilter (pDeviceObject, pIrp, NULL);
        }
    return Status;
    }

// =================================================================
// DRIVER UNLOADING
// =================================================================

void SpyDriverUnload (PDRIVER_OBJECT pDriverObject)
    {
    PFILTER_CONTEXT pFilterContext;
    UNICODE_STRING  usLinkName;

    pFilterContext = pFilterDevice->DeviceExtension;
    IoDetachDevice (pFilterContext->pTargetDevice);
    IoDeleteDevice (pFilterDevice);

    RtlInitUnicodeString (&usLinkName, uSymbolicLink);
    IoDeleteSymbolicLink (&usLinkName);
    IoDeleteDevice (pControlDevice);

    return;
    }

// =================================================================
// DRIVER INITIALIZATION
// =================================================================

NTSTATUS CreateDevices (PDRIVER_OBJECT  pDriverObject,
                        PUNICODE_STRING pusRegistryPath)
    {
    PFILTER_CONTEXT  pFilterContext;
    PCONTROL_CONTEXT pControlContext;
    UNICODE_STRING   usDeviceName, usLinkName;
    NTSTATUS         Status;

    RtlInitUnicodeString (&usDeviceName, uControlDevice);
    Status = IoCreateDevice (pDriverObject,
                             sizeof (CONTROL_CONTEXT),
                             &usDeviceName,
                             FILE_DEVICE_SPY_CONTROL,
                             0,
                             FALSE,
                             &pControlDevice);

    if (Status == STATUS_SUCCESS)
        {
        RtlInitUnicodeString (&usLinkName, uSymbolicLink);
        Status = IoCreateSymbolicLink (&usLinkName,
                                       &usDeviceName);

        if (Status != STATUS_SUCCESS)
            {
            IoDeleteDevice (pControlDevice);
            }
        else
            {
            pControlContext = pControlDevice->DeviceExtension;
            ProtocolReset (pControlContext);
            ProtocolSetStatus (pControlContext, 0);
            RtlInitUnicodeString (&usDeviceName, uFilterDevice);
            Status = IoCreateDevice (pDriverObject,
                                     sizeof (FILTER_CONTEXT),
                                     &usDeviceName,
                                     dFileDeviceSpyFilter,
                                     0,
                                     FALSE,
                                     &pFilterDevice);

            if (Status != STATUS_SUCCESS)
                {
                IoDeleteSymbolicLink (&usLinkName);
                IoDeleteDevice (pControlDevice);
                }
            else
                {
                pFilterContext = pFilterDevice->DeviceExtension;
                RtlInitUnicodeString (&usDeviceName, uTargetDevice);
                Status = IoAttachDevice (pFilterDevice,
                                         &usDeviceName,
                                         &pFilterContext
                                          ->pTargetDevice);

                if (Status != STATUS_SUCCESS)
                    {
                    IoDeleteDevice (pFilterDevice);
                    IoDeleteSymbolicLink (&usLinkName);
                    IoDeleteDevice (pControlDevice);
                    }
                else
                    {
                    pFilterDevice->Flags = pFilterContext
                                           ->pTargetDevice->Flags;
                    }
                }
            }
        }
    return Status;
    }

// -----------------------------------------------------------------

NTSTATUS DriverEntry (PDRIVER_OBJECT  pDriverObject,
                      PUNICODE_STRING pusRegistryPath)
    {
    PDRIVER_DISPATCH *Function;
    NTSTATUS         Status;

    Status = CreateDevices (pDriverObject, pusRegistryPath);
    if (Status == STATUS_SUCCESS)
        {
        Function = pDriverObject->MajorFunction;

        Function [IRP_MJ_CREATE]                   =
        Function [IRP_MJ_CLOSE]                    =
        Function [IRP_MJ_DEVICE_CONTROL]           =
        Function [IRP_MJ_CREATE_NAMED_PIPE]        =
        Function [IRP_MJ_READ]                     =
        Function [IRP_MJ_WRITE]                    =
        Function [IRP_MJ_QUERY_INFORMATION]        =
        Function [IRP_MJ_QUERY_EA]                 =
        Function [IRP_MJ_SET_EA]                   =
        Function [IRP_MJ_FLUSH_BUFFERS]            =
        Function [IRP_MJ_QUERY_VOLUME_INFORMATION] =
        Function [IRP_MJ_SET_VOLUME_INFORMATION]   =
        Function [IRP_MJ_DIRECTORY_CONTROL]        =
        Function [IRP_MJ_FILE_SYSTEM_CONTROL]      =
        Function [IRP_MJ_INTERNAL_DEVICE_CONTROL]  =
        Function [IRP_MJ_SHUTDOWN]                 =
        Function [IRP_MJ_LOCK_CONTROL]             =
        Function [IRP_MJ_CLEANUP]                  =
        Function [IRP_MJ_CREATE_MAILSLOT]          =
        Function [IRP_MJ_QUERY_SECURITY]           =
        Function [IRP_MJ_SET_SECURITY]             =
        Function [IRP_MJ_SET_INFORMATION]          = SpyHandler;

        pDriverObject->DriverUnload = SpyDriverUnload;
        }
    return Status;
    }

// =================================================================
// END OF PROGRAM
// =================================================================
