/**************************** MEMORY.C ********************************\
|*                                                                    *|
|* Displays in-memory tables and memory usage                         *|
|*                                                                    *|
|* (c) Copyright 1989, David Craig.  All rights reserved.             *|
|* Permission is granted to incorporate these functions into a larger *|
|* work and to distribute the resulting executable file.              *|
|*                                                                    *|
|* Usage: memory  [for PC/MS-DOS 4.0 and later]                       *|
|*                                                                    *|
|* Compilers: TurboC 2.0, MSC 5.1                                     *|
|* Memory models: any                                                 *|
\**********************************************************************/

/* must prevent MSC from word aligning contents of our structures */
#pragma pack (1)

#include "local.h"
#include "dos4.h"
#if !defined(MK_FP)
#define MK_FP(seg,ofs) (void far *) \
                       ( ( ( (unsigned long) ((unsigned)(seg)) ) << 16) \
                       |   ( (unsigned long) ((unsigned)(ofs)) )        )
#endif

#define CHARACTER_DEVICE        0x8000
#define IOCTL_SUPPORT           0x4000
#define NON_IBM_FORMAT          0x2000          /* Output-until-busy (char) */
#define OPEN_CLOSE_SUPPORT      0x0800
#define GENERIC_IOCTL           0x0040
#define IS_CLOCK                0x0008
#define IS_NUL                  0x0004
#define IS_STDOUT               0x0002          /* 32-bit sector numbers (blk) */
#define IS_STDIN                0x0001

int     attr_bit_masks[] =
{
    IOCTL_SUPPORT, NON_IBM_FORMAT, OPEN_CLOSE_SUPPORT,
    GENERIC_IOCTL, IS_CLOCK, IS_NUL, IS_STDOUT, IS_STDIN
};

char    *char_attr_bit_texts[] =
{
    "(ioctl)", "(output-until-busy)", "(rmv)", "(generic ioctl)",
    "(clock)", "(nul)", "(stdout)", "(stdin)"
};

char    *block_attr_bit_texts[] =
{
    "(ioctl)", "(non-ibm)", "(rmv)", "(generic ioctl)",
    "(clock)", "(nul)", "(32-bit sector numbers)", "(stdin)"
};

void    get_pointers(void);
void    process_mcbs(void);
void    print_func_52h_table(void);
void    list_devices(void);
void    print_device_info(struct device_hdr far *device);

int     block_drivers = 0;
int     character_drivers = 0;
word    line_count;
word    far *psp_ptr;
word    far *first_mcb;
struct  device_hdr far *current_device;
struct  mcb far *mcbp;
struct  mcb far *mcbt[1000];
struct  func_52h far *func_52h;

int main(int argc, char **argv)
{
    printf("%s (c) 1989 David Craig, All rights reserved.\n",
        argv[0]);
    if(argc > 1)
    {
        fprintf(stderr, "%s requires no parameters.\n", argv[0]);
        abort();
    }
    if(signal(SIGINT, SIG_IGN) == SIG_ERR)
    {
        cputs("\r\nCouldn't set SIGINT.\r\n");
        abort();
    }
    get_pointers();
    process_mcbs();
    print_func_52h_table();
    printf("\n");
    list_devices();
    return(0);
}

void print_func_52h_table()
{
    line_count = 0;
    printf("Dump of Function 52h table of pointers.\n\n");
    printf("Pointer to first disk table = %Fp.\n", func_52h->disk_table);
    printf("Pointer to System File Table (handle) = %Fp.\n",
        func_52h->sft_ptr);
    printf("Pointer to CLOCK$ device = %Fp.\n", func_52h->clock_ptr);
    printf("Pointer to CON device = %Fp.\n", func_52h->con_ptr);
    printf("Maximum sector size supported by DOS = 0x%04X.\n",
        func_52h->max_sec);
    printf("Pointer to Cache Control Block = %Fp.\n", func_52h->cache);
    printf("Pointer to Current Directory Structure = %Fp.\n",
        func_52h->drive_inf);
    printf("Pointer to System File Table (FCB) = %Fp.\n", func_52h->fcb_ptr);
    printf("Size of FCB System File Table = %d.\n", func_52h->size_fcb_table);
    printf("Drive count = %d.\n", func_52h->drive_count);
    printf("Last drive = %d.\n", func_52h->last_drive);
    return;
}

void process_mcbs()
{
    int     i;
    int     j;
    int     isgood;
    int     table_index;
    int     last_table_index;
    int     last_mcb_config = 0;
    byte    new[9];
    byte    far *env_ptr;

    table_index = 0;
    while(1)
    {
        mcbt[table_index++] = mcbp;
        if(mcbp->control == 'Z')
            break;
        mcbp = MK_FP(FP_SEG(mcbp) + mcbp->size + 1, FP_OFF(mcbp));
    }
    mcbp = MK_FP(*first_mcb, 0);
    last_table_index = table_index;
    printf("\n\n");
    printf("Addr  T  Own   Size   Name       Comments\n");
    printf("____  _  ____  ____   ________   ____________________________\n");
    line_count = 2;
    while(1)
    {
        if(line_count > 23)
        {
            line_count = 0;
        }
        printf("%4X  %c  %4X  %4X   ", FP_SEG(mcbp),
            mcbp->control, mcbp->owner, mcbp->size);
        isgood = 1;
        for(i = 0; i < 8; i++)
        {
            new[i] = mcbp->name[i];
            if(isgood == 1)
            {
                if(new[i] == '\0')
                {
                    isgood = 2;
                }
                else
                {
                    j = new[i];
                    if(!isprint(j))
                        isgood = 0;
                }
            }
        }
        new[8] = '\0';
        if(mcbp->owner == (FP_SEG(mcbp) + 1))
        {
            if(isgood)
                printf("%-8s   ", new);
            else
                printf("           ");
        }
        else if(mcbp->owner == 0)
        {
            printf("(MS-DOS)   ");
        }
        else if(mcbp->owner == 8)
        {
            printf("(IBMBIO)   ");
        }
        else
        {
            printf("           ");
        }
        if(_osmajor < 4)
        {
            if(last_mcb_config)
            {
                printf("(Shell)");
            }
            else
            {
                printf("       ");
            }
        }
        else
        {
            if(mcbp->owner == (FP_SEG(mcbp) + 1))
            {
                psp_ptr = MK_FP( (FP_SEG(mcbp) + 1), 0x2c);
                if(*psp_ptr > FP_SEG(psp_ptr))
                {
                    printf("(Shell)");
                }
                else
                {
                    printf("       ");
                }
            }
        }
        if(mcbp->owner == (FP_SEG(mcbp) + 1))
        {
            psp_ptr = MK_FP( (FP_SEG(mcbp) + 1), 0x2c);
            env_ptr = MK_FP(*psp_ptr, 0);
            psp_ptr = ( void far * ) env_ptr;
            for(table_index = 0; table_index < last_table_index &&
                FP_SEG(mcbt[table_index]) != (FP_SEG(env_ptr) - 1);
                table_index++) ;
            if(mcbt[table_index]->owner == (FP_SEG(mcbp) + 1))
            {
                printf("   Environment at %04X.   ", FP_SEG(env_ptr));
            }
            else
            {
                printf("   Environment at %04X (invalid).   ",
                    FP_SEG(env_ptr));
            }
            if(*psp_ptr)
            {
                for( ; *env_ptr; env_ptr++)
                {
                    for( ; *env_ptr; env_ptr++) ;
                }
                env_ptr++;
                psp_ptr = ( void far * ) env_ptr;
            }
            else
                psp_ptr++;
        }
        printf("\n");
        line_count++;
        if(mcbp->owner == 8)
            last_mcb_config = 1;
        else
            last_mcb_config = 0;
        if(mcbp->control == 'Z')
            break;
        mcbp = MK_FP(FP_SEG(mcbp) + mcbp->size + 1, FP_OFF(mcbp));
    }
    if(line_count > 23)
    {
        line_count = 0;
    }
    printf("\nNumber of entries = %d.\n\n", last_table_index);
    return;
}

void get_pointers()
{
    union   REGS regs;
    struct  SREGS sregs;

    segread(&sregs);
    regs.x.ax = 0x5200;
    (void) intdosx(&regs, &regs, &sregs);
    func_52h = MK_FP(sregs.es, regs.x.bx);
    regs.x.bx -= 2;
    first_mcb = MK_FP(sregs.es, regs.x.bx);
    mcbp = MK_FP(*first_mcb, 0);
    current_device = &func_52h->nul_dev;
    return;
}

void list_devices()
{
    line_count = 99;                    /* Force header to print */
    while(FP_OFF(current_device) != ( word ) 0xffff)
    {
        if(line_count > 23)
        {
            line_count = 2;
            printf("\n\n\n");
            printf("Address   Flags Name/Units Strategy  Interrupt  Comments\n");
            printf("--------- ----  ---------- --------- ---------  --------\n");
        }
        print_device_info(current_device);
        current_device = current_device->next_dev;
    }
    if(++line_count > 21)
    {
        line_count = 0;
    }
    printf("\nSystem has %d character drivers, and %d block drivers.\n",
        character_drivers, block_drivers);
    return;
}

void print_device_info(struct device_hdr far *device)
{
    char    device_string[9];
    int     i;

    printf("%Fp %04X  ", device, device->attribute);
    if(device->attribute & CHARACTER_DEVICE)
    {
        character_drivers++;
        for(i = 0; i < 8; i++)
            device_string[i] = device->name_unit[i];
        device_string[8] = '\0';
        printf("%s   ", device_string);
    }
    else
    {
        block_drivers++;
        printf("%2u units   ", (word) device->name_unit[0]);
    }
    printf("%04X:%04X %04X:%04X  ", FP_SEG(device), device->strategy,
        FP_SEG(device), device->intrupt);
    if(!strcmp(device_string, "NUL     "))
        printf("%u joined drives", (word) device->joined_drive_count);
    if(device->attribute & ~CHARACTER_DEVICE)
    {
        printf("\n          ");
        line_count++;
    }
    if(device->attribute & CHARACTER_DEVICE)
    {
        for(i = 0; i < 8; i++)
            if(attr_bit_masks[i] & device->attribute)
                printf("%s ", char_attr_bit_texts[i]);
    }
    else
    {
        for(i = 0; i < 8; i++)
            if(attr_bit_masks[i] & device->attribute)
                printf("%s ", block_attr_bit_texts[i]);
    }
    printf("\n");
    line_count++;
    return;
}

