///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// STATLINE.CPP - description of AStatusLineSample, sample-program for the progress-indicating status line
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////

#define INCL_WIN
#define INCL_GPI
#define INCL_WINERRORS
#define INCL_GPIERRORS
#define INCL_DOSPROCESS
#define INCL_DOSSEMAPHORES
#define INCL_DOSERRORS
  
#include <os2.h>
#include <imsgbox.hpp>
#include <ifiledlg.hpp>
#include <icolor.hpp>

#include "statline.hpp"
#include "statline.rch"
#include "about.hpp"

//*************************************************************************
// main  - Application entry point                                        *
//*************************************************************************
int main(int argc, char ** argv)
{
  AStatusLineSample mainWindow (WND_MAIN);        // Create our main window

  IApplication :: current (). run ();             // Get the current application and run it

  return (0);
}

//********************************************************************************
// AStatusLineSample :: AStatusLineSample - Constructor of AStatusLineSample     *
//********************************************************************************
AStatusLineSample :: AStatusLineSample(unsigned long windowId)
               : IFrameWindow (IFrameWindow::titleBar
                               | IFrameWindow::systemMenu
                               | IFrameWindow::sizingBorder
                               | IFrameWindow::minimizeButton
                               | IFrameWindow::maximizeButton
                               | IFrameWindow::windowList
                               | IFrameWindow::appDBCSStatus
                               | IFrameWindow::minimizedIcon
                               | IFrameWindow::menuBar
                               | IFrameWindow::accelerator,
                               windowId)
{
  setupVariables ();                            	// setup the variables

  setupStatusLine ();                             // setup the status line
  
  IKeyboardHandler :: handleEventsFor (this);     // set self as keyboard handler

  ICommandHandler :: handleEventsFor (this);      // set self as command handler

  AUserMessageHandler :: handleEventsFor (this);  // set self as user message handler

  setFocus ();                                    // set the focus

  show ();                                        // show the window
}

//********************************************************************************
// AStatusLineSample :: ~AStatusLineSample - Destructor of AStatusLineSample     *
//********************************************************************************
AStatusLineSample :: ~AStatusLineSample(void)

{
  if(m_pMenuBar)                                          // delete the menu bar
     delete m_pMenuBar;

  if(m_pFileThread)                                       // is a file thread active?
     {                                                    // yes
      if(m_pEditFile)                                     // does a file exists?
         m_pEditFile -> stopProcess ();                   // yes: stop the process
      else
         m_pFileThread -> stop ();                        // no: kill the thread
      IThread :: current(). waitFor (*m_pFileThread);     // wait till everything is done
      delete m_pFileThread;                               // delete the object
      m_pFileThread = 0L;                                 // reset the pointer
     }

  if(m_pEditFile)                                         // delete the file
     delete m_pEditFile;

  if(m_pText)                                             // delete the text
     delete m_pText;
}

//********************************************************************************
// AStatusLineSample :: setupVariables - variable set up                         *
//********************************************************************************
void AStatusLineSample :: setupVariables (void)
{
  m_pFileThread = 0L;

  m_pEditFile = 0L;

  m_pMenuBar = new IMenuBar (WND_MAIN, this);

  m_pText = new IStaticText (WND_TEXT, this, this);

  m_pText -> setColor (IStaticText :: foreground, IColor :: red);  // set the foreground color of the text to red

  m_pText -> setText (STR_TEXT);                                   // set the text displayed in the client area

  m_pText -> setAlignment (IStaticText :: centerCenter);           // set the position of the text

  setClient (m_pText);                                             // set the text as client of the framewindow
}

//********************************************************************************
// AStatusLineSample :: setupStatusLine - setting up the StatusLine              *
//********************************************************************************
void AStatusLineSample :: setupStatusLine (void)
{
  IPresSpaceHandle hps = m_pMenuBar -> presSpace ();          // get the presspace from the menu bar

  IFont * pMenueFont = new IFont (hps);                       // get the font of the menu bar

  releasePresSpace (hps);                                     // release the presspace

  m_pStatusLine = new AProcessStatus (this);                  // create the status line

  addExtension (m_pStatusLine, IFrameWindow :: belowClient,   // set the size and position of the statusline
                pMenueFont -> maxAscender() + (ULONG) 2 * pMenueFont -> maxDescender ());

  m_pStatusLine -> setFont (*pMenueFont);                     // set the font of the status line to the font of the menu bar

  delete pMenuFont;                                           // delete the font
}

//********************************************************************************
// AStatusLineSample :: virtualKeyPress- virtual-key-event processing            *
//********************************************************************************
Boolean AStatusLineSample :: virtualKeyPress (IKeyboardEvent& event)

{
  if(event. virtualKey () == IKeyboardEvent :: esc)  // did the user pressed the esc-key?
     {                                               // yes
      if(m_pFileThread                               // is there is a reading/writing thread running=
         && m_pEditFile)                             // yes
         m_pEditFile -> stopProcess ();              // stop it
      return (true);                                 // event handled
     }

  return (false);                                    // event not handled
}

//********************************************************************************
// AStatusLineSample :: command - handles Command-Events                         *
//********************************************************************************
Boolean AStatusLineSample :: command(ICommandEvent &evt)
{
  Boolean filtered = true;                                        // event handled
  ULONG ulCommandId = evt.commandId();

  switch(ulCommandId)                         	                  // handle the menu commands
         {
          case MI_FILE_OPEN:                                      // open a file
               startOpenFile ();
               break;
          case MI_FILE_SAVE:                                      // save the file
               startSaveFile ();
               break;
          case MI_FILE_EXIT:                                      // exit the program
               if(m_pFileThread)              	                  // is a file thread active?
                  {                                               // yes
                   if(m_pEditFile)                                // does a file object exists
                      m_pEditFile -> stopProcess ();              // yes: stop the processing
                   IThread :: current(). waitFor (*m_pFileThread);// wait till the thread has finished
                   delete m_pFileThread;                          // delete the thread object
                   m_pFileThread = 0L;                            // reset the pointer
                  }
               close ();                                          // close the framewindow
               break;
          case MI_HELP_ABOUT:                                     // show the product information
               {
                AAbout * pDlgAbout = new AAbout (this);           // create the dialog box
                pDlgAbout -> showModally ();                      // show it
                delete pDlgAbout;                                 // delete it
                break;
               }
          default:
              filtered = false;                                   // event not handled
              break;
         }

  return filtered;
}

//********************************************************************************
// AStatusLineSample :: startOpenFile - starts opening a file, starts thread     *
//********************************************************************************
Boolean AStatusLineSample :: startOpenFile (void)

{
  if(m_pFileThread)                                                         // is a file thread active?
     {
      IMessageBox msgBox (this);                                            // yes
      if(!loadString (STR_FILE_OPEN))                                       // load the message title
         m_String = "Open a file";
      msgBox.setTitle (m_String);                                           // set the title
      if(!loadString (STR_ALREADY_PROCEEDING))                              // load the message
         m_String = "A file process is still in progress.";
      msgBox.show (m_String, IMessageBox::okButton   |                      // show it
                             IMessageBox::errorIcon  |
                             IMessageBox::applicationModal |
                             IMessageBox::moveable);
      return (false);                                            	          // abort
     }

  if(!loadString (STR_FILE_OPEN))                                           // load the title for the open file dialog
     m_String = "Open a file";

  IFileDialog :: Settings fsettings;    

  fsettings. setTitle (m_String);                                           // set the title

  fsettings. setFileName ("*.*");                                           // set the filename

  IFileDialog fd( (IWindow*) desktopWindow (), (IWindow*) this, fsettings); // show the open file dialog

  if (!fd. pressedOK ())                                                    // if not pressed "OK", abort
      return (false);

  m_sFilename = fd.fileName ();                                             // save the filename

  m_pFileThread = new IThread ();                                           // create a thread object

  m_pFileThread -> start(new IThreadMemberFn <AStatusLineSample> (*this, AStatusLineSample :: openFile));   // start the read-thread

  return (true);                                                            // return
}

//********************************************************************************
// AStatusLineSample :: openFile - thread-function to open a file                *
//********************************************************************************
void AStatusLineSample :: openFile (void)

{
  APIRET ret;

  DosSetPriority (PRTYS_THREAD, PRTYC_REGULAR, 0L, 0UL);            // set the priority of the current thread

  Boolean bError = false;                                           // reset the error-flag

  if(m_pEditFile)                                                   // delete an existing file object
     {
      delete m_pEditFile;
      m_pEditFile = 0L;
     }

  if(!(m_pEditFile = new AEditFile ()))                             // create a new file object
     goto endOpenFile;

  if(m_pEditFile -> open (m_sFilename, 0L, FILE_NORMAL, FILE_OPEN)) // try to open the file
     {
      bError = true;
      goto endOpenFile;
     }

  try
     {
      m_pEditFile -> allocateDataBuffer ();                         // try to allocate the data buffer
     }

  catch (ULONG ulSize)                                              // can't get the file size or it is 0
         {
          bError = true;
          goto endOpenFile;
         }

  catch (char * pszBuffer)                                          // can't allocate the buffer
         {
          bError = true;
          goto endOpenFile;
         }

  catch (...)                                                       // some other exceptions
         {
          Boolean bPMAlreadyInit = true;
          ICurrentThread& rCurrThread = IThread :: current();       // get the current thread
          if (rCurrThread. isPMInitialized() != true)               // is the pm environment active for this thread?
              {                                                     // no
               bPMAlreadyInit = false;                              // set the flag
               rCurrThread. initializePM ();                        // initialize it
              }
          bError = true;                                            // set the error flag
          IMessageBox msgBox (this);                                // create and show a message box
          msgBox.setTitle ("Unexpected Error");
          msgBox.show ("Unexpected Error in AStatusLineSample :: openFile; AEditFile :: allocateDataBuffer", 
                       IMessageBox::okButton   |
                       IMessageBox::errorIcon  |
                       IMessageBox::applicationModal |
                       IMessageBox::moveable);
          if(!bPMAlreadyInit)                                       // was pm active for this thread?
             rCurrThread. terminatePM();                            // no: terminate it
          goto endOpenFile;                                         // quit
         }

  m_startParams. ulProcessSize = m_pEditFile -> getFileSize();      // set the start parameters for the status line
  m_startParams. ulTickSize = AEditFile :: eBytesToRead;
  m_startParams. lResourceId = STR_FILE_READ;

  WinPostMsg (m_pStatusLine -> handle(), MYM_STATUS_START_PROCESS, MPFROMP (&m_startParams), 0); // post them

  ret = m_pEditFile -> read (m_pStatusLine -> handle ());           // read the file

  if(ret)                                                           // was it successful?
     {                                                              // no
      bError = true;                                                // set the flag
      goto endOpenFile;                                             // quit
     }

  if(m_pEditFile -> close())                                        // try to close the file
     {
      bError = true;
      goto endOpenFile;
     }

  endOpenFile:

  if(bError)                                                        // did an error occured?
     {                                                              // yes
      delete m_pEditFile;                                           // delete the file
      m_pEditFile = 0L;                                             // reset the pointer
     }

  WinPostMsg (m_pStatusLine -> handle(), MYM_STATUS_END_PROCESS, 0, 0); // let the status line know the process has finished

  WinPostMsg (handle(), MYM_OPENFILE_READY, 0, 0);                  // let the frame window know the process has finished
}

//********************************************************************************
// AStatusLineSample :: openFileReady - perhabs the wanted file is open          *
//********************************************************************************
Boolean AStatusLineSample :: openFileReady (IEvent& evt)

{
  if(!m_pFileThread)                                // does a sub-thread exists?
     return (true);                                 // no: return

  IThread :: current(). waitFor (*m_pFileThread);   // yes: wait till it is ready

  delete m_pFileThread;                             // delete the object

  m_pFileThread = 0L;                               // reset the pointer

  return (true);
}

//********************************************************************************
// AStatusLineSample :: startSaveFile - starts saving a file, starts thread      *
//********************************************************************************
Boolean AStatusLineSample :: startSaveFile (void)

{
  if(m_pFileThread)                                                         // is a file thread active?
     {
      IMessageBox msgBox (this);                                            // yes
      if(!loadString (STR_FILE_SAVE))                                       // load the message title
         m_String = "Save a file";
      msgBox.setTitle (m_String);                                           // set the title
      if(!loadString (STR_ALREADY_PROCEEDING))                              // load the message
         m_String = "A file process is still in progress.";
      msgBox.show (m_String, IMessageBox::okButton   |                      // show it
                             IMessageBox::errorIcon  |
                             IMessageBox::applicationModal |
                             IMessageBox::moveable);
      return (false);                                            	          // abort
     }

  m_pFileThread = new IThread ();                                           // create a thread object

  m_pFileThread -> start(new IThreadMemberFn <AStatusLineSample> (*this, AStatusLineSample :: saveFile));   // start the write-thread

  return (true);                                                            // return
}

//********************************************************************************
// AStatusLineSample :: saveFile - thread-function to save a file                *
//********************************************************************************
void AStatusLineSample :: saveFile (void)

{
  APIRET ret;

  DosSetPriority( PRTYS_THREAD, PRTYC_REGULAR, 0L, 0UL );           // set the priority of the current thread

  Boolean bError = false;                                           // reset the error-flag

  if(!m_pEditFile)                                                  // is there a file to save?
     {                                                              // no
      goto endSaveFile;                                             // quit
     }

  if(m_pEditFile -> reOpen (m_pEditFile -> getFileSize (), FILE_NORMAL, FILE_OPEN)) // try to re-open the file
     {
      bError = true;
      goto endSaveFile;
     }

  m_startParams. ulProcessSize = m_pEditFile -> getFileSize();      // set the start parameters for the status line
  m_startParams. ulTickSize = AEditFile :: eBytesToRead;
  m_startParams .lResourceId = STR_FILE_WRITE;

  WinPostMsg (m_pStatusLine -> handle(), MYM_STATUS_START_PROCESS, MPFROMP (&m_startParams), 0); // post them

  ret = m_pEditFile -> write (m_pStatusLine -> handle ());          // write the file

  if(ret)                                                           // was it successful?
     {                                                              // no
      bError = true;                                                // set the flag
      goto endSaveFile;                                             // quit
     }

  if(m_pEditFile -> close())                                        // try to close the file
     {
      bError = true;
      goto endSaveFile;
     }

  endSaveFile:

  if(bError)                                                        // did an error occured?
     {                                                              // yes
      delete m_pEditFile;                                           // delete the file
      m_pEditFile = 0L;                                             // reset the pointer
     }

  WinPostMsg (m_pStatusLine -> handle(), MYM_STATUS_END_PROCESS, 0, 0); // let the status line know the process has finished

  WinPostMsg (handle(), MYM_SAVEFILE_READY, 0, 0);                  // let the frame window know the process has finished
}

//********************************************************************************
// AStatusLineSample :: saveFileReady - perhabs the wanted file is written       *
//********************************************************************************
Boolean AStatusLineSample :: saveFileReady (IEvent& evt)

{
  if(!m_pFileThread)                                // does a sub-thread exists?
     return (true);                                 // no: return

  IThread :: current(). waitFor (*m_pFileThread);   // yes: wait till it is ready

  delete m_pFileThread;                             // delete the object

  m_pFileThread = 0L;                               // reset the pointer

  return (true);
}

//********************************************************************************
// AStatusLineSample :: loadString - loads a string from the resource library    *
//********************************************************************************
Boolean AStatusLineSample :: loadString (unsigned long ulStringId)

{
  try
     {
      m_String = IApplication :: current (). userResourceLibrary (). loadString (ulStringId);  // try to load the string
     }

  catch (IException&)       // if not successful
         {
          return (false);   // return false
         }
  
  return (true);            // string loaded
}
