/*----------------------------------------------------------------------------
PROJECT config.exe
FILE    config.c
HISTORY
	Lee Woon Jeong  941020 Created
	Tilakraj Roy    941026 Modified
ABOUT
	Contains the Plug And Play configuration routines, to talk to the
	PnP card's EPROM.
----------------------------------------------------------------------------*/
#include <iostream.h>
#include <fstream.h>
#include <stdio.h> 
#include <conio.h>
#include "pnp.h"
#include "config.h"
#include "parser.h"


//----------------------------------------------------------------------------
// EnterSleepMode
// This function enables all the PnP cards on the ISA Bus by sending the 
// Initiation Key Sequence to the Universal ISA-PnP Address Port.
//----------------------------------------------------------------------------
void EnterSleepMode(void)
{ 
	unsigned short wIndex = 0;
	unsigned char bInitializationKey[32] = {
		0x6A,0xB5,0xDA,0xED,0xF6,0xFB,0x7D,0xBE,
		0xDF,0x6F,0x37,0x1B,0x0D,0x86,0xC3,0x61,
		0xB0,0x58,0x2C,0x16,0x8B,0x45,0xA2,0xD1,
		0xE8,0x74,0x3A,0x9D,0xCE,0xE7,0x73,0x39 };
						     
	// send two zeros to ADDRESS PORT to reset the LFSR              
	_outp ( ADDRESS_PORT, 0x00 );
	_outp ( ADDRESS_PORT, 0x00 );
		     
	// send initiation key sequence to put all PnP cards
	// into sleep state.
	for ( wIndex = 0 ; wIndex < 32 ; wIndex++ )
		_outp ( ADDRESS_PORT, bInitializationKey[ wIndex ] );
}        

//----------------------------------------------------------------------------
//	GoIntoIsolation
//	This function forcs all cards, except those who have a CSN to go into
//	isolation state.
//	ENTRY
//		bCSN = 0	: Put all cards into isolation state.
//		bCSN = XX	: Put the car whose CSN = XX into config state.
//----------------------------------------------------------------------------
void GoIntoIsolation(unsigned char bCSN)
{
	_outp ( ADDRESS_PORT, WAKE_CSN );
	_outp ( WRITE_DATA, bCSN );
	
}                                                                
//----------------------------------------------------------------------------
//	Reset
//	This function will reset the PnP cards CSN and registers to its
//	boot-up value
//----------------------------------------------------------------------------
void Reset(void)
{
       _outp(ADDRESS_PORT,CONFIG_CONTROL);
       _outp(WRITE_DATA, 0x07);
}                                

//----------------------------------------------------------------------------
//	SetReadPort
//	This function will set the readport of the plug and play system
//----------------------------------------------------------------------------
void SetReadPort(void)
{
	_outp(ADDRESS_PORT,SET_READ_PORT);
	_outp(WRITE_DATA,LOBYTE(readport));
}

//----------------------------------------------------------------------------
//	AssignCSN
//	This function will assign a CSN to the card that has won the isolation
//	process.
//----------------------------------------------------------------------------
void AssignCSN(void)
{
	_outp ( ADDRESS_PORT, CARD_SEL_NO);
	_outp ( WRITE_DATA, csn);
}            
									

//----------------------------------------------------------------------------
//Config
//Writes the settings supplied in the pConfig structure into the Plug & Play
//configuration registers.
//----------------------------------------------------------------------------
void Config(PCONFIG pConfig)
{   
	unsigned short wIndex;

	//Get to the particular logical device first
	_outp ( ADDRESS_PORT, LOGICAL_DEV_NO );
	_outp ( WRITE_DATA, pConfig->wLogicalDevice );

	//Start to configure the I/O
	for ( wIndex = 0 ; wIndex < pConfig->wNumIO ; wIndex++ )
	{
		_outp ( ADDRESS_PORT, ( IOBaseReg + ( wIndex * 2 ) ) );
		_outp ( WRITE_DATA, HIBYTE( pConfig->wIO[ wIndex ] ) );
		_outp ( ADDRESS_PORT, ( IOBaseReg + ( wIndex * 2 ) + 1 ) );
		_outp ( WRITE_DATA, LOBYTE( pConfig->wIO[ wIndex ] ) ); 
	}
	//What about DMA?
	for( wIndex = 0; wIndex < pConfig->wNumDMA ; wIndex++ )
	{
		_outp ( ADDRESS_PORT, ( DMABaseReg + wIndex ) );
		_outp ( WRITE_DATA, pConfig->wDMA[ wIndex ] );
	}

	//Any IRQ?
	for( wIndex = 0; wIndex < pConfig->wNumIRQ; wIndex ++ )
	{
		_outp ( ADDRESS_PORT, ( IRQBaseReg + ( wIndex * 2 ) ) );
		_outp ( WRITE_DATA, pConfig->wIRQ[ wIndex ] );
	}

	//Do you need memory?
	for( wIndex = 0 ; wIndex < pConfig->wNumMEM ; wIndex++ )
	{
		_outp ( ADDRESS_PORT, ( MEMBaseReg + ( wIndex * 8 ) ) );
		_outp ( WRITE_DATA, LOBYTE( HIWORD( pConfig->dwMEM[ wIndex ] ) ) );
		_outp ( ADDRESS_PORT, ( MEMBaseReg + 1 + ( wIndex * 8 ) ) );
		_outp ( WRITE_DATA, HIBYTE( LOWORD( pConfig->dwMEM[ wIndex ] ) ) );
		_outp ( ADDRESS_PORT, ( MEMBaseReg + 2 + ( wIndex * 8 ) ) );
		_outp ( WRITE_DATA, ( pConfig->bBits[ wIndex ] << 1 ) );

		if( ( _inp( readport ) & 0x01) == 0x01 )
		{
			_outp ( ADDRESS_PORT, ( MEMBaseReg + 3 + ( wIndex * 8 ) ) );
			_outp ( WRITE_DATA, LOBYTE( HIWORD( pConfig->dwMEM[ wIndex ] +
				pConfig->dwLength[ wIndex ] ) ) );
			_outp ( ADDRESS_PORT, ( MEMBaseReg + 4 + ( wIndex * 8 ) ) );
			_outp ( WRITE_DATA, HIBYTE(LOWORD(pConfig->dwMEM[ wIndex ] +
				pConfig->dwLength[ wIndex ] ) ) );
		}
		else
		{
			_outp ( ADDRESS_PORT, ( MEMBaseReg + 3 + ( wIndex * 8 ) ) );
			_outp ( WRITE_DATA,
				LOBYTE( HIWORD( pConfig->dwLength[ wIndex ] ) ) );
			_outp ( ADDRESS_PORT, ( MEMBaseReg + 4 + ( wIndex * 8 ) ) );
			_outp ( WRITE_DATA,
				HIBYTE( LOWORD( pConfig->dwLength[ wIndex ] ) ) );

		}

	}


	//Activate the card now 
	_outp ( ADDRESS_PORT, ACTIVATE );
	_outp ( WRITE_DATA, 0x01 );
}


//----------------------------------------------------------------------------
//			GLOBAL DATA
//----------------------------------------------------------------------------

// initialize the characteristics of the 4 resources.
PARSERTABLE ParserTable[ MAX_RESOURCE_TYPE ] =
{
	{ "IO", 8, TYPE_HEX, ID_IO },
	{ "DMA", 2, TYPE_DEC, ID_DMA },
	{ "IRQ", 2, TYPE_DEC, ID_IRQ },
	{ "MEM", 4, TYPE_HEX, ID_MEM }
};

unsigned short readport = 0x82;  //Set the readport to 0x20B first
unsigned char csn = 1;
unsigned long dwLogicalDevice = 0xff;

void main(int argc, char *argv[])
{
	unsigned short	wLine ;
	char szFileString[ 80 ];
	PCONFIG	pListHead = NULL;
	PCONFIG	pConfig = NULL;

	cerr << "\n\nPlug & Play H/W Configuration Utility (c) Tilakraj Roy & Lee Woon Jeong 1994\n";

	if ( argc < 2 )
	{
		cerr << "\nUsage : " << argv[0] << " ConfigFilename\n";
		return ;
	}

	fstream ConfigFile ( argv[1], ios::in );

	cerr << "\nProcessing File : " << argv[1] << endl;

	for ( wLine = 0 ; !ConfigFile.eof() ; wLine ++ )
	{
		ConfigFile.getline(szFileString, 80 );
		if ( ParseLine ( szFileString, &pConfig ) )
		{
			cerr << "PARSER : ERROR : Parsing Line # " << (wLine+1);
			return;
		}

		// special treatment for the first node.
		if ( ! pListHead ) // is NULL
			pListHead = pConfig;
	}

	ConfigFile.close();

	// do the actual plug and play stuff here.
	Reset();
	EnterSleepMode();
	GoIntoIsolation(0);
	SetReadPort();
	AssignCSN();

	// traverse the link list and call Config() for each logical device
	unsigned short wNum;
	for ( wNum = 0, pConfig = pListHead ; pConfig ;
		pConfig = pConfig->Next , wNum++)
	{
		cerr << "\nProgrammming Configuration [" << dec << wNum << "] \n";
		cerr << "Logical Device	[" << dec << pConfig->wLogicalDevice << "] \n";
		cerr << "Number of IO 	[" << dec << pConfig->wNumIO << "] \n";
		for ( unsigned short wIdx = 0 ; wIdx < pConfig->wNumIO ;
			wIdx++ )
		{
			cerr <<	"\t Resource # " << wIdx <<
				"\t[" << hex << pConfig->wIO[wIdx] << "] \n";
		}
		cerr << "Number of DMA 	[" << dec << pConfig->wNumDMA << "] \n";
		for ( wIdx = 0 ; wIdx < pConfig->wNumDMA ;
			wIdx++ )
		{
			cerr <<	"\t Resource # " << wIdx <<
				"\t[" << dec << pConfig->wDMA[wIdx] << "] \n";
		}
		cerr << "Number of IRQ 	[" << dec << pConfig->wNumIRQ << "] \n";
		for ( wIdx = 0 ; wIdx < pConfig->wNumIRQ ;
			wIdx++ )
		{
			cerr <<	"\t Resource # " << wIdx <<
				"\t[" << dec << pConfig->wIRQ[wIdx] << "] \n";
		}
		cerr << "\n";
		Config (pConfig);
	}
}          
