/*
 *  CONFIG.C
 *
 *  Written on 30-Jul-90 by jim nutt.  Changes on 10-Jul-94 by John Dennis.
 *  Released to the public domain.
 *
 *  Finds & reads configuration files, initializes everything.
 */

#define TEXTLEN    512
#define AREAS_BBS  0

#include "msged.h"
#include "main.h"
#include "config.h"
#include "nshow.h"
#include "bmg.h"
#include "strextra.h"
#include "memextra.h"

/* non-local prototypes */

void e_assignkey(unsigned int key, char *label);
void r_assignkey(unsigned int key, char *label);

/* local variables */

static GROUP *group = NULL;
static int areas_type = 0, num_groups = 0, check_area_files = 0;

static char *cfgverbs[] =
{
    "Name",
    "Address",
    "PrivateNet",
    "Alias",
    "OutFile",
    "Lastread",
    "TossLog",
    "Userlist",
    "SwapPath",
    "NodePath",
    "Nodelist",
    "HelpFile",
    "Xxxxx",
    "AreaFile",
    "UserOffset",
    "Quote",
    "Attribution",
    "ForwardLine",
    "RedirLine",
    "Followup",
    "ReplyArea",
    "ReplyExtra",
    "AlterFunc",
    "Switch",
    "QQuotes",
    "Sqllast",
    "Video",
    "Color",
    "Right",
    "QuoteRight",
    "TabSize",
    "CurStart",
    "CurEnd",
    "Fido",
    "Squish",
    "UUCP",
    "Domain",
    "Gate",
    "Origin",
    "ReadKey",
    "EditKey",
    "Function",
    "Include",
    "Speller",
    "VideoSeg",
    "MaxX",
    "MaxY",
    "Template",
    "UucpName",
    "Group",
    "Colour",
    "Editor",
    "RobotName",
    "QuickBBS",
    "Quick",
    NULL
};

static struct colorverb colortable[] =
{
    {"Black", 0x00},
    {"Blue", 0x01},
    {"Green", 0x02},
    {"Cyan", 0x03},
    {"Red", 0x04},
    {"Magenta", 0x05},
    {"Brown", 0x06},
    {"LGrey", 0x07},
    {"DGrey", 0x08},
    {"LBlue", 0x09},
    {"LGreen", 0x0A},
    {"LCyan", 0x0B},
    {"LRed", 0x0C},
    {"LMagenta", 0x0D},
    {"Yellow", 0x0E},
    {"White", 0x0F},
    {"Intense", 0x08},
    {"_Black", 0x00},
    {"_Blue", 0x10},
    {"_Green", 0x20},
    {"_Cyan", 0x30},
    {"_Red", 0x40},
    {"_Magenta", 0x50},
    {"_Brown", 0x60},
    {"_LGrey", 0x70},
    {"_DGrey", 0x80},
    {"_LBlue", 0x90},
    {"_LGreen", 0xA0},
    {"_LCyan", 0xB0},
    {"_LRed", 0xC0},
    {"_LMagenta", 0xD0},
    {"_Yellow", 0xE0},
    {"_White", 0xF0},
    {"_Blink", 0x80},
    {NULL, 0}
};

static char *cfgcolors[] =
{
    "MainNorm",
    "MainQuote",
    "MainKludge",
    "MainTempl",
    "MainInfo",
    "MainDivide",
    "MainHeader",
    "MainHdrTxt",
    "MainBlock",
    "MainEdit",
    "MainWarn",
    "MainNetInf",
    "MenuBorder",
    "MenuNorm",
    "MenuSelect",
    "HelpBorder",
    "HelpTitle",
    "HelpNorm",
    "HelpHilight",
    "InfoBorder",
    "InfoNorm",
    "InputBorder",
    "InputNorm",
    "InputEdit",
    "DlgBorder",
    "DlgNorm",
    "DlgChNorm",
    "DlgChSel",
    "DlgEnNorm",
    "DlgEnSel",
    "DlgButSel",
    "DlgButNorm",
    "DlgButShadow",
    "ListBdr",
    "ListTitle",
    "ListNorm",
    "ListInfo",
    "ListSelect",
    NULL
};

static char *cfgswitches[] =
{
    "HardQuote",
    "SaveCC",
    "RawCC",
    "SEEN-BYs",
    "ShowNotes",
    "Confirm",
    "ShowAddr",
    "MSGIDs",
    "OpusDate",
    "DateArvd",
    "ChopQuote",
    "QQuotes",
    "UseLastr",
    "ShowCR",
    "ShowEOL",
    "RealMsgN",
    "UseMouse",
    "EditCROnly",
    "UsePID",
    "SOTEOT",
    "ShowTime",
    "ImportFN",
    "DMore",
    "StatBar",
    "ShowSystem",
    "ExtFormat",
    NULL
};


/* Strips leading whitespaces from a string. */

char *striplwhite(char *str)
{
    while (*str && isspace((unsigned char)*str))
    {
        str++;
    }
    return str;
}


/* Strips trailing spaces from a string. */

char *striptwhite(char *str)
{
    char *p;

    if (str == NULL)
    {
        return str;
    }
    p = strend(str);
    while (p > str && *p && isspace((unsigned char)*p))
    {
        *p = '\0';
        p--;
    }
    return str;
}


/* Skips from current position to first blank found. */

char *skip_to_blank(char *str)
{
    if (str == NULL)
    {
        return str;
    }
    while (*str && !isspace((unsigned char)*str))
    {
        str++;
    }
    return str;
}


/* Kills a trailing slash from a path. */

void kill_trail_slash(char *str)
{
    char *p;

    p = strend(str);
    if (*p == '\\' || *p == '/')
    {
        *p = '\0';
    }
}


/*
 *  Returns the number of the passed verb in relation to the cfgverbs
 *  array, or -1 if the passed verb is not in the array.
 */

int GetVerbNum(char *verb)
{
    int i = 0;

    while (cfgverbs[i] != NULL)
    {
        if (stricmp(verb, cfgverbs[i]) == 0)
        {
            return i;
        }
        i++;
    }
    return -1;
}


/* Assigns the passed named area to the global colour array. */

void AssignColor(char *name, int color)
{
    int i = 0;

    while (cfgcolors[i] != NULL)
    {
        if (stricmp(name, cfgcolors[i]) == 0)
        {
            cm[i] = color;
            return;
        }
        i++;
    }
    printf("\nUnknown colour argument: '%s'\n", name);
}


/* Assigns the passed switch name to on or off. */

void AssignSwitch(char *swtch, int OnOff)
{
    int i = 0;

    while (cfgswitches[i] != NULL)
    {
        if (stricmp(swtch, cfgswitches[i]) == 0)
        {
            break;
        }
        i++;
    }

    if (cfgswitches[i] == NULL)
    {
        printf("\nUnknown switch: '%s'\n", swtch);
        return;
    }

    switch (i)
    {
    case 0:
        SW->hardquote = OnOff;
        break;
    case 1:
        SW->savecc = OnOff;
        break;
    case 2:
        SW->rawcc = OnOff;
        break;
    case 3:
        SW->seenbys = OnOff;
        break;
    case 4:
        SW->shownotes = OnOff;
        break;
    case 5:
        SW->confirmations = OnOff;
        break;
    case 6:
        SW->showaddr = OnOff;
        break;
    case 7:
        SW->msgids = OnOff;
        break;
    case 8:
        SW->opusdate = OnOff;
        break;
    case 9:
        SW->datearrived = OnOff;
        break;
    case 10:
        SW->chopquote = OnOff;
        break;
    case 11:
        SW->qquote = OnOff;
        break;
    case 12:
        SW->use_lastr = OnOff;
        break;
    case 13:
        SW->showcr = OnOff;
        break;
    case 14:
        SW->showeol = OnOff;
        break;
    case 15:
        SW->showrealmsgn = OnOff;
        break;
    case 16:
        SW->usemouse = OnOff;
        break;
    case 17:
        SW->editcronly = OnOff;
        break;
    case 18:
        SW->usepid = OnOff;
        break;
    case 19:
        SW->soteot = OnOff;
        break;
    case 20:
        SW->showtime = OnOff;
        break;
    case 21:
        SW->importfn = OnOff;
        break;
    case 22:
        SW->dmore = OnOff;
        break;
    case 23:
        SW->statbar = OnOff;
        break;
    case 24:
        SW->showsystem = OnOff;
        break;
    case 25:
        SW->extformat = OnOff;
        break;
    default:
        printf("\nUnknown switch: '%s'\n", swtch);
        break;
    }
}

/*
   ** Searches through the colortable for the matching
   ** colorname and returns the color, or -1 if not found.
 */

int GetColor(char *color)
{
    int i = 0;

    while (colortable[i].name != NULL)
    {
        if (stricmp(color, colortable[i].name) == 0)
            return colortable[i].color;
        i++;
    }
    printf("\nUnknown color: '%s'\n", color);
    return -1;
}

/*
   **
   ** Determines if the passed char is a token seperator.
   **
 */

int toksep(char ch)
{
    switch ((int)ch)
    {
    case ' ':
    case ',':
    case '\r':
    case '\t':
    case '\n':
        return TRUE;
    default:
        return FALSE;
    }
}

/*
   **
   ** Gets num tokens from a string, seperated by a default set of tokens.
   ** Handles stuff bewtween quotes as one token.
   **
 */

void parse_tokens(char *str, char *tokens[], int num)
{
    int i = 0;
    char *s = str;
    int done = 0;

    while (!done && i < num)
    {
        if (toksep(*s))
            while (*s && toksep(*s))
                s++;

        if (*s == ';')
        {
            /* We hit a comment... dinna process anymore. */

            tokens[i] = NULL;
            break;
        }
        if (*s == '\"')
        {
            tokens[i++] = ++s;

            while (*s && *s != '\"')
                s++;
        }
        else
        {
            tokens[i++] = s;
            while (*s && !toksep(*s))
                s++;
        }
        if (*s == '\0')
        {
            tokens[i] = NULL;
            done = TRUE;
        }
        else
            *s++ = '\0';
    }
    tokens[i] = NULL;
}

/*
   **
   ** Opens cfn first, and returns if successful, else it tries
   ** env.  env can be considered the default name.
   **
 */

static FILE *fileopen(char *env, char *cfn)
{
    FILE *fp;

    if (cfn)
    {
        if ((fp = fopen(cfn, "r")) != NULL)
            return fp;
    }
    fp = fopen(env, "r");

    return fp;
}

/*
   **
   ** Parses a macro.
   **
 */

static unsigned int *parse_macro(char *macro)
{
    unsigned int *m;
    unsigned int *t;
    int l;
    char tmp[5];

    t = xcalloc(strlen(macro) * 2, sizeof *t);
    m = t;

    if ((t == (void *)NULL) || (macro == NULL))
        return (NULL);

    while (*macro)
    {
        if (*macro == '^')
        {
            *t++ = (unsigned int)(*(macro + 1) == '^') ? '^' : (toupper(*(macro + 1)) - 64);
            macro += 2;
        }
        else if (*macro == '\\')
        {
            if (*(macro + 1) == '\\')
            {
                *t++ = (unsigned int)'\\';
                macro += 2;
            }
            else
            {
                memset(tmp, 0, sizeof tmp);
                strncpy(tmp, macro + 1, 4);
                *t++ = (unsigned int)strtol(tmp, NULL, 0) << 8;
                macro += 5;
            }
        }
        else
            *t++ = (unsigned int)*macro++;
    }
    *t = 0;
    l = (int)(t - m) + 1;

    t = xrealloc(m, l * sizeof(int));

    if (t)
        return (t);

    return (m);
}

/*
   **
   ** Sets the username and the template for an area.
   **
 */

static void SetAreaInfo(AREA * a)
{
    int i;

    for (i = 0; i < num_groups; i++)
    {
        if (bmg_find(a->description, group[i].search) != NULL)
            break;
    }

    if (i != num_groups)
    {
        a->username = group[i].username;
        a->template = group[i].template;
    }
}

/*
   **
   ** Adds a new area to the area-list.
   **
 */

static void AddArea(AREA * a)
{
    int i;
    char *d, *t, *p;

    if ((a->msgtype == QUICK) && (ST->quickbbs == NULL))
    {
        printf("\nFor QuickBBS areas, set QuickBBS path!\n");
        return;
    }
    for (i = 0; i < SW->areas; i++)
    {
        if ((arealist[i].path != NULL) && (((a->msgtype == QUICK) &&
             (arealist[i].msgtype == QUICK) && (a->board == a->board)) ||
                                           ((a->msgtype != QUICK) && (stricmp(a->path, arealist[i].path) == 0))))
            break;
    }

    if (i == SW->areas)
    {
        /* It's a new area, so we add it to the end of the list. */

        SW->areas++;
        SW->area = SW->areas - 1;
        arealist = xrealloc(arealist, SW->areas * sizeof(struct _area));

        memset(&CurArea, 0, sizeof(struct _area));

        CurArea = *a;

        if (a->addr.domain)
        {
            CurArea.addr.domain = xstrdup(a->addr.domain);
        }

        CurArea.description = xstrdup(a->description);
        CurArea.tag = xstrdup(a->tag);
        if (a->msgtype != QUICK)
        {
            CurArea.path = xstrdup(a->path);
            kill_trail_slash(CurArea.path);
        }
    }
    else
    {
        /* We redefine the area to the new defaults, with the * exception
           of the path, tag and desc. */

        release(arealist[i].addr.domain);
        p = arealist[i].path;
        t = arealist[i].tag;
        d = arealist[i].description;

        arealist[i] = *a;
        arealist[i].path = p;
        arealist[i].tag = t;
        arealist[i].description = d;
        if (a->addr.domain)
        {
            arealist[i].addr.domain = xstrdup(a->addr.domain);
        }
    }
    SetAreaInfo(&arealist[i]);

    /* Clean up after us... */

    release(a->tag);
    release(a->description);
    release(a->path);
    release(a->addr.domain);
}

/*
   **
   ** Checks a file in the areas.bbs format and gets all
   ** the area definitions. First attemps to open the file
   ** specified on the commandline, and if that fails, tries
   ** for the default name "areas.bbs".
   **
 */

static void checkareas(char *areafile)
{
    static AREA a;              /* current area */
    static char buffer[TEXTLEN];  /* line buffer */
    FILE *fp;                   /* file handle */
    char *s;                    /* pointer */
    int flag = 1;               /* found host name? */
    char *tokens[12];
#ifdef USE_MSGAPI
    int sq = 0;                 /* a squish base? */
#endif

    if ((fp = fileopen("areas.bbs", areafile)) == NULL)
        return;

    if (fp != NULL)
    {
        while (fgets(buffer, TEXTLEN, fp) != NULL)
        {
            if ((*buffer == '\r') || (*buffer == '\n'))
                continue;

            s = striplwhite(buffer);

            if ((*s == ';') || (*s == '-') || (*s == '#'))
                continue;

            if ((strchr(buffer, '!')) && flag)
            {
                char *p = strrchr(buffer, '!');
                if (p != NULL)
                {
                    *p = '\0';

                    release(ST->origin);
                    ST->origin = xstrdup(buffer);
                }
                flag = 0;
                continue;
            }
#ifdef USE_MSGAPI
            if (*s == '$')
            {
                sq = 1;
                s++;
            }
            else
            {
                sq = 0;
            }
#endif

            /* Grab the tokens on the line. */

            memset(tokens, 0, sizeof(tokens));
            parse_tokens(s, tokens, 4);

            if (tokens[0] == NULL || tokens[1] == NULL)
                continue;

            memset(&a, 0, sizeof(AREA));

            a.echomail = 1;
            a.msgtype = FIDO;
            a.path = xstrdup(tokens[0]);
            a.tag = xstrdup(tokens[1]);
            a.description = xstrdup(tokens[1]);
            a.addr.notfound = 1;

            kill_trail_slash(a.path);

            if (tokens[2] != NULL && strlen(tokens[2]) >= 7)
            {
                s = tokens[2];
                if (*s == '-' && toupper(*(s + 1)) == 'P')
                    a.addr = parsenode(s + 2);
            }
            if (a.addr.notfound)
            {
                a.addr = thisnode;
                if (thisnode.domain)
                {
                    a.addr.domain = xstrdup(thisnode.domain);
                }
            }
            strlwr(a.path);
            strupr(a.tag);

#ifdef USE_MSGAPI
            if (sq)
            {
                a.msgtype = SQUISH;
            }
#endif

            AddArea(&a);
        }
    }
    if (fp != NULL)
    {
        fclose(fp);
    }
}

/* check_squish - reads the areas in a squish configuration file */

static void check_squish(char *areafile)
{
    static AREA a;
    static char progress_indicators[4] =
    {'-', '\\', '|', '/'};
    static char buffer[TEXTLEN];
    char *tokens[20];
    FILE *fp;
    int i, line_num = 0;
    char *s;

    if ((fp = fileopen("squish.cfg", areafile)) == NULL)
        return;

    if (fp != NULL)
    {
        memset(buffer, 0, TEXTLEN);

        while (fgets(buffer, TEXTLEN, fp) != NULL)
        {
            line_num++;

            printf("%c\b", progress_indicators[line_num % 2]);

            if ((*buffer == '\r') || (*buffer == '\n'))
                continue;

            s = striplwhite(buffer);

            if ((*s == ';') || (*s == '-') || (*s == '#'))
                continue;

            memset(tokens, 0, sizeof(tokens));
            parse_tokens(s, tokens, 15);

            if (tokens[0] == NULL)
                continue;

            if (!stricmp(tokens[0], "EchoArea") ||
                !stricmp(tokens[0], "LocaLarea") ||
                !stricmp(tokens[0], "NetArea") ||
                !stricmp(tokens[0], "BadArea") ||
                !stricmp(tokens[0], "DupeArea"))
            {
                if (tokens[1] == NULL || tokens[2] == NULL)
                    continue;

                memset(&a, 0, sizeof(AREA));

                if (!stricmp(tokens[0], "EchoArea"))
                {
                    a.echomail = 1;
                }
                else if (!stricmp(tokens[0], "LocalArea"))
                {
                    a.local = 1;
                }
                else
                {
                    a.netmail = 1;
                    a.priv = 1;
                }

                a.msgtype = FIDO;
                a.addr.notfound = 1;
                a.tag = xstrdup(tokens[1]);
                a.path = xstrdup(tokens[2]);
                a.description = xstrdup(tokens[1]);

                strupr(a.tag);
                strlwr(a.path);

                i = 3;
                while ((s = tokens[i]) != NULL)
                {
                    if (*s == ';')
                    {
                        i = 0;
                        break;
                    }
#ifdef USE_MSGAPI
                    else if (*s == '-' && *(s + 1) && *(s + 1) == '$')
                    {
                        a.msgtype = SQUISH;
                    }
#endif
                    else if (*s == '-' && *(s + 1) && toupper(*(s + 1)) == 'P')
                    {
                        /* An address is specified... get it. */

                        release(a.addr.domain);
                        a.addr = parsenode(s + 2);
                    }
                    else if (*s == '-' && *(s + 1) && *(s + 1) == '0')
                    {
                        /* A passthru area... ignore it. */

                        release(a.tag);
                        release(a.description);
                        release(a.path);
                        release(a.addr.domain);
                        i = 0;
                        break;
                    }
                    i++;
                }

                if (i == 0)
                    continue;

                if (a.addr.notfound)
                {
                    a.addr = thisnode;
                    if (thisnode.domain)
                    {
                        a.addr.domain = xstrdup(thisnode.domain);
                    }
                }
                AddArea(&a);
            }
        }
    }
    if (fp != NULL)
        fclose(fp);
}

/*
   **
   ** Reads in an area definition in the msged.cfg format.
   **
 */

static void parsemail(char *keyword, char *value)
{
    char *tokens[20];
    char *s;
    static AREA a;

    memset(&a, 0, sizeof a);
    memset(tokens, 0, sizeof tokens);

    if (alias == NULL)
    {
        printf("\nError! Primary address must be defined.\n");
        exit(-1);
    }
    if (user_list[0].name == NULL)
    {
        printf("\nError! Name must be defined.\n");
        exit(-1);
    }
    switch (tolower(*keyword))  /* one letter is enough for now */
    {
    default:
    case 'f':
        a.msgtype = FIDO;
        break;
    case 'q':
        a.msgtype = QUICK;
        break;
#ifdef USE_MSGAPI
    case 's':
        a.msgtype = SQUISH;
        break;
#endif
    }

    parse_tokens(value, tokens, 5);

    if (tokens[2] == NULL)
        return;

    s = tokens[0];

    while (*s)
    {
        switch (tolower(*s))    /* one letter is enough */
        {
        case 'n':
            a.news = 1;
            break;
        case 'u':
            a.uucp = 1;
            break;
        case 'm':
            a.netmail = 1;
            break;
        case 'e':
            a.echomail = 1;
            break;
        case 'p':
            a.priv = 1;
            break;
        case 'h':
            a.hold = 1;
            break;
        case 'd':
            a.direct = 1;
            break;
        case 'c':
            a.crash = 1;
            break;
        case 'k':
            a.killsent = 1;
            break;
        case 'l':
            a.local = 1;
            break;
        default:
            break;
        }
        s++;
    }

    if (!a.echomail && !a.netmail && !a.uucp && !a.news)
        a.local = 1;

    a.description = xstrdup(tokens[1]);
    switch (a.msgtype)
    {
    case QUICK:
        a.board = atoi(tokens[2]);
        break;
    default:
        a.path = xstrdup(tokens[2]);
        break;
    }
    a.addr.notfound = 1;

    if (a.echomail)
    {
        if (tokens[3] != NULL)
        {
            a.tag = xstrdup(tokens[3]);
            if (tokens[4] != NULL)
                a.addr = parsenode(tokens[4]);
        }
        else
        {
            a.tag = xstrdup(a.description);
        }
    }
    else
    {
        a.tag = xstrdup(a.description);
        if (tokens[3] != NULL)
        {
            a.addr = parsenode(tokens[3]);
        }
    }

    if (a.addr.notfound)
    {
        a.addr = thisnode;
        if (thisnode.domain)
        {
            a.addr.domain = xstrdup(thisnode.domain);
        }
    }
    AddArea(&a);
}

/*
   **
   ** Parses an alias.
   **
   **
 */

static void parse_alias(char *value)
{
    static ALIAS a;
    char *tokens[6];
    char *s;

    memset(&a, 0, sizeof a);
    memset(tokens, 0, sizeof(tokens));

    parse_tokens(value, tokens, 5);

    if (tokens[2] == NULL)
        return;

    a.alias = xstrdup(tokens[0]);
    a.name = xstrdup(tokens[1]);

    if (!stricmp(tokens[2], "uucp"))
    {
        a.addr.internet = 1;
    }
    else
    {
        a.addr = parsenode(tokens[2]);
    }

    if ((s = tokens[3]) != NULL)
    {
        a.attrib.local = 1;
        a.attr = 1;
        while (*s)
        {
            switch (tolower(*s))
            {
            case 'p':
                a.attrib.priv = 1;
                break;
            case 'h':
                a.attrib.hold = 1;
                break;
            case 'd':
                a.attrib.direct = 1;
                break;
            case 'c':
                a.attrib.crash = 1;
                break;
            case 'k':
                a.attrib.killsent = 1;
                break;
            case 'n':
                a.attr = 0;
                break;
            default:
                break;
            }
            s++;
        }
        if (tokens[4] != NULL)
        {
            a.subj = xstrdup(tokens[4]);
        }
    }
    aliaslist = xrealloc(aliaslist, (++SW->otheraliases) * sizeof(struct _alias));
    aliaslist[SW->otheraliases - 1] = a;
}

/*
   **
   ** Handles the areafiles...
   **
 */

void do_areafile(char *value)
{
    char *tokens[3];

    memset(tokens, 0, sizeof(tokens));
    parse_tokens(value, tokens, 2);

    if (tokens[0] == NULL)
        return;

    areas_type = AREAS_BBS;

    if (toupper(*tokens[0]) == 'S')
        areas_type = SQUISH;

    if (tokens[1])
    {
        if (areas_type == AREAS_BBS)
            checkareas(tokens[1]);
        else
            check_squish(tokens[1]);
    }
    else
    {
        /* If no area-file was specified, then we want to check later. */

        check_area_files = 1;
    }
}

/*
   **
   ** Hanldes an entire config file.
   **
   **
 */

static void parseconfig(FILE * fp)
{
    static char progress_indicators[4] =
    {'-', '\\', '|', '/'};
    static char buffer[TEXTLEN];
    char *keyword;
    char *value = NULL;
    char *s = NULL;
    char *tokens[20];
    int i, line_num = 0;

    memset(buffer, 0, TEXTLEN);

    while (!feof(fp))
    {
        line_num++;

        printf("%c\b", progress_indicators[line_num % 2]);

        if (fgets(buffer, TEXTLEN, fp) == NULL)
            return;

        keyword = strtok(buffer, " \t\n\r");

        if (keyword)
            strlwr(keyword);
        else
            continue;

        if ((*keyword) == ';')
            continue;

        value = strtok(NULL, ";\n\r");
        value = striptwhite(value);

        /* clear the pointers in the array */

        memset(tokens, 0, sizeof(tokens));

        while (value && *value && isspace((unsigned char)*value))
            if ((*value == '\n') || (*value == ';'))
                break;
            else
                value++;

        switch (GetVerbNum(keyword))
        {
        case -1:
            printf("\nUnknown configuration keyword: '%s'\n", keyword);
            break;

        case 0:                /* name */
            parse_tokens(value, tokens, 3);
            if (tokens[0] != NULL)
            {
                for (i = 0; i < 11; i++)
                {
                    if (user_list[i].name == NULL)
                        break;
                }
                if (i < 11)
                {
                    user_list[i].name = xstrdup(tokens[0]);

                    if (tokens[1] != NULL)
                    {
                        user_list[i].lastread = xstrdup(tokens[1]);
                    }

                    if (tokens[2] != NULL)
                    {
                        user_list[i].offset = atol(tokens[2]);
                    }

                    if (i == 0)
                    {
                        release(ST->username);
                        ST->username = xstrdup(user_list[i].name);

                        if (user_list[i].lastread)
                        {
                            release(ST->lastread);
                            ST->lastread = xstrdup(user_list[i].lastread);
                        }
                        SW->useroffset = user_list[i].offset;
                    }
                }
            }
            break;

        case 1:                /* address */
            alias = xrealloc(alias, (++SW->aliascount) * sizeof(ADDRESS));
            alias[SW->aliascount - 1] = parsenode(value);
            break;

        case 2:                /* privatenet */
            SW->pointnet = (int)strtol(value, NULL, 0);
            break;

        case 3:                /* alias */
            parse_alias(value);
            break;

        case 4:                /* outfile */
            release(ST->outfile);
            ST->outfile = xstrdup(value);
            break;

        case 5:                /* lastread */
            if (user_list[0].name == NULL)
            {
                release(ST->lastread);
                ST->lastread = xstrdup(value);
                user_list[0].lastread = xstrdup(value);
            }
            break;

        case 6:                /* tosslog */
            release(ST->confmail);
            ST->confmail = xstrdup(value);
            break;

        case 7:                /* userlist */
            ST->fidolist = xstrdup(strtok(value, ",\n"));
            ST->userlist = strtok(NULL, ",\n");
            if (ST->userlist != NULL)
            {
                ST->userlist = xstrdup(ST->userlist);
            }
            break;
        case 8:                /* swappath */
            release(ST->swap_path);
            kill_trail_slash(value);
            ST->swap_path = xstrdup(value);
            break;
        case 9:                /* nodepath */
            release(ST->nodepath);
            ST->nodepath = xstrdup(value);
            break;

        case 10:               /* nodelist */
            parse_tokens(value, tokens, 3);
            if (tokens[2] != NULL)
            {
                node_lists = xrealloc(node_lists, (++SW->nodelists) * sizeof(D_LIST));
                node_lists[SW->nodelists - 1].name = xstrdup(tokens[0]);
                node_lists[SW->nodelists - 1].base_name = xstrdup(tokens[1]);
                node_lists[SW->nodelists - 1].sysop = xstrdup(tokens[2]);

                if (SW->nodelists == 1)
                {
                    ST->nodebase = node_lists[SW->nodelists - 1].base_name;
                    ST->sysop = node_lists[SW->nodelists - 1].sysop;
                }
            }
            break;

        case 11:               /* helpfile */
            release(ST->helpfile);
            ST->helpfile = xstrdup(value);
            break;

        case 12:               /* xxxxx */
            break;

        case 13:               /* areafile */
            do_areafile(value);
            break;

        case 14:               /* useroffset */
            SW->useroffset = atoi(value);
            break;

        case 15:               /* quote */
            release(ST->quotestr);
            striptwhite(value);
            ST->quotestr = xstrdup(value);
            s = ST->quotestr;
            while (*s)
            {
                *s = (*s == (char)'_') ? (char)' ' : *s;
                s++;
            }
            break;

        case 22:               /* alterfunc? */
            parse_tokens(value, tokens, 2);

            if (tokens[0] && tokens[1])
            {
                int var = 0;
                char *c;

                c = striplwhite(tokens[0]);
                s = striplwhite(tokens[1]);

                if ((*s != '\0') && (*c != '\0'))
                {
                    while (*c && !isspace((unsigned char)*c))
                    {
                        switch (toupper(*c))
                        {
                        case 'Q':
                            if (!(var & MT_REP) && !(var & MT_NEW))
                                var |= MT_QUO;
                            break;

                        case 'R':
                            if (!(var & MT_QUO) && !(var & MT_NEW))
                                var |= MT_REP;
                            break;

                        case 'N':
                            if (!(var & MT_QUO) && !(var & MT_REP))
                                var |= MT_NEW;
                            break;

                        case 'A':
                            var |= MT_ARC;
                            break;
                        case 'F':
                            var |= MT_FOL;
                            break;
                        default:
                            break;
                        }
                        c++;
                    }
                    if (!stricmp("replyquote", s))
                        SW->rquote = var;
                    else if (!stricmp("replyotherarea", s))
                        SW->rotharea = var;
                    else if (!stricmp("replyfollow", s))
                        SW->rfollow = var;
                    else if (!stricmp("replyextra", s))
                        SW->rextra = var;
                }
            }
            break;

        case 23:               /* switch */
            parse_tokens(value, tokens, 2);
            if (tokens[1] != NULL)
                AssignSwitch(value, (stricmp(tokens[1], "on") == 0) ? 1 : 0);
            break;

        case 24:               /* qqoutes */
            SW->qquote = 1;
            break;

        case 25:               /* sqllast */
            SW->use_lastr = 1;
            break;

        case 26:               /* video */
            /* do nothing */
            break;

        case 27:               /* color */
        case 50:               /* colour */
            parse_tokens(value, tokens, 3);
            if (tokens[2] != NULL)
            {
                int fcol, bcol;

                fcol = GetColor(tokens[1]);
                bcol = GetColor(tokens[2]);

                if (fcol == -1 || bcol == -1)
                    continue;

                AssignColor(tokens[0], fcol | bcol);
            }
            break;

        case 28:               /* right */
            SW->rm = (int)strtol(value, NULL, 0);
            break;

        case 29:               /* quoteright */
            SW->qm = (int)strtol(value, NULL, 0);
            break;

        case 30:               /* tabsize */
            SW->tabsize = (int)strtol(value, NULL, 0);
            break;

        case 31:               /* curstart */
            cur_start = atoi(value);
            break;

        case 32:               /* curend */
            cur_end = atoi(value);
            break;

        case 33:               /* fido */
        case 34:               /* squish */
            parsemail(keyword, value);
            break;

        case 35:               /* uucp */
            uucp_gate = parsenode(value);
            break;

        case 36:               /* domain */
            domain_list = xrealloc(domain_list, (++SW->domains) * sizeof(ADDRESS));
            domain_list[SW->domains - 1] = parsenode(value);
            break;

        case 37:               /* gate */
            if (value)
                strlwr(value);
            if (strncmp(value, "zones", 5) == 0)
                SW->gate = GZONES;
            else if (strncmp(value, "domains", 7) == 0)
                SW->gate = GDOMAINS;
            else if (strncmp(value, "both", 4) == 0)
                SW->gate = BOTH;
            else if (strncmp(value, "none", 4) == 0)
                SW->gate = 0;
            break;

        case 38:               /* origin */
            if (value != NULL)
            {
                release(ST->origin);
                ST->origin = xstrdup(value);
            }
            break;

        case 39:               /* readkey */
        case 40:               /* editkey */
            if (value)
            {
                strlwr(value);
                i = (int)strtol(value, &value, 0);
                value = striplwhite(value);
            }
            if (*keyword == 'e')
                e_assignkey(i, value);
            else
                r_assignkey(i, value);
            break;

        case 41:               /* function */
            i = (int)strtol(value, &s, 0);
            s = striplwhite(s);

            if ((i >= 0) && (i <= 40))
                macros[i] = parse_macro(s);
            break;

        case 42:               /* include */
            {
                FILE *ifp;
                if ((ifp = fopen(value, "r")) != NULL)
                {
                    parseconfig(ifp);
                    fclose(ifp);
                }
            }
            break;

        case 45:               /* maxx */
            maxx = (int)strtol(value, NULL, 0);
            break;

        case 46:               /* maxy */
            maxy = (int)strtol(value, NULL, 0);
            break;

        case 47:               /* template */
            templates = xrealloc(templates, (++SW->numtemplates) * sizeof(char *));

            templates[SW->numtemplates - 1] = xstrdup(value);

            if (SW->numtemplates == 1)
            {
                ST->template = xstrdup(templates[SW->numtemplates - 1]);
            }
            break;

        case 48:               /* uucpname */
            release(ST->uucpgate);
            ST->uucpgate = xstrdup(value);
            break;

        case 49:               /* group */
            group = xrealloc(group, (++num_groups) * sizeof(GROUP));

            parse_tokens(value, tokens, 3);

            if (tokens[2] != NULL)
            {
                group[num_groups - 1].search = xstrdup(strupr(tokens[0]));
                group[num_groups - 1].username = atoi(tokens[1]);
                /* mod PE 1995-04-12 to protect against wild user */
                if ((group[num_groups - 1].username >= MAXUSERS)
                    || (group[num_groups - 1].username >= MAXUSERS)
                    || (user_list[group[num_groups - 1].username].name
                        == NULL))
                {
                    group[num_groups - 1].username = 0;
                }
                group[num_groups - 1].template = atoi(tokens[2]);
                /* mod PE 1995-04-08 to protect against wild template */
                if (group[num_groups - 1].template >= SW->numtemplates)
                {
                    group[num_groups - 1].template = 0;
                }
            }
            break;

        case 51:               /* editor */
            release(ST->editorName);
            ST->editorName = xstrdup(value);
            break;

        case 52:               /* robotname */
            parse_tokens(value, tokens, 3);
            if (tokens[0] != NULL)
            {
                for (i = 0; i < 11; i++)
                {
                    if (user_list[i].robotname == NULL)
                        break;
                }
                if (i < 11)
                {
                    user_list[i].robotname = xstrdup(tokens[0]);
                }
            }
            break;

        case 53:               /* path to quickbbs file */
            release(ST->quickbbs);
            ST->quickbbs = xstrdup(value);
            break;

        case 54:               /* quickbbs area */
            parsemail(keyword, value);
            break;

        default:
            printf("\nUnknown configuration keyword: '%s'\n", keyword);
            break;
        }
        memset(buffer, 0, TEXTLEN);
    }
}

void show_debuginfo(int macro_count)
{
    char *szYes = "Yes";
    char *szNo = "No";

    printf(
              "\n"
              "%-30s; %s\n"
              "-------------------------------------------------------------------------------\n"
              "\n",
              PROG " " VERSION CLOSED "; Mail Reader",
              "Compiled on " __DATE__ " at " __TIME__
        );
    printf("Screen size       : %d columns, %d rows\n", maxx, maxy);
    printf("User              : \"%s\" (%s)\n", ST->username, show_address(&thisnode));
    printf("Origin            : \"%s\"\n", ST->origin);
    printf("Macros            : %d macros defined\n", macro_count);
    printf("Home directory    : %s\n", ST->home);
    printf("Quote string      : \"%s\"\n", ST->quotestr);
    printf("Export file       : %s (default)\n", ST->outfile);
    printf("Config file       : %s\n", ST->cfgfile);
    printf("Echotoss log      : %s\n", ST->confmail);
    printf("Template file     : %s\n", ST->template);
#ifndef __OS2__
    printf("Swap path         : %s\n", ST->swap_path);
#endif
    printf("Help file         : %s\n", ST->helpfile);
    printf("Command processor : %s\n", ST->comspec);
    printf("External editor   : %s\n", ST->editorName);
    printf("QuickBBS path     : %s\n", ST->quickbbs);
    printf("\n");
    printf("Areas                     : %d areas found\n", SW->areas);
    printf("Generate MSGIDs           : %s\n", SW->msgids ? szYes : szNo);
    printf("Generate Opus time stamps : %s\n", SW->opusdate ? szYes : szNo);
    printf("Show hidden lines         : %s\n", SW->shownotes ? szYes : szNo);
    printf("Show SEEN-BY lines        : %s\n", SW->seenbys ? szYes : szNo);
    printf("Confirm deletes, aborts   : %s\n", SW->confirmations ? szYes : szNo);
    printf("Show date arrived         : %s\n", SW->datearrived ? szYes : szNo);
    printf("Show current address      : %s\n", SW->showaddr ? szYes : szNo);
    printf("Use lastread/current      : %s\n", SW->use_lastr ? szYes : szNo);
    printf("Quote quotes              : %s\n", SW->qquote ? szYes : szNo);
    printf("Save original carbon copy : %s\n", SW->savecc ? szYes : szNo);
    printf("Save raw carbon copy      : %s\n", SW->rawcc ? szYes : szNo);
    printf("Chop end of quoted msgs   : %s\n", SW->chopquote ? szYes : szNo);
    printf("Use \"HardQuotes\" feature  : %s\n", SW->hardquote ? szYes : szNo);
    printf("Show paragraph markers    : %s\n", SW->showcr ? szYes : szNo);
    printf("Show end-of-line markers  : %s\n", SW->showeol ? szYes : szNo);
    printf("Show real message numbers : %s\n", SW->showrealmsgn ? szYes : szNo);
    printf("Enable mouse support      : %s\n", SW->usemouse ? szYes : szNo);
    printf("Expand tab character      : %s\n", SW->tabexpand ? szYes : szNo);
    printf("Only show CRs in editor   : %s\n", SW->editcronly ? szYes : szNo);
    printf("Generate PID              : %s\n", SW->usepid ? szYes : szNo);
    printf("Generate SOT/EOT          : %s\n", SW->soteot ? szYes : szNo);
    printf("Show system time          : %s\n", SW->showtime ? szYes : szNo);
    printf("Show file hdrs on import  : %s\n", SW->importfn ? szYes : szNo);
    printf("Msg num at top of screen  : %s\n", SW->dmore ? szYes : szNo);
    printf("Show status bar           : %s\n", SW->statbar ? szYes : szNo);
    printf("Perform address lookup    : %s\n", SW->showsystem ? szYes : szNo);
    printf("Format external messages  : %s\n", SW->extformat ? szYes : szNo);
    MouseOFF();
    exit(0);
}

/*
   **
   ** Handles the entire configuration process.
   **
   **
 */

void opening(char *cfgfile, char *areafile)
{
    WND *hCurr, *hWnd;
    FILE *fp;
    int count = 0, i;
    static char cfnname[] = "msged.cfg";
    char tmp[PATHLEN];

    InitVars();

    printf(PROG " " VERSION CLOSED " ... ");

    if ((fp = fileopen(cfnname, cfgfile)) == NULL)
    {
        if (cfgfile == NULL)
            cfgfile = cfnname;

        printf("\nCannot open " PROG " configuration file '%s'.\n", cfgfile);
        exit(-1);
    }

    if (fp)
    {
        parseconfig(fp);
        fclose(fp);
    }
    for (i = 0; i < 40; i++)
        count += (macros[i] != (void *)NULL) ? 1 : 0;

    if (check_area_files)
    {
        if (areas_type == AREAS_BBS)
            checkareas(areafile);
        else
            check_squish(areafile);
    }
    if (!arealist)
    {
        printf("\nError! At least one message area must be defined.\n");
        exit(-1);
    }
    printf(" \n");

    InitScreen();

    mygetcwd(tmp, PATHLEN);
    ST->home = xstrdup(tmp);

    if (cmd_dbginfo)
    {
        show_debuginfo(count);
    }

    cursor(0);

    SW->rm = (SW->rm > maxx) ? maxx : SW->rm;
    SW->qm = (SW->qm > SW->rm) ? SW->rm - strlen(ST->quotestr) : SW->qm;

    hCurr = Wtop();
    hWnd = WPopUp(42, 6, SBDR | SHADOW, cm[IN_BTXT], cm[IN_NTXT]);

    WPutsCen(0, cm[IN_NTXT], PROG " Mail Reader");
    WPutsCen(2, cm[IN_BTXT], "Version " VERSION CLOSED);
    WPutsCen(4, cm[IN_NTXT], "Press a key to continue");

    GetKey();

    WClose(hWnd);
    WCurr(hCurr);
    TTScolor(cm[CM_NTXT]);
    TTClear(hMnScr->x1, hMnScr->y1, hMnScr->x2, hMnScr->y2);
}
