/* Server for the Midnight Commander Virtual File System.
   Routines for the tcp connection, includes the primitive rpc routines.
   
   Copyright (C) 1995 Miguel de Icaza
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <config.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <signal.h>
#include <pwd.h>
#include <sys/types.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <malloc.h>

#ifdef HAVE_PMAP_SET
#include <rpc/rpc.h>
#include <rpc/pmap_prot.h>
#ifdef HAVE_RPC_PMAP_CLNT_H
#include <rpc/pmap_clnt.h>
#endif
#endif

#ifdef USE_TERMNET
#include <termnet.h>
#endif

#include <signal.h>
#include <errno.h>
#include "tcputil.h"
#include "../src/dialog.h"	/* for message () */
#include "../src/mem.h"		/* for bcopy */
#include "../src/util.h"	/* for unix_error_string */
#include "../src/mad.h"
#include "mcfs.h"		/* for mcserver_port definition */

#define CHECK_SIG_PIPE(sock) if (got_sigpipe) \
     { tcp_invalidate_socket (sock); return got_sigpipe = 0; }

extern void tcp_invalidate_socket (int);

int got_sigpipe;

/* Reads a block on dest for len bytes from sock */
/* Returns a boolean indicating the success status */
int socket_read_block (int sock, char *dest, int len)
{
    int nread, n;

    for (nread = 0; nread < len;){
	n = read (sock, dest+nread, len-nread);
	if (n <= 0){
	    tcp_invalidate_socket (sock);
	    return 0;
	}
	nread += n;
    }
    return 1;
}

int socket_write_block (int sock, char *buffer, int len)
{
    int left, status;

    for (left = len; left > 0;){
	status = write (sock, buffer, left);
	CHECK_SIG_PIPE (sock);
	if (status < 0)
	    return 0;
	left -= status;
	buffer += status;
    }
    return 1;
}

int send_string (int sock, char *string)
{
    return socket_write_block (sock, string, strlen (string));
}

int rpc_send (int sock, ...)
{
    long int tmp, len, cmd;
    char *text;
    va_list ap;

    va_start (ap, sock);

    for (;;){
	cmd = va_arg (ap, int);
	switch (cmd){
	case RPC_END:
	    va_end (ap);
	    return 1;

	case RPC_INT:
	    tmp = htonl (va_arg (ap, int));
	    write (sock, &tmp, sizeof (tmp));
	    CHECK_SIG_PIPE (sock);
	    break;

	case RPC_STRING:
	    text = va_arg (ap, char *);
	    len = strlen (text);
	    tmp = htonl (len);
	    write (sock, &tmp, sizeof (tmp));
	    CHECK_SIG_PIPE (sock);
	    write (sock, text, len);
	    CHECK_SIG_PIPE (sock);
	    break;	    

	case RPC_BLOCK:
	    len = va_arg (ap, int);
	    text = va_arg (ap, char *);
	    tmp = htonl (len);
	    write (sock, text, len);
	    CHECK_SIG_PIPE (sock);
	    break;

	default:
	    fprintf (stderr, "Unknown rpc message\n");
	    abort ();
	}
    }
}

typedef struct sock_callback_t {
    int  sock;
    void (*cback)(int);
    struct sock_callback_t *link;
} sock_callback_t;

sock_callback_t *sock_callbacks = 0;

static void check_hooks (int sock)
{
    sock_callback_t *callback, *prev;

    for (prev=callback = sock_callbacks; callback; callback = callback->link){
	if (callback->sock != sock){
	    prev = callback;
	    continue;
	}
	callback->sock = -1;
	(callback->cback)(sock);
	if (callback == sock_callbacks){
	    sock_callbacks = callback->link;
	} else {
	    prev->link = callback->link;
	}
	free (callback);
	return;
    }
}

int rpc_get (int sock, ...)
{
    long int tmp, len;
    char *text, **str_dest;
    int  *dest, cmd;
    va_list ap;

    va_start (ap, sock);

    check_hooks (sock);

    for (;;){
	cmd = va_arg (ap, int);
	switch (cmd){
	case RPC_END:
	    va_end (ap);
	    return 1;

	case RPC_INT:
	    if (socket_read_block (sock, (char *) &tmp, sizeof (tmp)) == 0)
		return 0;
	    dest = va_arg (ap, int *);
	    *dest = ntohl (tmp);
	    break;

	    /* returns an allocated string */
	case RPC_STRING:
	    if (socket_read_block (sock, (char *)&tmp, sizeof (tmp)) == 0)
		return 0;
	    len = ntohl (tmp);
	    text = malloc (len+1);
	    if (socket_read_block (sock, text, len) == 0)
		return 0;
	    str_dest = va_arg (ap, char **);
	    *str_dest = text;
	    text [len] = 0;
	    break;	    

	case RPC_BLOCK:
	    len = va_arg (ap, int);
	    text = va_arg (ap, char *);
	    if (socket_read_block (sock, text, len) == 0)
		return 0;
	    break;

	default:
	    fprintf (stderr, "Unknown rpc message\n");
	    abort ();
	}
    }
}

void rpc_add_get_callback (int sock, void (*cback)(int))
{
    sock_callback_t *new;

    new = malloc (sizeof (sock_callback_t));
    new->cback = cback;
    new->sock = sock;
    new->link = sock_callbacks;
    sock_callbacks = new;
}

static void sig_pipe (void)
{
    got_sigpipe = 1;
    signal (SIGPIPE, sig_pipe);	/* Reset the signal handler */
}

void tcp_init (void)
{
    got_sigpipe = 0;
    signal (SIGPIPE, sig_pipe);
}

int get_remote_port (struct sockaddr_in *sin)
{
    int port;
    
#ifdef HAVE_PMAP_GETPORT
    port = pmap_getport (sin, RPC_PROGNUM, RPC_PROGVER, IPPROTO_TCP);
    if (port == 0)
	return mcserver_port;
    else
	return port;
#else
    return mcserver_port;
#endif
}

int open_tcp_link  (char *host, int *port, char *caller)
{
    struct   sockaddr_in server_address;
    unsigned long inaddr;
    struct   hostent *hp;
    int      my_socket;

    if (!*host)
	return 0;
    
    bzero ((char *) &server_address, sizeof (server_address));
    server_address.sin_family = AF_INET;
    
    /*  Try to use the dotted decimal number */
    if ((inaddr = inet_addr (host)) != -1)
	bcopy ((char *) &inaddr, (char *) &server_address.sin_addr,
	       sizeof (inaddr));
    else {
	if ((hp = gethostbyname (host)) == NULL){
	    message (1, caller, " Can't locate hostname: %s ", host);
	    return 0;
	}
	bcopy ((char *) hp->h_addr, (char *) &server_address.sin_addr,
	       hp->h_length);
    }

    /* Try to contact a remote portmapper to obtain the listening port */
    if (*port == 0){
	*port = get_remote_port (&server_address);
	if (*port < 1)
	    return 0;
    }
    server_address.sin_port = htons (*port);
    
    if ((my_socket = socket (AF_INET, SOCK_STREAM, 0)) < 0){
	message (1, caller, " Can't create socket: %s ",
		 unix_error_string(errno));
	return 0;
    }
    if (connect (my_socket, (struct sockaddr *) &server_address,
	     sizeof (server_address)) < 0){
	message (1, caller, " Can't connect to server: %s ",
		 unix_error_string (errno));
	close (my_socket);
	return 0;
    }
    return my_socket;
}

/* Extract the hostname and username from the path */
/* path is in the form: [user@]hostname:port/remote-dir, e.g.:
 *
 * ftp://sunsite.unc.edu/pub/linux
 * ftp://miguel@sphinx.nuclecu.unam.mx/c/nc
 * ftp://tsx-11.mit.edu:8192/
 * ftp://joe@foo.edu:11321/private
 *
 * If the user is empty, e.g. ftp://@roxanne/private, then your login name
 * is supplied.
 * */

char *get_host_and_username (char *path, char **host, char **user, int *port,
			     int default_port, int default_is_anon,
			     char **pass)
{
    struct passwd *passwd_info;
    char *p, *q;

    *pass = NULL;
    *port = default_port;
    for (p = path; ((*p != '/') && (*p != ':')) && *p; p++);

    q = strchr (path, '@');
    if (q != NULL && q < p) {
        if (q == path)
            *user = NULL;
        else {
            *user = (char *) xmalloc (q  - path + 1, "get_host_and_username");
            strncpy (*user, path, q - path);
            (*user) [q - path] = 0;
        }
        q++;
        *host = (char *) xmalloc (p - q + 1, "get_host_and_username");
        strncpy (*host, q, p - q);
        (*host) [p-q] = 0;
    } else {
        *host = (char *) xmalloc (p - path + 1, "get_host_and_username");
        strncpy (*host, path, p - path);
        (*host) [p-path] = 0;
        *user = NULL;
#ifdef WE_NEED_TO_ACTIVATE_THE_CODE_IN_FTPFS
        if (use_netrc)
            if (lookup_netrc (*host, user, pass) < 0) {
                if (*user) { free (*user); *user = NULL; }
                if (*pass) { free (*pass); *pass = NULL; }
            }
#endif
        if (*user == NULL && default_is_anon)
	    *user = strdup ("anonymous");
    }

    /* If we got a port spec ... */
    if (*p == ':'){
	
	q = ++p;

	for (;*q != '/' && *q; q++)
	    ;
	
	if (!*q && q == p)
	    return 0;

	*port = atoi (p);
	
	if (*port <= 0 || *port >= 65536)
	    *port = 21;
	p = q;
    }

    if (!*user){
	if ((passwd_info = getpwuid (geteuid ())) == NULL)
	    *user = strdup ("anonymous");
	else {
	    *user = strdup (passwd_info->pw_name);
	}
	endpwent ();
    }
    if (p && *p)
	return strdup (p);
     else
	return strdup ("/");
}

