/*
*  gcal_rcu.c:  Central function, manages RC-file handling and displays the list
*
*
*  Copyright (C) 1994, 1995 Thomas Esken
*
*  This software doesn't claim completeness, correctness or usability.
*  On principle I will not be liable for any damages or losses (implicit
*  or explicit), which result from using or handling my software.
*  If you use this software, you agree without any exception to this
*  agreement, which binds you LEGALLY !!
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the `GNU General Public License' as published by
*  the `Free Software Foundation'; either version 2, or (at your option)
*  any later version.
*
*  You should have received a copy of the `GNU General Public License'
*  along with this program; if not, write to the:
*    Free Software Foundation
*    59 Temple Place, Suite 330
*    Boston, MA 02111-1307  USA
*/



/*
*  Include definition header file to see whether USE_RC is defined there.
*    compile this module only if USE_RC is defined, otherwise skip it.
*/
#include "gcal_tai.h"



#if USE_RC
#  ifdef RCSID
static char rcsid[]="$Id: gcal_rcu.c 0.39 1995/12/20 00:03:09 tom Exp $";
#  endif



/*
*  Include header files
*/
#  if HAVE_ASSERT_H
#    include <assert.h>
#  endif
#  if HAVE_CTYPE_H
#    include <ctype.h>
#  endif
#  include "gcal.h"



/*
*  Function prototypes
*/
#  if __cplusplus
extern "C"
{
#  endif
/*
************************************************** Defined in `gcal_fil.c'
*/
IMPORT FILE *
file_open __P_((      char       *filename,
                const Fmode_enum  mode));
IMPORT char *
file_read_line __P_((      FILE       *fp,
                           char       *line_buffer,
                           int        *in_pool,
                           char       *pool,
                           char       *ptr_pool,
                     const char       *filename,
                           int        *line,
                     const Fmode_enum  mode,
                           Bool       *is_include,
                           Bool       *is_dvar));
/*
************************************************** Defined in `gcal_hdy.c'
*/
IMPORT void
print_all_holidays __P_((      Bool init_data,
                         const Bool detect));
/*
************************************************** Defined in `gcal_rc.c'
*/
IMPORT void
rc_clean_flags __P_((void));
IMPORT char *
rc_get_date __P_((      char *ptr_char,
                        Bool *is_weekday_mode,
                        int  *d,
                        int  *m,
                        int  *y,
                        int  *n,
                        int  *len,
                        char *hc,
                        int  *hn,
                        int  *hwd,
                  const char *filename,
                  const int   line,
                  const char *line_buffer,
                  const Bool  on_error_exit));
/*
************************************************** Defined in `gcal_rcc.c'
*/
IMPORT void
rc_check __P_((      char *line_buffer,
               const char *filename,
                     int  *line,
                     int  *rc_elems,
                     int   day,
                     int   ed,
               const int   wd));
/*
************************************************** Defined in `gcal_tty.c'
*/
IMPORT void
print_text __P_((      FILE       *fp,
                       char       *txt_line,
                 const Dmode_enum  mode));
/*
************************************************** Defined in `gcal_utl.c'
*/
IMPORT VOID_PTR
my_malloc __P_((const int   amount,
                const int   exit_status,
                const char *module_name,
                const int   module_line,
                const char *var_name,
                const int   var_contents));
IMPORT void
my_error __P_((const int   exit_status,
               const char *module_name,
               const int   module_line,
               const char *var_name,
               const int   var_contents));
IMPORT int
asc_sort __P_((const char **a,
               const char **b));
IMPORT int
des_sort __P_((const char **a,
               const char **b));
IMPORT Bool
is_presorted __P_((char **table,
                   int   elems));
IMPORT void
resort __P_((      char **table,
             const int    elems));
#if !USE_GER
IMPORT const char *
day_suffix __P_((int day));
IMPORT const char *
short3_day_name __P_((const int day));
#endif
IMPORT const char *
short_day_name __P_((const int day));
IMPORT const char *
short_month_name __P_((const int month));
IMPORT Bool
jdate2sdate __P_((      int  jdate,
                  const int  is_leap_year,
                        int *day,
                        int *month));
IMPORT int
weekday_of_date __P_((const int day,
                      const int month,
                      const int year));
IMPORT int
day_of_year __P_((const int day,
                  const int month,
                  const int year));
IMPORT int
days_of_february __P_((const int year));
IMPORT int
week_number __P_((const int day,
                  const int month,
                  const int year));
IMPORT int
weekno2jday __P_((      int week,
                  const int year));
/*
************************************************** Defined in `gcal_rcu.c'
*/
EXPORT void
rc_use __P_((void));
#  if __cplusplus
}
#  endif



/*
*  Declare public(extern) variables
*/
IMPORT Hls_struct   ehls1s;                   /* Effective hls 1 start (current day) */
IMPORT Hls_struct   ehls1e;                   /* Effective hls 1 end (current day) */
IMPORT Hls_struct   ehls2s;                   /* Effective hls 2 start (holiday) */
IMPORT Hls_struct   ehls2e;                   /* Effective hls 2 end (holiday) */
#if HAVE_ASSERT_H
IMPORT Uint         testval;                  /* Set to INT_MAX for checking the maximum table range */
#endif
IMPORT int          len_year_max;             /* String length of the maximum year able to compute */
IMPORT int          start_day;                /* -s<0,1...7|day name> */
IMPORT int          warning_level;            /* --debug[=0...WARN_LVL_MAX] */
IMPORT int          day;                      /* Current day */
IMPORT int          month;                    /* Current month */
IMPORT int          year;                     /* Current year */
IMPORT int          act_day;                  /* Actual day */
IMPORT int          act_month;                /* Actual month */
IMPORT int          act_year;                 /* Actual year */
IMPORT int          fiscal_month;             /* Starting month of a fiscal year */
IMPORT int          is_leap_year;             /* Is current year a leap year? */
IMPORT int          holiday_vector[MONTH_MAX][MONTH_MAX]; /* Stores the holiday dates */
IMPORT char         s[MAXLEN+1];              /* General purpose text buffer */
IMPORT char         s2[MAXLEN+1];             /* General purpose text buffer */
IMPORT char         s3[MAXLEN+1];             /* General purpose text buffer */
IMPORT Bool         emu_hls;                  /* Must we emulate the highlighting sequences? */
IMPORT char        *hd_table[HD_ELEMS_MAX];   /* Stores the eternal holiday list texts */
IMPORT Bool         cal_julian_flag;          /* -j */
IMPORT Bool         holiday_flag;             /* -n|N */



/*
   Define public(extern) variables
*/
PUBLIC Dvar_struct  rc_dvar[RC_DVAR_MAX];       /* Date variables a[=mmdd]...z[] (`yyyy'@{a|b|...|z}[[-]<n>]) */
PUBLIC int          rc_period;                  /* Amount of period of fixed date warnings */
PUBLIC int          rc_elems_max=RC_ELEMS_MAX;  /* Actual size of `rc_table' */
PUBLIC int          rc_elems;                   /* Amount of resource file entries */
PUBLIC int          rc_have_today_in_list=0;    /* [-c]d */
PUBLIC int          len_fil_wt;                 /* Filler length of week number text */
PUBLIC char       **rc_table=(char **)NULL;     /* Stores the valid fixed date texts */
PUBLIC char        *rc_adate=(char *)NULL;      /* Text of modified actual date %... */
PUBLIC char        *rc_filename=(char *)NULL;   /* Name of alternative resource file(s) -f|F<NAME[+...]> */
PUBLIC char        *rc_grp_sep=(char *)NULL;    /* Fixed date list grouping separator [-c]g[txt] */
PUBLIC Bool         rc_julian_flag=FALSE;       /* -jc */
PUBLIC Bool         rc_both_dates_flag=FALSE;   /* -jcb */
PUBLIC Bool         rc_use_flag=FALSE;          /* -c */
PUBLIC Bool         rc_all_dates_flag=FALSE;    /* -C[] or -C[][T|W|M|Y] or -c[][T|W|M|Y] or -F<> */
PUBLIC Bool         rc_sort_des_flag=FALSE;     /* [-c]- */
PUBLIC Bool         rc_enable_fn_flag=FALSE;    /* [-c]a */
PUBLIC Bool         rc_enable_hda_flag=FALSE;   /* [-c]e */
PUBLIC Bool         rc_enable_hdl_flag=FALSE;   /* [-c]E */
PUBLIC Bool         rc_weekno_flag=FALSE;       /* [-c]k */
PUBLIC Bool         rc_period_list=FALSE;       /* [-c]l */
PUBLIC Bool         rc_omit_date_flag=FALSE;    /* [-c]o */
PUBLIC Bool         rc_title_flag=TRUE;         /* [-c]x */
PUBLIC Bool         rc_count_flag=FALSE;        /* [-c]z */
PUBLIC Bool         rc_period_flag=FALSE;       /* [-c]<<<<n>>[<d|w|+|-]>|`mmdd'|`mmww[w]'<n>> */
PUBLIC Bool         rc_tomorrow_flag=FALSE;     /* [-c]t */
PUBLIC Bool         rc_week_flag=FALSE;         /* [-c]w */
PUBLIC Bool         rc_month_flag=FALSE;        /* [-c]m */
PUBLIC Bool         rc_year_flag=FALSE;         /* [-c]y */
PUBLIC Bool         rc_week_year_flag=FALSE;    /* [-c<<n>>]w */
PUBLIC Bool         rc_forwards_flag=FALSE;     /* [-c<<n>|w|m|y>]+ */
PUBLIC Bool         rc_backwards_flag=FALSE;    /* [-c<<n>|w|m|y>]- */
PUBLIC Bool         rc_nzero_ldvar_flag=FALSE;  /* --hidden   don't reset local date variables */
PUBLIC Bool         is_date_given=FALSE;        /* Is a date given in the command line? */
PUBLIC Bool         is_1month_mode;             /* [-c]<n>w and complete week is in month */
PUBLIC Bool         is_2month_mode;             /* [-c]<n>w and only part of week is in month */
PUBLIC Bool         adate_set=FALSE;            /* [-c]<n>w and actual date modified */



#  ifdef ANSI_PROTO
PUBLIC void
rc_use (void)
#  else /* !ANSI_PROTO */
   PUBLIC void
rc_use ()
#  endif /* !ANSI_PROTO */
/*
   Processes a standard-/special resource file and displays the valid
   fixed dates found resp.,  displays valid fixed dates of eternal holiday list
*/
{
   auto     FILE  *fp=(FILE *)NULL;
   auto     FILE  *fp2=(FILE *)NULL;
   register int    wd=weekday_of_date (act_day, act_month, act_year);
   register int    ed;
   register int    len;
   register int    i;
   register int    j;
   register int    tmp_month=month;
   register int    tmp_fiscal_month=fiscal_month;
   register int    tmp_start_day=start_day;
   register int    tmp_ad=act_day;
   register int    tmp_am=act_month;
   register int    tmp_ay=act_year;
   auto     int    line=0;
   auto     int    line2;
   auto     int    i_dummy;
   static   char   line_buffer[MAXLEN+1];
   static   char   string[MAXLEN+1];
   static   char   str[MAXLEN+1];
   auto     char  *ptr_char;
   auto     char   c_dummy;
#if HAVE_ASSERT_H
   static   Bool   rc_table_range_checked=FALSE;
#endif
   static   Bool   rc_table_initialized=FALSE;
   auto     Bool   ok=FALSE;
   auto     Bool   is_include;
   auto     Bool   is_dvar;
   auto     Bool   b_dummy;


#if HAVE_ASSERT_H
   /*
      Check if value for maximum number of table entries
        fits to the positive range of a signed int!!
   */
   if (!rc_table_range_checked)
    {
      assert(rc_elems_max>0);
      assert((Uint)rc_elems_max<=testval);
      rc_table_range_checked = TRUE;
    }
#endif
   /*
      Initial memory allocation for `rc_table[]'
   */
   if (!rc_table_initialized)
    {
      rc_table = (char **)my_malloc (RC_ELEMS_MAX*sizeof(char *),
                                     124, __FILE__, __LINE__ -1,
                                     "rc_elems", RC_ELEMS_MAX);
      rc_table_initialized = TRUE;
    }
   /*
      Initialize some important module global variables
   */
   len_fil_wt=rc_elems = 0;
   /*
      Depending on fixed date mode,
        compute last valid day `ed' a fixed date may occur
   */
   ed=day = day_of_year (act_day, act_month, act_year);
   if (is_date_given)
    {
      /*
         NOT in simple month-/year mode (an explicit date is given in command line):
           compute the starting- and ending loop values of requested period
      */
      ed = DAY_LAST + is_leap_year + 1;
      if (!month)
        day = DAY_MIN;
      else
       {
         day = day_of_year (DAY_MIN, month, year);
         if (month < MONTH_MAX)
           ed = day_of_year (DAY_MIN, month+1, year);
       }
      /*
         Then clean all flags, which are related to the fixed date warning period
      */
      rc_clean_flags ();
    }
   else
    {
      auto Bool  swap_flag=FALSE;


      if (rc_tomorrow_flag)
       {
         /*
            [-c]t modifier found
         */
         rc_period_list = FALSE;
         ed += 2;
       }
      else
        if (rc_week_flag)
         {
           /*
              [-c]w[+|-] modifier found
           */
           rc_period_list = FALSE;
           if (   !rc_forwards_flag
               && !rc_backwards_flag)
             day += (-DAY_MAX + SDAY(DAY_MAX-wd+1, start_day));
           else
            {
              if (rc_forwards_flag)
               {
                 if (!rc_have_today_in_list)
                   day++;
               }
              else
                day += rc_have_today_in_list;
            }
           if (   rc_backwards_flag
               || (   !rc_forwards_flag
                   && !rc_backwards_flag
                   && (start_day > wd)))
            {
              swap_flag = TRUE;
              ed -= DAY_MAX;
            }
           ed += SDAY(DAY_MAX-wd+1, start_day);
         }
        else
          if (rc_month_flag)
           {
             /*
                [-c]m[+|-] modifier found
             */
             rc_period_list = FALSE;
             if (   !rc_forwards_flag
                 && !rc_backwards_flag)
               day = day_of_year (DAY_MIN, act_month, act_year);
             else
              {
                if (rc_forwards_flag)
                 {
                   if (!rc_have_today_in_list)
                     day++;
                 }
                else
                  day += rc_have_today_in_list;
              }
             if (rc_backwards_flag)
               ed = day_of_year (DAY_MIN, act_month, act_year);
             else
              {
                if (act_month < MONTH_MAX)
                  ed = day_of_year (DAY_MIN, act_month+1, act_year);
                else
                  ed = DAY_LAST + is_leap_year + 1;
              }
           }
          else
            if (rc_year_flag)
             {
               /*
                  [-c]y[+|-] modifier found
               */
               rc_period_list = FALSE;
               if (   !rc_forwards_flag
                   && !rc_backwards_flag)
                 day = DAY_MIN;
               else
                {
                  if (rc_forwards_flag)
                   {
                     if (!rc_have_today_in_list)
                       day++;
                   }
                  else
                    day += rc_have_today_in_list;
                }
               if (rc_backwards_flag)
                 ed = DAY_MIN;
               else
                 ed = DAY_LAST + is_leap_year + 1;
             }
            else
              if (rc_period_flag)
               {
                 if (rc_week_year_flag)
                  {
                    /*
                       [-c]<n>w flag found
                         (those ISO-8601:1988 weeks start at monday)
                    */
                    wd=start_day = 1;
                    /*
                       Compute day number of year the week start at
                    */
                    j=day = weekno2jday (rc_period, act_year);
                    if (day != -WEEK_MAX)
                     {
                       if (rc_period_list)
                        {
                          /*
                             [-c]<n>w modifier found
                          */
                          if (day > ed)
                           {
                             if (day < DAY_LAST+is_leap_year)
                              {
                                rc_forwards_flag = TRUE;
                                day = ed + 1 - rc_have_today_in_list;
                                ed = j + DAY_MAX;
                                if (ed > DAY_LAST+is_leap_year+1)
                                  ed = DAY_LAST + is_leap_year + 1;
                              }
                             else
                               /*
                                  No fixed date messages to display
                               */
                               day = ed;
                           }
                          else
                           {
                             if (ed > DAY_MIN)
                              {
                                rc_backwards_flag = TRUE;
                                day = ed + rc_have_today_in_list;
                                ed = j;
                                if (ed < DAY_MIN)
                                  ed = DAY_MIN;
                              }
                             else
                               /*
                                  No fixed date messages to display
                               */
                               day = ed;
                           }
                        }
                       else
                        {
                          /*
                             [-c]<n>w modifier found:
                               set actual date to monday's date of week
                               and omit highlighting that date; boolean
                               `adate_set' is set to distinct this
                               SPECIAL mode from other modes...
                          */
                          rc_week_flag = TRUE;
                          ed = day + DAY_MAX;
                          if (day < DAY_MIN)
                           {
                             act_year--;
                             j += (DAY_LAST + (days_of_february (act_year) == 29));
                           }
                          else
                            j = day;
                          (void)jdate2sdate (j, is_leap_year, &act_day, &act_month);
                          month = act_month;
                          fiscal_month = MONTH_MIN;
                          adate_set = TRUE;
                        }
                     }
                    else
                      /*
                         Wanted week doesn't exist:
                           don't display any fixed date messages
                      */
                      day = ed;
                  }
                 else
                   /*
                      [-c]<n>d,
                      [-c]`mmdd',
                      [-c]`mmww[w]'<n>,
                      [-c]*d|w<n>[`ww[w]'] and
                      [-c]@<e|`dvar'>[[-]<n>]
                        modifiers are managed in this subsection too
                   */
                   if (   rc_forwards_flag
                       && (day < DAY_LAST+is_leap_year))
                    {
                      if (rc_period_list)
                       {
                         /*
                            [-c]<n>+ modifier found
                         */
                         if (!rc_have_today_in_list)
                           day++;
                         ed += (rc_period + 1);
                         if (ed > DAY_LAST+is_leap_year+1)
                           ed = DAY_LAST + is_leap_year + 1;
                       }
                      else
                       {
                         /*
                            [-c]<n>+ modifier found
                         */
                         rc_forwards_flag = FALSE;
                         day += rc_period;
                         ed = day;
                         if (day < DAY_LAST+is_leap_year+1)
                          {
                            rc_period_flag = FALSE;
                            (void)jdate2sdate (day, is_leap_year, &act_day, &act_month);
                            month = act_month;
                            ed++;
                          }
                       }
                    }
                   else
                     if (   rc_backwards_flag
                         && (day > DAY_MIN))
                      {
                        if (rc_period_list)
                         {
                           /*
                              [-c]<n>- modifier found
                           */
                           day += rc_have_today_in_list;
                           ed -= rc_period;
                           if (ed < DAY_MIN)
                             ed = DAY_MIN;
                         }
                        else
                         {
                           /*
                              [-c]<n>- modifier found
                           */
                           rc_backwards_flag = FALSE;
                           day -= rc_period;
                           ed = day;
                           if (day > 0)
                            {
                              rc_period_flag = FALSE;
                              (void)jdate2sdate (day, is_leap_year, &act_day, &act_month);
                              month = act_month;
                              ed++;
                            }
                         }
                      }
               }
              else
               {
                 /*
                    Only simple -c (without any modifiers) found
                 */
                 rc_period_list = FALSE;
                 ed++;
               }
      /*
         Swap start date `day' and ending date `ed' of period
      */
      if (   swap_flag
          || rc_backwards_flag)
        day ^= (ed ^= (day ^= ed));
    }
   if (day != ed)
    {
      /*
         Now include the eternal holidays, which are valid fixed dates,
           into `*rc_table[]'
      */
      if (   rc_enable_hda_flag
          || rc_enable_hdl_flag)
       {
         while (   (hd_table[line] != (char *)NULL)
                && (line < HD_ELEMS_MAX))
          {
            strcpy(line_buffer, hd_table[line]);
            i = LEN_HD_NAME + len_year_max + 4 + 2;
            if (   rc_enable_hda_flag
                || (   rc_enable_hdl_flag
                    && (line_buffer[i] != *DIS_HLS_PREF)))
             {
               i -= 2;
               while (   i
                      && MY_ISSPACE(line_buffer[i]))
                 line_buffer[i--] = '\0';
               if (i > len_year_max+4)
                 rc_check (line_buffer, "`"HD_LIST_TITLE"'", &line, &rc_elems, day, ed, wd);
             }
            if (!holiday_flag)
             {
               free(hd_table[line]);
               hd_table[line] = (char *)NULL;
             }
            line++;
          }
         /*
            If fixed date warning mode is related to a week or to tomorrow
              and the warning period is off the current year, include those
              eternal holidays in year +/- 1 into `rc_table', which are
              valid fixed dates
         */
         if (   (   rc_tomorrow_flag
                 || rc_week_flag)
             && !is_date_given
             && !is_1month_mode
             && !is_2month_mode
             && (   (   !rc_forwards_flag
                     && !rc_backwards_flag
                     && (   (   (year+1 <= YEAR_MAX)
                             && (ed > DAY_LAST+is_leap_year+1))
                         || (   (year-1 >= YEAR_MIN)
                             && (day < DAY_MIN))))
                 || (   rc_forwards_flag
                     && (   (year+1 <= YEAR_MAX)
                         && (ed > DAY_LAST+is_leap_year+1)))
                 || (   rc_backwards_flag
                     && (   (year-1 >= YEAR_MIN)
                         && (day < DAY_MIN)))))
          {
            register int  jtmp_ad=act_day;
            register int  tay=act_year;
            register int  tam=act_month;
            register int  tad=act_day;


            if (holiday_flag)
              for (i=0 ; i < line ; i++)
               {
                 free(hd_table[i]);
                 hd_table[i] = (char *)NULL;
               }
            line = 0;
            if (day < DAY_MIN)
             {
               year--;
               month = MONTH_MAX;
             }
            else
             {
               year++;
               month = MONTH_MIN;
             }
            is_leap_year = (days_of_february (year) == 29);
            if (cal_julian_flag)
              act_day = day_of_year (jtmp_ad, act_month, act_year);
            if (adate_set)
              fiscal_month = tmp_fiscal_month;
            if (   (   (year == EASTER_MIN-1)
                    && (fiscal_month > MONTH_MIN))
                || (   (year >= EASTER_MIN)
                    && (year <= EASTER_MAX)))
              print_all_holidays (TRUE, TRUE);
            if (adate_set)
              fiscal_month = MONTH_MIN;
            if (day < DAY_MIN)
             {
               year++;
               month = MONTH_MIN;
             }
            else
             {
               year--;
               month = MONTH_MAX;
             }
            while (   (hd_table[line] != (char *)NULL)
                   && (line < HD_ELEMS_MAX))
             {
               strcpy(line_buffer, hd_table[line]);
               i = LEN_HD_NAME + len_year_max + 4 + 2;
               if (   rc_enable_hda_flag
                   || (   rc_enable_hdl_flag
                       && (line_buffer[i] == *DIS_HLS_PREF)))
                {
                  i -= 2;
                  while (   i
                         && MY_ISSPACE(line_buffer[i]))
                    line_buffer[i--] = '\0';
                  if (i > len_year_max+4)
                    rc_check (line_buffer, "`"HD_LIST_TITLE"'", &line, &rc_elems, day, ed, wd);
                }
               free(hd_table[line]);
               hd_table[line] = (char *)NULL;
               line++;
             }
            is_leap_year = (days_of_february (year) == 29);
            if (adate_set)
             {
               fiscal_month = tmp_fiscal_month;
               act_year = tmp_ay;
               act_month = tmp_am;
               jtmp_ad=act_day = tmp_ad;
             }
            if (cal_julian_flag)
              act_day = day_of_year (jtmp_ad, act_month, act_year);
            if (   holiday_flag
                && (   (   (year == EASTER_MIN-1)
                        && (fiscal_month > MONTH_MIN))
                    || (   (year >= EASTER_MIN)
                        && (year <= EASTER_MAX))))
              print_all_holidays (TRUE, TRUE);
            if (cal_julian_flag)
              act_day = jtmp_ad;
            if (adate_set)
             {
               fiscal_month = MONTH_MIN;
               act_year = tay;
               month=act_month = tam;
               act_day = tad;
             }
            else
              month = tmp_month;
          }
       }
      /*
         Try to open the resource file(s)
      */
      if (rc_filename == (char *)NULL)
        *s3 = '\0';
      else
       {
         strncpy(s3, rc_filename, MAXLEN+1);
         s3[MAXLEN] = '\0';
       }
      while (!ok)
       {
         if (*s3)
          {
            if (strchr(s3, *RC_FNAME_SEP) == (char *)NULL)
             {
               /*
                  Single alternative resource file name given in command line
               */
               ok = TRUE;
               strcpy(str, s3);
               fp = file_open (str, REsource);
             }
            else
             {
               auto char  *ptr2_char;
               auto Bool   ok2=FALSE;


               /*
                  List of alternative resource file names given in command line
               */
               while (   !ok2
                      && (fp == (FILE *)NULL))
                {
                  ok2 = (Bool)((ptr2_char=strchr(s3, *RC_FNAME_SEP)) == (char *)NULL);

                  if (ok2)
                    i = (int)strlen(s3);
                  else
                    i = (int)strlen(s3) - strlen(ptr2_char);
                  strcpy(str, s3);
                  str[i] = '\0';
                  if (*str)
                    fp = file_open (str, REsource);
                  if (ok2)
                   {
                     if (!s3[i])
                       ok = TRUE;
                     break;
                   }
                  ptr2_char = s3 + i + 1;
                  strcpy(s3, ptr2_char);
                  if (!*s3)
                    ok=ok2 = TRUE;
                }
             }
          }
         if (   !ok
             && (fp == (FILE *)NULL))
          {
            /*
               Use standard resource file, always processed at last
            */
            ok = TRUE;
            strcpy(str, PRGR_RC_NAME);
            fp = file_open (str, REsource);
          }
         /*
            Now read and check contents of a resource file `str'
              and include valid fixed dates into `*rc_table[]'
         */
         if (fp != (FILE *)NULL)
          {
            auto int    in_pool=0;
            auto char  *pool=(char *)NULL;
            auto char  *ptr_pool=(char *)NULL;


            pool = (char *)my_malloc (BUF_LEN+1, 124, __FILE__, __LINE__, "pool", 0);
            /*
               First copy the contents of global date variables
                 to local variables (if one of these isn't defined)
                 so we can perform local operations (++, --, +=, -=)
                 on global variables
            */
            for (i=0 ; i < RC_DVAR_MAX ; i++)
              if (   rc_dvar[i].g.month
                  && !rc_dvar[i].l.month)
               {
                 rc_dvar[i].l.month = rc_dvar[i].g.month;
                 rc_dvar[i].l.day = rc_dvar[i].g.day;
               }
            line = 0;
            while ((ptr_pool=file_read_line (fp, line_buffer, &in_pool, pool, ptr_pool,
                                             str, &line, REsource, &is_include, &is_dvar)) != (char *)NULL)
             {
               /*
                  Check whether an "#include" statement is found
               */
               if (is_include)
                {
                  auto Bool  is_usr_file=FALSE;
                  auto Bool  is_sys_file=FALSE;


                  len = (int)strlen(RC_INCL_STMENT);
                  strcpy(string, line_buffer);
                  for (i=0 ; i < len ; i++)
                    string[i] = (char)tolower(string[i]);
                  if (strncmp(string, RC_INCL_STMENT, len))
                    /*
                       Error, misspelled "#include" directive found
                    */
                    my_error (120, str, line, line_buffer, 0);
                  /*
                     Skip leading whitespace characters
                  */
                  while (MY_ISSPACE(string[i]))
                    i++;
                  if (   !string[i]
                      || string[i] == '\n')
                    /*
                       Error, no include file "argument" encoded
                    */
                    my_error (120, str, line, line_buffer, 0);
                  if (string[i] == *RC_INCL_USR_ID)
                    is_usr_file = TRUE;
                  else
                    if (string[i] == *RC_INCL_SYS_ID)
                      is_sys_file = TRUE;
                    else
                      /*
                         Error, illegal leading include file name delimiter found
                      */
                      my_error (120, str, line, line_buffer, 0);
                  i++;
                  /*
                     Skip leading whitespace characters before include file name
                  */
                  while (MY_ISSPACE(string[i]))
                    i++;
                  j = 0;
                  /*
                     Get include file name
                  */
                  while (   string[i]
                         && !MY_ISSPACE(string[i])
                         && (string[i] != *RC_INCL_USR_OD)
                         && (string[i] != *RC_INCL_SYS_OD))
                    string[j++] = string[i++];
                  /*
                     Skip trailing whitespace characters after include file name
                  */
                  while (MY_ISSPACE(string[i]))
                    i++;
                  if (   (   is_usr_file
                          && (string[i] != *RC_INCL_USR_OD))
                      || (   is_sys_file
                          && (string[i] != *RC_INCL_SYS_OD)))
                    /*
                       Error, illegal trailing include file name delimiter found
                    */
                    my_error (120, str, line, line_buffer, 0);
                  if (!j)
                    /*
                       Error, no include file "name" encoded
                    */
                    my_error (120, str, line, line_buffer, 0);
                  /*
                     Now we have the name of include file
                       and the mode of operation
                  */
                  string[j] = '\0';
                  /*
                     Try to open include file
                  */
                  fp2 = file_open (string, (is_usr_file) ? USr_include : SYs_include);
                  /*
                     Include file does exist so read it in
                  */
                  if (fp2 != (FILE *)NULL)
                   {
                     auto Dvar_base_struct   tmp_dvars[RC_DVAR_MAX];
                     auto int                in_2pool=0;
                     auto char              *pool2=(char *)NULL;
                     auto char              *ptr_2pool=(char *)NULL;


                     pool2 = (char *)my_malloc (BUF_LEN+1, 124, __FILE__, __LINE__, "pool2", 0);
                     /*
                        Store all local date variables into `tmp_dvars[]'
                     */
                     for (i=0 ; i < RC_DVAR_MAX ; i++)
                      {
                        tmp_dvars[i].month = rc_dvar[i].l.month;
                        tmp_dvars[i].day = rc_dvar[i].l.day;
                      }
                     line2 = 0;
                     while ((ptr_2pool=file_read_line (fp2, line_buffer, &in_2pool, pool2, ptr_2pool,
                                                       string, &line2, REsource, &is_include, &is_dvar))
                                      != (char *)NULL)
                       if (is_include)
                         /*
                            Error, invalid sub-include statement found
                         */
                         my_error (119, string, line2, line_buffer, 0);
                       else
                         if (   *line_buffer
                             && !is_dvar)
                           rc_check (line_buffer, string, &line2, &rc_elems, day, ed, wd);
                     fclose(fp2);
                     fp2 = (FILE *)NULL;
                     free(pool2);
                     /*
                        Restore all local date variables from `tmp_dvars[]'
                     */
                     for (i=0 ; i < RC_DVAR_MAX ; i++)
                      {
                        rc_dvar[i].l.month = tmp_dvars[i].month;
                        rc_dvar[i].l.day = tmp_dvars[i].day;
                      }
                   }
                }
               else
                 if (   *line_buffer
                     && !is_dvar)
                   rc_check (line_buffer, str, &line, &rc_elems, day, ed, wd);
             }
            fclose(fp);
            fp = (FILE *)NULL;
            free(pool);
            /*
               Next file -> reset all local date variables to 0,
                 if --hidden flag set, don't reset them !!
            */
            if (!rc_nzero_ldvar_flag)
              for (i=0 ; i < RC_DVAR_MAX ; i++)
                rc_dvar[i].l.month = (char)0;
          }
       }
      fiscal_month = tmp_fiscal_month;
      /*
         Now display the valid resource file(s) entries in sorted order
      */
      if (   rc_elems
          || rc_all_dates_flag)
       {
         auto     int  d=0;
         auto     int  m=0;
         auto     int  y=0;
         auto     int  dd=0;
         auto     int  mm=0;
         auto     int  yy=0;



         len = (rc_julian_flag && !rc_both_dates_flag) ? 3 : 2;
         j = 0;
         /*
            If `rc_all_dates_flag' is set, complete the contents of `*rc_table[]',
              i.e., generate "empty" warnings for each day in requested period
         */
         if (rc_all_dates_flag)
          {
            register int  rc_e=rc_elems;


            /*
               Lets sort ascending first only in case `rc_table[]' isn't presorted!
            */
            if (   (rc_elems > 1)
                && !is_presorted (rc_table, rc_elems))
              qsort((VOID_PTR)rc_table, rc_elems, sizeof *rc_table, (Func_cmp)asc_sort);
            if (rc_elems)
              (void)rc_get_date (rc_table[j], &b_dummy, &d, &m, &y, &i_dummy, &i_dummy,
                                 &c_dummy, &i_dummy, &i_dummy, str, line, rc_table[j], TRUE);
            for (i=day ; i < ed ; i++)
             {
               yy = year;
               ok = jdate2sdate (i, is_leap_year, &dd, &mm);
               if (!ok)
                {
                  if (   rc_tomorrow_flag
                      || rc_week_flag)
                   {
                     if (i < DAY_MIN)
                      {
                        yy = year - 1;
                        if (yy < YEAR_MIN)
                          break;
                        line = (days_of_february (yy) == 29);
                        (void)jdate2sdate (DAY_LAST+line+i, line, &dd, &mm);
                      }
                     else
                       if (i > DAY_LAST+is_leap_year)
                        {
                          yy = year + 1;
                          if (yy > YEAR_MAX)
                            break;
                          line = (days_of_february (yy) == 29);
                          (void)jdate2sdate (i-(DAY_LAST+is_leap_year), line, &dd, &mm);
                        }
                   }
                  else
                    break;
                }
               if (   j == rc_e
                   || mm < m
                   || yy < y
                   || (   (dd < d)
                       && (mm <= m)
                       && (yy <= y)))
                {
                  sprintf(line_buffer, "%0*d%02d%02d ", len_year_max, yy, mm, dd);
                  rc_check (line_buffer, "`"RC_INTERNAL_TXT"'", &line, &rc_elems, day, ed, wd);
                }
               else
                 while (   (dd == d)
                        && (mm == m)
                        && (yy == y))
                   if (j+1 < rc_e)
                    {
                      j++;
                      (void)rc_get_date (rc_table[j], &b_dummy, &d, &m, &y, &i_dummy, &i_dummy,
                                         &c_dummy, &i_dummy, &i_dummy, str, line, rc_table[j], TRUE);
                    }
                   else
                    {
                      j++;
                      break;
                    }
             }
            /*
               Respect fiscal year mode...
            */
            if (   (fiscal_month > MONTH_MIN)
                && (year < YEAR_MAX))
             {
               day = DAY_MIN;
               yy = year + 1;
               is_leap_year = (days_of_february (yy) == 29);
               if (fiscal_month < MONTH_MAX)
                 ed = day_of_year (DAY_MIN, fiscal_month+1, yy) + 1;
               else
                 ed = DAY_LAST + is_leap_year + 1;
               for (i=day ; i < ed ; i++)
                {
                  ok = jdate2sdate (i, is_leap_year, &dd, &mm);
                  if (!ok)
                    break;
                  if (   j == rc_e
                      || mm < m
                      || yy < y
                      || (   (dd < d)
                          && (mm <= m)
                          && (yy <= y)))
                   {
                     sprintf(line_buffer, "%0*d%02d%02d ", len_year_max, yy, mm, dd);
                     rc_check (line_buffer, "`"RC_INTERNAL_TXT"'", &line, &rc_elems, day, ed, wd);
                   }
                  else
                    while (   (dd == d)
                           && (mm == m)
                           && (yy == y))
                      if (j+1 < rc_e)
                       {
                         j++;
                         (void)rc_get_date (rc_table[j], &b_dummy, &d, &m, &y, &i_dummy, &i_dummy,
                                            &c_dummy, &i_dummy, &i_dummy, str, line, rc_table[j], TRUE);
                       }
                      else
                       {
                         j++;
                         break;
                       }
                }
             }
            is_leap_year = (days_of_february (year) == 29);
          }
         if (rc_elems)
          {
            register int   hls1_len=(ehls1s.len!=1) ? ehls1s.len+ehls1e.len : 0;
            register int   hls2_len=(ehls1s.len!=1) ? ehls2s.len+ehls2e.len : 0;
            register int   skip=0;
            register int   len_rce=0;
            register int   len_fn_part;
            register int   len_s;
            register int   len_ptr_char;
            auto     Bool  same_date=FALSE;
            auto     Bool  hls1_set;
            auto     Bool  hls2_set;


            /*
               At last, sort the fixed date warnings only if they are not presorted
            */
            if (rc_elems > 1)
             {
               if (!is_presorted (rc_table, rc_elems))
                {
                  if (rc_sort_des_flag)
                    qsort((VOID_PTR)rc_table, rc_elems, sizeof *rc_table, (Func_cmp)des_sort);
                  else
                    qsort((VOID_PTR)rc_table, rc_elems, sizeof *rc_table, (Func_cmp)asc_sort);
                }
               else
                 if (rc_sort_des_flag)
                   /*
                      `rc_table[]' is presorted and must be shown in descending sort order,
                        rearrange its internal sort order from ascending to descending sort order.
                   */
                   resort (rc_table, rc_elems);
             }
            /*
               Copy `rc_grp_sep' [-c]g[] to text buffer variable `s3' for further use
                 and perform the `~'-TILDE expansion
            */
            if (rc_grp_sep != (char *)NULL)
             {
               strncpy(s3, rc_grp_sep, MAXLEN+1);
               s3[MAXLEN] = '\0';
               ptr_char = s3;
               if (*ptr_char)
                 if (strchr(ptr_char, RC_NL_CHAR) != (char *)NULL)
                  {
                    /*
                       `~'-TILDE or quoted `\~'-TILDE found in group separator `txt':
                         quote TILDE resp., insert a real `\n'-NEWLINE character
                    */
                    i = 0;
                    while (*ptr_char)
                     {
                       if (*ptr_char == RC_NL_CHAR)
                        {
                          if (i)
                           {
                             /*
                                RC_NL_CHAR is single `\~' quoted
                                           or double `\\~' quoted:
                                  replace QUOTE_CHAR by RC_NL_CHAR resp.,
                                  replace last QUOTE_CHAR by RC_NL_CHAR
                             */
                             if (s3[i-1] == QUOTE_CHAR)
                               s3[i-1] = *ptr_char;
                             else
                               /*
                                  RC_NL_CHAR is not quoted `~':
                                    insert a real `\n'-NEWLINE character
                               */
                               s3[i++] = '\n';
                           }
                          else
                            /*
                               RC_NL_CHAR is not quoted `~':
                                 insert a real `\n'-NEWLINE character
                            */
                            s3[i++] = '\n';
                        }
                       else
                         s3[i++] = *ptr_char;
                       ptr_char++;
                     }
                    s3[i] = '\0';
                  }
             }
            /*
               Now display a leading newline before text/title
            */
            *s = '\0';
            print_text (stdout, s, INternal);
            /*
               Now display a leading text/title
            */
            if (rc_title_flag)
             {
               sprintf(s, "%s:", RC_LIST_TITLE);
               print_text (stdout, s, INternal);
               print_text (stdout, s, INternal);
             }
            /*
               Detect the actual length of `rc_elems'
            */
            sprintf(s, "%d", rc_elems);
            len_rce = (int)strlen(s) + 3;   /* +3 because of the ")  " text */
            /*
               Display all fixed date warnings in `rc_table[]'
            */
            for (line=0 ; line < rc_elems ; line++)
             {
               ptr_char = rc_get_date (rc_table[line], &b_dummy, &day, &m, &y, &i_dummy, &i_dummy,
                                       &c_dummy, &i_dummy, &i_dummy, str, line, rc_table[j], TRUE);
               if (   !line
                   && (   rc_omit_date_flag
                       || rc_grp_sep != (char *)NULL))
                {
                  dd = day;
                  mm = m;
                  yy = y;
                }
               /*
                  Avoid displaying of duplicate resource file entries
               */
               if (   !line
                   || (   line
                       && strcmp(line_buffer, rc_table[line])))
                {
                  strcpy(line_buffer, rc_table[line]);
                  /*
                     Now display fixed date warning list group separator resp.,
                       detect if date of warning differs from previous warning
                  */
                  if (   rc_omit_date_flag
                      || rc_grp_sep != (char *)NULL)
                   {
                     if (   (day == dd)
                         && (m == mm)
                         && (y == yy))
                      {
                        /*
                           Same date:
                             avoid displaying of group separator
                        */
                        if (line)
                          same_date = TRUE;
                      }
                     else
                      {
                        /*
                           Date differs
                        */
                        same_date = FALSE;
                        dd = day;
                        mm = m;
                        yy = y;
                        if (rc_grp_sep != (char *)NULL)
                         {
                           /*
                              Display constructed group separator `txt' [-c]g[]
                           */
                           strcpy(s, s3);
                           print_text (stdout, s, INternal);
                         }
                      }
                   }
                  /*
                     Display leading (highlighted)date of fixed date text
                  */
                  if (rc_weekno_flag)
                   {
                     j = week_number (day, m, y);
                     if (!j)
                      {
                        /*
                           Week is both last week of previous year and first week of year
                        */
                        if (y == GREG_YEAR+1)
                          j = ((DAY_LAST + (days_of_february (GREG_YEAR) == 29)
                              - (GREG_L_DAY - GREG_F_DAY + 1)) / DAY_MAX) + 1;
                        else
                          j = WEEK_MAX + 1;
                        sprintf(s2, "%02d/1", j);
                        if (rc_count_flag)
#  if USE_GER
                          sprintf(s, "%0*d)  |%s| %s, ", len_rce-3, (line-skip)+1, s2,
                                  short_day_name (weekday_of_date (day, m, y)));
#  else /* !USE_GER */
                          sprintf(s, "%0*d)  |%s| %s, ", len_rce-3, (line-skip)+1, s2,
                                  short3_day_name (weekday_of_date (day, m, y)));
#  endif /* !USE_GER */
                        else
#  if USE_GER
                          sprintf(s, "|%s| %s, ", s2,
                                  short_day_name (weekday_of_date (day, m, y)));
#  else /* !USE_GER */
                          sprintf(s, "|%s| %s, ", s2,
                                  short3_day_name (weekday_of_date (day, m, y)));
#  endif /* !USE_GER */
                      }
                     else
                      {
                        /*
                           Ordinary week 1...52|53
                        */
                        if (rc_count_flag)
#  if USE_GER
                          sprintf(s, "%0*d)  |%02d|%*s %s, ", len_rce-3, (line-skip)+1, j,
                                  len_fil_wt, "", short_day_name (weekday_of_date (day, m, y)));
#  else /* !USE_GER */
                          sprintf(s, "%0*d)  |%02d|%*s %s, ", len_rce-3, (line-skip)+1, j,
                                  len_fil_wt, "", short3_day_name (weekday_of_date (day, m, y)));
#  endif /* !USE_GER */
                        else
#  if USE_GER
                          sprintf(s, "|%02d|%*s %s, ", j, len_fil_wt, "",
                                  short_day_name (weekday_of_date (day, m, y)));
#  else /* !USE_GER */
                          sprintf(s, "|%02d|%*s %s, ", j, len_fil_wt, "",
                                  short3_day_name (weekday_of_date (day, m, y)));
#  endif /* !USE_GER */
                      }
                   }
                  else
                   {
                     if (rc_count_flag)
#  if USE_GER
                       sprintf(s, "%0*d)  %s, ", len_rce-3, (line-skip)+1,
                               short_day_name (weekday_of_date (day, m, y)));
#  else /* !USE_GER */
                       sprintf(s, "%0*d)  %s, ", len_rce-3, (line-skip)+1,
                               short3_day_name (weekday_of_date (day, m, y)));
#  endif /* !USE_GER */
                     else
#  if USE_GER
                       sprintf(s, "%s, ", short_day_name (weekday_of_date (day, m, y)));
#  else /* !USE_GER */
                       sprintf(s, "%s, ", short3_day_name (weekday_of_date (day, m, y)));
#  endif /* !USE_GER */
                   }
                  d = day;
                  wd = 0;
                  j = (int)strlen(s);
                  if (   rc_julian_flag
                      || rc_both_dates_flag)
                    d = day_of_year (d, m, y);
                  hls1_set=hls2_set = FALSE;
#  if USE_GER
                  if (   (y == tmp_ay)
                      && (m == tmp_am)
                      && (day == tmp_ad))
                   {
                     j += 2;
                     hls1_set = TRUE;
                     if (rc_both_dates_flag)
                      {
                        sprintf(s2, "%*s%s%2d(%3d)%s%*s", !emu_hls, "", ehls1s.seq,
                                day, d, ehls1e.seq, !emu_hls, "");
                        j += 7;
                      }
                     else
                      {
                        sprintf(s2, "%*s%s%*d%s%*s",
                                !emu_hls, "", ehls1s.seq, len, d, ehls1e.seq, !emu_hls, "");
                        j += len;
                      }
                   }
                  else
                   {
                     if (*holiday_vector[m-1])
                       for (i=0 ; holiday_vector[m-1][i] ; i++)
                         if (holiday_vector[m-1][i] == ((rc_both_dates_flag||rc_julian_flag) ? day : d))
                          {
                            wd = day;
                            break;
                          }
                     if (wd)
                      {
                        j += 2;
                        hls2_set = TRUE;
                        if (rc_both_dates_flag)
                         {
                           sprintf(s2, "%*s%s%2d(%3d)%s%*s", !emu_hls, "", ehls2s.seq,
                                   day, d, ehls2e.seq, !emu_hls, "");
                           j += 7;
                         }
                        else
                         {
                           sprintf(s2, "%*s%s%*d%s%*s", !emu_hls, "", ehls2s.seq,
                                   len, d, ehls2e.seq, !emu_hls, "");
                           j += len;
                         }
                      }
                     else
                      {
                        if (rc_both_dates_flag)
                          sprintf(s2, NO_HLS"%2d(%3d)"NO_HLS, day, d);
                        else
                          sprintf(s2, NO_HLS"%*d"NO_HLS, len, d);
                        j += (int)strlen(s2);
                      }
                   }
                  strcat(s, s2);
                  sprintf(s2, "%s %0*d:",
                          (rc_julian_flag&&!rc_both_dates_flag)
                          ? "Tag" : (char *)short_month_name (m), len_year_max, y);
#  else /* !USE_GER */
                  sprintf(s2, "%s%s",
                          (rc_julian_flag&&!rc_both_dates_flag)
                          ? "" : (char *)short_month_name (m),
                          (rc_julian_flag&&!rc_both_dates_flag) ? "" : " ");
                  j += (int)strlen(s2);
                  strcat(s, s2);
                  if (   (y == tmp_ay)
                      && (m == tmp_am)
                      && (day == tmp_ad))
                   {
                     j += 5;
                     hls1_set = TRUE;
                     if (rc_both_dates_flag)
                      {
                        sprintf(s2, "%*s%s%2d%s(%3d)%s%*s", !emu_hls, "", ehls1s.seq,
                                day, day_suffix (day), d, ehls1e.seq, !emu_hls, "");
                        j += 7;
                      }
                     else
                      {
                        sprintf(s2, "%*s%s%*d%s%s%*s", !emu_hls, "", ehls1s.seq,
                                len, d, day_suffix (d), ehls1e.seq, !emu_hls, "");
                        j += len;
                      }
                   }
                  else
                   {
                     if (*holiday_vector[m-1])
                       for (i=0 ; holiday_vector[m-1][i] ; i++)
                         if (holiday_vector[m-1][i] == ((rc_both_dates_flag||rc_julian_flag) ? day : d))
                          {
                            wd = day;
                            break;
                          }
                     if (wd)
                      {
                        j += 5;
                        hls2_set = TRUE;
                        if (rc_both_dates_flag)
                         {
                           sprintf(s2, "%*s%s%2d%s(%3d)%s%*s", !emu_hls, "", ehls2s.seq,
                                   day, day_suffix (day), d, ehls2e.seq, !emu_hls, "");
                           j += 7;
                         }
                        else
                         {
                           sprintf(s2, "%*s%s%*d%s%s%*s", !emu_hls, "", ehls2s.seq,
                                   len, d, day_suffix (d), ehls2e.seq, !emu_hls, "");
                           j += len;
                         }
                      }
                     else
                      {
                        if (rc_both_dates_flag)
                          sprintf(s2, NO_HLS"%2d%s(%3d)"NO_HLS, day, day_suffix (day), d);
                        else
                          sprintf(s2, NO_HLS"%*d%s"NO_HLS, len, d, day_suffix (d));
                        j += (int)strlen(s2);
                      }
                   }
                  strcat(s, s2);
                  sprintf(s2, "%s%0*d:",
                          (rc_julian_flag&&!rc_both_dates_flag) ? "day " : "", len_year_max, y);
#  endif /* !USE_GER */
                  j += (int)strlen(s2);
                  strcat(s, s2);
                  /*
                     Check if `ptr_char' must be truncated
                  */
                  len_s = (int)strlen(s);
                  len_ptr_char = (int)strlen(ptr_char);
                  if (len_s+len_ptr_char >= MAXLEN)
                    *(ptr_char + (MAXLEN-len_s)) = '\0';
                  /*
                     Now modify fixed date text
                  */
                  ok = FALSE;
                  len_fn_part = 0;
                  /*
                     "Empty" warnings texts have no trailing space character!
                  */
                  if (len_ptr_char > 1)
                   {
                     ok = (Bool)(strchr(ptr_char, RC_NL_CHAR) == (char *)NULL);
                     if (ok)
                       strcat(s, ptr_char);
                     else
                      {
                        /*
                           `~'-TILDE or quoted `\~'-TILDE found in `text'-part of fixed date:
                             quote TILDE resp., insert a real `\n'-NEWLINE character
                        */
                        if (rc_enable_fn_flag)
                         {
                           auto char  *ptr_fn_part;


                           /*
                              Add the length of the filename+3 (+3 because the
                                filename itself is leaded by a blank character
                                and is enclosed in parantheses --> " (%s)")
                                to the number of leading blanks, which must be
                                displayed if the line wrapped by a given `~'-TILDE
                           */
                           ptr_fn_part = ptr_char;
                           while (   *ptr_fn_part
                                  && (*ptr_fn_part != ')'))
                            {
                              len_fn_part++;
                              ptr_fn_part++;
                            }
                           if (!*ptr_fn_part)
                             /*
                                Internal error, a maintainer has modified the output
                                  format of a line and forgots to respect this change here!
                             */
                             my_error (99, "", 0, "", 0);
                           len_fn_part++;
                         }
                        while (   *ptr_char
                               && (len_s < MAXLEN))
                         {
                           if (*ptr_char == RC_NL_CHAR)
                            {
                              /*
                                 RC_NL_CHAR is single `\~' quoted
                                            or double `\\~' quoted:
                                   replace QUOTE_CHAR by RC_NL_CHAR resp.,
                                   replace last QUOTE_CHAR by RC_NL_CHAR
                              */
                              if (s[len_s-1] == QUOTE_CHAR)
                                s[len_s-1] = *ptr_char;
                              else
                               {
                                 /*
                                    RC_NL_CHAR is not quoted `~':
                                      insert a real `\n'_NEWLINE character
                                 */
                                 s[len_s++] = '\n';
                                 /*
                                    Insert `j+len_fn_part+1' leading blanks
                                 */
                                 i_dummy = j + len_fn_part + 1;
                                 for (i=0 ; (i < i_dummy) && (len_s < MAXLEN) ; i++)
                                   s[len_s++] = ' ';
                               }
                            }
                           else
                             s[len_s++] = *ptr_char;
                           ptr_char++;
                         }
                        s[len_s] = '\0';
                      }
                   }
                  if (   rc_omit_date_flag
                      && same_date)
                   {
                     /*
                        Overwrite leading date part with blanks
                     */
                     if (rc_count_flag)
                       i = len_rce;
                     else
                       i = 0;
                     for ( ; i < j ; i++)
                       s[i] = ' ';
                     if (ok)
                      {
                        s[i] = '\0';
                        strcat(s, ptr_char);
                      }
                     else
                       if (   (ehls1s.len != 1)
                           && (   hls1_set
                               || hls2_set))
                        {
                          register int  hls_len=(hls1_set) ? hls1_len : hls2_len;


                          ptr_char = &s[j+hls_len];
                          for (i=j ; *ptr_char ; i++)
                            s[i] = *ptr_char++;
                          s[i] = '\0';
                        }
                   }
                  /*
                     Now display constructed fixed date warning message
                  */
                  print_text (stdout, s, INternal);
                }
               else
                 skip++;
               free(rc_table[line]);
               rc_table[line] = (char *)NULL;
             }
          }
       }
    }
   start_day = tmp_start_day;
   month = tmp_month;
   act_day = tmp_ad;
   act_month = tmp_am;
   act_year = tmp_ay;
}
#endif /* USE_RC */
