/*
*  gcal_prt.c:  Create, construct and print the calendar.
*
*
*  Copyright (C) 1994, 1995 Thomas Esken
*
*  This software doesn't claim completeness, correctness or usability.
*  On principle I will not be liable for any damages or losses (implicit
*  or explicit), which result from using or handling my software.
*  If you use this software, you agree without any exception to this
*  agreement, which binds you LEGALLY !!
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the `GNU General Public License' as published by
*  the `Free Software Foundation'; either version 2, or (at your option)
*  any later version.
*
*  You should have received a copy of the `GNU General Public License'
*  along with this program; if not, write to the:
*    Free Software Foundation
*    59 Temple Place, Suite 330
*    Boston, MA 02111-1307  USA
*/



#ifdef RCSID
static char rcsid[]="$Id: gcal_prt.c 0.38 1995/11/29 00:03:08 tom Exp $";
#endif



/*
*  Include header files
*/
#include "gcal_tai.h"
#include "gcal.h"



/*
*  Function prototypes
*/
#if __cplusplus
extern "C"
{
#endif
/*
************************************************** Defined in `gcal_hdy.c'
*/
IMPORT void
print_all_holidays __P_((      Bool init_data,
                         const Bool detect));
#if USE_RC
/*
************************************************** Defined in `gcal_rc.c'
*/
IMPORT Bool
jdate2sdate __P_((      int  jdate,
                  const int  is_leap_year,
                        int *day,
                        int *month));
/*
************************************************** Defined in `gcal_rcu.c'
*/
IMPORT void
rc_use __P_((void));
#endif /* USE_RC */
/*
************************************************** Defined in `gcal_tty.c'
*/
IMPORT void
print_text __P_((      FILE       *fp,
                       char       *txt_line,
                 const Dmode_enum  mode));
/*
************************************************** Defined in `gcal_utl.c'
*/
IMPORT Bool
get_actual_date __P_((void));
IMPORT const char *
short_day_name __P_((const int day));
IMPORT const char *
day_name __P_((const int day));
IMPORT const char *
month_name __P_((const int month));
IMPORT int
weekday_of_date __P_((const int day,
                      const int month,
                      const int year));
IMPORT int
day_of_year __P_((const int day,
                  const int month,
                  const int year));
IMPORT int
days_of_february __P_((const int year));
IMPORT Bool
valid_date __P_((const int day,
                 const int month,
                 const int year));
IMPORT int
weekno2jday __P_((      int week,
                  const int year));
/*
************************************************** Defined in `gcal_prt.c'
*/
EXPORT void
print_calendar __P_((void));
LOCAL void
fill_year_vector __P_((int year));
LOCAL Bool
print_single_date __P_((Bool marker_flag));
LOCAL Bool
print_highlighted_date __P_((const Bool  marker_flag,
                             const char *hls_start,
                             const int   hls_slen,
                             const char *hls_end,
                             const int   hls_elen));
#if __cplusplus
}
#endif



/*
*  Declare public(extern) variables
*/
IMPORT const int   dvec[MONTH_MAX];           /* Amount of days in months' */
IMPORT Ml_struct   month_list;                /* Used if month/year is a list/range */
IMPORT Hls_struct  ehls1s;                    /* Effective hls 1 start (current day) */
IMPORT Hls_struct  ehls1e;                    /* Effective hls 1 end (current day) */
IMPORT Hls_struct  ehls2s;                    /* Effective hls 2 start (holiday) */
IMPORT Hls_struct  ehls2e;                    /* Effective hls 2 end (holiday) */
IMPORT int         len_year_max;              /* String length of the maximum year able to compute */
IMPORT int         start_day;                 /* -s<0,1...7|day name> */
IMPORT int         day;                       /* Current day */
IMPORT int         month;                     /* Current month */
IMPORT int         year;                      /* Current year */
IMPORT int         act_day;                   /* Actual day */
IMPORT int         act_month;                 /* Actual month */
IMPORT int         act_year;                  /* Actual year */
IMPORT int         fiscal_month;              /* Starting month of a fiscal year */
IMPORT int         out_rows;                  /* Number of month rows of a year calendar */
IMPORT int         out_cols;                  /* Number of month columns of ... */
IMPORT int         fmt_len;                   /* Format lenght of a standard/julian/both day */
IMPORT int         is_leap_year;              /* Is current year a leap year? */
IMPORT int         holiday_vector[MONTH_MAX][MONTH_MAX]; /* Stores the holiday dates */
IMPORT char        s[MAXLEN+1];               /* General purpose text buffer */
IMPORT char        s2[MAXLEN+1];              /* General purpose text buffer */
#if USE_RC
IMPORT int         rc_period;                 /* Amount of period of fixed date warnings */
IMPORT Bool        rc_use_flag;               /* -c */
IMPORT Bool        rc_period_flag;            /* [-c]<<<<n>>[<d|w|+|-]>|[0]`mmdd'|[0]`mmww[w]'<n>> */
IMPORT Bool        rc_period_list;            /* [-c]l */
IMPORT Bool        rc_week_year_flag;         /* [-c<<n>>]w */
IMPORT Bool        rc_forwards_flag;          /* [-c<<n>|w|m|y>]+ */
IMPORT Bool        rc_backwards_flag;         /* [-c<<n>|w|m|y>]- */
IMPORT Bool        is_date_given;             /* Is a date given in the command line? */
IMPORT Bool        is_1month_mode;            /* [-c]<n>w and complete week is in month */
IMPORT Bool        is_2month_mode;            /* [-c]<n>w and only part of week is in month */
#endif
IMPORT Bool        special_calsheet_flag;     /* -i */
IMPORT Bool        highlight_flag;            /* -H<yes> or -H<no> */
IMPORT Bool        cal_julian_flag;           /* -j */
IMPORT Bool        cal_both_dates_flag;       /* -jb */
IMPORT Bool        holiday_flag;              /* -n|N */
IMPORT Bool        is_fiscal_year;            /* `:' char found in argument (`mm':`yyyy') */
IMPORT Bool        is_3month_mode;            /* Argument is "." or ".+" or ".-" */
IMPORT Bool        is_3month_mode2;           /* Argument is ".." -> current quarter of actual year */
IMPORT Bool        is_ext_year;               /* Is extended year mode? */
IMPORT Bool        is_ext_list;               /* Is extended list mode? */
IMPORT Bool        is_ext_range;              /* Is extended range mode? */



/*
   Define local(static) variables
*/
LOCAL int  year_vector[VEC_ELEMS];     /* Standard-/ Julian year dates */
LOCAL int  julian_vector[VEC_ELEMS];   /* Julian year dates only */



#ifdef ANSI_PROTO
PUBLIC void
print_calendar (void)
#else /* !ANSI_PROTO */
   PUBLIC void
print_calendar ()
#endif /* !ANSI_PROTO */
/*
   Prints one or more single month-/year calendar(s)
*/
{
   auto     Slint  amount=0L;
   auto     Slint  count;
   register int    tmp_ad=act_day;
   register int    i;
   register int    d;
   register int    m;
   register int    mm;
   register int    yy;
   register int    hday;
   register int    hmonth;
   register int    outer_end=(special_calsheet_flag) ? MONTH_COLS : DAY_MAX;
   register int    inner_end;
   register int    blanks_between=(special_calsheet_flag) ? 5 : 2;
   auto     Bool   is_marked=FALSE;
   auto     Bool   marker_flag=FALSE;
   auto     Bool   backwards=FALSE;


   if (cal_julian_flag)
     act_day = day_of_year (tmp_ad, act_month, act_year);
   /*
      Evaluate entries of `month_list':
        set the actual list/range/fiscal mode
        initialize the control variables of main loop
   */
   if (!is_ext_range)
     for (i=0 ; month_list.month[i] ; i++)
      {
        if (!month_list.year[i])
         {
           if (year != act_year)
             month_list.year[i] = year;
           else
             month_list.year[i] = act_year;
         }
        amount++;
      }
   else
    {
      if (!is_ext_year)
       {
         if (!*month_list.year)
           *month_list.year = act_year;
         if (!month_list.year[1])
           month_list.year[1] = act_year;
         backwards = (Bool)(   *month_list.year > month_list.year[1]
                            || (   (*month_list.year == month_list.year[1])
                                && (*month_list.month > month_list.month[1])));
         if (backwards)
           amount = (((*month_list.year - 1L) - month_list.year[1]) * MONTH_MAX)
                      + *month_list.month + ((MONTH_MAX - month_list.month[1]) + 1L);
         else
           amount = (((month_list.year[1] - 1L) - *month_list.year) * MONTH_MAX)
                      + month_list.month[1] + ((MONTH_MAX - *month_list.month) + 1L);
         month = *month_list.month;
         year = *month_list.year;
       }
      else
       {
         if (is_fiscal_year)
          {
            backwards = (Bool)(*month_list.year > month_list.year[1]);
            if (backwards)
              amount = (*month_list.year - month_list.year[1]) + 1L;
            else
              amount = (month_list.year[1] - *month_list.year) + 1L;
            year = *month_list.year;
          }
         else
          {
            backwards = (Bool)(*month_list.month > month_list.month[1]);
            if (backwards)
              amount = (*month_list.month - month_list.month[1]) + 1L;
            else
              amount = (month_list.month[1] - *month_list.month) + 1L;
            year = *month_list.month;
          }
       }
    }
   if (is_fiscal_year)
     fiscal_month = *month_list.month;
   if (!fiscal_month)
     fiscal_month = act_month;
   if (!amount)
     amount++;
#if USE_RC
   /*
      [-c]<n>w option and no explicit date given:
        set the correct month/year to display
   */
   if (   rc_week_year_flag
       && !rc_period_list
       && !is_date_given)
    {
      i = weekno2jday (rc_period, act_year);
      if (i != -WEEK_MAX)
       {
         if (   i < DAY_MIN
             || i+DAY_MAX-1 > DAY_LAST+is_leap_year)
          {
            is_2month_mode=is_fiscal_year = TRUE;
            *month_list.month=fiscal_month = MONTH_MAX;
            if (i < DAY_MIN)
              *month_list.year = --year;
          }
         else
          {
            (void)jdate2sdate (i, is_leap_year, &day, &month);
            m = month;
            (void)jdate2sdate (i+DAY_MAX-1, is_leap_year, &day, &month);
            if (m != month)
             {
               is_2month_mode=is_fiscal_year = TRUE;
               month=fiscal_month = m;
             }
            else
              is_1month_mode = TRUE;
            *month_list.month = month;
          }
         if (is_2month_mode)
          {
            if (cal_both_dates_flag)
             {
               out_rows = B2_OUT_ROWS;
               out_cols = B2_OUT_COLS;
             }
            else
             {
               out_rows = A2_OUT_ROWS;
               out_cols = A2_OUT_COLS;
             }
          }
       }
    }
#endif
   /*
      All necessary global initializations done,
        so enter loop
   */
   for (count=0L ; count < amount ; count++)
    {
      /*
         If loop must be processed multiple,
           so re-initialize the affected variables according to actual mode
      */
      mm=yy = 0;
      if (   !is_ext_list
          && !is_ext_range)
       {
         /*
            If a month calendar of current year only, resp.,
              a year calendar of current year only is wanted:
              initialize the affected variables
         */
         month = month_list.month[(int)count];
         if (month_list.year[(int)count])
           year = month_list.year[(int)count];
         if (count)
           yy = year;
#if USE_RC
         /*
            Modify actual date
         */
         if (   rc_period_flag
             && (   rc_forwards_flag
                 || rc_backwards_flag)
             && !rc_period_list
             && !is_date_given)
          {
            if (cal_julian_flag)
              i = act_day;
            else
              i = day_of_year (tmp_ad, act_month, act_year);
            if (rc_forwards_flag)
              i += rc_period;
            else
              i -= rc_period;
            if (   (i > 0)
                && (i < DAY_LAST+is_leap_year+1))
             {
               (void)jdate2sdate (i, is_leap_year, &day, &month);
               act_month = month;
               if (cal_julian_flag)
                 act_day = day_of_year (day, month, act_year);
               else
                 act_day = day;
             }
          }
#endif /* USE_RC */
       }
      else
        if (is_ext_list)
         {
           if (!is_ext_year)
             {
               month = month_list.month[(int)count];
               if (   count
                   && (year == month_list.year[(int)count]))
                 yy = year;
               else
                 year = month_list.year[(int)count];
             }
           else
            {
              month = 0;
              if (is_fiscal_year)
               {
                 if (   count
                     && (fiscal_month == month_list.month[(int)count])
                     && (year == month_list.year[(int)count]))
                  {
                    mm = fiscal_month;
                    yy = year;
                  }
                 else
                  {
                    fiscal_month = month_list.month[(int)count];
                    year = month_list.year[(int)count];
                  }
               }
              else
               {
                 if (   count
                     && (year == month_list.month[(int)count]))
                   yy = year;
                 else
                   year = month_list.month[(int)count];
               }
            }
         }
        else
          if (   is_ext_range
              && count)
            {
              if (!is_ext_year)
               {
                 yy = year;
                 if (backwards)
                  {
                    month--;
                    if (month < MONTH_MIN)
                      month = MONTH_MAX, year--;
                  }
                 else
                  {
                    month++;
                    if (month > MONTH_MAX)
                      month = MONTH_MIN, year++;
                  }
               }
              else
               {
                 month = 0;
                 if (backwards)
                   year--;
                 else
                   year++;
               }
            }
          else
            if (is_ext_year)
              month = 0;
      if (   yy != year
          || (   is_fiscal_year
              && (mm != fiscal_month)))
       {
         is_leap_year = (days_of_february (year) == 29);
         if (count)
           for (i=0 ; i < VEC_ELEMS ; i++)
             year_vector[i]=julian_vector[i] = 0;
         /*
            Build year calendar data structure (vector)
         */
         fill_year_vector (year);
         /*
            Get the actual dates of holidays, used for highlighting the calendar
         */
         if (   (   (year == EASTER_MIN-1)
                 && (fiscal_month > MONTH_MIN))
             || (   (year >= EASTER_MIN)
                 && (year <= EASTER_MAX)))
           print_all_holidays (TRUE, TRUE);
         else
           if (count)
             for (i=0 ; i < MONTH_MAX ; i++)
               holiday_vector[i][0] = 0;
       }
      else
        /*
           Get the actual dates of holidays, used for highlighting the calendar
        */
        if (   (   (year == EASTER_MIN-1)
                && (fiscal_month > MONTH_MIN))
            || (   (year >= EASTER_MIN)
                && (year <= EASTER_MAX)))
          print_all_holidays (TRUE, TRUE);
      *s = '\0';
      /*
         Ok, all necessary initializations done,
           so print a month calendar / year calendar of required year
      */
      if (   !is_ext_year
          && !is_fiscal_year
#if USE_RC
          && !is_2month_mode
#endif
          && (   month
              || is_ext_list
              || is_ext_range))
       {
         /*
            Print the month calendar and if needed:
              fixed dates related to the month
         */
         print_text (stdout, s, CAlendar);
         if (special_calsheet_flag)
          {
            /*
               Print the month calendar in the special format
            */
            /*
               Initialize the terminating value of inner loop
            */
            inner_end = DAY_MAX;
            /*
               Print the month header in a centered manner
            */
            strcpy(s2, month_name (month));
            i = ((fmt_len * DAY_MAX) >> 1)
                + ((strlen(s2) + len_year_max + 1) >> 1) - len_year_max;
            sprintf(s, "%*s %0*d", i, month_name (month), len_year_max, year);
            print_text (stdout, s, CAlendar);
            /*
               Print the day names
            */
            for (i=DAY_MIN ; i <= DAY_MAX ; i++)
             {
               sprintf(s2, "%*s", fmt_len, short_day_name (SDAY(i, start_day)));
               strcat(s, s2);
             }
            print_text (stdout, s, CAlendar);
          }
         else
          {
            /*
               Print the month calendar in standard format
            */
            /*
               Initialize the terminating value of inner loop
            */
            inner_end = MONTH_COLS;
            /*
               Print the month header
            */
            sprintf(s, "%s %0*d", month_name (month), len_year_max, year);
            print_text (stdout, s, CAlendar);
            print_text (stdout, s, CAlendar);
          }
         /*
            Print the days
         */
         for (i=1 ; i <= outer_end ; i++)
          {
            /*
               Print the day name
            */
            if (!special_calsheet_flag)
#if USE_GER
              sprintf(s, "%-*s", 10+blanks_between,
                      day_name (SDAY(i, start_day)));
#else /* !USE_GER */
              sprintf(s, "%-*s", 9+blanks_between,
                      day_name (SDAY(i, start_day)));
#endif /* !USE_GER */
            for (d=1 ; d <= inner_end ; d++)
             {
               /*
                  Compute the day's position in `year_vector'
               */
               if (special_calsheet_flag)
                 day = (month - 1) * VEC_BLOCK - 1
                       + (i - 1) * DAY_MAX + d;
               else
                 day = (month - 1) * VEC_BLOCK - 1
                       + d * DAY_MAX - MONTH_COLS + i - 1;
               /*
                  Is `day' the actual day?
               */
               if (   highlight_flag
                   && (year_vector[day] == act_day)
                   && (month == act_month)
                   && (year == act_year))
                {
                  if (   is_marked
                      && (ehls1s.len == 1))
                    marker_flag = print_highlighted_date (is_marked,
                                                          "", 0,
                                                          ehls1e.seq, ehls1e.len);
                  else
                   {
                     marker_flag = print_highlighted_date (is_marked,
                                                           ehls1s.seq, ehls1s.len,
                                                           ehls1e.seq, ehls1e.len);
                     is_marked = TRUE;
                   }
                }
               else
                {
                  /*
                     Is `day' a holiday?
                  */
                  hday = 0;
                  if (   year_vector[day]
                      && *holiday_vector[month-1])
                   {
                     register int  k;


                     for (k=0 ; holiday_vector[month-1][k] ; k++)
                       if (holiday_vector[month-1][k] == year_vector[day])
                        {
                          hday = holiday_vector[month-1][k];
                          break;
                        }
                   }
                  if (   hday
                      && highlight_flag)
                   {
                     /*
                        `day' is a holiday!
                     */
                     if (   is_marked
                         && (ehls2s.len == 1))
                       marker_flag = print_highlighted_date (is_marked,
                                                             "", 0,
                                                             ehls2e.seq, ehls2e.len);
                     else
                      {
                        marker_flag = print_highlighted_date (is_marked,
                                                              ehls2s.seq, ehls2s.len,
                                                              ehls2e.seq, ehls2e.len);
                        is_marked = TRUE;
                      }
                   }
                  else
                   {
                     /*
                        `day' is no holiday!
                     */
                     marker_flag = print_single_date (marker_flag);
                     is_marked = FALSE;
                   }
                }

             }
            /*
               Print the constructed line
            */
            print_text (stdout, s, CAlendar);
            is_marked=marker_flag = FALSE;
          }
#if USE_RC
         /*
            Print fixed date texts
         */
         if (   rc_use_flag
             && (   is_ext_list
                 || is_ext_range
                 || amount > 1))
           rc_use ();
#endif
         /*
            Print eternal holiday list
         */
         if (   holiday_flag
             && (   is_ext_list
                 || is_ext_range
                 || amount > 1)
             && (   (   (year == EASTER_MIN-1)
                     && (fiscal_month > MONTH_MIN))
                 || (   (year >= EASTER_MIN)
                     && (year <= EASTER_MAX))))
           print_all_holidays (FALSE, FALSE);
       }
      else
       {
         /*
           Print the year calendar and if needed:
             fixed dates related to the year
             eternal holidays related to the year
         */
         print_text (stdout, s, CAlendar);
         print_text (stdout, s, CAlendar);
         if (special_calsheet_flag)
          {
            /*
               Print the year calendar in the special format
            */
            /*
               Initialize the terminating value of inner loop
            */
            inner_end = DAY_MAX * out_cols;
            /*
               Print the year header
            */
            d = ((out_cols - 1) * blanks_between + out_cols * fmt_len * DAY_MAX) >> 1;
            if (   (   !is_3month_mode
#if USE_RC
                    && !is_2month_mode
#endif
                    && !is_3month_mode2
                    && (fiscal_month > MONTH_MIN)
                    && (year+1 <= YEAR_MAX))
#if USE_RC
                || (   is_2month_mode
                    && (fiscal_month == MONTH_MAX))
#endif
                || (   is_3month_mode
                    && (fiscal_month >= MONTH_MAX-1)))
              sprintf(s, "%*s%0*d/%0*d", d-len_year_max, "",
                      len_year_max, year, len_year_max, year+1);
            else
              sprintf(s, "%*s%0*d", d-(len_year_max>>1), "", len_year_max, year);
          }
         else
          {
            /*
               Print the year calendar in standard format
            */
            /*
               Initialize the terminating value of inner loop
            */
            inner_end = MONTH_COLS * out_cols;
            /*
               Print the year header
            */
            d = (out_cols * fmt_len * MONTH_COLS + 2 + blanks_between) >> 1;
            if (   (   !is_3month_mode
#if USE_RC
                    && !is_2month_mode
#endif
                    && !is_3month_mode2
                    && (fiscal_month > MONTH_MIN)
                    && (year+1 <= YEAR_MAX))
#if USE_RC
                || (   is_2month_mode
                    && (fiscal_month == MONTH_MAX))
#endif
                || (   is_3month_mode
                    && (fiscal_month >= MONTH_MAX-1)))
              sprintf(s, "%*s%0*d/%0*d", d-len_year_max, "",
                      len_year_max, year, len_year_max, year+1);
            else
              sprintf(s, "%*s%0*d", d-(len_year_max>>1), "", len_year_max, year);
          }
         print_text (stdout, s, CAlendar);
         print_text (stdout, s, CAlendar);
         print_text (stdout, s, CAlendar);
         for (m=0 ; m < out_rows ; m++)
          {
            if (special_calsheet_flag)
             {
               /*
                  Print the month header in a centered manner
               */
               for (i=1 ; i <= out_cols ; i++)
                {
                  strcpy(s2, month_name (SMONTH(m*out_cols+i, fiscal_month)));
                  d = ((fmt_len * DAY_MAX) >> 1) + (strlen(s2) >> 1);
                  sprintf(s2, "%*s", d, month_name (SMONTH(m*out_cols+i, fiscal_month)));
                  strcat(s, s2);
                  if (i != out_cols)
                   {
                     sprintf(s2, "%*s", (fmt_len*DAY_MAX)-(d-blanks_between), "");
                     strcat(s, s2);
                   }
                }
               print_text (stdout, s, CAlendar);
               /*
                  Print the day names
               */
               for (i=1 ; i <= out_cols ; i++)
                {
                  for (d=DAY_MIN ; d <= DAY_MAX ; d++)
                   {
                     sprintf(s2, "%*s", fmt_len, short_day_name (SDAY(d, start_day)));
                     strcat(s, s2);
                   }
                  if (i != out_cols)
                   {
                     sprintf(s2, "%*s", blanks_between, "");
                     strcat(s, s2);
                   }
                }
             }
            else
             {
               /*
                  Print the month header
               */
               for (i=1 ; i <= out_cols ; i++)
                {
                  strcpy(s2, month_name (SMONTH(m*out_cols+i, fiscal_month)));
                  d = (int)strlen(s2);
                  sprintf(s2, "%*s%s", 2+blanks_between+1, "",
                          month_name (SMONTH(m*out_cols+i, fiscal_month)));
                  strcat(s, s2);
                  if (i != out_cols)
                   {
                     sprintf(s2, "%*s", (fmt_len*MONTH_COLS)-(d+2+blanks_between+1), "");
                     strcat(s, s2);
                   }
                }
               print_text (stdout, s, CAlendar);
             }
            print_text (stdout, s, CAlendar);
            /*
               Print the days
            */
            for (i=1 ; i <= outer_end ; i++)
             {
               /*
                  Print the day name
               */
               if (!special_calsheet_flag)
                 sprintf(s, "%-*s", 2+blanks_between,
                         short_day_name (SDAY(i, start_day)));
               for (d=1 ; d <= inner_end ; d++)
                {
                  /*
                     Compute the day's position in `year_vector'
                  */
                  if (special_calsheet_flag)
                    day = m * out_cols * VEC_BLOCK - 1
                          + (((d - 1) / DAY_MAX) * VEC_BLOCK)
                          + ((i - 1) * DAY_MAX) + ((d - 1) % DAY_MAX) + 1;
                  else
                    day = m * out_cols * VEC_BLOCK - 1
                          + (d * DAY_MAX - MONTH_COLS) + (i - 1);
                  hday = (m * out_cols) + ((d-1)
                         / ((special_calsheet_flag) ? DAY_MAX : MONTH_COLS)) + 1;
                  hmonth = SMONTH(hday, fiscal_month);
                  /*
                     Is `day' the actual day?
                  */
                  if (   highlight_flag
                      && (year_vector[day] == act_day)
                      && (hmonth == act_month)
                      && (act_year == ((hday>hmonth) ? year+1 : year)))
                   {
                     if (   is_marked
                         && (ehls1s.len == 1))
                       marker_flag = print_highlighted_date (is_marked,
                                                             "", 0,
                                                             ehls1e.seq, ehls1e.len);
                     else
                      {
                        marker_flag = print_highlighted_date (is_marked,
                                                              ehls1s.seq, ehls1s.len,
                                                              ehls1e.seq, ehls1e.len);
                        is_marked = TRUE;
                      }
                   }
                  else
                   {
                     /*
                        Is `day' a holiday?
                     */
                     hday = 0;
                     if (   year_vector[day]
                         && *holiday_vector[hmonth-1])
                      {
                        register int  k;


                        for (k=0 ; holiday_vector[hmonth-1][k] ; k++)
                          if (holiday_vector[hmonth-1][k] == year_vector[day])
                           {
                             hday = holiday_vector[hmonth-1][k];
                             break;
                           }
                      }
                     if (   hday
                         && highlight_flag)
                      {
                        /*
                           `day' is a holiday!
                        */
                        if (   is_marked
                            && (ehls2s.len == 1))
                          marker_flag = print_highlighted_date (is_marked,
                                                                "", 0,
                                                                ehls2e.seq, ehls2e.len);
                        else
                         {
                           marker_flag = print_highlighted_date (is_marked,
                                                                 ehls2s.seq, ehls2s.len,
                                                                 ehls2e.seq, ehls2e.len);
                           is_marked = TRUE;
                         }
                      }
                     else
                      {
                        /*
                           `day' is no holiday!
                        */
                        marker_flag = print_single_date (marker_flag);
                        is_marked = FALSE;
                      }
                   }
                  /*
                     If we have finished a day line of a month:
                       concatenate some separating blanks between the months
                  */
                  if (   special_calsheet_flag
                      && !(d % DAY_MAX)
                      && (d != DAY_MAX * out_cols))
                   {
                     sprintf(s2, "%*s", blanks_between, "");
                     strcat(s, s2);
                   }
                }
               /*
                  Print the constructed line
               */
               print_text (stdout, s, CAlendar);
               is_marked=marker_flag = FALSE;
             }
            /*
               Print two newlines between month rows
            */
            if (m < out_rows-1)
             {
               print_text (stdout, s, CAlendar);
               print_text (stdout, s, CAlendar);
             }
          }
#if USE_RC
         /*
            Print fixed date texts
         */
         if (   rc_use_flag
             && (   is_ext_list
                 || is_ext_range))
           rc_use ();
#endif
         /*
            Print eternal holiday list
         */
         if (   is_ext_year
             && holiday_flag
             && (   (   (year == EASTER_MIN-1)
                     && (fiscal_month > MONTH_MIN))
                 || (   (year >= EASTER_MIN)
                     && (year <= EASTER_MAX))))
           print_all_holidays (FALSE, FALSE);
       }
    }
#if USE_RC
   /*
      [-c]<n>w option and no explicit date given:
        reset some affected global variables
   */
   if (   is_1month_mode
       || is_2month_mode)
    {
      is_fiscal_year = TRUE;
      fiscal_month = month;
      year = act_year;
    }
   else
     /*
        Reeinitialize actual date
     */
     if (   rc_period_flag
         && (   rc_forwards_flag
             || rc_backwards_flag)
         && !rc_period_list
         && !is_date_given)
       (void)get_actual_date ();
     else
#endif /* USE_RC */
       if (cal_julian_flag)
         act_day = tmp_ad;
}



#ifdef ANSI_PROTO
LOCAL void
fill_year_vector (int year)
#else /* !ANSI_PROTO */
   LOCAL void
fill_year_vector (year)
   int year;
#endif /* !ANSI_PROTO */
/*
   Builds the "standard" year vector and the "julian" year vector
     according to delivered `year'
*/
{
#if !USE_RC
   register int  days_of_feb=days_of_february (year);
#endif
   register int  i;
   register int  j;
   register int  count;
   register int  d;
   register int  m=fiscal_month;


   /*
      First, detect starting day of the fiscal-/ standard year
   */
   i = weekday_of_date (DAY_MIN, m, year);
   i=day = SYEAR(i, start_day);
   j=d = 0;
   if (fiscal_month > MONTH_MIN)
     count = day_of_year (DAY_MIN, m, year) - 1;
   else
     count = 0;
   /*
      Then, fill the year_vector
   */
   LOOP
    {
      d++;
      count++;
#if USE_RC
      if (!valid_date (d, m, year))
#else /* !USE_RC */
      if (   (   (m != 2)
              && (d > dvec[m-1]))
          || (   (m == 2)
              && (d > days_of_feb)))
#endif /* !USE_RC */
       {
         j++;
         if (   (m < MONTH_MAX)
             && (j < MONTH_MAX))
          {
            if (fiscal_month > MONTH_MIN)
              i = j * VEC_BLOCK + day;
            else
              i = m * VEC_BLOCK + day;
            d = DAY_MIN;
            m++;
          }
         else
          {
            if (fiscal_month > MONTH_MIN)
             {
               year++;
               if (   j == MONTH_MAX
                   || year > YEAR_MAX)
                 break;
               i = j * VEC_BLOCK + day;
               count=d = DAY_MIN;
               m = MONTH_MIN;
#if !USE_RC
               days_of_feb = days_of_february (year);
#endif
             }
            else
              break;
          }
       }
      if (   (year == GREG_YEAR)
          && (m == GREG_MONTH)
          && (   (d >= GREG_F_DAY)
              && (d <= GREG_L_DAY)))
       {
         i--;
         count--;
       }
      else
       {
         day++;
         if (cal_julian_flag)
           year_vector[i-1] = count;
         else
           year_vector[i-1] = d;
         julian_vector[i-1] = count;
       }
      i++;
      if (day > DAY_MAX)
        day = DAY_MIN;
    }
}



#ifdef ANSI_PROTO
LOCAL Bool
print_single_date (Bool marker_flag)
#else /* !ANSI_PROTO */
   LOCAL Bool
print_single_date (marker_flag)
   Bool marker_flag;
#endif /* !ANSI_PROTO */
/*
   Like the function name says
*/
{
   if (cal_both_dates_flag)
    {
      if (year_vector[day])
       {
         if (marker_flag)
          {
            sprintf(s2, "%2d(%3d)", year_vector[day], julian_vector[day]);
            marker_flag = FALSE;
          }
         else
           sprintf(s2, "%3d(%3d)", year_vector[day], julian_vector[day]);
       }
      else
       {
         if (marker_flag)
          {
            sprintf(s2, "%*s", fmt_len-1, "");
            marker_flag = FALSE;
          }
         else
           sprintf(s2, "%*s", fmt_len, "");
       }
    }
   else
    {
      if (year_vector[day])
       {
         if (marker_flag)
          {
            sprintf(s2, "%*d", fmt_len-1, year_vector[day]);
            marker_flag = FALSE;
          }
         else
           sprintf(s2, "%*d", fmt_len, year_vector[day]);
       }
      else
       {
         if (marker_flag)
          {
            sprintf(s2, "%*s", fmt_len-1, "");
            marker_flag = FALSE;
          }
         else
           sprintf(s2, "%*s", fmt_len, "");
       }
    }
   strcat(s, s2);

   return(marker_flag);
}



#ifdef ANSI_PROTO
LOCAL Bool
print_highlighted_date (const Bool  marker_flag,
                        const char *hls_start,
                        const int   hls_slen,
                        const char *hls_end,
                        const int   hls_elen)
#else /* !ANSI_PROTO */
   LOCAL Bool
print_highlighted_date (marker_flag, hls_start, hls_slen, hls_end, hls_elen)
   const Bool  marker_flag;
   const char *hls_start;
   const int   hls_slen;
   const char *hls_end;
   const int   hls_elen;
#endif /* !ANSI_PROTO */
/*
   Like the function name says
*/
{
   if (cal_both_dates_flag)
     sprintf(s2, "%s%s%2d(%3d)%s%s",
             ((hls_slen>1)&&!marker_flag) ? NO_HLS : "", hls_start,
             year_vector[day], julian_vector[day],
             hls_end, (hls_elen>1) ? NO_HLS : "");
   else
     sprintf(s2, "%s%s%*d%s%s",
             ((hls_slen>1)&&!marker_flag) ? NO_HLS : "", hls_start,
             fmt_len-1, year_vector[day],
             hls_end, (hls_elen>1) ? NO_HLS : "");
   strcat(s, s2);

   return(TRUE);
}
