/*
*  gcal_fil.c:  Managing and accessing resource, include and response files
*
*
*  Copyright (C) 1994, 1995 Thomas Esken
*
*  This software doesn't claim completeness, correctness or usability.
*  On principle I will not be liable for any damages or losses (implicit
*  or explicit), which result from using or handling my software.
*  If you use this software, you agree without any exception to this
*  agreement, which binds you LEGALLY !!
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the `GNU General Public License' as published by
*  the `Free Software Foundation'; either version 2, or (at your option)
*  any later version.
*
*  You should have received a copy of the `GNU General Public License'
*  along with this program; if not, write to the:
*    Free Software Foundation
*    59 Temple Place, Suite 330
*    Boston, MA 02111-1307  USA
*/



#ifdef RCSID
static char rcsid[]="$Id: gcal_fil.c 0.38 1995/11/29 00:03:08 tom Exp $";
#endif



/*
*  Include header files
*/
#include "gcal_tai.h"
#if HAVE_ASSERT_H
#  include <assert.h>
#endif
#if HAVE_CTYPE_H
#  include <ctype.h>
#endif
#ifdef GCAL_SHELL
#  if HAVE_UNISTD_H
#    include <unistd.h>
#  endif
#endif
#include "gcal.h"



/*
*  Function prototypes
*/
#if __cplusplus
extern "C"
{
#endif
#if USE_RC
/*
************************************************** Defined in `gcal.c'
*/
IMPORT int
eval_longopt __P_((char *longopt,
                   int  *longopt_symbolic));
/*
************************************************** Defined in `gcal_rc.c'
*/
IMPORT Bool
set_dvar __P_((const char      *line_buffer,
               const char      *filename,
               const int        line,
               const Dvar_enum  mode));
#endif
/*
************************************************** Defined in `gcal_tty.c'
*/
IMPORT void
print_text __P_((      FILE       *fp,
                       char       *txt_line,
                 const Dmode_enum  mode));
/*
************************************************** Defined in `gcal_utl.c'
*/
IMPORT VOID_PTR
my_malloc __P_((const int   amount,
                const int   exit_status,
                const char *module_name,
                const int   module_line,
                const char *var_name,
                const int   var_contents));
IMPORT VOID_PTR
my_realloc __P_((      VOID_PTR  ptr_memblock,
                 const int       amount,
                 const int       exit_status,
                 const char     *module_name,
                 const int       module_line,
                 const char     *var_name,
                 const int       var_contents));
IMPORT void
my_error __P_((const int   exit_status,
               const char *module_name,
               const int   module_line,
               const char *var_name,
               const int   var_contents));
IMPORT int
days_of_february __P_((const int year));
/*
************************************************** Defined in `gcal_fil.c'
*/
EXPORT FILE *
file_open __P_((      char       *filename,
                const Fmode_enum  mode));
EXPORT char *
file_read_line __P_((      FILE       *fp,
                           char       *line_buffer,
                           int        *in_pool,
                           char       *pool,
                           char       *ptr_pool,
                     const char       *filename,
                           int        *line,
                     const Fmode_enum  mode,
                           Bool       *is_include,
                           Bool       *is_dvar));
EXPORT void
insert_response_file __P_((      char *filename,
                           const char *opt_list,
                                 int  *my_argc_max,
                                 int  *my_argc,
                                 char *my_argv[]));
EXPORT void
write_log_file __P_((const char       *filename,
                     const Fmode_enum  mode,
                     const char       *mode_msg,
                     const int         argc,
                           char       *argv[]));
LOCAL void
make_absolute_filename __P_((const char *directory,
                             const char *filename));
LOCAL FILE *
get_file_ptr __P_((      FILE       *fp,
                   const char       *filename,
                   const Fmode_enum  mode,
                         Bool       *is_first));
#if __cplusplus
}
#endif



/*
*  Declare public(extern) variables
*/
IMPORT const int    dvec[MONTH_MAX];           /* Amount of days in months */
IMPORT Hls_struct   ehls1s;                    /* Effective hls 1 start (current day) */
IMPORT Hls_struct   ehls1e;                    /* Effective hls 1 end (current day) */
IMPORT Hls_struct   ehls2s;                    /* Effective hls 2 start (holiday) */
IMPORT Hls_struct   ehls2e;                    /* Effective hls 2 end (holiday) */
#if HAVE_ASSERT_H
IMPORT Uint         testval;                   /* Set to INT_MAX for checking the maximum table range */
#endif
IMPORT int          warning_level;             /* --debug[=0...WARN_LVL_MAX] */
IMPORT int          year;                      /* Current year */
IMPORT int          act_sec;                   /* Actual second */
IMPORT int          act_min;                   /* Actual minute */
IMPORT int          act_hour;                  /* Actual hour */
IMPORT int          buf_ad;                    /* Buffer of actual day */
IMPORT int          buf_am;                    /* Buffer of actual month */
IMPORT int          buf_ay;                    /* Buffer of actual year */
IMPORT char        *prgr_name;                 /* Stores the actual program name */
IMPORT char         s[MAXLEN+1];               /* General purpose text buffer */
IMPORT char         s2[MAXLEN+1];              /* General purpose text buffer */



/*
   Define local(static) variables
*/
#if USE_RC
LOCAL char  str[MAXLEN+1];   /* General purpose text buffer */
#endif



#ifdef ANSI_PROTO
PUBLIC FILE *
file_open (      char       *filename,
           const Fmode_enum  mode)
#else /* !ANSI_PROTO */
   PUBLIC FILE *
file_open (filename, mode)
         char       *filename;
   const Fmode_enum  mode;
#endif /* !ANSI_PROTO */
/*
   Tries to open a resource/response (mode `REsource', `REsponse' or `USr_include')
     file in following order:
       1) actual directory
       2) $HOME
       3) $PATH(DOS) resp., $DPATH(OS/2) resp., $GCALPATH(UN*X and others)
       4) GCAL_USR_LIBDIR (environment variable first; if unset then default name)
       5) GCAL_SYS_LIBDIR (environment variable first; if unset then default name)
     if success, return a file pointer to this file and it's name,
     otherwise return a NULL pointer.
   If mode is set to `SYs_include', this function tries to open an
     include file (#include <file> directive found) only in:
       1) GCAL_USR_LIBDIR (environment variable first; if unset then default name)
       2) GCAL_SYS_LIBDIR (environment variable first; if unset then default name)
     and returns a file pointer to this file and it's name,
     otherwise a NULL pointer.
   If mode is set to `COmmon', this function tries to open a
     common file in
       1) $PATH
     and returns a file pointer to this file and it's name,
     otherwise return a NULL pointer. NO informational messages will
     be emitted in this mode!!
*/
{
   auto FILE   *fp=(FILE *)NULL;
   auto char   *ptr_env;
   auto char   *ptr_char;
#if USE_RC
   auto Bool    is_first=(Bool)((mode==REsource) ? TRUE : FALSE);
#else /* !USE_RC */
   auto Bool    is_first=FALSE;
#endif /* !USE_RC */


   strncpy(s, filename, MAXLEN+1);
   s[MAXLEN] = '\0';
#if USE_RC
   if (mode != SYs_include)
#endif
    {
      if (mode != COmmon)
       {
         /*
            Try to open file (in actual directory)
         */
         fp = get_file_ptr (fp, filename, mode, &is_first);
         /*
            Absolute or relative file name delivered:
              return to caller immediately
         */
         if (strchr(filename, *DIR_SEP) != (char *)NULL)
          {
#if USE_RC
            if (   (fp == (FILE *)NULL)
                && (warning_level >= WARN_LVL_MAX))
              /*
                 Error, absolute file name not found
              */
              my_error (118, filename, 0, filename, 0);
#endif /* USE_RC */
            return(fp);
          }
       }
#if !defined(AMIGA) || defined(__GNUC__)
      if (mode != COmmon)
       {
         /*
            Simple file name delivered and file not found in actual directory:
              search for file
         */
         if (fp == (FILE *)NULL)
          {
            /*
               File not found in actual directory:
                 search file in the directory, which is
                 stored in the environment variable:  HOME
            */
            ptr_env = getenv(ENV_VAR_HOME);
            if (ptr_env != (char *)NULL)
              if (*ptr_env)
               {
                 make_absolute_filename (ptr_env, filename);
                 fp = get_file_ptr (fp, s, mode, &is_first);
               }
          }
       }
      if (fp == (FILE *)NULL)
       {
         /*
            File not found in HOME directory:
              search file in the directory(s), which are stored
              in the environment variable:  [[D]|[GCAL]]PATH
         */
         if (mode == COmmon)
           ptr_env = getenv(ENV_VAR_PATH);
         else
           ptr_env = getenv(ENV_VAR_DPATH);
         if (ptr_env != (char *)NULL)
           if (*ptr_env)
            {
              register int    i;
              auto     Bool   ok=FALSE;


              strncpy(s2, ptr_env, MAXLEN+1);
              s2[MAXLEN] = '\0';
              while (   !ok
                     && (fp == (FILE *)NULL))
               {
#  ifdef SUFFIX_SEP
                 if ((ptr_char=strchr(s2, *SUFFIX_SEP)) != (char *)NULL)
                   *ptr_char = '\0';
#  endif
                 ok = (Bool)((ptr_char=strchr(s2, *PATH_SEP)) == (char *)NULL);
                 if (ok)
                   i = (int)strlen(s2);
                 else
                   i = (int)strlen(s2) - strlen(ptr_char);
                 strcpy(s, s2);
                 s[i] = '\0';
                 ptr_char = strrchr(s, *DIR_SEP);
                 if (ptr_char != (char *)NULL)
                   if (strlen(ptr_char) > 1)
                     strcat(s, DIR_SEP);
                 strcat(s, filename);
                 if (*s)
                   fp = get_file_ptr (fp, s, mode, &is_first);
                 if (ok)
                   break;
                 ptr_char = s2 + i + 1;
                 strcpy(s2, ptr_char);
                 if (*s2 == '\0')
                   ok = TRUE;
               }
            }
       }
    }
#  if USE_RC
   if (mode != COmmon)
    {
      if (fp == (FILE *)NULL)
       {
         /*
            File not found in one of the directories, which are
            stored in the environment variable [[D]|[GCAL]]PATH:
              search file in user library directory --> $HOME/GCAL_USR_LIBDIR
         */
         ptr_env = getenv(ENV_VAR_HOME);
         if (ptr_env != (char *)NULL)
           if (*ptr_env)
            {
              ptr_char = getenv(ENV_VAR_USR_LIBDIR);
              /*
                 Search file in directory specified in GCAL_USR_LIBDIR
                   environment variable first
              */
              if (ptr_char != (char *)NULL)
               {
                 if (*ptr_char)
                  {
                    make_absolute_filename (ptr_env, ptr_char);
                    strcpy(s2, s);
                    make_absolute_filename (s2, filename);
                    fp = get_file_ptr (fp, s, mode, &is_first);
                  }
               }
              /*
                 If GCAL_USR_LIBDIR environment variable unset,
                   search file in burned-in user library directory --> GCAL_USR_LIBDIR
              */
              if (fp == (FILE *)NULL)
               {
                 make_absolute_filename (ptr_env, GCAL_USR_LIBDIR);
                 strcpy(s2, s);
                 make_absolute_filename (s2, filename);
                 fp = get_file_ptr (fp, s, mode, &is_first);
              }
            }
       }
      if (fp == (FILE *)NULL)
       {
         /*
            File not found in user library directory $HOME/GCAL_USR_LIBDIR:
              search file in system library directory --> GCAL_SYS_LIBDIR
         */
         ptr_env = getenv(ENV_VAR_SYS_LIBDIR);
         /*
            Search file in directory specified in GCAL_SYS_LIBDIR
              environment variable first
         */
         if (ptr_env != (char *)NULL)
          {
            if (*ptr_env)
             {
               make_absolute_filename (ptr_env, filename);
               fp = get_file_ptr (fp, s, mode, &is_first);
             }
          }
       }
    }
#  endif /* USE_RC */
#else /* AMIGA && !__GNUC__ */
#  if USE_RC
   if (mode != COmmon)
    {
      /*
         This part is for compilers/systems which don't have a getenv()-function
      */
      if (fp == (FILE *)NULL)
       {
         /*
            File not found in actual directory:
              search file in burned-in user library directory --> GCAL_SYS_LIBDIR
         */
         make_absolute_filename (GCAL_USR_LIBDIR, filename);
         fp = get_file_ptr (fp, s, mode, &is_first);
       }
    }
#  endif /* USE_RC */
#endif /* AMIGA && !__GNUC__ */
#if USE_RC
   if (mode != COmmon)
    {
      if (fp == (FILE *)NULL)
       {
         /*
            File not found in user library directory GCAL_USR_LIBDIR:
              search file in burned-in system library directory --> GCAL_SYS_LIBDIR
         */
         make_absolute_filename (GCAL_SYS_LIBDIR, filename);
         fp = get_file_ptr (fp, s, mode, &is_first);
       }
    }
#endif /* USE_RC */
   /*
      If file found:
        copy real (absolute) name of file to `filename'
   */
   if (fp != (FILE *)NULL)
     strcpy(filename, s);
#if USE_RC
   else
     if (   (warning_level >= WARN_LVL_MAX)
         && (mode != COmmon))
       /*
          Terminate if --debug=abort option is given
       */
       my_error (118, filename, 0, filename, 0);
#endif

   return(fp);
}



#ifdef ANSI_PROTO
PUBLIC char *
file_read_line (      FILE       *fp,
                      char       *line_buffer,
                      int        *in_pool,
                      char       *pool,
                      char       *ptr_pool,
                const char       *filename,
                      int        *line,
                const Fmode_enum  mode,
                      Bool       *is_include,
                      Bool       *is_dvar)
#else /* !ANSI_PROTO */
   PUBLIC char *
file_read_line (fp, line_buffer, in_pool, pool, ptr_pool, filename, line, mode, is_include, is_dvar)
         FILE       *fp;
         char       *line_buffer;
         int        *in_pool;
         char       *pool;
         char       *ptr_pool;
   const char       *filename;
         int        *line;
   const Fmode_enum  mode;
         Bool       *is_include;
         Bool       *is_dvar;
#endif /* !ANSI_PROTO */
/*
   Reads a line of delivered resource/response file into `line_buffer'
     using delivered char vector `pool', which must be allocated by caller
     with size BUF_LEN+1 (BUF_LEN should be a power of 2, e.g., 4096).
     returns the position in buffer of last managed character by char pointer
     `ptr_pool', which must be defined by caller; or NULL if EOF is detected.
*/
{
   register int    i;
   auto     char  *ptr_char=line_buffer;
   auto     char   ch;
   auto     Bool   is_error=FALSE;



   *is_include=*is_dvar = FALSE;
   /*
      Initial fill/refill `pool'
   */
   if (!*in_pool)
    {
      *in_pool = read(fileno(fp), (char *)pool, BUF_LEN);
      if (!*in_pool)
        /*
           At end of file
        */
        return((char *)NULL);
      else
        if (*in_pool < 0)
          /*
             File read error
          */
          my_error (109, filename, 0, "", 0);
      ptr_pool = pool;
    }
   if (*ptr_pool == '\n')
     (*line)++;
   while (   (*ptr_pool != REM_CHAR)
          && (
#if USE_RC
                 (   (mode == REsource)
                  && (*ptr_pool != *RC_INCL_STMENT)
                  && (*ptr_pool != RC_HDY_CHAR)
                  && (*ptr_pool != RC_NWD_CHAR)
                  && !isalnum(*ptr_pool))
              ||
#endif
                 (   (mode == REsponse)
                  && (*ptr_pool != *MONTH3_LIT)
                  && (*ptr_pool != *FYEAR_SEP)
                  && (*ptr_pool != RSP_CHAR)
                  && !isalnum(*ptr_pool)
#if USE_RC
                  && (*ptr_pool != RC_ADATE_CHAR)
#endif
                  && (*ptr_pool != *SWITCH)
                  && (*ptr_pool != *SWITCH2))))
    {
      ptr_pool++;
      (*in_pool)--;
      if (!*in_pool)
       {
         /*
            Refill `pool', because the line we work on isn't complete
         */
         *in_pool = read(fileno(fp), (char *)pool, BUF_LEN);
         if (!*in_pool)
           /*
              At end of file
           */
           return((char *)NULL);
         else
           if (*in_pool < 0)
             /*
                File read error
             */
             my_error (109, filename, 0, "", 0);
         ptr_pool = pool;
       }
      if (*ptr_pool == '\n')
        (*line)++;
    }
   (*line)++;
   /*
      Skip whole line
   */
   if (*ptr_pool == REM_CHAR)
    {
      /*
         Read until NEWLINE or EOF
      */
      while (*ptr_pool != '\n')
       {
         /*
            Refill `pool', because the line we work on isn't complete
         */
         if (!*in_pool)
          {
            *in_pool = read(fileno(fp), (char *)pool, BUF_LEN);
            if (!*in_pool)
              /*
                 At end of file
              */
              return((char *)NULL);
            else
              if (*in_pool < 0)
                /*
                   File read error
                */
                my_error (109, filename, 0, "", 0);
            ptr_pool = pool;
          }
         else
          {
            ptr_pool++;
            (*in_pool)--;
          }
       }
      /*
         Skip NEWLINE character
      */
      ptr_pool++;
      if (*in_pool)
        (*in_pool)--;
    }
   else
    {
#if USE_RC
      if (mode == REsource)
       {
         if (*ptr_pool == *RC_INCL_STMENT)
           *is_include = TRUE;
         else
           if (   isupper(*ptr_pool)
               || islower(*ptr_pool))
             *is_dvar = TRUE;
           else
             if (!isdigit(*ptr_pool))
               is_error = TRUE;
       }
#endif
      ch=(*ptr_char++) = *ptr_pool++;
      (*in_pool)--;
      i = 1;
      LOOP
       {
         if (*in_pool)
          {
            i++;
            /*
               Character sequence `\\'`\n' (BACKSLASH-NEWLINE) found:
                 quote newline char and append next line to current line
            */
            if (   (*ptr_pool == '\n')
                && (ch == QUOTE_CHAR))
             {
               if (is_error)
                {
                  (*in_pool)--;
                  break;
                }
               (*line)++;
               if (i < MAXLEN)
                {
                  ptr_char--;
                  i -= 2;
                  if (*in_pool)
                    ptr_pool++;
                }
             }
            else
              /*
                 Single NEWLINE character found:
                   finish line
              */
              if (*ptr_pool == '\n')
               {
                 ptr_pool++;
                 (*in_pool)--;
                 break;
               }
              else
                if (i < MAXLEN)
                  ch=(*ptr_char++) = *ptr_pool++;
            (*in_pool)--;
          }
         /*
            Refill `pool', because the line we work on isn't complete
         */
         if (!*in_pool)
          {
            *in_pool = read(fileno(fp), (char *)pool, BUF_LEN);
            if (!*in_pool)
              /*
                 At end of file
              */
              return((char *)NULL);
            else
              if (*in_pool < 0)
                /*
                   File read error
                */
                my_error (109, filename, 0, "", 0);
            ptr_pool = pool;
          }
       }
    }
   *ptr_char = '\0';
#if USE_RC
   /*
      Check for local date variable definition
   */
   if (*is_dvar)
     if (!set_dvar (line_buffer, filename, *line, LOcal))
       /*
          Error, unable to store date variable
       */
       my_error (114, filename, *line, line_buffer, 0);
   if (   is_error
       && (mode == REsource))
     /*
        Error, invalid date field given
     */
     my_error (123, filename, *line, line_buffer, 0);
#endif

   return(ptr_pool);
}



#ifdef ANSI_PROTO
PUBLIC void
insert_response_file (      char *filename,
                      const char *opt_list,
                            int  *my_argc_max,
                            int  *my_argc,
                            char *my_argv[])
#else /* !ANSI_PROTO */
   PUBLIC void
insert_response_file (filename, opt_list, my_argc, my_argv)
         char *filename;
   const char *opt_list;
         int  *my_argc_max;
         int  *my_argc;
         int  *my_argv[];
#endif /* !ANSI_PROTO */
/*
   Tries to manage a response file "@file" argument given in command line,
     the '@' character must be the first character of file name.  inserts the
     options and commands found in file "@file" (name delivered in `filename')
     into `*my_argc[]' and sets delivered `*my_argc' and perhaps `*my_argc_max'
     to according "new" values.  uses global variable `s' internally.
*/
{
   auto FILE  *fp=(FILE *)NULL;


   fp = file_open (filename, REsponse);
   if (fp != (FILE *)NULL)
    {
      auto int    line=0;
      auto int    in_pool=0;
      auto char  *pool=(char *)NULL;
      auto char  *ptr_pool=(char *)NULL;
      auto Bool   b_dummy;   /* Dummy */


      pool = (char *)my_malloc (BUF_LEN+1, 124, __FILE__, __LINE__, "pool", 0);
      while ((ptr_pool=file_read_line (fp, s, &in_pool, pool, ptr_pool,
                                       filename, &line, REsponse, &b_dummy, &b_dummy)) != (char *)NULL)
        if (*s)
         {
           register int  len=0;


           /*
              Leading whitespace characters eliminated by file_read_line(),
                so truncate line at first whitespace character found.
           */
           while (   s[len]
                  && !MY_ISSPACE(s[len]))
             len++;
           if (   (   *s == *SWITCH
                   || *s == *SWITCH2)
               && s[1]
               && (s[1] != *SWITCH))
             /*
                Short-style option found
             */
             if (strchr(opt_list, s[1]) != (char *)NULL)
              {
                register int  spaces=0;
                register int  newpos=0;


                /*
                   Short-style option requires an argument, which could be separated
                     by whitespace characters from the option character; respect this!
                     this means all separating whitespace characters between the
                     option character and argument will be eliminated.
                */
                while (   s[len+spaces]
                       && MY_ISSPACE(s[len+spaces]))
                  spaces++;   /* Count the separating whitespace characters. */
                /*
                   Store needed argument of option adjacent to option character.
                */
                while (   s[len+spaces]
                       && !MY_ISSPACE(s[len+spaces]))
                 {
                   s[len+newpos++] = s[len+spaces];
                   len++;
                 }
             }
           s[len] = '\0';
           /*
              Avoid one or two letter combinitions of '-', '/' or '%' characters only!
           */
           if (       (   (len == 1)
                   && (   *s == *SWITCH
#if USE_RC
                       || *s == RC_ADATE_CHAR
#endif
                       || *s == *SWITCH2))
               || (   (len == 2)
                   && (   *s == *SWITCH
#if USE_RC
                       || *s == RC_ADATE_CHAR
#endif
                       || *s == *SWITCH2)
                   && (   s[1] == *SWITCH
#if USE_RC
                       || s[1] == RC_ADATE_CHAR
#endif
                       || s[1] == *SWITCH2)))
             ;   /* Void, don't allocate memory */
           else
            {
              if (*my_argc >= *my_argc_max)
               {
#if HAVE_ASSERT_H
                 /*
                    Check if value for maximum number of table entries
                      fits to the positive range of a signed int(INT_MAX)!!
                 */
                 assert((Uint)*my_argc_max+MY_ARGC_MAX<=testval);
#endif
                 /*
                    Resize the table
                 */
                 *my_argc_max += MY_ARGC_MAX;
                 my_argv = (char **)my_realloc (my_argv, *my_argc_max*sizeof(char *),
                                                124, __FILE__, __LINE__ -1,
                                                "my_argc", *my_argc_max);
               }
              my_argv[*my_argc] = (char *)my_malloc (len+1,
                                                     124, __FILE__, __LINE__ -1,
                                                     "my_argc", *my_argc);
              strcpy(my_argv[(*my_argc)++], s);
            }
         }
      fclose(fp);
      free(pool);
    }
}



#ifdef ANSI_PROTO
PUBLIC void
write_log_file (const char       *filename,
                const Fmode_enum  mode,
                const char       *mode_msg,
                const int         argc,
                      char       *argv[])
#else /* !ANSI_PROTO */
   PUBLIC void
write_log_file (filename, mode, mode_msg, argc, argv)
   const char       *filename;
   const Fmode_enum  mode;
   const char       *mode_msg;
   const int         argc;
         char       *argv[];
#endif /* !ANSI_PROTO */
/*
   Writes the contents of the environment variable GCAL, which is already
     stored in `argc' and the arguments of command line either into a
     response file (mode==REsponse) or into a shell script (mode==SCript);
     other settings to `mode' will `abort()' the program immediately.
*/
{
   auto     FILE  *fp=(FILE *)NULL;
   auto     int    len=0;
   register int    i=0;


   fp = fopen(filename, "w");
   if (fp != (FILE *)NULL)
    {
      switch (mode)
       {
         case REsponse:
           len = fprintf(fp, "%c `%s' %s `%s' -- "
                         "%04d/%02d/%02d %02d"TIME_SEP"%02d"TIME_SEP"%02d\n%c\n",
                         REM_CHAR, prgr_name, mode_msg, filename,
                         buf_ay, buf_am, buf_ad, act_hour, act_min, act_sec,
                         REM_CHAR);
           break;
#  ifdef GCAL_SHELL
         case SCript:
#    if HAVE_SYS_INTERPRETER
           len = fprintf(fp, "%s\n%c\n%c `%s' %s `%s' -- "
                         "%04d/%02d/%02d %02d"TIME_SEP"%02d"TIME_SEP"%02d\n%c\n%s",
                         SHELL, *SHL_REM, *SHL_REM, prgr_name, mode_msg, filename,
                         buf_ay, buf_am, buf_ad, act_hour, act_min, act_sec,
                         *SHL_REM, prgr_name);
#    else /* !HAVE_SYS_INTERPRETER */
           len = fprintf(fp, "%c `%s' %s `%s' -- "
                         "%04d/%02d/%02d %02d"TIME_SEP"%02d"TIME_SEP"%02d\n%c\n%s",
                         *SHL_REM, prgr_name, mode_msg, filename,
                         buf_ay, buf_am, buf_ad, act_hour, act_min, act_sec,
                         *SHL_REM, prgr_name);
#    endif /* !HAVE_SYS_INTERPRETER */
           break;
#endif /* GCAL_SHELL */
         default:
           /*
              This case MUST be an internal error!
           */
           abort();
       }
      if (len != EOF)
       {
         for (i=1 ; i < argc ; i++)
          {
            /*
               Don't write the name of a response file or of a
                shell script into the response file!
            */
            if (   *argv[i] == *SWITCH
                || *argv[i] == *SWITCH2)
             {
               /*
                  If short option -R<NAME> or -S<NAME> is given, skip!
               */
               if (   *(argv[i]+1) == 'R'
#ifdef GCAL_SHELL
                   || *(argv[i]+1) == 'S'
#endif
                  )
                 continue;
               if (*(argv[i]+1) == *SWITCH)
                {
                  /*
                     Detect whether long option --response-file=ARG
                       or --shell-script=ARG is given and if found,
                       don't write this option in the response file!
                  */
                  strcpy(str, argv[i]+2);
                  (void)eval_longopt (str, &len);
                  if (len == SYM_RESPONSE_FILE)
                    continue;
#ifdef GCAL_SHELL
                  if (len == SYM_SCRIPT_FILE)
                    continue;
#endif
                }
             }
            switch (mode)
             {
               case REsponse:
                 len = fprintf(fp, "%s\n", argv[i]);
                 break;
#ifdef GCAL_SHELL
               case SCript:
                 len = fprintf(fp, " \"%s\"", argv[i]);
                 break;
#endif
               default:
                 /*
                    This case can't occur anymore, if so,
                      it MUST be an internal error!
                 */
                 abort();
             }
            if (len == EOF)
              break;
          }
#ifdef GCAL_SHELL
         if (mode == SCript)
           len = fprintf(fp, " $*\n");
#endif
         i = fclose(fp);
       }
    }
   else
     len = EOF;
   if (i == EOF)
     len = EOF;
   if (   (   (len != EOF)
           && (warning_level == 1))
       || (   (len == EOF)
           && (warning_level == 2))
       || warning_level > 2)
    {
#if USE_GER
      sprintf(str, "Versuche %s `%s' zu schreiben... %s%s%s", mode_msg, filename,
              (ehls1s.len!=1) ? ((len==EOF) ? ehls2s.seq : ehls1s.seq) : "",
              (len==EOF) ? "Versagt" : "Erfolg",
              (ehls1s.len!=1) ? ((len==EOF) ? ehls2e.seq : ehls1e.seq) : "");
#else /* !USE_GER */
      sprintf(str, "Try to write %s `%s'... %s%s%s", mode_msg, filename,
              (ehls1s.len!=1) ? ((len==EOF) ? ehls2s.seq : ehls1s.seq) : "",
              (len==EOF) ? "failed" : "success",
              (ehls1s.len!=1) ? ((len==EOF) ? ehls2e.seq : ehls1e.seq) : "");
#endif /* !USE_GER */
      print_text (stderr, str, INternal);
      /*
         Terminate program in case the file can't be written
      */
      if (   (len == EOF)
          && (warning_level >= WARN_LVL_MAX))
        my_error (115, filename, 0, filename, 0);
    }
#ifdef GCAL_SHELL
   if (   (mode == SCript)
       && (len != EOF))
    {
      /*
         Try to make the created shell script executable
      */
      sprintf(str, CHMOD_PRGR" "CHMOD_OPTS" %s", filename);
      /*
         And ignore any errors...
      */
      (void)system(str);
    }
#endif /* GCAL_SHELL */
}



#ifdef ANSI_PROTO
LOCAL FILE *
get_file_ptr (      FILE       *fp,
              const char       *filename,
              const Fmode_enum  mode,
                    Bool       *is_first)
#else /* !ANSI_PROTO */
   LOCAL FILE *
get_file_ptr (fp, filename, mode, is_first)
         FILE       *fp;
   const char       *filename;
   const Fmode_enum  mode;
         Bool       *is_first;
#endif /* !ANSI_PROTO */
/*
   Tries to open the file (with optional diagnostic messages
     --debug[=0...WARN_LVL_MAX]) and returns a pointer to that file
     or NULL if fails
*/
{
   fp = fopen(filename, "r");
   if (   (warning_level >= 0)
       && (mode != REsponse)
       && (mode != COmmon))
#if USE_RC
    {
      if (   (   (fp != (FILE *)NULL)
              && (warning_level == 1))
          || (   (fp == (FILE *)NULL)
              && (warning_level == 2))
          || warning_level > 2)
       {
         /*
            If this function is entered the first time:
              print a leading newline before any warning texts
         */
         if (*is_first)
          {
            *is_first = FALSE;
            *str = '\0';
            print_text (stderr, str, INternal);
          }
#  if USE_GER
         sprintf(str, "Versuche %s-Datei `%s' zu "OE"ffnen... %s%s%s",
                 (mode==REsource) ? "Resourcen" : "Include", filename,
                 (ehls1s.len!=1) ? ((fp==(FILE *)NULL) ? ehls2s.seq : ehls1s.seq) : "",
                 (fp==(FILE *)NULL) ? "Versagt" : "Erfolg",
                 (ehls1s.len!=1) ? ((fp==(FILE *)NULL) ? ehls2e.seq : ehls1e.seq) : "");
#  else /* !USE_GER */
         sprintf(str, "Try to open %s file `%s'... %s%s%s",
                 (mode==REsource) ? "resource" : "include", filename,
                 (ehls1s.len!=1) ? ((fp==(FILE *)NULL) ? ehls2s.seq : ehls1s.seq) : "",
                 (fp==(FILE *)NULL) ? "failed" : "success",
                 (ehls1s.len!=1) ? ((fp==(FILE *)NULL) ? ehls2e.seq : ehls1e.seq) : "");
#  endif /* !USE_GER */
         print_text (stderr, str, INternal);
       }
    }
#else /* !USE_RC */
     ;   /* Void */
#endif /* !USE_RC */

   return(fp);
}



#ifdef ANSI_PROTO
LOCAL void
make_absolute_filename (const char *directory,
                        const char *filename)
#else /* !ANSI_PROTO */
   LOCAL void
make_absolute_filename (directory, filename)
   const char *directory;
   const char *filename;
#endif /* !ANSI_PROTO */
/*
   Creates an absolute file name (directory+file name) of a delivered
     file name and directory and returns the result to the concatenated
     absolute file name in the global variable `s'
*/
{
   register int  len=(int)strlen(DIR_SEP)+strlen(filename);
   register int  len2;


   strncpy(s, directory, MAXLEN+1);
   s[MAXLEN] = '\0';
   len2 = (int)strlen(s);
   if (len + len2 > MAXLEN)
    {
      if (len < MAXLEN)
      strcpy(s, filename);
    }
   else
    {
      if (s[len2-1] != *DIR_SEP)
        strcat(s, DIR_SEP);
      strcat(s, filename);
    }
}
