/**
 ** sipp - SImple Polygon Processor
 **
 **  A general 3d graphic package
 **
 **  Copyright Jonas Yngvesson  (jonas-y@isy.liu.se) 1988/89/90/91
 **            Inge Wallin      (ingwa@isy.liu.se)         1990/91
 **
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 1, or any later version.
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 ** You can receive a copy of the GNU General Public License from the
 ** Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 **/

/**
 ** sipp.h - Public inteface to the sipp rendering library.
 **/


#ifndef _SIPP_H
#define _SIPP_H

#include <geometric.h>


#ifndef M_PI
#define M_PI 3.1415926535897932384626
#endif

#ifndef FALSE
typedef int bool;
#define FALSE  0
#define TRUE   1
#endif 

/*
 * Customize for those that don't have memcpy() and friends, but
 * have bcopy() instead.
 */

#ifdef NOMEMCPY
#define memcpy(to, from, n) bcopy((from), (to), (n))
#endif


/*
 * The macro RANDOM() should return a random number
 * in the range [-1, 1].
 */
#include <stdlib.h>
#define RANDOM()  (-1.0+2.0*(double)rand()/(double)RAND_MAX)


/*
 * Modes for rendering
 */
#define PHONG      0
#define GOURAUD    1
#define FLAT       2
#define LINE       3


/*
 * Interface to shader functions.
 */
typedef void Shader();


/*
 * Colors are handled as an rgb-triple
 * with values between 0 and 1.
 */
typedef struct {
    double   red;
    double   grn;
    double   blu;
} Color;


/*
 * Structure storing the vertices in surfaces. The vertices for a
 * surface are stored in a binary tree sorted first on x, then y and last z.
 */
typedef struct vertex_t {
    double            x, y, z;    /* vertex position */
    double            a, b, c;    /* average normal at vertex */
    double            u, v, w;    /* texture parameters (if any) */
    struct vertex_t  *big, *sml;  /* pointers to children in the tree */
} Vertex;


/*
 * Structure to keep a list of vertex references.
 */
typedef struct vertex_ref_t {
    Vertex       *vertex;
    struct vertex_ref_t *next;
} Vertex_ref;


/*
 * Polygon definition. A polygon is defined by a list of
 * references to its vertices (counterclockwize order).
 */
typedef struct polygon_t {
    Vertex_ref *vertices;     /* vertex list */
    bool        backface;   /* polygon is backfacing (used at rendering) */
    struct polygon_t  *next;
} Polygon;


/*
 * Surface definition. Each surface consists of a vertex tree, 
 * a polygon list, a pointer to a surface description and a pointer
 * to a shader function.
 */
typedef struct surface_t {
    Vertex           *vertices;          /* vertex tree */
    Polygon          *polygons;          /* polygon list */
    void             *surface;           /* surface description */
    Shader           *shader;            /* shader function */
    int               ref_count;         /* no of references to this surface */
    struct surface_t *next;              /* next surface in the list */
} Surface;


/*
 * Object definition. Object consists of one or more
 * surfaces and/or one or more subojects. Each object
 * has its own transformation matrix that affects itself
 * and all its subobjects.
 */
typedef struct object_t {
    Surface         *surfaces;       /* List of surfaces */
    struct object_t *sub_obj;        /* List of subobjects */
    Transf_mat       transf;         /* Transformation matrix */
    int              ref_count;      /* No of references to this object */
    struct object_t *next;           /* Next object in this list */
} Object;



/*
 * Lightsource definition. 
 */
typedef struct lightsource {
    double              intensity;  /* intensity, same for r, g, b */
    Vector              dir;        /* direction from origo */  
    struct lightsource *next;       /* next lightsource in the list */
} Lightsource;


/*
 * Surface description used by the basic shader. This shader
 * does simple shading of surfaces of a single color.
 */
typedef struct {
    double  ambient;       /* Fraction of color visible in ambient light */
    double  specular;      /* Fraction of colour specularly reflected */
    double  c3;            /* "Shinyness" 0 = shiny,  1 = dull */
    Color   color;         /* Colour of the surface */
} Surf_desc;


extern char  * SIPP_VERSION;


/*
 * This defines all public functions implemented in sipp.
 */
extern void          sipp_init();
extern void          sipp_show_backfaces();
extern void          vertex_push();
extern void          vertex_tx_push();
extern void          polygon_push();
extern Surface      *surface_create();
extern Surface      *surface_basic_create();
extern void          surface_set_shader();
extern void          surface_basic_shader();
extern Object       *object_create();
extern Object       *object_instance();
extern Object       *object_dup();
extern Object       *object_deep_dup();
extern void          object_delete();
extern void          object_add_surface();
extern void          object_sub_surface();
extern void          object_add_subobj();
extern void          object_sub_subobj();
extern void          object_install();
extern void          object_uninstall();
extern void          object_set_transf();
extern Transf_mat   *object_get_transf();
extern void          object_clear_transf();
extern void          object_transform();
extern void          object_rot_x();
extern void          object_rot_y();
extern void          object_rot_z();
extern void          object_rot();
extern void          object_scale();
extern void          object_move();
extern void          lightsource_push();
extern void          view_from();
extern void          view_at();
extern void          view_up();
extern void          view_focal();
extern void          viewpoint();
extern void          render_image_file();
extern void          render_image_pixmap();
extern void          basic_shader();

/*
 * Macros to ensure compatibility with older versions.
 */
#define render_image(xres, yres, image_file)\
                render_image_file((xres), (yres), (image_file), PHONG, 2)


#endif /* _SIPP_H */
