/* -- rawedit.c  View/edit a file in hex/ascii             -- */
 
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>
 
#include "rawedit.h"

/* --------------------- **
**     M A C R O S       **
** --------------------- */

#define HEX_SIDE       0
#define ASCII_SIDE     1
#define STARTROW       3  
#define MAXROW        24
#define BYTES_PER_ROW 24
#define BYTES_PER_PAGE ((MAXROW - STARTROW + 1) * BYTES_PER_ROW)
#define HEX_COL_START     1
#define ASCII_COL_START  56
#define ASCII_LEN (BYTES_PER_ROW)
#define HEX_LEN   ( (BYTES_PER_ROW * 2) + 6 )


/* -------------------------------- **
**  G L O B A L   V A R I A B L E S **
** -------------------------------- */

FILE *file;
char *filename = (char *)0;
short curx = 1, cury = 1;
long  filelen = 0, filepos = 0;
char  orig_buffer[ BYTES_PER_PAGE + 3 ];
long  orig_filepos = -1;
char  buffer[ BYTES_PER_PAGE + 3 ];
short bufpos=0, buflen=0;
short nibblestate = 0;
short tabstate = HEX_SIDE;
char  search_direction = 'F';   /* F = forward, B = backward */
char  search_type      = 'A';   /* H = hex, A = ascii */
char  search_string[200];
short search_string_len = 0;
long  last_srch_pos  = 0;
short read_only = 0;

char *cursor_help[] = {
                 "                                                   ",
                 "  -----------------------------------------------  ", 
                 "  |   RAWEDIT v1.0    By: Ken Martin            |  ",
                 "  |        C U R S O R   K E Y S                |  ",
                 "  |                                             |  ",
                 "  | Up ............... UP-ARROW, CTRL-U         |  ",
                 "  | Down ............. DOWN-ARROW, CTRL-D       |  ",
                 "  | Previous Page .... PAGE-UP, CTRL-P          |  ",
                 "  | Next Page ........ PAGE-DOWN, CTRL-N        |  ",
                 "  | Left ............. LEFT-ARROW, CTRL-L       |  ",
                 "  | Right ............ RIGHT-ARROW, CTRL-R      |  ",
                 "  | Beginning of file. HOME, CTRL-H             |  ",
                 "  | End of file ...... END, CTRL-E              |  ",
                 "  | Jump ............. TAB, CTRL-I              |  ",
                 "  | Menu ............. ENTER, CTRL-M            |  ",
                 "  | Exit ............. ESCAPE, CTRL-C           |  ",
                 "  | Help (in menu) ... ?                        |  ",
                 "  |                                             |  ",
                 "  | Hit any key to continue...                  |  ",
                 "  |                                             |  ",
                 "  -----------------------------------------------  ",
                 "                                                   ",
                 "\0"
               };

char *main_menu_help[] = {
                 "                                                   ",
                 "  -----------------------------------------------  ", 
                 "  |   RAWEDIT HELP:  Main Menu                  |  ",
                 "  |                                             |  ",
                 "  | Exit ............ Leave the RAWEDIT program |  ",
                 "  | Goto ............ Goto an offset# in file   |  ",
                 "  | Search ...... ... Search for a string       |  ",
                 "  | Repeat_search ... Repeat search using       |  ",
                 "  |                   previous search parameters|  ",
                 "  | re-Draw_screen .. Redisplay entire screen   |  ",
                 "  | Help ............ Show cursor keys          |  ",
                 "  | ? ............... Show this help screen.    |  ",
                 "  |                                             |  ",
                 "  | Access any menu option by pressing the      |  ",
                 "  | corresponding capitalized letter.  For      |  ",
                 "  | example, to exit RAWEDIT, press 'E'.        |  ",
                 "  |                                             |  ",
                 "  | Hit a key to continue...                    |  ",
                 "  |                                             |  ",
                 "  -----------------------------------------------  ",
                 "                                                   ",
                 "\0"
               };

char *offset_num_help[] = {
                 "                                                   ",
                 "  -----------------------------------------------  ", 
                 "  |   RAWEDIT HELP:  Offset Number              |  ",
                 "  |                                             |  ",
                 "  | The offset # should be from 0 to file       |  ",
                 "  | length (shown at the top of the screen).    |  ",
                 "  |                                             |  ",
                 "  | Offset # may be an integer (base 10), or    |  ",
                 "  | hexadecimal.  For hexadecimal, prefix with  |  ",
                 "  | '0x'.                                       |  ",
                 "  |                                             |  ",
                 "  | Examples:                                   |  ",
                 "  |    Enter offset# to go: 1234                |  ",
                 "  |    Enter offset# to go: 0xF1C4              |  ",
                 "  |                                             |  ",
                 "  | Hit a key to continue...                    |  ",
                 "  |                                             |  ",
                 "  -----------------------------------------------  ",
                 "                                                   ",
                 "\0"
               };

char *search_direction_help[] = {
                 "                                                   ",
                 "  -----------------------------------------------  ", 
                 "  |   RAWEDIT HELP:  Search Direction           |  ",
                 "  |                                             |  ",
                 "  | To                                  Type    |  ",
                 "  | ----------------------------------- ------- |  ",
                 "  | Search forward from current offset  F       |  ",
                 "  | Search backward from current offset B       |  ",
                 "  | Return to previous menu             [ENTER] |  ",
                 "  | Show this help screen               ?       |  ",
                 "  |                                             |  ",
                 "  |                                             |  ",
                 "  | Hit a key to continue...                    |  ",
                 "  |                                             |  ",
                 "  -----------------------------------------------  ",
                 "                                                   ",
                 "\0"
               };

char *search_type_help[] = {
                 "                                                   ",
                 "  -----------------------------------------------  ", 
                 "  |   RAWEDIT HELP:  Search Type                |  ",
                 "  |                                             |  ",
                 "  | To                                  Type    |  ",
                 "  | ----------------------------------- ------- |  ",
                 "  | Search using hexadecimal numbers    H       |  ",
                 "  | Search using normal ASCII text      A       |  ",
                 "  | Return to previous menu             [ENTER] |  ",
                 "  | Show this help screen               ?       |  ",
                 "  |                                             |  ",
                 "  |                                             |  ",
                 "  | Hit a key to continue...                    |  ",
                 "  |                                             |  ",
                 "  -----------------------------------------------  ",
                 "                                                   ",
                 "\0"
               };

char *search_string_help[] = {
                 "                                                   ",
                 "  -----------------------------------------------  ", 
                 "  |   RAWEDIT HELP:  Search String              |  ",
                 "  |                                             |  ",
                 "  | Enter the ASCII text string to be used for  |  ",
                 "  | searching.  RAWEDIT will then go to the     |  ",
                 "  | first matching position in the file.        |  ",
                 "  | A failure a match will return to the        |  ",
                 "  | current offset position.                    |  ",
                 "  |                                             |  ",
                 "  | Hit a key to continue...                    |  ",
                 "  |                                             |  ",
                 "  -----------------------------------------------  ",
                 "                                                   ",
                 "\0"
               };

char *search_hex_nums_help[] = {
                 "                                                   ",
                 "  -----------------------------------------------  ", 
                 "  |   RAWEDIT HELP:  Search Hex Nums            |  ",
                 "  |                                             |  ",
                 "  | Enter the hexadecimal numbers to be used for|  ",
                 "  | searching.  RAWEDIT will then go to the     |  ",
                 "  | first matching position in the file.        |  ",
                 "  | A failure a match will return to the        |  ",
                 "  | current offset position.                    |  ",
                 "  |                                             |  ",
                 "  | Examples:                                   |  ",
                 "  |   search hex nums: A1C4010004               |  ",
                 "  |   search hex nums: 00 01 39 CF              |  ",
                 "  |   search hex nums: 0xF00F 0xC1 0x001A03     |  ",
                 "  |                                             |  ",
                 "  | Hit a key to continue...                    |  ",
                 "  |                                             |  ",
                 "  -----------------------------------------------  ",
                 "                                                   ",
                 "\0"
               };

char *write_changes_help[] = {
                 "                                                   ",
                 "  -----------------------------------------------  ", 
                 "  |   RAWEDIT HELP:  Write changes              |  ",
                 "  |                                             |  ",
                 "  | Yes ...... Save changes made to this screen |  ",
                 "  |                                             |  ",
                 "  | No ....... Discard changes, this page will  |  ",
                 "  |            not be altered.                  |  ",
                 "  |                                             |  ",
                 "  |                                             |  ",
                 "  | Hit a key to continue...                    |  ",
                 "  |                                             |  ",
                 "  -----------------------------------------------  ",
                 "                                                   ",
                 "\0"
               };

/* -------------------------------- **
**     P R O T O T Y P E S          **
** ---------------------------------*/
 
int   rawedit( char *fname );
void  display_header( void );
void  refresh_screen( short firstrow, short lastrow );
void  flush_buffer( void );
void  get_buffer( void );
void  compute_xy( short *x, short *y ); 
char *get_hex( char ch );
void  print_filename( void );
void  update_position_display( void );
void  do_page_up( short newbufpos );
void  do_page_down( short newbufpos );
void  do_right( void );
int   do_menu( void );
void  do_goto_offset( void );
void  do_help( char *help_text[] );
void  strip( char *s );
long  hex_to_long( char *s );
char  hex_to_nibble( char c );
void  message( char *msg );
void  do_search( void );
void  do_search_direction( void );
void  do_search_type( void );
void  do_repeat_search( void );
void  search_file( long start_pos );
void get_file_length( void );
char *long_to_hex( long l );
short input_line( char *editstr, short col, short row, short len );
void lstrip( char * );
void rstrip( char * );


/* -------------------- **
**      M A I N         **
** ---------------------*/

int main(int argc, char *argv[])
{
  long i;
    
 
  if ( argc < 2 )
  {
     printf(
   "\n RAWEDIT  -  View a file in hex/ascii         "
   "\n"
   "\n       Syntax: rawedit -r <filename1> {<filename2> ...}"  
   "\n"
   "\n       where: -r          Read-only mode.  No file changes allowed."
   "\n              <filename>  The file to view/modify."
   "\n"
              );
     exit(1);
  }

  io_init(); 
  display_header();

  /* --- Check for switch --- */
  for (i = 1; i < argc; i++ )
  {
    if ( strcmp( argv[i], "-r" ) == 0 )
    {
       read_only = 1;
    }
  }

  for (i = 1; i < argc; i++ )
  {
    if ( strcmp( argv[i], "-r" ) == 0 ) continue;
    if ( !rawedit ( argv[i] ) )
    {
      io_refresh();
      io_end();
      exit(1);
    }
  }
 
  io_end();
 
  return 0;
}


/* ----------------------------------- **
**  H E X E D I T   F U N C T I O N S  **
** ----------------------------------- */

void display_header( void )
{
   io_clrscr();
   io_gotoxy( 26, 1 ); io_printf( "%s", "Len:" ); 
   io_gotoxy( 41, 1 ); io_printf( "%s", "File Offset:" ); 
   io_gotoxy( 1, 2 ); io_printf( "%s", "[ENTER]=Menu" ); 
   io_refresh();
}
 
int rawedit( char *fname )
{
  char tmp[100];
  short keyhit;
  short tmpx, tmpy;
 
  filename = fname;

  print_filename();
 
  /* ------------------------------- **
  **   Open filename                 **
  ** ------------------------------- */
 
  if ( read_only )
  {
    file = fopen( filename, "rb" );
  }
  else
  {
    file = fopen( filename, "r+b" );
  }

  if ( file == (FILE *)0 )
  {
    io_clrscr();
    sprintf( tmp, "Unable to open %s.  Hit a key...", filename );
    message( tmp );
    return 0;
  }
 
 
  /* ------------------------------- **
  **   Display initial file info     **
  ** ------------------------------- */
  get_file_length();
 
  fseek( file, 0L, SEEK_SET );  /* Reset to beginning */
 
  get_buffer();
  refresh_screen( STARTROW, MAXROW );
 
 
  /* ------------------------------- **
  **   Get keystroke and process     **
  ** --------------------------------*/
  while( 1 )
  {
     update_position_display();
 
     compute_xy( &curx, &cury );
     io_gotoxy( curx, cury );

     io_refresh(); 
     keyhit = io_inchar();
 
     if ( keyhit == io_ESCAPE || keyhit ==  io_CTRL_C ) 
     {
          break;
     }
     else if ( keyhit == io_TAB || keyhit == io_CTRL_I )  /* Jump  */
     {
          tabstate = 1 - tabstate;
          nibblestate = 0;
     }
     else if ( keyhit == io_PAGEUP || keyhit == io_CTRL_P )
     {
          do_page_up( bufpos );
     }
     else if ( keyhit == io_PAGEDOWN || keyhit == io_CTRL_N )
     {
          do_page_down( bufpos );
     }
     else if ( keyhit == io_UP || keyhit == io_CTRL_U )
     {
          if ( bufpos - BYTES_PER_ROW  >= 0 )
          {
             bufpos -= BYTES_PER_ROW;
          }
          else
          {
            if ( filepos > 0 )
            {
              if ( filepos < BYTES_PER_PAGE )
              {
                do_page_up( filepos - BYTES_PER_ROW );
              }
              else
              {
                do_page_up( BYTES_PER_PAGE - BYTES_PER_ROW + bufpos );
              }
            }
          }
     }
     else if ( keyhit == io_DOWN || keyhit == io_CTRL_D )
     {
          if ( bufpos + BYTES_PER_ROW  < buflen )
          {
             bufpos += BYTES_PER_ROW;
          }
          else
          {
            do_page_down( bufpos - BYTES_PER_PAGE + BYTES_PER_ROW );
          }
     }
     else if ( keyhit == io_LEFT || keyhit == io_CTRL_L )
     {
          if ( tabstate == ASCII_SIDE ) /* --- Ascii area --- */
          {
            if ( bufpos > 0 )
                bufpos--;
            else
            {
              if ( filepos < BYTES_PER_PAGE )
              {
                do_page_up( filepos - 1 );
              }
              else
              {
                do_page_up( BYTES_PER_PAGE );
              }
            }
          }
          else /* --- Hex text area --- */
          {
            if ( nibblestate )
            {
              nibblestate = 0;
            }
            else
            {
              if ( bufpos > 0 )
              {
                 bufpos--;
                 nibblestate = 1;
              }
              else
              {
                if ( filepos > 0 )
                {
                  if ( filepos < BYTES_PER_PAGE )
                  {
                    do_page_up( filepos - 1 );
                  }
                  else
                  {

                    do_page_up( BYTES_PER_PAGE );
                  }
                }
              }
            }
          }
     }
     else if ( keyhit == io_RIGHT || keyhit == io_CTRL_R )
     {
       do_right();
     }
     /* --- Beginning of file --- */
     else if ( keyhit == io_HOME || keyhit == io_CTRL_H )
     {
       filepos = 0;
       bufpos = 0;

       get_buffer();
       refresh_screen( STARTROW, MAXROW );

     }

     /* --- End of file --- */
     else if ( keyhit == io_END || keyhit == io_CTRL_E )
     {
       get_file_length( );
       
       filepos = filelen - BYTES_PER_PAGE + BYTES_PER_ROW;

       if ( filepos < 0 ) filepos = 0;

       bufpos = filelen - filepos;

       get_buffer();
       refresh_screen( STARTROW, MAXROW );
     }

     /* ----- Activate Menu ----- */
     else if ( keyhit == io_ENTER || keyhit == io_CTRL_M )
     {
        if ( do_menu() ) break;
        io_gotoxy( 1, 2 );
        io_printf("[ENTER]=Menu %65.65s", " " );
     }
     /* --- Modify the buffer --- */
     else if ( !read_only && keyhit >=(short)32 && keyhit <= (short)io_MAX_VALID_CHARACTER )
     {
       if ( tabstate == HEX_SIDE )
       { /*  --- Hex side, allow only hex characters --- */
         keyhit = (short)toupper( (int)keyhit );
         if ( (keyhit >= (short)'0' && keyhit <= (short)'9') ||
              (keyhit >= (short)'A' && keyhit <= (short)'F')
            )
         {
            if ( nibblestate == 0 )  /* --- Upper nibble --- */
            {
                buffer[bufpos] =
                  ( buffer[bufpos] & 0x0F ) |
                  ( hex_to_nibble( (char)keyhit ) << 4 );
            }
            else  /* --- Lower nibble --- */
            {
                buffer[bufpos] =
                  ( buffer[bufpos] & 0xF0 ) |
                  hex_to_nibble( (char)keyhit );
            }

            /* --- update other side --- */
            tabstate = ASCII_SIDE;
            compute_xy( &tmpx, &tmpy );
            io_gotoxy( tmpx, tmpy );
            io_printf( "%c", buffer[bufpos] );
            tabstate = HEX_SIDE;

            io_gotoxy( curx, cury );
            io_printf("%c", keyhit );
            do_right();

         }
       }
       else  /* --- Ascii side, allow regular characters --- */
       {
          buffer[bufpos] = (char)keyhit;

          /* --- Update other side --- */
          tabstate = HEX_SIDE;
          compute_xy( &tmpx, &tmpy );
          io_gotoxy( tmpx, tmpy );
          io_printf( "%02s", get_hex( keyhit ) );
          tabstate = ASCII_SIDE;

          io_gotoxy( curx, cury );
          io_printf("%c", keyhit );
          do_right();

       }


     }
     else
     {
       ;
     }
  }
  flush_buffer();

  fclose( file );
 
  return 1;
}

void  refresh_screen( short firstrow, short lastrow ) 
{
  short y, i, j;
  char hex_text[81];
  char ascii_text[81];
  short bufidx = 0;
  char ch;
 
  for( y = firstrow, bufidx = (firstrow-STARTROW)*BYTES_PER_ROW;
       y <= lastrow;  y++, bufidx+=BYTES_PER_ROW )
  {
     /* --- build hex/ascii -number text for display --- */
     strcpy( hex_text, "" );
 
     for ( i = bufidx, j=0;
           i < bufidx + BYTES_PER_ROW && i < buflen; i++, j++ )
     {
       ch = buffer[i];
       /* --- hex number --- */
       strcat( hex_text, get_hex(ch) );
       if ( (i+1) % 4 == 0 ) strcat( hex_text, " " );
 
       /* --- ascii number --- */
       if ( (short)ch < (short)32 ||
            (short)ch > (short)io_MAX_VALID_CHARACTER )
          ascii_text[j] = '.';
       else
          ascii_text[j] = ch; 
 
     }
 
     ascii_text[j] = '\0';
 
     /* --- Display the row --- */
 
     io_gotoxy( HEX_COL_START, y );
     io_printf( "%-*.*s %-*.*s", HEX_LEN, HEX_LEN, hex_text,
                                 ASCII_LEN, ASCII_LEN, ascii_text );
  }
 
}

void flush_buffer( void )
{
     char tmp[100];
     short keyhit;
     short status;

     /* --- Check for changes to buffer --- */
     if ( !read_only && orig_filepos >= 0 &&
          memcmp( orig_buffer, buffer, buflen ) != 0 )
     {
        while( 1 )
        {
          io_gotoxy( 1, 2 );
          sprintf( tmp, "Write changes?: Yes No  " );
          io_printf( "%-79.79s", tmp );
          io_gotoxy( strlen( tmp ), 2 );
          io_refresh();

          keyhit = io_inchar( );

          if ( keyhit == 'Y' || keyhit == 'y' )  
          {
            /* --- write changes to the file --- */
            fseek( file, orig_filepos, SEEK_SET );
            status = fwrite( buffer, buflen, 1L, file );
            if ( !status )
            {
               message( "Unable to write changes.  Hit a key..." );
               memcpy( buffer, orig_buffer, buflen );
            }
            break;
          }
          else if ( keyhit == 'N' || keyhit == 'n' )
          {
             break;
          }
          else if ( keyhit == '?' )
          {
            do_help( write_changes_help );
          }
          else 
          {
            ;
          }
        } /* End while loop */

        io_gotoxy( 1, 2 );
        io_printf("[ENTER]=Menu %65.65s", " " );
     }

}

void  get_buffer( void )
{
     short status;

     flush_buffer();

     /* -- Always get end-of-file in case file length changes while editing -*/
     get_file_length();

     if ( filepos > filelen )
        filepos = filelen;

     if ( filepos < 0 ) filepos = 0;

     if ( BYTES_PER_PAGE < filelen - filepos + 1 )
       buflen = BYTES_PER_PAGE;
     else
       buflen = filelen - filepos + 1;

     /* -- Clear out keyboard buffer before performing a disk read -- */
     io_flush_keyboard();

      fseek( file, filepos, SEEK_SET );
      status = fread( buffer, 1, buflen, file );
      if ( !status )
      {
        message( "Unable to read file.  Hit a key..." );
        memset( buffer, '\0', buflen );
        filepos = 0;
      }

      if ( !read_only )
      {
        memcpy( orig_buffer, buffer, buflen );
        orig_filepos = filepos;
      }
}
 
void compute_xy( short *x, short *y ) 
{
   /* tabstate 0 means hex area
   ** tabstate other means ascii area
   */
   short ascii_col_start = 56;
   short hex_col_start   =  1;
   short col_adder       = 0;
 
   if ( bufpos > buflen ) bufpos = buflen;
 
   if ( tabstate == ASCII_SIDE ) /* --- Ascii area --- */
   {
      *y = (bufpos / ASCII_LEN);
      *x = (bufpos - (*y * BYTES_PER_ROW ) ) + ascii_col_start;
      *y = *y + STARTROW;
   }
   else   /* --- Hex area --- */
   {
      *y = (bufpos / ASCII_LEN);
      *x = (bufpos - (*y * BYTES_PER_ROW ) ); 
      *y = *y + STARTROW;
      col_adder = *x / 4;
      *x = (*x * 2) + col_adder + nibblestate + hex_col_start;
   }
 
}

void print_filename( void )
{
  char *ptr;

  /* ------------------------------- **
  **   Print filename on header      **
  ** ------------------------------- */
 
  io_gotoxy( 1, 1 );
  ptr = strrchr( filename, '/' );
  if ( ptr == (char *)0 )
  {
     ptr = strrchr( filename, '\\' );
  }
 
  if ( ptr == (char *)0 )
  {
    io_printf( "%-24.24s", filename );
  }
  else
  {
    io_printf( "%-24.24s", &ptr[1] );
  }

}
 
void update_position_display( void )
{
   char tmp[30];
   long offset;

   offset = filepos + bufpos;
   io_gotoxy( 54, 1 );
   sprintf( tmp, "%ld, 0x%s", offset, long_to_hex( offset ) );
   io_printf( "%-24s", tmp );
 
}
 
void do_page_up( short newbufpos )   
{                                                          
     filepos -= BYTES_PER_PAGE;                              
     if (filepos < 0) filepos = 0;                           
     bufpos = 0;                                             
     get_buffer();                                           
     refresh_screen( STARTROW, MAXROW );                     
     if ( (newbufpos) > buflen - 1 )                         
        bufpos = buflen - 1;                                 
     else                                                    
        bufpos = (newbufpos);                                
}

void do_page_down( short newbufpos )  
{                                                          
    if ( filepos + BYTES_PER_PAGE < filelen )               
    {                                                       
       filepos += BYTES_PER_PAGE;                           
       if ( filepos > filelen ) filepos = filelen;          
       bufpos = 0;                                          
       get_buffer();                                        
       refresh_screen( STARTROW, MAXROW );                  
       if ( (newbufpos) > buflen - 1 )                      
            bufpos = buflen - 1;                               
       else                                                 
            bufpos = (newbufpos);                              
    }                                                       
}

void do_right( void )
{
          if ( tabstate == ASCII_SIDE ) /* --- Ascii area --- */
          {
            if ( bufpos < buflen - 1 ) 
               bufpos++;
            else
               do_page_down( 0 );
          }
          else   /* --- Hex text area --- */
          {
            if ( nibblestate == 0)
            {
              nibblestate = 1;
            }
            else
            {
              if ( bufpos < buflen - 1 )
              {
                 bufpos++;
                 nibblestate = 0;
              }
              else
                 do_page_down( 0 );
            }
          }
}

/* --------------------------------- **
**  M E N U   F U N C T I O N S      **
** --------------------------------- */

int do_menu( void )
{
   /* -- RETURN STATUS:  0 = continue normally, 1 = exit program --- */
   short keyhit;
   char  tmp[82];

   while( 1 )
   {
      io_gotoxy( 1, 2 );
      sprintf( tmp, "Exit Goto Search Repeat_search re-Draw_screen Help  " );
      io_printf( "%-79.79s", tmp );
      io_gotoxy( strlen( tmp ), 2 );
      io_refresh();

      keyhit = io_inchar( );

      if ( keyhit == 'E' || keyhit == 'e' )  
      {
         return 1;
      }
      else if ( keyhit == 'G' || keyhit == 'g' )
      {
         do_goto_offset();
         return 0;
      }
      else if ( keyhit == 'S' || keyhit == 's' )
      {
         do_search();
         return 0;
      }
      else if ( keyhit == 'R' || keyhit == 'r' )
      {
         do_repeat_search();
         return 0;
      }
      else if ( keyhit == 'D' || keyhit == 'd' )
      {
         /* --- Redraw screen -- */
         display_header();
         update_position_display();
         print_filename();
         get_file_length();
         refresh_screen( STARTROW, MAXROW );
         return 0;
      }
      else if ( keyhit == 'H' || keyhit == 'h' )
      {
         do_help( cursor_help );
         return 0;
      }
      else if ( keyhit == '?' )
      {
         do_help( main_menu_help );
      }
      else if ( keyhit == io_ESCAPE || keyhit == io_ENTER ||
                keyhit == io_CTRL_M || keyhit == io_CTRL_C )
      {
         return 0;
      }
      else
      {
         ;
      }
   } /* End while loop */
}

void do_goto_offset( void )
{
   short keyhit;
   char tmp[100];
   long offset;

   while( 1 )
   {
      io_gotoxy( 1, 2 );
      sprintf( tmp, "Enter offset# to go:  " );
      io_printf( "%-79.79s", tmp );
      io_gotoxy( strlen( tmp ), 2 );

      io_refresh();

      keyhit = input_line( tmp, strlen( tmp ), 2, 15 );

      if ( keyhit == io_ESCAPE || keyhit == io_CTRL_C ) return;

      if ( keyhit != io_ENTER && ( keyhit < 32 ||
           keyhit > io_MAX_VALID_CHARACTER ) ) continue;

      if ( strcmp( tmp, "" ) == 0 ) return;

      if ( strcmp( tmp, "?" ) == 0 )
      {
        do_help( offset_num_help );
        continue;
      }

      strip( tmp );

      if ( strcmp( tmp, "" ) == 0 ) return;

      if ( strlen(tmp) > 2 && tmp[0] == '0' && tmp[1] == 'X' )
      /* -- Hex value entered -- */
      {
         offset = hex_to_long( &tmp[2] );
         if ( offset < 0 ) continue;
      }
      else  /* -- Numeric value entered -- */
      {
         offset = atol( tmp );
         if ( offset == 0 && strcmp( tmp, "0" ) != 0 ) continue;
      }
      break;
   }

   filepos = offset;
   bufpos = 0;

   get_file_length();

   if ( filepos > filelen )
   {
       filepos = filelen - BYTES_PER_PAGE + BYTES_PER_ROW;
       bufpos = filelen - filepos;
   }

   if ( filepos < 0 )
   {
     filepos = 0;
     bufpos = 0;
   }
 
   get_buffer();
   refresh_screen( STARTROW, MAXROW );
}

void do_search( void )
{
   short keyhit;
   char tmp[82];
   char tmp2[82];
   char *ptr;
   short tmp_hex_len, i, j;

   /* ---- Get search direction ---- */
   while( 1 )
   {
      io_gotoxy( 1, 2 );

      sprintf( tmp, "search direction: Forward Backward  " );

      io_printf( "%-79.79s", tmp );
      io_gotoxy( strlen( tmp ), 2 );
      io_refresh();

      keyhit = io_inchar( );

      if ( keyhit == 'F' || keyhit == 'f' )  
      {
         search_direction = 'F';
         break;
      }
      else if ( keyhit == 'B' | keyhit == 'b' )
      {
         search_direction = 'B';
         break;
      }
      else if ( keyhit == '?' )
      {
        do_help( search_direction_help );
      }

      else if ( keyhit == io_ESCAPE || keyhit == io_ENTER || 
                keyhit == io_CTRL_M || keyhit == io_CTRL_C )
      {
         return;
      }
      else
      {
         ;
      }
   } /* End while loop */


   /* ---- Get search type ---- */
   while( 1 )
   {
      io_gotoxy( 1, 2 );

      sprintf( tmp, "search type: Hex Ascii_text  " );

      io_printf( "%-79.79s", tmp );
      io_gotoxy( strlen( tmp ), 2 );
      io_refresh();

      keyhit = io_inchar( );

      if ( keyhit == 'H' || keyhit == 'h' )  
      {
         search_type = 'H';
         break;
      }
      else if ( keyhit == 'A' | keyhit == 'a' )
      {
         search_type = 'A';
         break;
      }
      else if ( keyhit == '?' )
      {
         do_help( search_type_help );
      }
      else if ( keyhit == io_ESCAPE || keyhit == io_ENTER || 
                keyhit == io_CTRL_M || keyhit == io_CTRL_C )
      {
         return;
      }
      else
      {
         ;
      }
   } /* End while loop */


   /* --- Get search data --- */
   while( 1 )
   {
     switch ( search_type )
     {
       case 'A':
         io_gotoxy( 1, 2 );

         sprintf( tmp, "search string:  " );

         io_printf( "%-79.79s", tmp );
         io_gotoxy( strlen( tmp ), 2 );
         io_refresh();

         keyhit = input_line( search_string, strlen( tmp ), 2,
                              80 - strlen( tmp ) );

         if ( keyhit == io_ESCAPE || keyhit == io_CTRL_C ) return;
         if ( keyhit != io_ENTER && ( keyhit < 32 ||
              keyhit > io_MAX_VALID_CHARACTER ) ) continue;

         lstrip( search_string );
         rstrip( search_string );

         if ( strcmp( search_string, ""  ) == 0 ) return;
         if ( strcmp( search_string, "?" ) == 0 )
         {
           do_help( search_string_help );
           continue;
         }

         search_string_len = strlen( search_string );
         break;

       case 'H':
         io_gotoxy( 1, 2 );

         sprintf( tmp, "search hex nums: " );

         io_printf( "%-79.79s", tmp );
         io_gotoxy( strlen( tmp ), 2 );
         io_refresh();

         keyhit = input_line( tmp2, strlen( tmp ), 2, 80 - strlen( tmp ) );

         if ( keyhit == io_ESCAPE || keyhit == io_CTRL_C ) return;
         if ( keyhit != io_ENTER && ( keyhit < 32 ||
              keyhit > io_MAX_VALID_CHARACTER ) ) continue;

         if ( strcmp( tmp2, "" ) == 0 ) return;
         if ( strcmp( tmp2, "?" ) == 0 )
         {
            do_help( search_hex_nums_help );
            continue;
         }

         /* --- convert this input to the search_string --- */
         strip( tmp2 );
         strcpy( search_string, "" );
  
         /* --- Strip off any leading 0x prefix --- */
         while(1)
         {
           ptr = strstr( tmp2, "0X" );
           if ( ptr == (char *)0 ) break;
           ptr[0] = ' ';
           ptr[1] = ' ';
         }
         strip( tmp2 );
         ptr = &tmp2[0];

         tmp_hex_len = strlen( ptr );
         if ( tmp_hex_len % 2 != 0 )
         {
            message( "Must enter an even number of hex values." );
            continue;
         }

         search_string_len = tmp_hex_len / 2;

         /* --- Cycle through list of hex values --- */
         for ( i = 0, j=0; i < tmp_hex_len; i += 2, j++ )
         {
           search_string[j] = ( hex_to_nibble( ptr[i] ) << 4 ) | 
                                hex_to_nibble( ptr[i+1] );
         }
         break;

     default:
       return;
     } /* -- End switch -- */

     break;

   } /* -- End while loop -- */

   /* --- Perform actual search --- */
   search_file( filepos + bufpos );
}

void do_repeat_search( void )
{

   /* --- Perform actual search --- */
   if ( search_direction == 'F' )
      search_file( filepos + bufpos + 1);
   else
      search_file( filepos + bufpos - 1);
}

void do_help( char *help_text[] )
{
   short i = 0;

   while( strcmp( help_text[i], "\0" ) != 0 )
   {
      io_gotoxy( 11, STARTROW + i );
      io_printf( "%s", help_text[i] );
      i++;
   }

    (void)io_inchar();
    refresh_screen( STARTROW, STARTROW + i - 1 );

}

void  search_file( long start_pos )
{
   long old_file_pos, i, j;
   short old_buf_pos, match;

   old_file_pos = filepos;
   old_buf_pos = bufpos;

   /* --- last_search_pos hold current filepos --- */
   if ( start_pos > filelen )
   {
      message( "* End of file.  Hit a key.." );
      return;
   }

   filepos = start_pos;  /* Set to current cursor location */
   bufpos = 0;

   switch ( search_direction )
   {
     case 'F':  /* Forward */
       while( filepos < filelen )
       {
         io_gotoxy(60,2);      
         io_printf("filepos: %10ld", filepos ); 
         io_refresh();

         get_buffer();

         /* --- Go through each position in the buffer --- */
         for ( i = 0; i < buflen; i++ )
         {
            /* --- Compare string with current string in buffer --- */
            match = 1;
            for ( j = 0; j < search_string_len; j++ )
            {
              if ( buffer[i+j] != search_string[j] )
              {
                match = 0;
                break;
              }
            }
            if ( match )
            {
              filepos += i;
              bufpos = 0;
              get_buffer();
              refresh_screen( STARTROW, MAXROW );
              /* message( "Match found.  Hit a key..." ); */
              refresh_screen( STARTROW, MAXROW );
              return;
            }
         }
         filepos += ( BYTES_PER_PAGE - search_string_len );
       } /* --- End of while loop -- */

       message( "No match found.  Hit a key..." );
       filepos = old_file_pos;
       bufpos = old_buf_pos;
       get_buffer();
       refresh_screen( STARTROW, MAXROW );

       break;

     case 'B':  /* Backward */
       while( filepos > 0 )
       {
         if ( filepos >= BYTES_PER_PAGE )
         {
            bufpos = BYTES_PER_PAGE;
           filepos -= (BYTES_PER_PAGE - search_string_len);
         }
         else
         {
            bufpos = filepos;
            filepos = 0;
         }

         io_gotoxy(60,2);  
         io_printf("filepos: %10ld", filepos ); 
         io_refresh();
    
         get_buffer();

         /* --- Go through each position in the buffer --- */
         for ( i = buflen-1; i >= 0; i-- )
         {
            /* --- Compare string with current string in buffer --- */
            match = 1;
            for ( j = 0; j < search_string_len; j++ )
            {
              if ( buffer[i+j] != search_string[j] )
              {
                match = 0;
                break;
              }
            }
            if ( match )
            {
              filepos += i;
              bufpos = 0;
              get_buffer();
              refresh_screen( STARTROW, MAXROW );
            /* message( "Match found.  Hit a key..." ); */
              refresh_screen( STARTROW, MAXROW );
              return;
            }
         }
       } /* --- End of while loop -- */

       message( "No match found.  Hit a key..." );
       filepos = old_file_pos;
       bufpos = old_buf_pos;
       get_buffer();
       refresh_screen( STARTROW, MAXROW );

       break;

     default:
      ;
   }

}



/* ----------------------------------- **
**  U T I L I T Y   F U N C T I O N S  **
** ----------------------------------- */

char *get_hex( char ch )
{
  static char tmp[3] = {0,0,0};
  char lo, hi;
 
  lo = ch & 0x0F;
  hi = (ch >> 4) & 0x0F;
 
  if ( lo < 10 )
    tmp[1] = lo + (char)'0';
  else
    tmp[1] = lo + (char)'A' - 10;
 
  if ( hi < 10 )
    tmp[0] = hi + (char)'0';
  else
    tmp[0] = hi + (char)'A' - 10;
 
  return tmp;
}

void strip( char *s )
{
  long source=0, target=0, len=0;

  len = strlen( s );
  if ( len == 0 ) return;

  while( source < len )
  {
     if ( s[source] >= 'a' && s[source] <= 'z' ) 
     {
       s[source] = (char)toupper( s[source] );
     }
     if ( ( s[source] >= '0' && s[source] <= '9' ) ||
          ( s[source] >= 'A' && s[source] <= 'Z' )
        )
     {
       if ( source != target )
       {
         s[target] = s[source];
       } 
       target++;
     }
     source++;
  }

  s[target] = (char)0;
}

long hex_to_long( char *s )
{
   long len, i, ret_val = 0;
   char ch;

   len = strlen( s );
   if ( len > 8 ) return -1;

   for (i = 0; i < len; i++ )
   {
      ch = s[i];
      ch = (char)toupper( ch );

      if ( ch >= '0' && ch <= '9' )
      {
         ret_val = ( ret_val << 4 ) | (long)( ch - '0' );
      }
      else if ( ch >= 'A' && ch <= 'F' )
      {
         ret_val = ( ret_val << 4 ) | (long)( ch - 'A' + 10 );
      }
      else
      {
         return -1;
      }   
   }  
   return ret_val;
}

char hex_to_nibble( char c )
{
   char ch;

   ch = (char)toupper( c );

   if ( ch >= '0' && ch <= '9' )
   {
       return (char)( ch - '0' );
   }
   else if ( ch >= 'A' && ch <= 'F' )
   {
      return (char)( ch - 'A' + 10 );
   }
   else  /* --- Failure --- */
   {
      return 0xFF;
   }   
}

char *long_to_hex( long l )
{
  static char s[10];
  long source=0, target=0, len=0;
  short flag=0;

  strcpy( s,"" );
  strcat( s, get_hex( (char)(l >> 24) ) );
  strcat( s, get_hex( (char)((l >> 16) & 0x000000FF) ) );
  strcat( s, get_hex( (char)((l >> 8) & 0x000000FF) ) );
  strcat( s, get_hex( (char)(l & 0x000000FF) ) );

  /* --- clean off leading zeros --- */
  len = strlen( s );
  if ( len == 0 ) return (char *)0;

  while( source < len )
  {
     if ( flag )
     {
       if ( source != target )
       {
         s[target] = s[source];
       } 
       target++;

     }
     else
     {
       if ( s[source] != '0' )
       {
         flag = 1;
         source--;
       }
     }
     source++;
  }

  s[target] = (char)0;
  return s;
}

void message( char *msg )
{
   io_gotoxy( 1, 2 );

   io_printf( "%-79.79s", msg );
   io_gotoxy( strlen( msg ), 2 );
   io_refresh();

   (void)io_inchar( );
}

void get_file_length( void )
{
    static old_file_length = -1;

     /* -- Always get end-of-file in case file length changes while editing -*/
     fseek( file, 0L, SEEK_END );
     filelen = ftell( file );   /* Get file length */ 

     /* --- Update display to screen of file length --- */
     if ( filelen != old_file_length )
     {
       io_gotoxy( 31, 1 );
       io_printf( "%-10ld", filelen );
       old_file_length = filelen;
     }
}

short input_line( char *str, short col, short row, short len )
{
   short i;
   short curpos;
   short startcol;
   short keyhit = 0;

   for (i=0; i<=len; i++)  /* Clear out extra with nulls */
   {
     str[i]=0;
   }

   startcol = col;
   if (startcol < 1) startcol = 1;

   /*----------------------------------------------
       Process keystrokes
    *----------------------------------------------*/

   io_gotoxy(startcol, row);
   io_printf("%s",str);


   while(1)                       /* Beginning of key-processing block */
   {
     curpos = col-startcol+1;

     io_gotoxy(col,row);
     io_refresh();

     keyhit = io_inchar();

     if ( keyhit == io_BACKSPACE || keyhit == io_CTRL_H )
     {
         if (col > startcol) col--;
         if (curpos <= len) str[curpos-1] = ' ';
         io_printf(" ");

     }
     else
     {
         if (keyhit >= 32 && keyhit <= io_MAX_VALID_CHARACTER )  
         {

           /* --- Regular character entered -- */
           if (col < startcol + len - 1) /* Not past end */
           {
             io_printf("%c", (char)keyhit);
             str[curpos-1]=(char)keyhit;
             col++;
           }
         }
         else
         {
           break;
         }

     }     
   } 

   return keyhit;
}

void rstrip( char *s )
{
   long i;

   for( i = strlen(s)-1; i >= 0; i--)
   {
     if ( s[i] == ' ' || s[i] == '\t' || s[i] == '\n')
       s[i] = 0;
     else
       break;
     
   }

}

void lstrip( char *s )
{
   long i;
   long len;
   long ptr=0;

   len = strlen(s);
   
   /* -- find first non-blank -- */
   for (i = 0; i < len; i++)
   {
     if ( s[i] != ' ' && s[i] != '\t' && s[i] != '\n')
     {
       ptr = i;
       break;
     }
   }

   if ( ptr == 0 ) return;  /* -- Nothing to strip -- */

   /* -- Copy elements over -- */

   for (i = 0; ptr < len; i++, ptr++)
   {
      s[i] = s[ptr];

   }
   s[i] = 0;
}
