// PS_Stream -- rewrite of ostream for PostScript Printers

/*[]------------------------------------------------------------[]*/
/*|                                                              |*/
/*|     PSstream.cpp                                             |*/
/*|                                                              |*/
/*|     Class oprnstream                                           |*/
/*|          oprnstream::oprnstream()                                |*/
/*|                                                              |*/
/*[]------------------------------------------------------------[]*/

/*
 *      C/C++ Run Time Library - Version 5.0
 *
 *      Copyright (c) 1990, 1992 by Borland International
 *      All Rights Reserved.
 *
 */

#undef _BIG_INLINE          // disable expansion of large inline functions

const MaxCharsInLong        = 16;
const StreambufSize         = 1024;
const MinStrstreamAlloc     = 16;
const MinStrstreamIncr      = 4;


#include "oprnstr.h"
#include <string.h>
#include <stdio.h>

oprnstream::oprnstream(int flag)
{
	psflag = flag;
	newpage = True;
	header  = False;
	typesize = 12;
}

oprnstream::oprnstream(streambuf _FAR *s, int flag)
{
	ios::init(s);
	psflag = flag;
	newpage = True;
	header = False;
	typesize = 12;
}

/*  ******************************************************************
	This section contains the routines that are unique to oprnstream
	they deal with printer margins pagesize and formatting commands for
	PostScript

	***********************************************************************/


int oprnstream::putn(const char *c, int len)
{
	int 	i;

	for (i=0;i<len;i++)	{
		if (putbuf((int) *c++) == EOF)
			break;
	}
	return i;
}

int oprnstream::putbuf(int c)
{
	int rc;							// return code
				// if this is a new line, prefix data by
				// '(' to tell printer it is a string
	if (psflag && !header)	{
		if (c >= 127)
			c = ' ';
		else {                       // this is only valid for PostScript
			switch((char) c) {
				case '(':
				case ')':
				case '\\': 	if ((rc = bp->sputc('\\')) == EOF)
								break;
							rc = bp->sputc(c);
							break;
				case '\n':	bp->sputn(") show\n nl (", 12);
							break;
				case '\t':	bp->sputn(") show\n tab (", 13);
							break;
				case '\f':  bp->sputn(") show\n np (", 12);
							break;
				default:   	rc = bp->sputc(c);
							break;
			}
		}
	}
	else
		bp->sputc(c);
	return rc;
}


/* **********************************************************************
	This section contains the original ostream code for the most part
	the changes involve the constructors (setting initial values for some
	new member data and replacing bp->sputc and bp->sputn functions with
	putbuf and putn functions to allow the formatting of the PostScript text
	and commands.
	***********************************************************************/

// insert newline and flush

oprnstream _FAR & endl(oprnstream _FAR & os)
{
	os << '\n';
	return os;
}
// insert null to terminate string

oprnstream _FAR & ends(oprnstream _FAR & os)
{
	os << char(0);
	return os;
}


/*
	This code assumes the IEEE Standard 754-1985 model of floating-point
	numbers, but makes no assumptions about the details of the floating-point
	representation.  There are several IEEE kinds (classes) of number.
	Positive or Negative Infinity result on overflow, and propagate
	as a mathematical infinity does.  Overflow and underflow may or may not
	produce a signal (interrupt).  Operations on infinities may or may not
	produce a signal.
	An undefined operation, such as (0.0 / 0.0) or (0.0 * Infinity),
	produces a special value called Not-A-Number, or NaN.
	Any operation on a NaN results in a NaN.  Production of or operations on
	a NaN may or may not produce a signal (interrupt).
	When a result is smaller in magnitude than the smallest standard value
	but is not zero, it may be represented as a Denormalized number.
	Operations on denormalized numbers may or may not produce a signal.

	The IEEE standard recommends that any implementation supply certain
	auxiliary functions to help in working with special values.  These
	recommended functions are in a header file called <fakeieee.h>.

	This code uses the following IEEE functions:
	// return the kind of number
	int fpclass( long double );

	// return the first value with the sign of the second,
	// even if one is Infinity or NaN, which would otherwise be
	// an illegal operation
	long double fpcopysign( long double, long double );

	// return true if -INFINITY < parameter < +INFINITY
	int fpfinite( long double );

	// return true if parameter is a NaN
	int fpisnan( long double );

	In addition, we generate special strings for +Infinity, -Infinity,
	and NotANumber.  You may wish to generate other strings, or just
	set Infinities to HUGE_VAL and emit them normally.  It is not
	obvious what to do about NaN's in that case.

	If your system does not have an <ieee.h> or equivalent header, or
	if your floating-point system does not fit this model, you may as
	an expedient define the macro name _NOIEEE_ immediately below, and
	use a simplified header "fakeieee.h" supplied with this package.
	You will later want to go through the code to provide any needed
	functionality depending on your floating-point model.

	Long doubles are not directly supported by this package.  The
	modifications to support them should be obvious and straightforward.

	NOTES:
	1.  Function "round" below contains a rounding table which must
	have as many entries as there are significant decimal digits in
	the floating-point representation.  IEEE double-precision provides
	about 16 significant decimal digits.
	2.  Function "eout" below assumes that the maximum exponent of 10
	which can be represented is no bigger than 999.  This may not be
	true for long doubles, and provision for extra digits will be
	needed.


	This code has now been modified to use type long double exclusively,
	under the assumption that the math chips and the software floating-
	point library all convert all floats and doubles to long double
	internally.  It makes more sense to do this conversion once on input
	and once on output rather than multiple times.
*/

#define _NOIEEE_

#include <string.h>
#include <math.h>

#ifdef _NOIEEE_
#   include "fakeieee.h"
#else
#   include <ieee.h>
#endif

static char Infinity[]    = "[+Infinity]";
static char NegInfinity[] = "[-Infinity]";
static char NotANumber[]  = "[NotANumber]";

// structure used to pass around the current status of the value

struct float_data {
	int exp;        // the power of 10
	long double remain; // normalized range [1.0, 10.0), unprinted digits
	int is_neg;     // is this a negative value
	int is_zero;    // is value zero
	int showpoint;  // do we force display of decimal point
	int precision;  // precision to display (number of digits)
    int expchar;    // 'e' or 'E' in scientific format
};


/*
 * Funny version of frexp() for long doubles which just returns
 * the exponent.  We don't have long double format published in any
 * header, so we just use non-portable magic code to do the job.
 * Some day we may have a real library version of frexpl(), which
 * should then be used.
 */

static int lfrexp(long double x)
{
	// get last 15 bits of the long double
    int res = (((unsigned int*) &x)[4]) & 0x7FFF;
    return (res == 0) ? 0 : res - 0x3FFF;
}

//  Normalize representation of floating value as signed magnitude in
//  range [1.0, 10.0) times 10^exp, or as 0.0

static void normalize( long double val, float_data& eval )
{
    eval.is_neg = 0;
    eval.is_zero = 0;
    eval.exp = 0;

    switch( fpclass(val) )
        {
        case FPSIGNAN:
        case FPQUIETNAN:
            break;              // cannot process NaN's

        case FPNEGINF:
            eval.is_neg = 1;
            val = fpcopysign(val, 1.0); // make into positive infinity
            // fall through

        case FPPOSINF:
            break;

        case FPNEGZERO:
            eval.is_neg = 1;
            val = 0.0;          // make into non-negative zero
            // fall through

        case FPPOSZERO:
            eval.is_zero = 1;
            break;

        case FPNEG:
        case FPNEGDENOR:
            eval.is_neg = 1;
            val = -val;         // make positive
            // fall through

        case FPPOSDENOR:
        case FPPOS:
		// processing for non-special values

			// this calculation is not exact, but we fix it up later
            int exp = lfrexp(val);             // get exponent base 2
            exp = (int) (((long) exp * 2466L) / 8192); // approx exponent base 10
            eval.exp = exp;

            if( exp != 0 )
                {
                int neg = 0;
                if( exp < 0 )
                    {
                    exp = -exp;
                    neg = 1;
                    }
                long double power = 1.0;
                long double temp = 10.0;
                do  {
                    while( exp != 0  &&  (exp & 1) == 0 )
						{
                        temp *= temp;
                        exp >>= 1;
                        }
                    power *= temp;
                    exp--;
                    } while( exp != 0 );
                if( neg )
                    val *= power;
                else
                    val /= power;
                }

            // fix up if out of range (we know it can't be >= 10.0)
            if( val < 1.0 )
                {
                val *= 10.0L;
                eval.exp -= 1;
                }
            break;
        }

    eval.remain = val;
}


// Return the ASCII representation of the next digit of the value,
// adjusting remainder.

static char next_digit( float_data& eval )
{
    int digit;

    digit = (int) eval.remain;
    eval.remain = 10.0L * (eval.remain - (long double) digit);
    return digit + '0';
}


// Generate ASCII digits for value in float_data.
// Return pointer to the end of the string.

static char* flt_out( float_data& eval, char *pt )
{
    int count, exp;
#pragma -eff
    if( fpisnan(eval.remain) )
        {
        strcpy(pt, NotANumber);
        pt += strlen(pt);
        }
#pragma .eff

    else if( ! fpfinite(eval.remain) )
        {
        if( eval.is_neg )
            strcpy(pt, NegInfinity);
        else
            strcpy(pt, Infinity);
        pt += strlen(pt);
        }
    else
        {
        // integer portion
        exp = eval.exp + 1; // add 1 to make subsequent tests cheaper
        if( exp <= 0 )
            *(pt++) = '0';
        else
            for( ;  exp > 0;  exp-- )
                *(pt++) = next_digit(eval);

		// possible decimal point
        if( eval.precision  ||  eval.showpoint )
            *(pt++) = '.';

        // fraction portion
        for( count = eval.precision;  count > 0;  count-- )
            {
            if( exp < 0 )
                {
                *(pt++) = '0';
                exp++;
                }
            else
                *(pt++) = next_digit(eval);
            }

        // maybe strip trailing fractional zeros
        if( eval.precision  &&  ! eval.showpoint )
            {
			while( pt[-1] == '0' )
                --pt;
            if( pt[-1] == '.' )
                --pt;   // don't leave a bare decimal point
            }

        *pt = '\0';
        }

    return pt;
}


// "digits" = normalized fraction digits affected by rounding.

static void round( float_data& eval, int digits )
{
    // roundamt[n] == 1/2 of nth digit
    const int sigdig = 20;  // see note in introduction
    static long double roundamt[sigdig+1] =
    { 5.0e-1,  5.0e-2,  5.0e-3,  5.0e-4,  5.0e-5,  5.0e-6,
      5.0e-7,  5.0e-8,  5.0e-9,  5.0e-10, 5.0e-11, 5.0e-12,
      5.0e-13, 5.0e-14, 5.0e-15, 5.0e-16, 5.0e-17, 5.0e-18,
      5.0e-19, 5.0e-20, 5.0e-21
    };

	switch( fpclass(eval.remain) )
        {
        case FPSIGNAN:
        case FPQUIETNAN:
        case FPNEGINF:
        case FPNEGZERO:
        case FPPOSZERO:
        case FPPOSINF:
            break;      // no rounding needed or possible

        default:      // finite value may require rounding
            if(digits >= 0 && digits <= sigdig)
                {
                eval.remain += roundamt[digits];
                if( eval.remain >= 10.0L )
                    {
                    eval.remain /= 10.0L;
                    eval.exp++;
                    }
				}
            break;
        }
}


// generate a number in scientific format
// if values greater than 10^999 are possible, see note in introduction

static void eout( float_data& eval, char* buf )
{
    // rounding can affect exponent, so do it first
    round(eval, eval.precision);

    // save exponent data
    int negexp = 0;     // is exponent negative
    int exp = eval.exp;     // absolute value of exponent
    if( exp < 0 )
        {
        negexp = 1;
        exp = -exp;
        }

    // generate digits field
    eval.exp = 0;   // for flt_out
    char* pt = flt_out(eval, buf);

    if( fpfinite(eval.remain) )
        {
        // generate exponent field
        *(pt++) = eval.expchar;
        if( negexp )
            *(pt++) = '-'; // always a sign
        else
            *(pt++) = '+';
        if(exp >= 100)
            {
            if (exp >= 1000)
                {
                *(pt++) = '0' + exp / 1000; // fourth digit needed
                exp %= 1000;
                }
            *(pt++) = '0' + exp / 100;      // third digit needed
            exp %= 100;
            }
		*(pt++) = '0' + exp / 10;           // at least two digits
        *(pt++) = '0' + exp % 10;
        *(pt++) = '\0';
        }
}


oprnstream _FAR & oprnstream::operator<< (long double d)
{
	float_data eval;
	char buf[60];   // big enough for any floating result

	normalize(d, eval);
	eval.expchar = (flags() & ios::uppercase) ? 'E' : 'e';
	eval.precision = precision();
	if( eval.precision <= 0 )
		eval.precision = 6; // default value for precision
	eval.showpoint = (flags() & ios::showpoint) != 0;

	if( flags() & ios::fixed )
		{
		round(eval, eval.exp + eval.precision);
		flt_out(eval, buf);
		}
	else if( flags() & ios::scientific )
		eout(eval, buf);
	else if( eval.exp < -4  ||  eval.precision < eval.exp )
		eout(eval, buf);    // default to scientific
	else
		{          // default to fixed
		round(eval, eval.exp + eval.precision);
		flt_out(eval, buf);
		}

	char* prefix = 0;
	if( eval.is_neg )
		prefix = "-";
	else if( ! eval.is_zero  &&  (flags() & ios::showpos) )
		prefix = "+";

	// now we have a formatted string for output, to be possibly padded
	outstr(buf, prefix);
	return *this;
}

// extract from streambuf, insert into this oprnstream
oprnstream _FAR & oprnstream::operator<< (streambuf* s)
{
	if( opfx() )		{
		int c;
		while( (c = s->sbumpc()) != EOF )	{
			if( putbuf(c) == EOF )  {
				setstate(ios::badbit);
				break;
			}
		}
	}
	osfx();
	return *this;
}

/*
 * Convert val>=0 to ascii in buffer b, with implied numeric base.
 * Digits are acquired in reverse order, so we start at the end of the
 * buffer and work forward.
 * 'b' initially points to the end of the buffer, and is assumed big enough.
 * Return a pointer to the beginning address of the null-terminated string.
 */

// decimal conversion

static char* todec(char *b, unsigned long val)
{
	*b = '\0';
	do  {
		*--b = ((int) (val % 10)) + '0';
		val /= 10;
		} while( val );
	return b;
}

// octal conversion

static char* tooct(char *b, unsigned long val)
{
	*b = '\0';
    do  {
        *--b = (val & 7) + '0';
        val /= 8;   // let compiler make this a shift if appropriate
        } while( val );
    return b;
}

// hex conversion, with indicator for uppercase or lowercase letters

static char *tohex(char *b, unsigned long val, int upper)
{
    static char digits[2][16] = {
    {'0','1','2','3','4','5','6','7','8','9','a','b','c','d','e','f'},
    {'0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F'}
    };
    char *d = upper ? &digits[1][0] : &digits[0][0];
    *b = '\0';
    do  {
        *--b = d[(int)val & 0xf];
        val /= 16;  // let compiler make this a shift if appropriate
        } while( val );
    return b;
}


// format and insert a signed long

oprnstream _FAR & oprnstream::operator<< (long l)
{
    char buf[MaxCharsInLong];   // result of conversion
    char *prefix = 0;           // displayed numeric prefix string
    char *p;

    // find conversion base
    int base = (flags() & ios::hex) ? 16 : ((flags() & ios::oct) ? 8 : 10);

    // do we treat this as negative?  (oct and hex are unsigned)
    int neg = base == 10  &&  l < 0;

    // value to use, exclusive of sign
    unsigned long ul = neg ? -l : l;

    if( base == 10 )
        {
		p = todec(buf + MaxCharsInLong - 1, ul);

        // compute any sign prefix
        if( ul )
            if( neg )
                prefix = "-";
            else if( flags() & ios::showpos )
                prefix = "+";
        }
    else if( base == 16 )
        {
        int upper = (flags() & ios::uppercase) != 0;
        p = tohex(buf + MaxCharsInLong - 1, ul, upper);

        // compute any base prefix
        if( flags() & ios::showbase )
            prefix = upper ? "0X" : "0x";
        }
    else /* base == 8 */
        {
        p = tooct(buf + MaxCharsInLong - 1, ul);

        // compute any base prefix
        if( flags() & ios::showbase )
            prefix = "0";
        }

    // now we have a formatted string for output, to be possibly padded
	outstr((char*)p, prefix);
    return *this;
}


// format and insert an unsigned long
oprnstream _FAR & oprnstream::operator<< (unsigned long ul)
{
    char buf[MaxCharsInLong];
    char *prefix = 0;       // displayed numeric prefix string
    char *p;


    if( flags() & ios::hex )
		{
        int upper = (flags() & ios::uppercase) != 0;
        p = tohex(buf + MaxCharsInLong - 1, ul, upper);

        // compute any base prefix
        if( flags() & ios::showbase )
            prefix = upper ? "0X" : "0x";
        }
    else if( flags() & ios::oct )
        {
        p = tooct(buf + MaxCharsInLong - 1, ul);

        // compute any base prefix
        if( flags() & ios::showbase )
            prefix = "0";
        }
    else
        {
        p = todec(buf + MaxCharsInLong - 1, ul);

        // compute any sign prefix
        if( ul  &&  (flags() & ios::showpos) )
            prefix = "+";
        }

    // now we have a formatted string for output, to be possibly padded
    outstr((char*)p, prefix);
    return *this;
}


oprnstream _FAR & oprnstream::operator<< (signed char c)
{
	if( opfx() )
		{
		int pad = width(0) - 1;

		// pad on left (right-adjust) if needed -- the default case
		if( ! (x_flags & (ios::left | ios::internal)))
			{
			while( --pad >= 0 )
				{
				if( putbuf(x_fill) == EOF )
					{
					setstate(ios::badbit);
					break;
					}
				}
			}

		// output the data
		if( ! fail() )
			if( putbuf(c) == EOF )
                setstate(ios::badbit);

        // pad on right (left-adjust) if needed
        if( ! fail()  &&  (x_flags & ios::left) )
            {
            while( --pad >= 0 )
                {
				if( putbuf(x_fill) == EOF )
                    {
                    setstate(ios::badbit);
                    break;
                    }
                }
            }
        }
    osfx();

    return *this;
}


#include <string.h>

// Perform the prefix routine, output the string with any needed padding,
// and perform the suffix routine.
// 'd' is the data portion, 'p' is the prefix portion.

void oprnstream::outstr(const char *d, const char *p)
{
	if( opfx() ) {
		int plen = p ? strlen(p) : 0;
		int dlen = d ? strlen(d) : 0;
		int pad = width(0) - plen - dlen;

		// pad on left (right-adjust) if needed -- the default case
		if( ! (x_flags & (ios::left | ios::internal)) )	{
			while( --pad >= 0 )	{
				if( putbuf(x_fill) == EOF )	{
					setstate(ios::badbit);
					break;
				}
			}
		}

		// output the prefix
		if( ! fail()  &&  plen )	{
			if( putn(p, plen) != plen )
				setstate(ios::badbit);
		}
		// internal padding if needed
		if( ! fail()  &&  (x_flags & ios::internal) )	{
			while( --pad >= 0 )	{
				if( putbuf(x_fill) == EOF )	{
					setstate(ios::badbit);
					break;
				}
			}
		}

		// output the data
		if( ! fail()  &&  dlen )
			if( putn(d, dlen) != dlen )
				setstate(ios::badbit);

		// pad on right (left-adjust) if needed
		if( ! fail()  &&  (x_flags & ios::left) )	{
			while( --pad >= 0 )	{
				if( putbuf(x_fill) == EOF )	{
					setstate(ios::badbit);
					break;
				}
			}
		}
	}
	osfx();
}


// insert character representation of the value of the pointer
oprnstream _FAR & oprnstream::operator<< (void* p)
{
	long f = flags();
	setf((ios::hex | ios::showbase | ios::internal),
		 (ios::basefield | ios::showbase | ios::adjustfield));
	fill('0');

#ifndef SHOW_COLON
	/* Output without a separator */
#if defined(__COMPACT__) || defined(__LARGE__) || defined(__HUGE__)
	width(10);
	*this << (unsigned long) p;
#else
	width(6);
	*this << (unsigned) p;
#endif

#else
	/* Output with ':' separator */
	width(6);
#if defined(__LARGE__) || defined(__HUGE__) || defined(__COMPACT__)
	*this << (unsigned)((unsigned long)p >> 16);
	*this << ':';
#ifndef SHOWBASE_ON_OFFSET
	/* Don't show 0x on offset */
	unsetf(ios::showbase);
	width(4);
#else
	width(6);
#endif  /* SHOWBASE_ON_OFFSET */
#endif
	*this << (unsigned)p;
#endif  /* SHOW_COLON */

	flags(f);
	return *this;
}


oprnstream _FAR & oprnstream::write(const signed char* s, int n)
{
	if( ! fail() )
		{
		if( putn((const char*)s, n) != n )
			setstate(badbit);
		}
	return *this;
}

oprnstream _FAR & oprnstream::flush()
{
	if( bp->sync() == EOF )
		setstate(ios::badbit);
	return *this;
}

// manipulators
oprnstream _FAR & oprnstream::operator<< (ios _FAR & (*f)(ios _FAR &))
{
	(*f)(*((ios*)this));
	return *this;
}

// flush the oprnstream
oprnstream _FAR & flush(oprnstream _FAR & os)
{
	os.flush();
	return os;
}

// implementation of opfx -- output prefix operations
int oprnstream::do_opfx()
{
	if( ! fail() )
		{
		if( x_tie )
			x_tie->flush();
		return 1;
		}

	return 0;
}

// implementation of osfx -- output suffix operations
void oprnstream::do_osfx()
{
	if( ! fail()  && (x_flags & ios::unitbuf) )
		flush();
#if !defined( _RTLDLL )
	if( x_flags & ios::stdio )
		{
		cout.flush();
		clog.flush();
		}
#endif
}


// set the put pointer's position
oprnstream _FAR & oprnstream::seekp(streamoff off, ios::seek_dir dir)
{
	if( bad()  ||  bp->seekoff(off, dir, ios::out) == EOF )
		setstate(ios::failbit);
	return *this;
}

// set the put pointer's position
oprnstream _FAR & oprnstream::seekp(streampos pos)
{
	if( bad()  ||  bp->seekpos(pos, ios::out) == EOF )
		setstate(ios::failbit);
	return *this;
}

// read the put pointer's position
streampos oprnstream::tellp()
{
	streampos p = EOF;
	if( bad()  ||  (p = bp->seekoff(0, ios::cur, ios::out)) == EOF )
		setstate(ios::failbit);
	return p;
}

// does no initialization

oprnstream_withassign::oprnstream_withassign(int flag) :
		oprnstream( flag )
{
}

// gets buffer from oprnstream and does entire initialization

oprnstream_withassign _FAR & oprnstream_withassign::operator= (oprnstream _FAR & os)
{
	ios::init(os.rdbuf());
	return *this;
}



// associates streambuf with stream and does entire initialization

oprnstream_withassign _FAR & oprnstream_withassign::operator= (streambuf* s)
{
	ios::init(s);
	return *this;
}


oprnstream::~oprnstream()
{
    flush();
}

oprnstream_withassign::~oprnstream_withassign()
{
}
