// ==========================================================================
// 							Class Specification : CDirSpec
// ==========================================================================

// Header file : dir.h

// Source : Periphere NV (R.Mortelmans)
// Creation Date : 	   2nd November. 1995
// Last Modification : 2nd November. 1995
                          
// //////////////////////////////////////////////////////////////////////////

// Properties:
//	NO	Abstract class (does not have any objects)
//	YES	Derived from CObject

//	NO	Is a Cwnd.                     
//	NO	Two stage creation (constructor & Create())
//	NO	Has a message map
//	NO Needs a resource (template)

//	NO	Persistent objects (saveable on disk)      
//	NO	Uses exceptions

// //////////////////////////////////////////////////////////////////////////

// Desciption :         
//		This class encapsulates the directory specification on Windows NT
//		supported file systems (FAT, VFAT, NTFS, HPFS)
//		A directory consists of a drive specification, followed a subdirectory

// Remark:
//		Directory specifications are always represented in upper case characters
//		 An implicit conversion is done when the supplied names are in lower case;
//		All member functions starting with "Do" will physically change something
//		 on disk.  Other functions only change the internal data.
//		E.g. The normal destructor only destroys the internal representation,
//		 to actually remove a directory use DoRemove()

// Prerequisites (necessary conditions):
//		

/////////////////////////////////////////////////////////////////////////////
#ifndef __DIR_H__
#define __DIR_H__

class CDirSpec : public CObject
{
DECLARE_DYNAMIC(CDirSpec)

// Data members -------------------------------------------------------------
public:
	
protected:
	CString 	m_sDrive;
	CString		m_sSubdirectory;

private:
	
// Member functions ---------------------------------------------------------
public:
	CDirSpec();
	// --- In  :
	// --- Out : 
	// --- Returns :
	// --- Effect : Contructor of object
	//				It will initialize the internal state

    CDirSpec(const char* pszDirectory);
	// --- In  : pszDirectory : The new drive and directory of the object
	// --- Out : 
	// --- Returns : 
	// --- Effect : 
	// --- Effect : Contruction of an object together with a SetDirectory()
	//				Notice that whether the directory specification is valid or not
	//				cannot be checked in this way

	#ifdef WIN32
	BOOL IsChildDir(LPWIN32_FIND_DATA lpFindFileData) const;
	#endif
	
    CDirSpec(const CDirSpec& dirSrc);
	// --- In :	dirSrc : Directory object which will be copied
	// --- Out : 
	// --- Returns :
	// --- Effect : Copy contruction.
	
    BOOL AppendDirectory(const CDirSpec dirSecond);
	// --- In :	dirSecond : The directory object which will be appendec to the object
	// --- Out : 
	// --- Returns : Whether drive and directory specification have a correct format
	// --- Effect : The drive specification of the second directory specification
	//				must be empty or the same as that of this object.
	//				The subdirectory of the second dir spec may not start with
	//				back slash.
	//				It is interpreted as a subdirectory beneath the subdirectory
	//				of this object
	
	const CDirSpec& operator=(const CDirSpec& dirSrc);
	// --- In :	dirSrc : Directory object which will be assign to 'this' directory object
	// --- Out:
	// --- Returns: 
	// --- Effect : Assignment operator
	
	CString GetDrive() const;
	// --- In  :
	// --- Out : 
	// --- Returns :The drive specification of this object
	//			 	It consists of a drive letter and a colon
	//			 	or an empty string
	// --- Effect : 
	BOOL SetDrive(const char* pszDrive);
	// --- In  : pszDrive : The new drive letter of the object
	// --- Out : 
	// --- Returns : Whether drive specification has a correct format
	//				 Whether it actually exitst is not checked
	// --- Effect : 
	void ForceSetDrive(const char* pszDrive);
	// --- In  : pszDrive : The new drive letter of the object
	// --- Out : 
	// --- Returns : 
	// --- Effect : This function extracts illegal characters and thus
	//				will always succeeds
	
	CString GetSubdirectory() const;
	// --- In  :
	// --- Out : 
	// --- Returns :The subdirectory specification of this object
	//			 	It only ends in a back slash when it is the root directory
	//			 	The result may be an empty string
	// --- Effect : 
	BOOL SetSubdirectory(const char* pszSubdirectory);
	// --- In  : pszSubdirectory : The new subdirectory of the object
	// --- Out : 
	// --- Returns : Whether subdirectory specification has a correct format
	//				 Wild characters are not allowed
	//				 Whether it actually exitst is not checked
	// --- Effect : 
	void ForceSetSubdirectory(const char* pszSubdirectory);
	// --- In  : pszSubdirectory : The new subdirectory of the object
	// --- Out : 
	// --- Returns : 
	// --- Effect : This function extracts illegal characters and thus
	//				will always succeeds
	
	CDirSpec GetLastSubdirectory() const;
	// --- In  :
	// --- Out : 
	// --- Returns :The last subdirectory specification of this object
	//				Several subdirectories can be seperated by black slashes,
	//				only the last one is returned
	//				(together with the drive specification)
	//			 	It only ends in a back slash when it is the root directory
	// --- Effect : 
	void RemoveLastSubdirectory();
	// --- In  :
	// --- Out : 
	// --- Returns :
	// --- Effect : Removes the last subdirectory specification of this object
	//				Several subdirectories can be seperated by black slashes,
	//				only the last one is removed
	//			 	The root directory is never removed

	CString GetDirectory() const;
	// --- In  :
	// --- Out : 
	// --- Returns :The drive and directory specification of this object
	//			 	It only ends in a back slash when it is the root directory
	//			 	The result may be an empty string
	// --- Effect : 
	BOOL SetDirectory(const char* pszDirectory);
	// --- In  : pszDirectory : The new drive and directory of the object
	// --- Out : 
	// --- Returns : Whether drive and directory specification have a correct format
	//				 Whether it actually exitst is not checked
	// --- Effect : 
	void ForceSetDirectory(const char* pszDirectory);
	// --- In  : pszDirectory : The new drive and directory of the object
	// --- Out : 
	// --- Returns : 
	// --- Effect : This function extracts illegal characters and thus
	//				will always succeeds
	
	CString GetFileSystemType();
	// --- In  : 
	// --- Out : 
	// --- Returns : The name identifying the file system of this drive
	//				 When the drive spec is not specified, an empty string
	//				 is returned
	// --- Effect : Determines the file system.  
	//				Possible values are (among others):
	//				FAT :  File Allocation Table
	//				HPFS : High Performance File System
	//				NTFS : New Technology File Sytem
	//				CDFS : Compact Disk File System
	//				Notice that WIN16 can only detect FAT-systems

	BOOL MakeTemp();
	// --- In  : 
	// --- Out : 
	// --- Returns : Whether the unique directory could be determined
	// --- Effect : Replaces the directory specification by that of the
	//				temporary directory
	
	BOOL MakeUnique();
	// --- In  : 
	// --- Out : 
	// --- Returns : Whether a unique name could be made
	// --- Effect : Replaces the directory specification by another 
	//				directory specification	that does not yet exist on disk
	//				So this can be used to make unique directory names
	
	BOOL MakeLargestExisting();
	// --- In  : 
	// --- Out : 
	// --- Returns : Whether a unique name could be made
	// --- Effect : Removes subdirectories from the directory specification 
	// 				until a directory specification is found that actually exists on disk
	//				(This will become the current directory)
	//				The drive spec is never changed
	//				The resulting dir spec is an absolute spec

	BOOL Exists() const;
	// --- In  : 
	// --- Out : 
	// --- Returns : Whether the directory (not a file!) physically exists on disk
	// --- Effect :
	
	BOOL IsEmpty() const;
	// --- In  : 
	// --- Out : 
	// --- Returns : Whether the directory specification is empty
	// --- Effect :

	void Empty();
	// --- In  : 
	// --- Out : 
	// --- Returns :
	// --- Effect : CLears the directory specification
	
	BOOL IsEmptyDir() const;
	// --- In  : 
	// --- Out : 
	// --- Returns : Whether the directory is empty = contains no files or other subdirs
	// --- Effect : 
 
	BOOL DoGetCurrentDir();
	// --- In  :
	// --- Out : 
	// --- Returns :Whether it succeeded or not
	// --- Effect : Copies the current DOS drive and directory to the object
	BOOL DoSetCurrentDir() const;	
	// --- In  :
	// --- Out : 
	// --- Returns : Whether it succeeded or not
	// --- Effect : Makes the direcory specified by this object the current
	//				directory
	
	BOOL DoMakeNew() const;
	// --- In  :
	// --- Out : 
	// --- Returns :Whether it succeeded or not
	// --- Effect : Makes a new subdirectory
	//				Several directories can be created in one call
	BOOL DoRemove(BOOL bRecursively = FALSE, BOOL bAlsoRemoveReadOnly = FALSE) const;	
	// --- In  : bRecursively : Whether all underlying subdirectories
	//		    				sould be deleted first
	//			 bAlsoRemoveReadOnly : Whether to also remove files that are 
	//							marked as read only (TRUE) or not (FALSE)
	// --- Out : 
	// --- Returns : Whether it succeeded or not
	// --- Effect : Removes the directory

	BOOL operator==(CDirSpec dirSpec) const;
	BOOL operator!=(CDirSpec dirSpec) const;
	BOOL operator<=(CDirSpec dirSpec) const;
	BOOL operator<(CDirSpec dirSpec) const;
	BOOL operator>=(CDirSpec dirSpec) const;
	BOOL operator>(CDirSpec dirSpec) const;
	// --- In  : dirSpec : Second object to which this object will be compared
	// --- Out : 
	// --- Returns : FALSE or TRUE
	// --- Effect : Comparison operators
	
#ifdef _DEBUG
	virtual void Dump(CDumpContext&) const;
	virtual void AssertValid() const;
#endif //_DEBUG

	virtual ~CDirSpec();
	// --- In  :
	// --- Out : 
	// --- Returns :
	// --- Effect : Destructor of object

protected:      
	CDirSpec FindLowerDirectory() const;
	BOOL RemoveAllFiles(BOOL bAlsoRemoveReadOnly = FALSE) const;

private:

};

#endif
// ==========================================================================
