{ =========================================================================

                                           @@
         @@@@@@@@ @@@@@@                   @@          @@@@
         @@ @@ @@  @@  @@                               @@
         @  @@  @  @@  @@  @@@@    @@@@@  @@@   @@@@@   @@  @@ @@@
            @@     @@@@@      @@  @@   @@  @@  @@   @@  @@   @@  @@
            @@     @@  @@  @@@@@   @@@     @@  @@       @@   @@  @@
            @@     @@  @@ @@  @@     @@@   @@  @@       @@   @@  @@
            @@     @@  @@ @@  @@  @@   @@  @@  @@   @@  @@   @@  @@
           @@@@   @@@@@@   @@@ @@  @@@@@  @@@@  @@@@@  @@@@  @@  @@


{ ========================================================================= }
{                     Copyright  1995, Greg Truesdell                      }
{ ========================================================================= }

Unit TBasicIn;

{ ========================================================================= }
                                  Interface
{ ========================================================================= }

Uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, DCossAPI, KWList;

Const MAX_STACK = 64; { for..next and gosub..return stack depth }

Type

  { trace record }
  TBasTraceState = Record
    Action        : String;
    CurrentLine   : Integer;
    CurrentIndex  : Integer;
    LineText      : String;
  end;

  { for..next stack record }
  TForNextStackRecord = Record
    ControlVar: String[MAX_KEYWORD_LEN];
    ControlVal: String[MAX_KEYWORD_LEN];
    ControlTyp: Integer;
  end;
  TForNextStack = Array [0..MAX_STACK-1] of TForNextStackRecord;

  { File type record for Open/Close/Input/Print }
  OpenRecord = Record
    IOToken   : Integer;  { io type by token tkRead/tkWrite }
    IOResult  : Integer;  { Delphi result code for last operation }
    Filename  : String;   { filename of opened file }
    FileVar   : Text;     { Delphi's file variable }
    Opened    : Boolean;  { Is the file open? }
  end;

  { gosub..return stack record }
  TStackRecord = Record
    FFilename : String[80]; { source filename }
    FLine     : Integer;    { line number }
    FIndex    : Integer;    { line index (character) }
  end;
  TStack = Array [1..MAX_STACK] of TStackRecord;

  { enumerated modes for Trim/LTrim/RTrim }
  TrimMode = (tmBoth, tmLeft, tmRight);

  { new events for TBasicInterpreter }
  TBasTraceEvent = Procedure( var Cancel : Boolean; var BasTraceState : TBasTraceState ) of Object;
  TSyntaxEvent = Procedure( var Continue : Boolean; msg : String ) of Object;

  { the TBasicInterpreter Component }
  TBasicInterpreter = class(TScriptParser)
  Private
    { Private declarations }
    Cancelled     : Boolean;
    FAlphaParser  : Boolean;
    FBasTrace     : TBasTraceEvent;
    FBasicTrace   : Boolean;
    FBeforeExec   : TCancelEvent;
    FOnBadToken   : TCancelEvent;
    FOnEndOfLine  : TLineEvent;
    FOnEndOfText  : TLineEvent;
    FOnError      : TErrorEvent;
    FOnNewLine    : TNewLineEvent;
    FOnNextParse  : TCancelEvent;
    FOnNumExpr    : TNumExprEvent;
    FOnProcedure  : TCancelEvent;
    FOnStrExpr    : TStrExprEvent;
    FOnStrLoop    : TStrExprEvent;
    FOnTrace      : TTraceEvent;
    FOnTypedVar   : TCancelEvent;
    FRunScript    : Boolean;
    FSyntaxError  : TSyntaxEvent;
    FilePtr       : Integer;
    Files         : Array [0..14] of OpenRecord;
    ForNextStack  : TForNextStack;
    IfLevel       : Integer;
    NextPtr       : Integer;
    RetStack      : TStack;
    ScriptName    : String;
    StackPtr      : Integer;
    Trace         : TBasTraceState;
    WhilePtr      : Integer;

    Procedure AssertTokens;
    Procedure EndOfText( LineNo : Integer );
    Procedure EnterStrExpr(var Results: OpenString; var Break, Cancel: Boolean);
    Procedure LoopStrExpr(var Results: OpenString; var Break, Cancel: Boolean);
    Procedure ParserMain( var Cancel: Boolean);
    Function  PopReturn : Boolean;
    Function  PushGosub : Boolean;
    Procedure ScriptEnterNumExpr(var Results: OpenString; vType: Integer; var Break, Cancel: Boolean);
    Procedure ScriptStrExprLoop(var Results: OpenString; var Break, Cancel: Boolean);
    Procedure ScriptTypedVar(var Cancel: Boolean);
    Procedure biExecute(Value: Boolean);
  Protected
    { Protected declarations }

  Public
    { Public declarations }
    Constructor Create(AOwner:TComponent); Override;
    Destructor  Destroy; Override;

    Procedure BasicTrace( Act:String; LText:String );
    Function  ChDirConstruct : Boolean;
    Procedure CloseFiles;
    Function  CloseText : Boolean;
    Procedure ErrorMessage( msg : String );
    Function  EvalComplexRelation(var RetVal : Boolean) : Boolean;
    Function  EvalFileNumber : Integer;
    Function  EvalNumeric( var Results : String; vType : Integer ) : Boolean;
    Function  EvalString( var Results : String ) : Boolean;
    Function  ExistsConstruct( var RetKey : Integer ) : Boolean;
    Function  Expected( str : String ) : Boolean;
    Function  ForNext : Boolean;
    Function  FreeFile( var Results : Integer ) : Boolean;
    Function  GetRelationalExpr( var strLeft, relop, strRight : String; var vType : Integer ) : Boolean;
    Function  GosubConstruct : Boolean;
    Function  GotoConstruct : Boolean;
    Function  IfThen : Boolean;
    Function  InStrConstruct( var Results : String ) : Boolean;
    Function  InputConstruct( var RetStr : String ) : Boolean;
    Function  KillConstruct : Boolean;
    Function  LCaseConstruct( var Results : String ) : Boolean;
    Function  LenConstruct( var Results : String ) : Boolean;
    Function  LookAhead : Integer;
    Function  MsgBoxConstruct( var RetKey : Integer ) : Boolean;
    Function  NowVariable( var Results : String ) : Boolean;
    Function  OpenText : Boolean;
    Function  ReadText : Boolean;
    Function  ReturnConstruct : Boolean;
    Function  ScanForElseEndIf : Boolean;
    Function  ScanForEndIf : Boolean;
    Procedure SkipComment;
    Function  SubStrConstruct( var Results : String ) : Boolean;
    Function  TestRelation( strLeft, relop, strRight : String; vType : Integer ) : Boolean;
    Function  TrimConstruct( var Results : String; Mode : TrimMode ) : Boolean;
    Function  UCaseConstruct( var Results : String ) : Boolean;
    Function  VarConstruct : Boolean;
    Function  WhileWEnd : Boolean;
    Function  WriteText : Boolean;

  Published
    { Published declarations }
    Property Aborted : Boolean Read Cancelled Write Cancelled;
    Property About;
    Property AlphaParser    : Boolean   Read FAlphaParser;
    Property CurrentIndex;
    Property CurrentLine;
    Property Exceptions;
    Property Execute        : Boolean   Read FRunScript   Write biExecute;
    Property LineText;
    Property Text;
    Property TraceBasic     : Boolean   Read FBasicTrace Write FBasicTrace;

    Property OnBadToken     : TCancelEvent  Read FOnBadToken    Write FOnBadToken;
    Property OnBasTrace     :TBasTraceEvent Read FBasTrace      Write FBasTrace;
    Property OnBeforeExec   : TCancelEvent  Read FBeforeExec    Write FBeforeExec;
    Property OnEndOfLine    : TLineEvent    Read FOnEndOfLine   Write FOnEndOfLine;
    Property OnEndOfText    : TLineEvent    Read FOnEndOfText   Write FOnEndOfText;
    Property OnEnterStrExpr : TStrExprEvent Read FOnStrExpr     Write FOnStrExpr;
    Property OnError        : TErrorEvent   Read FOnError       Write FOnError;
    Property OnNewLine      : TNewLineEvent Read FOnNewLine     Write FOnNewLine;
    Property OnNextParse    : TCancelEvent  Read FOnNextParse   Write FOnNextParse;
    Property OnNumExprLoop  : TNumExprEvent Read FOnNumExpr     Write FOnNumExpr;
    Property OnProcedure    : TCancelEvent  Read FOnProcedure   Write FOnProcedure;
    Property OnStrExprLoop  : TStrExprEvent Read FOnStrLoop     Write FOnStrLoop;
    Property OnSyntaxError  : TSyntaxEvent  Read FSyntaxError   Write FSyntaxError;
    Property OnTrace        : TTraceEvent   Read FOnTrace       Write FOnTrace;
    Property OnTypedVar     : TCancelEvent  Read FOnTypedVar    Write FOnTypedVar;
  end;

{ ========================================================================= }
{                             C O N S T A N T S                             }
{ ========================================================================= }

Const

  { token constants }
  { WARNING!  These constants MUST be aligned to the sequence }
  {           used to Add() keywords.                         }

  tkBegin       = 100;
  tkFloat       = 101;
  tkString      = 102;
  tkInteger     = 103;
  tkVar         = 104;
  tkMessageBox  = 105;
  tkInputBox    = 106;
  tkGoto        = 107;
  tkDo          = 108;
  tkIf          = 109;
  tkThen        = 110;
  tkElse        = 111;
  tkEndIf       = 112;
  tkWhile       = 113;
  tkWEnd        = 114;
  tkAs          = 115;
  tkDecimal     = 116;
  tkFor         = 117;
  tkOpen        = 118;
  tkClose       = 119;
  tkRead        = 120;
  tkWrite       = 121;
  tkFreeFile    = 122;
  tkInput       = 123;
  tkPrint       = 124;
  tkSubStr      = 125;
  tkInStr       = 126;
  tkTrim        = 127;
  tkLTrim       = 128;
  tkRTrim       = 129;
  tkLen         = 130;
  tkNow         = 131;
  tkKill        = 132;
  tkExists      = 133;
  tkNot         = 134;
  tkGosub       = 135;
  tkReturn      = 136;
  tkAnd         = 137;
  tkOr          = 138;
  tkUCase       = 139;
  tkLCase       = 140;
  tkChDir       = 141;
  tkNext        = 142;
  tkCStr        = 143;
  tkCInt        = 144;
  tkCFlt        = 145;
  tkCurDir      = 146;
  tkEnd         = 147;

Type
  { local keyword database structure }
  TKeyAddRecord = Record
    Keyword : TKeywordStr;
    Token   : Integer;
    RegNum  : Boolean;
    RegStr  : Boolean;
  end;

Const
  DefaultKeys : Array [tkBegin..tkEnd] of TKeyAddRecord = (
    (Keyword:'Begin';     Token:tkBegin;      RegNum:False; RegStr:False),
    (Keyword:'Float';     Token:tkFloat;      RegNum:False; RegStr:False),
    (Keyword:'String';    Token:tkString;     RegNum:False; RegStr:False),
    (Keyword:'Integer';   Token:tkInteger;    RegNum:False; RegStr:False),
    (Keyword:'Dim';       Token:tkVar;        RegNum:False; RegStr:False),
    (Keyword:'MessageBox';Token:tkMessageBox; RegNum:True;  RegStr:False),
    (Keyword:'InputBox';  Token:tkInputBox;   RegNum:True;  RegStr:True),
    (Keyword:'Goto';      Token:tkGoto;       RegNum:False; RegStr:False),
    (Keyword:'Do';        Token:tkDo;         RegNum:False; RegStr:False),
    (Keyword:'If';        Token:tkIf;         RegNum:False; RegStr:False),
    (Keyword:'Then';      Token:tkThen;       RegNum:False; RegStr:False),
    (Keyword:'Else';      Token:tkElse;       RegNum:False; RegStr:False),
    (Keyword:'EndIf';     Token:tkEndIf;      RegNum:False; RegStr:False),
    (Keyword:'While';     Token:tkWhile;      RegNum:False; RegStr:False),
    (Keyword:'WEnd';      Token:tkWEnd;       RegNum:False; RegStr:False),
    (Keyword:'As';        Token:tkAs;         RegNum:False; RegStr:False),
    (Keyword:'Decimal';   Token:tkDecimal;    RegNum:False; RegStr:False),
    (Keyword:'For';       Token:tkFor;        RegNum:False; RegStr:False),
    (Keyword:'Open';      Token:tkOpen;       RegNum:False; RegStr:False),
    (Keyword:'Close';     Token:tkClose;      RegNum:False; RegStr:False),
    (Keyword:'Read';      Token:tkRead;       RegNum:False; RegStr:False),
    (Keyword:'Write';     Token:tkWrite;      RegNum:False; RegStr:False),
    (Keyword:'FreeFile';  Token:tkFreeFile;   RegNum:True;  RegStr:False),
    (Keyword:'Input';     Token:tkInput;      RegNum:False; RegStr:False),
    (Keyword:'Print';     Token:tkPrint;      RegNum:False; RegStr:False),
    (Keyword:'SubStr';    Token:tkSubStr;     RegNum:False; RegStr:True ),
    (Keyword:'InStr';     Token:tkInStr;      RegNum:True;  RegStr:False),
    (Keyword:'Trim';      Token:tkTrim;       RegNum:False; RegStr:True ),
    (Keyword:'RTrim';     Token:tkRTrim;      RegNum:False; RegStr:True ),
    (Keyword:'LTrim';     Token:tkLTrim;      RegNum:False; RegStr:True ),
    (Keyword:'Len';       Token:tkLen;        RegNum:True;  RegStr:False),
    (Keyword:'Now';       Token:tkNow;        RegNum:False; RegStr:True ),
    (Keyword:'Kill';      Token:tkKill;       RegNum:False; RegStr:False),
    (Keyword:'Exists';    Token:tkExists;     RegNum:True;  RegStr:False),
    (Keyword:'Not';       Token:tkNot;        RegNum:False; RegStr:False),
    (Keyword:'Gosub';     Token:tkGosub;      RegNum:False; RegStr:False),
    (Keyword:'Return';    Token:tkReturn;     RegNum:False; RegStr:False),
    (Keyword:'And';       Token:tkAnd;        RegNum:False; RegStr:False),
    (Keyword:'Or';        Token:tkOr;         RegNum:False; RegStr:False),
    (Keyword:'UCase';     Token:tkUCase;      RegNum:False; RegStr:True ),
    (Keyword:'LCase';     Token:tkLCase;      RegNum:False; RegStr:True ),
    (Keyword:'ChDir';     Token:tkChDir;      RegNum:False; RegStr:False),
    (Keyword:'Next';      Token:tkNext;       RegNum:False; RegStr:False),
    (Keyword:'CStr';      Token:tkCStr;       RegNum:False; RegStr:True),
    (Keyword:'CInt';      Token:tkCInt;       RegNum:True;  RegStr:False),
    (Keyword:'CFlt';      Token:tkCFlt;       RegNum:True;  RegStr:False),
    (Keyword:'CurDir';    Token:tkCurDir;     RegNum:False; RegStr:True),
    (Keyword:'End';       Token:tkEnd;        RegNum:False; RegStr:False)
    );

{ ========================================================================= }
                               Implementation
{ ========================================================================= }


{ ========================================================================= }
{                            B a s i c  T r a c e                           }
{ ========================================================================= }
Procedure TBasicInterpreter.BasicTrace( Act:String; LText:String );
var
  Cancel : Boolean;
begin

  if (not FBasicTrace) or (not Assigned(FBasTrace)) then Exit;

  Trace.Action := Act;
  Trace.CurrentLine := CurrentLine;
  Trace.CurrentIndex := CurrentIndex;
  Trace.LineText := LineText;

  Cancel := False;
  OnBasTrace( Cancel, Trace );
  if Cancel then Execute := False;

end;

{ ========================================================================= }
{                         E v a l  S t r i n g                              }
{ =========================================================================
    I override the EvalString and EvalNumeric Methods so I can control
    the generation of error messages.
{ ========================================================================= }
Function TBasicInterpreter.EvalString( var Results : String ) : Boolean;
begin
  { call the inherited method }
  Result := inherited EvalString( Results );
  { generate an error only if the script is still running }
  { this avoids the problem of displaying two error messages }
  { when a function calls a function }
  if (not Result) and (inherited Execute) then begin
    ErrorMessage( Results );
  end;
end;

{ ========================================================================= }
{                         E v a l  N u m e r i c                            }
{ =========================================================================
    I override the EvalString and EvalNumeric Methods so I can control
    the generation of error messages.
{ ========================================================================= }
Function TBasicInterpreter.EvalNumeric( var Results : String; vType : Integer ) : Boolean;
begin
  { call the inherited method }
  Result := inherited EvalNumeric( Results, vType );
  { generate an error only if the script is still running }
  { this avoids the problem of displaying two error messages }
  { when a function calls a function }
  if (not Result) and (inherited Execute) then begin
    ErrorMessage( Results );
  end;
end;

{ ========================================================================= }
{           T  B a s i c  I n t e r p r e t e r .  D e s t r o y            }
{ ========================================================================= }
Destructor TBasicInterpreter.Destroy;
begin
  inherited Destroy;
end;

{ ========================================================================= }
{            T  B a s i c  I n t e r p r e t e r .  C r e a t e             }
{ ========================================================================= }
Constructor TBasicInterpreter.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);
  FRunScript := False;
  FAlphaParser := False;
  Inherited AlphaParser := False;
  { lock keywords for KWEditor }
  { in a future version of DCossAPI the }
  { Add() method will provide for locking }
  { of keywords }
  LockedKeywords := (tkEnd-tkBegin) + 1;
  KeyList.Clear;
  FBasicTrace := False;
  AssertTokens;
  StackPtr := 0;
  FillChar(RetStack,SizeOf(TStack),#0);
end;

{ ========================================================================= }
{                            P o p  R e t u r n                             }
{ ========================================================================= }
Function TBasicInterpreter.PopReturn : Boolean;
begin
  Result := False;
  if (StackPtr > 0) then with RetStack[StackPtr] do begin
    if FFilename <> '' then begin
      { load the other file }
    end;
    GotoLocation(FLine,FIndex);
    dec(StackPtr);
  end
  else Exit;
  Result := True;
end;

{ ========================================================================= }
{                            P u s h  G o s u b                             }
{ ========================================================================= }
Function TBasicInterpreter.PushGosub : Boolean;
begin
  Result := False;
  if (StackPtr < MAX_STACK) then begin
    Inc(StackPtr);
    with RetStack[StackPtr] do begin
      FFilename := '';
      FLine := CurrentLine;
      FIndex := CurrentIndex;
    end
  end
  else Exit;
  Result := True;
end;

{ ========================================================================= }
{                           P a r s e r  M a i n                            }
{ ========================================================================= }
Procedure TBasicInterpreter.ParserMain( var Cancel: Boolean);
var
  rc  : Integer;  { generic return code }
  tx  : String;   { generic text buffer }
  ln  : Integer;  { line number }
  ix  : Integer;  { line index }

  { ======================================================================= }
  {                            p t k  W  E n d                              }
  { =======================================================================
      Process a WEnd statement by looking for the corresponding While
      statement label.  While statement labels are stored using the
      template '_While_WEnd_##' as the label name.
  { ======================================================================= }
  Procedure ptkWEnd;
  begin
    if WhilePtr > -1 then begin     { it's ok to try looking }
      { build the label name }
      tx := '_While_Wend_' + IntToHex( WhilePtr, 2 );
      { decrement the while loop level }
      dec(WhilePtr);
      { go }
      GotoLabel(tx);
      ZapVariable(tx);
    end
    else begin
      ErrorMessage('WEnd without While');
      Cancel := True;
    end;
  end;

  { ======================================================================= }
  {                             p t k  N e x t                              }
  { ======================================================================= }
  Procedure ptkNext;
  var
    workint : LongInt;  { working integer value }
    cint    : LongInt;  { control integer }
    workflt : Comp;     { working float value }
    cflt    : Comp;     { control float }
    code    : Integer;  { conversion return code }
    workstr : String[MAX_KEYWORD_LEN]; { working conversion string }
  begin
    if NextPtr > -1 then begin
      { define current for next label name }
      tx := '_FOR_NEXT_' + IntToHex( NextPtr, 2 );
      with ForNextStack[NextPtr] do begin
        { handle integer control variables }
        if ControlTyp = vInteger then begin
          { get the control value }
          workint := StrToInt(ControlVal);
          { get the value of the control variable }
          cint := StrToInt(GetVariable(ControlVar,ControlTyp));
          { increment (currently only by integer 1)}
          inc(cint);
          { set the control variable }
          SetVariable(ControlVar,IntToStr(cint));
          { test : time to exit loop? }
          if cint <= workint then
            { no .. loop back to for..next }
            GotoLabel(tx)
          else begin
            { yes .. clear this stack record and pop }
            FillChar(ForNextStack[NextPtr],SizeOf(TForNextStackRecord),#0);
            dec(NextPtr);
            { will continue at statement after 'NEXT'}
          end;
          { release the for..next label }
          ZapVariable(tx);
        end
        else begin
          val(ControlVal, workflt, code);
          val(GetVariable(ControlVar,ControlTyp), cflt, code);
          cflt := cflt + 1.0;
          str(cflt,workstr);
          SetVariable(ControlVar,workstr);
          if cflt <= workflt then
            GotoLabel(tx)
          else begin
            FillChar(ForNextStack[NextPtr],SizeOf(TForNextStackRecord),#0);
            dec(NextPtr);
          end;
          ZapVariable(tx);
        end;
      end;
    end;
  end;


  { ======================================================================= }
  {                           p t k  E n d  I f                             }
  { =======================================================================
      Process the EndIf statement.  All I do here is decrement the If
      level counter.
  { ======================================================================= }
  Procedure ptkEndIf;
  begin
    if IfLevel < 0 then begin
      ErrorMessage('EndIf without If');
      Cancel := True;
    end
    else dec(IfLevel);
  end;

  { ======================================================================= }
  {                            p t k  E l s e                               }
  { =======================================================================
      Process the Else statement.
  { ======================================================================= }
  Procedure ptkElse;
  begin
    if not ScanForEndIf then begin
      { could not find a corresponding ENDIF statement }
      ErrorMessage('Expected "EndIf"');
      Cancel := True;
    end;
  end;

begin

  { now implement the language }
  FRunScript := inherited Execute;

  case Token of

    0 : ; { ignore null tokens }

    NT_VARIABLE_FOUND:  { do nothing, handled by OnTypedVar event };

    (* I have abandoned the # comment characters.
    NT_POUND,           { comment } *)
    NT_EXCLAMATION  :   SkipComment;
    NT_SEMICOLON,
    NT_COLON        :   ;
    NT_LABEL_FOUND  :   NextToken;
    NT_STRING_CONSTANT : {allows ' or " as a comment character};

    NT_PAST_EOF     : FRunScript := False;

    tkExists        : Cancel := Not ExistsConstruct( rc );
    tkKill          : Cancel := Not KillConstruct;
    tkVar           : Cancel := Not VarConstruct;
    tkMessageBox    : Cancel := Not MsgBoxConstruct( rc );
    tkInputBox      : Cancel := Not InputConstruct( tx );
    tkGoto          : Cancel := Not GotoConstruct;
    tkWhile         : Cancel := Not WhileWEnd;
    tkWEnd          : ptkWend;
    tkIf            : Cancel := Not IfThen;
    tkElse          : ptkElse;
    tkEndIf         : ptkEndIf;
    tkNext          : ptkNext;
    tkOpen          : Cancel := Not OpenText;
    tkClose         : Cancel := Not CloseText;
    tkInput         : Cancel := Not ReadText;
    tkPrint         : Cancel := Not WriteText;
    tkGosub         : Cancel := Not GosubConstruct;
    tkReturn        : Cancel := Not ReturnConstruct;
    tkFor           : Cancel := Not ForNext;
    tkChDir         : Cancel := Not ChDirConstruct;

    else begin
      if Assigned(FOnNextParse) then

        OnNextParse(Cancel)

      else begin
        ErrorMessage( 'Syntax Error: "' + Keyword + '"' );
        CloseFiles;
        Cancel := True;
      end;
    end;

  end;

  { remember to close all open files when the script is terminated }
  if Cancel then begin
    CloseFiles;
    StackPtr := 0;
    FRunScript := False;
  end;

  Application.ProcessMessages;

end;

{ ========================================================================= }
{                         A s s e r t  T o k e n s                          }
{ ========================================================================= }
Procedure TBasicInterpreter.AssertTokens;
var ii : Integer;
begin

  { clear the keyword database }
  ZapKeywords;

  for ii := tkBegin to tkEnd do with DefaultKeys[ii] do begin
    KeyList.AddKey(Keyword,True);
    if RegNum then RegisterNumeric(Token);
    if RegStr then RegisterFunction(Token);
  end;

end;

{ ========================================================================= }
{                            E n d  O f  T e x t                            }
{ ========================================================================= }
Procedure TBasicInterpreter.EndOfText( LineNo : Integer );
begin
  FRunScript := False;
  if Assigned(FOnEndOfText) and (LineNo > 0) then OnEndOfText( LineNo );
end;

{ ========================================================================= }
{                             E x p e c t e d                               }
{ =========================================================================
    Expect a keyword.  If it is not the current keyword, then return false
    and initiate the error message.
{ ========================================================================= }
Function TBasicInterpreter.Expected( str : String ) : Boolean;
var
  tmpToken  : Integer;
  tmpKeyword: String;
begin

  Result := False;

  { peek ahead at the next Keyword }
  PeekNextToken( tmpToken, tmpKeyword );

  if not (UpperCase(tmpKeyword) = UpperCase(str)) then begin
    { not as expected }
    ErrorMessage( 'Expected "' + str + '"' );
    Exit;
  end
  else begin
    NextToken;
    Result := True;
  end;

end;

{ ========================================================================= }
{                            L o o k  A h e a d                             }
{ =========================================================================
    Take a peek into the future: what token lies ahead?  Returns the next
    token without updating the parser's cursor.
{ ========================================================================= }
Function TBasicInterpreter.LookAhead : Integer;
var
  tmpToken  : Integer;
  tmpKeyword: String;
begin
  PeekNextToken( tmpToken, tmpKeyword );
  Result := tmpToken;
end;

{ ========================================================================= }
{                         E r r o r  M e s s a g e                          }
{ ========================================================================= }
Procedure TBasicInterpreter.ErrorMessage( msg : String );
var continue : Boolean;
begin

  { display and handle the error dialog }
  if not Assigned(FSyntaxError) then
    MessageDlg( 'TBasicInterpreter: '+ msg, mtError, [mbOk], 0 );

  { kill everything that needs killing when }
  { the script fails or completes }
  Continue := False;
  if Assigned(FSyntaxError) then OnSyntaxError(Continue, msg);
  Cancelled := not Continue;
  if Cancelled then begin
    CloseFiles;
    Execute := False;
  end;

end;

{ ========================================================================= }
{                           C l o s e  F i l e s                            }
{ ========================================================================= }
Procedure TBasicInterpreter.CloseFiles;
var
  ii : Integer;
begin
  {$I-}
  for ii := 0 to 14 do
    if Files[ii].Opened then begin
      CloseFile(Files[ii].FileVar);
      Files[ii].Opened := false;
      Files[ii].IOResult := IOResult;
      try
        SetVariable('Result',IntToStr(Files[ii].IOResult));
      except { don't wory about it }
      end;
    end;
  {$I+}
end;

{ ========================================================================= }
{                            C l o s e  T e x t                             }
{ =========================================================================
  Syntax:
    Close [#]<filehandle:integer>
{ ========================================================================= }
Function TBasicInterpreter.CloseText : Boolean;
var
  fh : Integer;
begin
  Result := False;

  fh := EvalFileNumber;
  if fh < 1 then Exit;

  BasicTrace('Close File', LineText);

  {$I-}
  if not Files[fh].Opened then begin
    ErrorMessage('File #'+IntToStr(fh)+' not open.');
    Exit;
  end
  else begin
    CloseFile(Files[fh].FileVar);
    Files[fh].Opened := False;
  end;

  {$I+}

  Files[fh].IOResult := fh;
  SetVariable('Result',IntToStr(Files[fh].IOResult));
  Result := True;
  Token := 0; { reset to short-circuit OnTypedVar event }
end;


{ ========================================================================= }
{                       E v a l  F i l e  N u m b e r                       }
{ =========================================================================
  Syntax
    [#]<1..14>

  Notes:
    This is an adjuct function for Open/Close/Input/Print ... etc
{ ========================================================================= }
Function TBasicInterpreter.EvalFileNumber : Integer;
var
  str : String;
begin

  Result := -1;
  NextToken;
  if IsKeyword('#') then NextToken;
  if not EvalNumeric( str, vInteger ) then Exit;

  Result := StrToInt( str );

  if not (Result in [1..14]) then begin
    ErrorMessage('Invalid file number.');
    Exit;
  end;

end;

{ ========================================================================= }
{                             F r e e  F i l e                              }
{ =========================================================================
  Support function for FreeFile Variable
{ ========================================================================= }
Function TBasicInterpreter.FreeFile( var Results : Integer ) : Boolean;
var
  ii : Integer;
begin
  Results := -1;
  for ii := 1 to 14 do
    if not Files[ii].Opened then begin
      Results := ii;
      Exit;
    end;
  BasicTrace(Keyword, IntToStr(Results));
end;

{ ========================================================================= }
{                    G e t  R e l a t i o n a l  E x p r                    }
{ =========================================================================
    This support function parses the stream looking for the following
    grammer:
      <expression> <relop> <expression>

    and returns the expression split in three.

    If your language requires some other relational syntax, say FORTH,
    then you can modify the code to piece it together here.
{ ========================================================================= }
Function TBasicInterpreter.GetRelationalExpr( var strLeft, relop, strRight : String; var vType : Integer ) : Boolean;
var
  tmpKeyword : String;
  tmpToken : Integer;
  ok : Boolean;
begin

  GetRelationalExpr := False;

  strLeft := '';
  strRight := '';
  relop := '';
  vType := -1;

  vType := ExprType;
  if not (vType in [vFloat, vInteger, vString]) then begin
    ErrorMessage('Syntax Error in relational expression: Expected an Expression.');
    Exit;
  end
  else begin
    if vType = vString then begin
      if not EvalString(strLeft) then Exit;
    end
    else if not EvalNumeric(strLeft,vType) then Exit;

    PeekNextToken( tmpToken, tmpKeyword );
    if (not (tmpKeyword[1] in ['<','>','='])) and (vType = vInteger) then begin
      relop := '##';
      GetRelationalExpr := True;
      strRight := '';
      Exit;
    end;

    NextToken;
    relop := '';

    if not (IsKeyword('<') or IsKeyword('=') or IsKeyword('>')) then begin
      ErrorMessage('Syntax Error in relational expression: Expected <,>,<>,>=,<=,=');
      Exit;
    end
    else begin
      relop := Keyword;

      PeekNextToken( tmpToken, tmpKeyword );
      if (tmpKeyword='<') or (tmpKeyword='=') or (tmpKeyword='>') then begin
        NextToken;
        relop := relop + Keyword;
      end;

      NextToken;

      case vType of
        vString:  ok := EvalString( strRight );
        vInteger,
        vFloat:   ok := EvalNumeric( strRight, vType );
      else
        ok := False;
      end;

      GetRelationalExpr := ok;
      {if not ok then ErrorMessage( strRight );}

    end;
  end;
end;

{ ========================================================================= }
{                        G o t o  C o n s t r u c t                         }
{ =========================================================================
  Syntax:
    Goto <label>
{ ========================================================================= }
Function TBasicInterpreter.GotoConstruct : Boolean;
begin

  GotoConstruct := False;
  NextToken;

  BasicTrace('Goto '+Keyword, LineText);
  if IsToken( NT_LABEL_FOUND )
    then GotoConstruct := GotoLabel( Keyword )
    else ErrorMessage('Expected a Label');
  BasicTrace('Goto Line #'+IntToStr(CurrentLine), LineText);

end;

{ ========================================================================= }
{                       G o s u b  C o n s t r u c t                        }
{ ========================================================================= }
Function TBasicInterpreter.GosubConstruct : Boolean;
begin

  Result := False;

  NextToken;

  BasicTrace('Gosub '+Keyword, LineText);
  if not PushGosub then begin
    ErrorMessage('Gosub..Return stack overflow');
    Exit;
  end;
  if IsToken( NT_LABEL_FOUND )
    then Result := GotoLabel( Keyword )
    else ErrorMessage('Expected a Label');
  BasicTrace('Gosub Line #'+IntToStr(CurrentLine), LineText);

end;

{ ========================================================================= }
{                      R e t u r n  C o n s t r u c t                       }
{ ========================================================================= }
Function TBasicInterpreter.ReturnConstruct : Boolean;
begin
  Result := False;
  NextToken;
  BasicTrace('Return ', LineText);
  if not PopReturn then begin
    ErrorMessage('Gosub..Return stack underflow');
    Exit;
  end;
  BasicTrace('Return to Line #'+IntToStr(CurrentLine), LineText);
  Result := True;
end;

{ ========================================================================= }
{                  E v a l  C o m p l e x  R e l a t i o n                  }
{ ========================================================================= }
Function TBasicInterpreter.EvalComplexRelation(var RetVal : Boolean) : Boolean;
var
  Failure   : Boolean;
  vType     : Integer;
  vData     : String;
  relop     : String;
  tmpToken  : Integer;
  tmpKeyword: String;
  expstr    : String;
  Inverse   : Boolean;
  Pipe      : Boolean;
  AndNext   : Boolean;
  OrNext    : Boolean;

begin
  Failure := False;
  Pipe := True;
  AndNext := False;
  OrNext := False;

  repeat

    Inverse := False;
    if token = tkNot then begin
      Inverse := True;
      NextToken;
    end;

    Failure := not GetRelationalExpr( vData, relop, expstr, vType );

    if not Failure then begin
      if AndNext then begin
        if inverse
          then Pipe := Pipe and (not TestRelation( vdata, relop, expstr, vtype ))
          else Pipe := Pipe and TestRelation( vdata, relop, expstr, vtype );
      end
      else if OrNext then begin
        if inverse
          then Pipe := Pipe or (not TestRelation( vdata, relop, expstr, vtype ))
          else Pipe := Pipe or TestRelation( vdata, relop, expstr, vtype );
      end
      else begin
        if inverse
          then Pipe := (not TestRelation( vdata, relop, expstr, vtype ))
          else Pipe := TestRelation( vdata, relop, expstr, vtype );
      end;
    end
    else begin
      Result := False;
      RetVal := False;
      Exit;
    end;

    NextToken;

    AndNext := (Token=tkAnd);
    OrNext := (Token=tkOr);
    if Token in [tkAnd, tkOr] then NextToken;

    { now trap un-connected expressions }
    if not Token in [tkThen, tkGoto, tkAnd, tkOr] then begin
      RetVal := False;
      Result := False;
      Exit;
    end;

  until (Token in [tkThen,tkGoto]) or Failure;
  RetVal := Pipe;
  Result := True;

end;

{ ========================================================================= }
{                               I f  T h e n                                }
{ =========================================================================
  Syntax:
    If <relational-expression> Then <statement>
      .or.
    If relational-expression> Then
      .. statements ..
    Else
      .. statement ..
    EndIf
{ ========================================================================= }
Function TBasicInterpreter.IfThen : Boolean;
var
  vType     : Integer;
  vData     : String;
  relop     : String;
  tmpToken  : Integer;
  tmpKeyword: String;
  Proceed   : Boolean;
  Ok        : Boolean;
  tmpLine   : Integer;
  tmpIndex  : Integer;

  { ======================================================================= }
  {                           S a m e  L i n e                              }
  { =======================================================================
      This support function is used to determine if the next statement
      is on the same line as the If..Then statement.  This is used to
      implement the If .. Then <statement> construct.
  { ======================================================================= }
  Function SameLine : Boolean;
  begin
    SameLine := False;
    { save the current parse environment }
    tmpKeyword := Keyword;
    tmpToken := Token;
    tmpLine := CurrentLine;
    tmpIndex := CurrentIndex;
    { skip to next }
    NextToken;
    { determine if the next token is on the same line }
    if CurrentLine = tmpLine then begin
      SameLine := True;
      Keyword := tmpKeyword;
      Token := tmpToken;
      CurrentIndex := tmpIndex;
      Exit;
    end
    else begin
      Keyword := tmpKeyword;
      Token := tmpToken;
      CurrentLine := tmpLine;
      CurrentIndex := tmpIndex;
    end;
  end;

begin { IfThen }

  IfThen := False;
  Proceed := False;

  BasicTrace(Keyword, LineText);

  NextToken; {skip IF}

  { collect the relational expression }
  if EvalComplexRelation(Proceed) then begin
    { 'Then' is expected to be current }
    if not IsToken(tkThen) then begin
      ErrorMessage('Expected "Then"');
      Exit;
    end;

    { handle if ... then goto <label> construct }
    if LookAhead = tkGoto then begin
      NextToken;
      NextToken;
      { a label is required }
      if not IsToken(NT_LABEL_FOUND) then begin
        ErrorMessage('Expected a Line Label, found "'+Keyword+'"');
        Exit;
      end;
      { syntax ok }
      IfThen := True;
      { decide }
      if Proceed
        then GotoLabel( Keyword )
        else Exit;
    end

    { not GOTO, so continue }
    else begin
      { test the relation for truth }
      if not Proceed then begin

        { The test failed. Now determine how to handle the }
        { process of skipping all intervening statements }
        { until end of line, 'Else' or 'EndIf' }

        { check for single statement on line }
        if SameLine then begin { move to the end of the line }
          { skip and exit }
          CurrentIndex := Length(Text.Strings[CurrentLine]);
          IfThen := True;
          Exit;
        end;

        { Scan for either Else or EndIf. Either one is fine. }
        { if 'Else' is encountered, then continue parsing because }
        { that's how 'Else' works.  If 'EndIf' then it's all over. }
        if ScanForElseEndIf then begin
          IfThen := True;
          { tell the interpreter that it's ok to encounter an EndIf }
          Inc(IfLevel);
        end;

      end

      { handling truth means continuing }
      else begin
        IfThen := True;
        { nest this 'If' if appropriate }
        if not SameLine then Inc(IfLevel);
        Exit;
      end;

    end;

  end else Exit; { error, no relational expression }

end;

{ ========================================================================= }
{                       I n p u t  C o n s t r u c t                        }
{ =========================================================================
  Syntax:
    <variable> = InputBox( <caption>, <prompt>, <default> )
{ ========================================================================= }
Function TBasicInterpreter.InputConstruct( var RetStr : String ) : Boolean;
var
  expstr  : String;
  msg,
  title,
  default : String;
  pMsg,
  pTitle  : PChar;
begin

  BasicTrace(Keyword, LineText);
  InputConstruct := False;
  msg := '';
  title := '';
  default := '';

  NextToken;
  if IsKeyword('(') then begin
    NextToken;
    { retrieve caption string }
    if EvalString(msg) then begin
      NextToken;
      if IsKeyword(',') then begin
        NextToken;
        { retrieve prompt string }
        if EvalString(title) then begin
          NextToken;
          if IsKeyword(',') then begin
            NextToken;
            { retrieve default }
            if EvalString(default) then begin
              NextToken;
              if not IsKeyword(')') then begin
                ErrorMessage('Expected ")"');
                Exit;
              end;
            end
            else Exit;
          end
          else if not IsKeyword(')') then begin
            ErrorMessage('Expected ")"');
            Exit;
          end;
        end
        else Exit;
      end
      else if not IsKeyword(')') then begin
        ErrorMessage('Expected ")"');
        Exit;
      end
    end
    else Exit;
  end else begin
    ErrorMessage('Expected "("');
    Exit;
  end;

  InputConstruct := True;
  RetStr := default;

  Application.ProcessMessages;

  if not InputQuery( title, msg, RetStr ) then begin
    SetVariable('RESULT','0');
    RetStr := '';
  end
  else SetVariable('RESULT','-1');

end;

{ ========================================================================= }
{                       I n  S t r  C o n s t r u c t                       }
{ =========================================================================
  Syntax:
    InStr( <start:integer>, <search:string>, <lookfor:string> )

  Notes:
    Attached to the numeric evaluation loop with RegisterNumeric.
{ ========================================================================= }
Function TBasicInterpreter.InStrConstruct( var Results : String ) : Boolean;
var
  source  : String;   { string to search }
  target  : String;   { string to find }
  start   : Integer;  { start position }
  expstr  : String;   { working expression string }
begin

  Results := '0';
  Result := False;
  start := 1;

  BasicTrace(Keyword, LineText);

  if not Expected('(') then Exit;
  NextToken;
  if not EvalNumeric( expstr, vInteger ) then Exit;
  start := StrToInt(expstr);
  if not Expected(',') then Exit;
  NextToken;
  if not EvalString( source ) then Exit;
  if (start < 1) or (start > (length(source)-start)) then begin
    ErrorMessage('Invalid InStr start value');
    Exit;
  end;
  if not Expected(',') then Exit;
  NextToken;
  if not EvalString( target ) then Exit;
  if not Expected(')') then Exit;

  Result := True;

  { trim source down to the requested start point }
  source := Copy( source, start, 255 );
  start := pos( target, source );
  Results := IntToStr(start);     { numeric return value }

end;

{ ========================================================================= }
{                        K i l l  C o n s t r u c t                         }
{ ========================================================================= }
Function  TBasicInterpreter.KillConstruct : Boolean;
var exp : String;
begin
  Result := False;
  BasicTrace(Keyword, LineText);
  NextToken;
  if not EvalString(exp) then Exit;
  Token := 0;
  DeleteFile(exp);
  Result := True;
end;

{ ========================================================================= }
{                      E x i s t s  C o n s t r u c t                       }
{ ========================================================================= }
Function TBasicInterpreter.ExistsConstruct( var RetKey : Integer ) : Boolean;
var exp : String;
begin
  Result := False;

  BasicTrace(Keyword, LineText);

  if not Expected('(') then Exit;           { expect "(" }
  NextToken;                                { skip to expression }
  if not EvalString( exp ) then Exit;       { expect string expression }
  if not Expected(')') then Exit;           { expect ")" }
  {NextToken;}

  if FileExists(exp)
    then RetKey := -1
    else RetKey := 0;

  Result := True;
end;

{ ========================================================================= }
{                         L e n  C o n s t r u c t                          }
{ =========================================================================
  Syntax:
    Len( <string-expression> )

  Notes:
    Attached to the numeric evaluation loop with RegisterNumeric.

    This, as with many registered functions, returns the result in the
    pass-by-referenced Results variable.  This will then be included in
    the expression being parsed. Like magic.
{ ========================================================================= }
Function TBasicInterpreter.LenConstruct( var Results : String ) : Boolean;
begin

  Results := '0';   { default return is zero }
  Result := False;

  BasicTrace(Keyword, LineText);

  if not Expected('(') then Exit;           { expect "(" }
  NextToken;                                { skip to expression }
  if not EvalString( Results ) then Exit;   { expect string expression }
  if not Expected(')') then Exit;           { expect ")" }

  Results := IntToStr( Length(Results) );
  Result := True;

end;

{ ========================================================================= }
{                       U  C a s e  C o n s t r u c t                       }
{ ========================================================================= }
Function TBasicInterpreter.UCaseConstruct( var Results : String ) : Boolean;
begin

  Results := '';
  Result := False;
  BasicTrace(Keyword,LineText);

  if not Expected('(') then Exit;           { expect "(" }
  NextToken;                                { skip to expression }
  if not EvalString( Results ) then Exit;   { expect string expression }
  if not Expected(')') then Exit;           { expect ")" }

  Results := UpperCase(Results);
  Result := True;

end;

{ ========================================================================= }
{                       L  C a s e  C o n s t r u c t                       }
{ ========================================================================= }
Function TBasicInterpreter.LCaseConstruct( var Results : String ) : Boolean;
begin

  Results := '';
  Result := False;
  BasicTrace(Keyword,LineText);

  if not Expected('(') then Exit;           { expect "(" }
  NextToken;                                { skip to expression }
  if not EvalString( Results ) then Exit;   { expect string expression }
  if not Expected(')') then Exit;           { expect ")" }

  Results := LowerCase(Results);
  Result := True;

end;

{ ========================================================================= }
{                       C h  D i r  C o n s t r u c t                       }
{ ========================================================================= }
Function TBasicInterpreter.ChDirConstruct : Boolean;
var Results : String;
begin
  Result := False;
  NextToken;
  if not EvalString(Results) then Exit;
  Result := True;
  try
    ChDir(Results);
  except
    ErrorMessage('Directory "'+Results+'" does not exist.');
  end;
end;


{ ========================================================================= }
{                      M s g  B o x  C o n s t r u c t                      }
{ =========================================================================
  Syntax:
    MessageBox( <string-expression> )
{ ========================================================================= }
Function TBasicInterpreter.MsgBoxConstruct( var RetKey : Integer ) : Boolean;
var
  msg : String;
begin

  MsgBoxConstruct := False;
  msg := '';
  BasicTrace(Keyword, LineText);

  NextToken;

  if IsKeyword('(') then begin

    NextToken;

    { retrieve message string }
    if EvalString(msg) then begin

      NextToken;

      if not IsKeyword(')') then begin
        ErrorMessage('Expected ")"');
        Exit;
      end

    end
    else begin
      Exit;
    end;

  end else begin
    ErrorMessage('Expected "("');
    Exit;
  end;

  MsgBoxConstruct := True;
  Application.ProcessMessages;
  RetKey := MessageDlg( Msg, mtInformation, mbOkCancel, 0 );

end;

{ ========================================================================= }
{                          N o w  V a r i a b l e                           }
{ ========================================================================= }
Function TBasicInterpreter.NowVariable( var Results : String ) : Boolean;
begin

  Result := True;
  Results := FormatDateTime('yyyy/mm/dd hh:nn:ss', Now );
  BasicTrace(Keyword, Results);

end;

{ ========================================================================= }
{                             O p e n  T e x t                              }
{ =========================================================================
  Syntax:
    Open <filename:string> for <READ|WRITE> as [#]<filenumber:integer>
{ ========================================================================= }
Function TBasicInterpreter.OpenText : Boolean;
var
  Filename : String;
  FileNo   : String;
  FileHandle : Integer;
  FileToken : Integer;
  Code : Integer;
begin

  Result := False;

  BasicTrace(Keyword, LineText);
  NextToken;

  if not EvalString( Filename ) then Exit;

  if Filename = '' then begin
    ErrorMessage( 'Invalid Filename.' );
    Exit;
  end;

  NextToken;

  if Not IsToken(tkFor) then begin
    ErrorMessage( 'Expected "For" in Open statement.' );
    Exit;
  end;

  NextToken;

  if not (Token in [tkRead, tkWrite]) then begin
    ErrorMessage( 'Expected "Read" or "Write" in Open statement.');
    Exit;
  end else FileToken := Token;

  NextToken;

  if not IsToken(tkAs) then begin
    ErrorMessage('Expected "As" in Open statement.');
    Exit;
  end;

  FileHandle := EvalFileNumber;

  if FileHandle < 1 then begin
    {ErrorMessage('Invalid File Number');}
    Exit;
  end;

  { =============== }
  Token := 0;  { reset the token to avoid variable evaluation }
                      { since OnTypedVar is activated after OnNextParse }

  if Files[FileHandle].Opened then begin
    ErrorMessage( 'File #' + IntToStr(FileHandle) + ' is already open.');
    exit;
  end;

  Files[FileHandle].Filename := Filename;

  case FileToken of

    tkRead: with Files[FileHandle] do begin

              {$I-}
              AssignFile(FileVar, Filename);
              Reset(FileVar);
              {$I+}

              Code := System.IOResult;
              Files[FileHandle].IOResult := Code;
              SetVariable('Result',IntToStr(Code));
              if not (Code = 0) then begin
                ErrorMessage( 'Error #' + IntToStr(Code) + ' while opening ' + Filename );
                Exit;
              end;

              Opened := True;
              IOToken := tkRead;

            end;

    tkWrite: with Files[FileHandle] do begin

              {$I-}
              AssignFile(FileVar, Filename);
              Rewrite(FileVar);
              {$I+}

              Code := System.IOResult;
              Files[FileHandle].IOResult := Code;
              SetVariable('Result',IntToStr(Code));

              if not (Code = 0) then begin
                ErrorMessage( 'Error #' + IntToStr(Code) + ' while opening ' + Filename );
                Exit;
              end;

              Opened := True;
              IOToken := tkWrite;

            end;

  end;

  Result := True;

end;

{ ========================================================================= }
{                             R e a d  T e x t                              }
{ =========================================================================
  Syntax:
    Input [#]<filenumber:integer>, <variable>
{ ========================================================================= }
Function TBasicInterpreter.ReadText : Boolean;
var
  fh : Integer;
  str : String;
begin

  Result := False;
  BasicTrace(Keyword, LineText);

  { collect the file number }
  fh := EvalFileNumber;
  if fh < 1 then Exit;

  { report error if the file is not open }
  if not Files[fh].Opened then begin

    ErrorMessage('File #'+IntToStr(fh)+' not open.');
    Exit;

  end;

  { check file mode for READ }
  if not (Files[fh].IOToken = tkRead) then begin
    ErrorMessage('File not opened for Read.');
    Exit;
  end;

  NextToken;

  if not IsKeyword(',') then begin
    ErrorMessage('Expected ","');
    Exit;
  end;

  NextToken;

  { only a string variable is acceptable }
  if not (IsToken(NT_VARIABLE_FOUND) and (AsVarType = vString)) then begin
    ErrorMessage('String variable expected.');
    Exit;
  end;

  { if already at EOF, then return IOResult = -1 }
  if Eof(Files[fh].FileVar) then

    Files[fh].IOResult := -1

  else begin

    { read a line of text }
    str := '';
    {$I-}
    ReadLn( Files[fh].FileVar, str );
    {$I+}

    { record the IO Result status}
    Files[fh].IOResult := System.IOResult;

  end;

  { set the status variable and the input variable }
  SetVariable('Result',IntToStr(Files[fh].IOResult));
  SetVariable( Keyword, str );
  Result := True;

  { NOTE: REMEMBER TO SET TOKEN TO ZERO WHENEVER A VARIABLE }
  {       IS THE LAST ITEM IN A COMMAND.  THIS TELLS THE    }
  {       PARSER NOT TO PASS THIS TOKEN TO OnTypedVar       }
  Token := 0; { always needed when last item is a variable }

end;

{ ========================================================================= }
{                    S c a n  F o r  E l s e  E n d  I f                    }
{ =========================================================================
    This support function scans the stream for 'Else' or 'EndIf'
    statements that belong to the current If nesting level.

    You can modify this code if you need a different syntax, say something
    like IFF(A=B,X$="HELLO",X$="NEVER MIND")
{ ========================================================================= }
Function TBasicInterpreter.ScanForElseEndIf : Boolean;
var
  ok : Boolean;
  tmpLine : Integer;
begin

  { scan and skip until EndIf, Else or end of line. }
  while not (IsToken(tkEndIf) or IsToken(tkElse) or IsToken(NT_PAST_EOF)) do begin

    { skip comments }
    SkipComment;

    { determine if it is another nested level }
    if IsToken(tkIf) then begin

      { record current line to determine what style of If it is }
      tmpLine := CurrentLine;
      NextToken;

      { if more lines, then scan for EndIf for this block }
      if (tmpLine < CurrentLine) and (not ScanForEndIf) then begin
        { endif not found ... this is an error }
        ScanForElseEndIf := False;
        Exit;
      end;

    end;

    { It's also an error if the end of the world is found }
    { before the end of the construct }
    if IsToken(NT_PAST_EOF) then begin
      ScanForElseEndIf := False;
      Exit;
    end;

    { continue scanning }
    NextToken;

  end;

  ScanForElseEndIf := True;

end;

{ ========================================================================= }
{                        S c a n  F o r  E n d  I f                         }
{ ========================================================================= }
Function TBasicInterpreter.ScanForEndIf : Boolean;
var ok : Boolean;
begin

  while not (IsToken(tkEndIf) or IsToken(NT_PAST_EOF)) do begin

    SkipComment;

    if IsToken(tkIf) then begin
      NextToken;
      if not ScanForEndIf then begin
        ScanForEndIf := False;
        Exit;
      end;
    end;

    if IsToken(NT_PAST_EOF) then begin
      ScanForEndIf := False;
      Exit;
    end;

    NextToken;

  end;

  ScanForEndIf := True;

end;

{ ========================================================================= }
{                      S u b  S t r  C o n s t r u c t                      }
{ =========================================================================
  Syntax:
    SubStr(<source:string>,<start:integer>[,<length:integer>)

  Notes:
    Attached to the string evaluation loop with RegisterFunction.
{ ========================================================================= }
Function TBasicInterpreter.SubStrConstruct( var Results : String ) : Boolean;
var
  expstr : String;
  target : String;
  start, len : Integer;
begin

  SubStrConstruct := False;
  Results := '';
  BasicTrace(Keyword, LineText);

  if not Expected('(') then Exit;

  NextToken;
  if not EvalString(target) then Exit;

  start := 1;
  len := Length(target);

  if not Expected(',') then exit;

  NextToken;
  if not EvalNumeric(expstr,vInteger) then exit;

  start := StrToInt(expstr);

  if LookAhead = NT_COMMA then begin

    NextToken;
    NextToken;

    if not EvalNumeric(expstr,vInteger) then Exit;

    len := StrToInt(expstr);

  end;

  if (start > length(target))
  {or ((len-start) < 0)}
  or (start < 1)
  or (len < 1) then begin
    ErrorMessage('SubStr parameters out of range');
    Exit;
  end;

  if not Expected( ')') then Exit;

  Results := Copy( target, start, len );
  SubStrConstruct := True;

end;

{ ========================================================================= }
{                         T e s t  R e l a t i o n                          }
{ =========================================================================
    This support function returns TRUE if the relational expression
    defined as <strLeft> <relop> <strRight> is true.

    You can modify the treatment of relational operators any way you
    please by modifying this function.
{ ========================================================================= }
Function TBasicInterpreter.TestRelation( strLeft, relop, strRight : String; vType : Integer ) : Boolean;
var
  Proceed : Boolean;
  llen,
  rlen    : Integer;
  ft      : String;
  lreal,
  rreal   : Extended;
  code    : Integer;
begin

  TestRelation := False;
  Proceed := False;

  if relop = '##' then begin { boolean expression }

    case vType of
      vString : TestRelation := (Length(strLeft) > 0);
      vInteger: TestRelation := (strLeft <> '0');
      vFloat  : TestRelation := (strLeft <> '0.0');
    end;

    Exit;

  end;

  case vType of
    vString:
      if      ((relop = '=') or (relop = '==')) and (strRight = strLeft) then
        Proceed := True
      else if ((relop = '<') or (relop = '<<')) and (strLeft < strRight) then
        Proceed := True
      else if ((relop = '>') or (relop = '>>')) and (strLeft > strRight) then
        Proceed := True
      else if ((relop = '<>') or (relop = '><')) and (strLeft <> strRight) then
        Proceed := True
      else if ((relop = '>=') or (relop = '=>')) and (strLeft >= strRight) then
        Proceed := True
      else if ((relop = '<=') or (relop = '=<')) and (strLeft <= strRight) then
        Proceed := True;
    vInteger,
    vFloat: begin

      Val( strLeft, lreal, code );
      Val( strRight, rreal, code );

      if      ((relop = '=') or (relop = '==')) and (rreal = lreal) then
        Proceed := True
      else if ((relop = '<') or (relop = '<<')) and (lreal < rreal) then
        Proceed := True
      else if ((relop = '>') or (relop = '>>')) and (lreal > rreal) then
        Proceed := True
      else if ((relop = '<>') or (relop = '><')) and (lreal <> rreal) then
        Proceed := True
      else if ((relop = '>=') or (relop = '=>')) and (lreal >= rreal) then
        Proceed := True
      else if ((relop = '<=') or (relop = '=<')) and (lreal <= rreal) then
        Proceed := True;
    end;

  end;

  TestRelation := Proceed;

end;

{ ========================================================================= }
{                        T r i m  C o n s t r u c t                         }
{ =========================================================================
  Syntax:
    Trim( <string> )
    LTrim( <string> )
    RTrim( <string> )

  Notes:
    Attached to the string evaluation loop with RegisterFunction.
{ ========================================================================= }
Function TBasicInterpreter.TrimConstruct( var Results : String; Mode : TrimMode ) : Boolean;
var
  expstr : String;  { expression buffer }

  { ======================================================================= }
  {                                T r i m                                  }
  { ======================================================================= }
  function Trim( str : String ) : String;
  begin
    while (length(str)> 1) and (str[1] = ' ') do
      delete( str, 1, 1 );
    while (length(str)> 1) and (str[length(str)] = ' ') do
      delete( str, length(str), 1 );
    Result := str;
  end;

  { ======================================================================= }
  {                              L  T r i m                                 }
  { ======================================================================= }
  function LTrim( str : String ) : String;
  begin
    while (length(str)> 1) and (str[1] = ' ') do
      delete( str, 1, 1 );
    Result := str;
  end;

  { ======================================================================= }
  {                              R  T r i m                                 }
  { ======================================================================= }
  function RTrim( str : String ) : String;
  begin
    while (length(str)> 1) and (str[length(str)] = ' ') do
      delete( str, length(str), 1 );
    Result := str;
  end;

begin

  Results := '';
  TrimConstruct := False;
  BasicTrace(Keyword, LineText);

  if not Expected('(') then Exit;

  NextToken;
  if not EvalString(expstr) then Exit;

  if not Expected(')') then Exit;
  {NextToken;}

  case Mode of
    tmBoth:   Results := Trim(expstr);
    tmLeft:   Results := LTrim(expstr);
    tmRight:  Results := RTrim(expstr);
  end;

  TrimConstruct := True;

end;

{ ========================================================================= }
{                         V a r  C o n s t r u c t                          }
{ =========================================================================
  Syntax:
    Dim <variablename> As <STRING|FLOAT|DECIMAL|INTEGER>[, variablename...]
{ ========================================================================= }
Function TBasicInterpreter.VarConstruct : Boolean;
var
  kw         : String;
  vname      : String;
  vtype      : Integer;
  rc         : Integer;
begin

  VarConstruct := False;
  BasicTrace(Keyword, LineText);

  { prime the variable dimensioning pump }
  NextToken;

  { only un-recognized tokens are candidates for variables}
  while Token = NT_TOKEN_NOTFOUND do begin

    { collect the variable name }
    vname := Keyword;
    NextToken;

    { 'As' is required }
    if not IsToken(tkAs) then begin

      ErrorMessage( 'Syntax Error: Expected "As"');
      Exit;

    end;

    { collect the variable type token }
    NextToken;

    case Token of

      { FLOAT, DECIMAL }
      tkFloat, tkDecimal: begin

          rc := AddVariable( vname, vFloat, '0.0' );
          if rc < 0 then begin

            ErrorMessage('Variable create error #' + IntToStr(rc));
            Exit;

          end;
          BasicTrace('Create Variable', vname);

        end;

      { INTEGER }
      tkInteger: begin

          rc := AddVariable( vname, vInteger, '0' );
          if rc < 0 then begin

            ErrorMessage('Variable create error #' + IntToStr(rc));
            Exit;

          end;
          BasicTrace('Create Variable', vname);

        end;

      { STRING }
      tkString: begin

          rc := AddVariable( vname, vString, '' );
          if rc < 0 then begin

            ErrorMessage('Variable create error #' + IntToStr(rc));
            Exit;

          end;
          BasicTrace('Create Variable', vname);
          {NextToken;}

        end;

    else { un-recognized variable type }

      ErrorMessage('Syntax Error: Expected variable type');
      Exit;

    end;

    { if a comma is encountered, then skip it }
    if LookAhead = NT_COMMA then begin
      NextToken;
      NextToken;
    end;

  end;

  VarConstruct := True;

end;

{ ========================================================================= }
{                            W h i l e  W  E n d                            }
{ =========================================================================
  Syntax:
    While <relational-expression>
      ... statements ...
    WEnd

  Notes:
    When the While statement is processed, a system label is created using
    the template: '_WHILE_WEND_hh' where hh is a hex number of 00-FF. This
    label (actually a pair of integers) is used to locate the jump point
    when the 'WEnd' keyword is encountered.

    Nested WHILE...WEND constructs are handled by encoding the nest level
    in the label, and by maintaining a nesting counter in WhilePtr.
{ ========================================================================= }
Function TBasicInterpreter.WhileWEnd : Boolean;
var
  strLeft,
  relop,
  strRight,
  wLabel    : String;
  vType     : Integer;
  lType     : Integer;
  lb        : LabelRecordType;
  indent    : Integer;
  Line,
  Index     : Integer;
  tmpToken  : Integer;
  tmpKeyword: String;
  excep     : Boolean;

  { ======================================================================= }
  {                       S c a n  F o r  W  E n d                          }
  { =======================================================================
      Scan the input stream until the appropriate 'WEnd' is located.
      Note that this function is called recursively when nested While..WEnd
      statements are encountered.
  { ======================================================================= }
  Function ScanForWEnd : Boolean;
  var ok : Boolean;
  begin

    while not (IsToken(tkWEnd) or IsToken(NT_PAST_EOF)) do begin

      SkipComment;
      if IsToken(tkWhile) then begin
        NextToken;
        if not ScanForWEnd then begin
          ScanForWend := False;
          Exit;
        end;
      end;

      if IsToken(NT_PAST_EOF) then begin
        ScanForWend := False;
        Exit;
      end;

      NextToken;

    end;

    ScanForWEnd := True;
    dec(WhilePtr);

  end;

begin

  WhileWEnd := False;
  BasicTrace(Keyword,LineText);

  { Begin by recording the current location in the label }
  { record.  The current location will be the relational }
  { test.  When 'WEnd' is processed, the parser will continue }
  { at this point and re-test the expression }
  if not LocateKeyword( lb.LineNo, lb.Index ) then begin

    lb.LineNo := CurrentLine;
    lb.Index := CurrentIndex;

  end;

  NextToken;

  { collect the required expression }
  if GetRelationalExpr( strLeft, relop, strRight, vType ) then begin

    { define a jump label for this loop }
    inc(WhilePtr);
    wLabel := '_WHILE_WEND_' + IntToHex(WhilePtr,2);

    { if the label already exists, then re-use it }

    excep := Exceptions;
    Exceptions := False;
    GetVariable( wLabel, lType );
    Exceptions := excep;
    if (lType = -1)
      then AddVariable( wLabel, vLabel, IntToStr(lb.LabelLong) )
      else SetVariable( wLabel, IntToStr(lb.LabelLong) );

    { skip optional 'Do' clause }
    if LookAhead = tkDo then NextToken;

    { syntax is correct }
    WhileWEnd := True;

    { test the relation for truth }
    if not TestRelation( strLeft, relop, strRight, vType ) then begin

      try
        ZapVariable(wLabel);
      except
      end;

      { False: skip to the WEnd statement }
      NextToken;

      if Not ScanForWEnd then begin
        ErrorMessage( 'Expected "WEnd"' );
        WhileWEnd := False;
        Exit;
      end;

    end
    { since the relation is TRUE, simply continue parsing }
    { the script until the 'WEnd' statement is located }
    else Exit;

  end
  else begin
    Exit;
  end;

end;

{ ========================================================================= }
{                              F o r  N e x t                               }
{ ========================================================================= }
Function TBasicInterpreter.ForNext : Boolean;
var
  wLabel    : String[MAX_KEYWORD_LEN];
  cvar      : String[MAX_KEYWORD_LEN];
  vType     : Integer;
  lType     : Integer;
  lb        : LabelRecordType;
  excep     : Boolean;
  FromVar,
  ToVar     : String[MAX_KEYWORD_LEN];
  workint   : LongInt;
  workflt   : Comp;
  cflt      : Comp;
  code      : Integer;

  { locate the Next for this For }
  Function ScanForNext : Boolean;
  var ok : Boolean;
  begin
    while not (IsToken(tkNext) or IsToken(NT_PAST_EOF)) do begin
      SkipComment;
      if IsToken(tkFor) then begin
        NextToken;
        if not ScanForNext then begin
          ScanForNext := False;
          Exit;
        end;
      end;
      if IsToken(NT_PAST_EOF) then begin
        ScanForNext := False;
        Exit;
      end;
      NextToken;
    end;
    ScanForNext := True;
    dec(NextPtr);
  end;

  Procedure SetNext;
  begin
    wLabel := '_FOR_NEXT_' + IntToHex(NextPtr,2);
    try
      AddVariable( wLabel, vLabel, IntToStr(lb.LabelLong) );
    except
      SetVariable( wLabel, IntToStr(lb.LabelLong) );
    end;
  end;

begin

  ForNext := False;
  BasicTrace(Keyword,LineText);

  if not LocateKeyword( lb.LineNo, lb.Index ) then begin
    lb.LineNo := CurrentLine;
    lb.Index := CurrentIndex;
  end;
  NextToken;

  if not IsToken(NT_VARIABLE_FOUND) then begin
    ErrorMessage('For/Next requires a declared control variable.');
    Exit;
  end;

  cVar := Keyword;
  GetVariable( cVar, vType );

  if not (vType in [vInteger,vFloat]) then begin
    ErrorMessage('For/Next requires an Integer or Float control variable.');
    Exit;
  end;

  if not Expected('=') then Exit;
  NextToken;
  if not EvalNumeric( FromVar, vType ) then Exit;
  if not Expected('TO') then Exit;
  NextToken;
  if not EvalNumeric( ToVar, vType ) then Exit;
  { do this for any construct that may end with a variable }
  { to short-circuit the OnTypedVar Event }
  Token := 0;
  Result := True; { syntax correct }

  if NextPtr > -1 then begin
    if Uppercase(ForNextStack[NextPtr].ControlVar) <> Uppercase(cvar) then begin
      SetVariable(cvar,FromVar);
    end;
  end;

  { sanity check ... should we continue? }
  if vType = vInteger then begin
    workint := StrToInt(ToVar);
    if StrToInt(GetVariable(cvar,vType)) > workint then begin
      if not ScanForNext then begin
        ErrorMessage('Expected "Next"');
        Result := False;
      end;
      Exit;
    end;
  end
  else begin
    val(ToVar,workflt,code);
    val(GetVariable(cvar,vType),cflt,code);
    if cflt > workflt then begin
      if not ScanForNext then begin
        ErrorMessage('Expected "Next"');
        Result := False;
      end;
      Exit;
    end;
  end;

  if NextPtr > -1 then begin
    { check stack ... are we there? }
    { if not, then add this to the stack }
    if not (UpperCase(ForNextStack[NextPtr].ControlVar) = UpperCase(cvar)) then begin
      inc(NextPtr);
      ForNextStack[NextPtr].ControlVar := cvar;
      ForNextStack[NextPtr].ControlVal := ToVar;
      ForNextStack[NextPtr].ControlTyp := vType;
      SetVariable(cVar,FromVar);
      SetNext;
    end
    else SetNext;
  end
  else begin
    inc(NextPtr);
    with ForNextStack[NextPtr] do begin
      ForNextStack[NextPtr].ControlVar := cvar;
      ForNextStack[NextPtr].ControlVal := ToVar;
      ForNextStack[NextPtr].ControlTyp := vType;
      SetVariable(cVar,FromVar);
      SetNext;
    end;
  end;
end;

{ ========================================================================= }
{                            W r i t e  T e x t                             }
{ =========================================================================
  Syntax:
    Print [#]<filenumber:integer>, <string-expression>
{ ========================================================================= }
Function TBasicInterpreter.WriteText : Boolean;
var
  fh : Integer;
  str : String;
begin

  Result := False;
  BasicTrace(Keyword,LineText);

  fh := EvalFileNumber;

  if fh < 1 then Exit;

  if not Files[fh].Opened then begin

    ErrorMessage('File #'+IntToStr(fh)+' not open.');
    Exit;

  end;

  if not (Files[fh].IOToken = tkWrite) then begin
    ErrorMessage('File not opened for Write.');
    Exit;
  end;

  NextToken;

  if not IsKeyword(',') then begin
    ErrorMessage('Expected ","');
    Exit;
  end;

  NextToken;

  if not EvalString( str ) then Exit;

  {$I-}
  if LookAhead = NT_SEMICOLON then begin
    NextToken;
    Write( Files[fh].FileVar, str );
  end
  else WriteLn( Files[fh].FileVar, str );
  {$I+}

  Files[fh].IOResult := System.IOResult;
  SetVariable('Result',IntToStr(Files[fh].IOResult));
  Result := True;

  { NOTE: REMEMBER TO SET TOKEN TO ZERO WHENEVER A VARIABLE }
  {       IS THE LAST ITEM IN A COMMAND.  THIS TELLS THE    }
  {       PARSER NOT TO PASS THIS TOKEN TO OnTypedVar       }
  Token := 0; { always needed when last item is a variable }

end;

{ ========================================================================= }
{                          S k i p  C o m m e n t                           }
{ ========================================================================= }
Procedure TBasicInterpreter.SkipComment;
begin
  if Token = NT_EXCLAMATION then begin
    CurrentLine := CurrentLine + 1;
    CurrentIndex := 0;
    Token := 0;
  end;
end;

{ ========================================================================= }
{                  S c r i p t  E n t e r  S t r  E x p r                   }
{ ========================================================================= }
Procedure TBasicInterpreter.ScriptStrExprLoop(var Results: OpenString; var Break,
  Cancel: Boolean);
var
  expstr : String;
  tmpToken : Integer;
  tmpKeyword : String;
  success : Boolean;
begin

  expstr := '';
  Results := '';

  { first, check for special string Functions }

  { InputBox function }
  if IsToken(tkInputBox) then begin
    if InputConstruct( expstr ) then begin
      Results := expstr;
      Break := True;
    end
    else Cancel := True;
    if Cancel then ErrorMessage( expstr );
  end

  { CurDir function }
  else if IsToken(tkCurDir) then begin
    GetDir(0,Results);
  end

  { CStr function }
  else if IsToken(tkCStr) then begin
    Cancel := True;
    if not Expected('(') then Exit;
    NextToken;
    if not EvalNumeric(Results,vFloat) then Exit;
    if not Expected(')') then Exit;
    Cancel := False;
  end

  { SubStr function }
  else if IsToken(tkSubStr) then begin
    if SubStrConstruct( expstr )
      then Results := expstr
      else Cancel := True;
  end

  { UCase function }
  else if IsToken(tkUCase) then begin
    if UCaseConstruct( expstr )
      then Results := expstr
      else Cancel := True;
  end

  { LCase function }
  else if IsToken(tkLCase) then begin
    if LCaseConstruct( expstr )
      then Results := expstr
      else Cancel := True;
  end

  { NOW function }
  else if Token = tkNow then begin
    NowVariable( expstr );
    Results := expstr;
  end

  { TRIM functions }
  else if Token in [tkTrim, tkLTrim, tkRTrim] then begin
    case Token of
      tkTrim : success := TrimConstruct( expstr, tmBoth );
      tkLTrim: success := TrimConstruct( expstr, tmLeft );
      tkRTrim: success := TrimConstruct( expstr, tmRight );
    end;
    if not success then begin
      Cancel := True;
      Exit;
    end;
    Results := expstr;
  end;

end;

{ ========================================================================= }
{                         E n t e r  S t r  E x p r                         }
{ ========================================================================= }
Procedure TBasicInterpreter.EnterStrExpr(var Results: OpenString; var Break,
  Cancel: Boolean);
begin
  if Assigned(FOnStrExpr) then
    OnEnterStrExpr(Results, Break, Cancel)
end;

{ ========================================================================= }
{                          L o o p  S t r  E x p r                          }
{ ========================================================================= }
Procedure TBasicInterpreter.LoopStrExpr(var Results: OpenString; var Break,
  Cancel: Boolean);
begin
  ScriptStrExprLoop( Results, Break, Cancel );
  if Cancel or Break then Exit;

  if Assigned(FOnStrLoop) then
    OnStrExprLoop(Results, Break, Cancel)
end;

{ ========================================================================= }
{                       S c r i p t  T y p e d  V a r                       }
{ =========================================================================
    Whenever a free-floating variable is located, this event is triggered.
    This is useful for syntax like A := -.23 or something.  It can also
    be used to implement functions and procedures by determining if the
    variable represents one.  Then jump to a previously defined label.
{ ========================================================================= }
Procedure TBasicInterpreter.ScriptTypedVar(var Cancel: Boolean);
label
  SyntaxError;        { shared syntax error entry point }
var
  vname  : String;   { variable's NAME container }
  vtype  : Integer;  { variable's TYPE container }
  expstr : String;
begin

  vname := Keyword;     { get variable name }
  vtype := AsVarType;   { get the variable's type }

  NextToken;            { skip to ':=' }

  { implementing <varname> = <expression> }

  if IsKeyword('=') then begin

    NextToken;

    { determine what to do based on variable type }

    case vtype of
      vString : Cancel := Not EvalString( expstr );
      vInteger: Cancel := Not EvalNumeric( expstr, vInteger );
      vFloat  : Cancel := Not EvalNumeric( expstr, vFloat );
    end;

    if not Cancel
      then SetVariable( vname, expstr )
      ;{else if not (vtype = vString) then ErrorMessage( expstr );}

  end
  else begin
    Cancel := True;
    ErrorMessage('Syntax Error: Expected "="');
    Exit;
  end;

  if not (vType = vString)
    then BasicTrace( 'Set Numeric Variable "' + vname + '" = '+ expstr, LineText )
    else BasicTrace( 'Set String Variable "' + vname + '" = "'+ expstr +'"', LineText )

end;

{ ========================================================================= }
{                  S c r i p t  E n t e r  N u m  E x p r                   }
{ =========================================================================
    Very powerful event.  Here you can implement any numeric function
    you like.  It could be anything; even involving calling another
    computer or executing an SQL script.  It's up to you.
{ ========================================================================= }
Procedure TBasicInterpreter.ScriptEnterNumExpr(var Results: OpenString;
  vType: Integer; var Break, Cancel: Boolean);
var
  rc : Integer;
  expstr : String;
  wLong : LongInt;
  wFloat : Comp;

  { ======================================================================= }
  {                         S t r i p  F l o a t                            }
  { ======================================================================= }
  Function StripFloat( exp : String ) : String;
  var
    p : Byte;
  begin

    p := Pos('.', exp);
    if p > 0
      then StripFloat := Copy( exp, 1, p-1 )
      else StripFloat := exp;

  end;

  { ======================================================================= }
  {                       V a l i d  N u m e r i c                          }
  { ======================================================================= }
  Function ValidNumeric( st : String ) : Boolean;
  var i,j : Integer;
  begin

    ValidNumeric := False;
    if st[0] = #0 then Exit;
    j := 1;
    if st[1] in ['-','+'] then j := 2;
    for i := j to Length(st) do
      if not (st[i] in ['0'..'9','.']) then Exit;
    ValidNumeric := True;

  end;

begin

  Cancel := False;
  { First, check for special Functions that return numeric values }

  { MessageBox() }
  if IsToken(tkMessageBox) then begin

    if MsgBoxConstruct(rc) then begin
      Break := True;
      Results := IntToStr(rc);
      Exit;
    end
    else Cancel := True;

  end

  { CInt }
  else if IsToken(tkCInt) then begin
    Cancel := True;
    if not Expected('(') then Exit;
    NextToken;
    if not EvalString( Results ) then Exit;
    try
      wLong := StrToInt( StripFloat(Results) );
      if not Expected(')') then Exit;
      Cancel := False;
      Exit;
    except
      ErrorMessage('String to Numeric conversion error.');
      Exit;
    end;
    Cancel := False;
  end

  { CFlt }
  else if IsToken(tkCFlt) then begin
    Cancel := True;
    if not Expected('(') then Exit;
    NextToken;
    if not EvalString( Results ) then Exit;
    Val(Results,wFloat,rc);
    if rc <> 0 then begin
      ErrorMessage('String to Numeric conversion error.');
      Exit;
    end;
    if not Expected(')') then Exit;
    Cancel := False;
    Exit;
  end

  { Exists }
  else if IsToken(tkExists) then begin
    if ExistsConstruct(rc) then begin
      Break := True;
      Results := IntToStr(rc);
      Exit;
    end
    else Cancel := True;
  end

  { InputBox() }
  else if IsToken(tkInputBox) then begin
    if InputConstruct(expstr) then begin
      if not (vType = vString) then
        if expstr = '' then expstr := '0.0';
      case vtype of
        vInteger:  Results := StripFloat(expstr);
        vFloat:    Results := expstr;
      end;
      Cancel := Not ValidNumeric(Results);
      if Cancel
        then ErrorMessage('Numeric conversion error in InputBox')
        else Break := True;
      Exit;
    end
    else Cancel := True;

  end

  { FreeFile }
  else if IsToken(tkFreeFile) then begin

    Cancel := Not FreeFile( rc );
    if not Cancel
      then Results := IntToStr(rc)
      else Results := 'Invalid File Number';
    Break := True;
    Exit;

  end

  { InStr }
  else if IsToken(tkInStr) then begin

    if not InStrConstruct( expstr )
      then Cancel := True
      else Results := expstr;

  end

  { Len }
  else if IsToken(tkLen) then begin

    if not LenConstruct( expstr )
      then Cancel := True
      else Results := expstr;

  end
  else if Assigned(FOnNumExpr) then
    OnNumExprLoop(Results, vType, Break, Cancel);

end;

{ ========================================================================= }
{                       b i  E x e c u t e  C l i c k                       }
{ ========================================================================= }
Procedure TBasicInterpreter.biExecute(Value : Boolean);
var
  ii : Integer;
begin

  inherited OnBadToken     := OnBadToken;
  inherited OnEndOfLine    := OnEndOfLine;
  inherited OnEndOfText    := EndOfText;
  inherited OnError        := OnError;
  inherited OnNewLine      := OnNewLine;
  inherited OnNextParse    := ParserMain;
  inherited OnProcedure    := OnProcedure;
  inherited OnTypedVar     := ScriptTypedVar;
  inherited OnNumExprLoop  := ScriptEnterNumExpr;
  inherited OnEnterStrExpr := EnterStrExpr;
  inherited OnStrExprLoop  := LoopStrExpr;
  inherited OnTrace        := OnTrace;

  if not Value then begin
    inherited Execute := Value;
    Cancelled := True;
    FRunScript := Value;
    Exit;
  end;

  FRunScript := True;
  Application.ProcessMessages;

  AssertTokens;

  { defaults }
  FilePtr     := -1;

  { initialize working variables and buffers }
  FillChar( Files, SizeOf(Files), #0 );
  FillChar( ForNextStack, SizeOf(TForNextStack), #0);
  {StackPtr := -1;}
  WhilePtr := -1;
  NextPtr := -1;
  IfLevel := -1;
  FilePtr := -1;
  CurrentLine := 0;
  CurrentIndex := 0;

  { clear all variables }
  ZapVariables;

  { scan for standard labels (<label>:) }
  ScanForLabels;

  { create global variables }
  AddVariable('TRUE',vInteger,IntToStr(-1));
  AddVariable('FALSE',vInteger,IntToStr(0));
  AddVariable('Result',vInteger,IntToStr(-1));    { true }

  { reset cancel flag }
  Cancelled := False;

  FRunScript := False;
  if Assigned(FBeforeExec) then OnBeforeExec(Cancelled);
  if Cancelled then Exit;
  FRunScript := True;
  StackPtr := 0;
  Application.ProcessMessages;

  { execute the script }
  try
    inherited Execute := True;  { Click }
  except
    FRunScript := False;
  end;

  FRunScript := False;

end;

{ ========================================================================= }
{                              R e g i s t e r                              }
{ ========================================================================= }
Procedure Register;
begin
  RegisterComponents('Samples', [TBasicInterpreter]);
end;

{ ========================================================================= }
{                        I N I T I A L I Z A T I O N                        }
{ ========================================================================= }
End.

{ ========================================================================= }
{                                   E O F                                   }
{ ========================================================================= }


