\ ***********************************************************
\ Example program to illustrate the use of multi-tasking and
\ Forth interrupt service routines
\ by Andy Yuen 1995 (C)
\ ***********************************************************
\ eForth has no CONSTANT, define one
\ (I have added DOES> to eForth)
: CONSTANT CREATE , DOES> @ ;

\ musical note generation frequency divider
6087 CONSTANT <G
5423 CONSTANT <A
4560 CONSTANT C
4064 CONSTANT D
3630 CONSTANT E
3044 CONSTANT G

\ hardware port and control definitions
HEX 
61 CONSTANT CONTROL-PORT        \ speaker control port
43 CONSTANT TIMER-PORT          \ timer mode control port
42 CONSTANT COUNT-PORT          \ timer count register
0B6 CONSTANT SETUP              \ counter 2 square wave generator mode
0FE CONSTANT MASK               \ timer-driven speaker disable control mask
1C CONSTANT TIMER-INT           \ PC clock tick interrupt
0B800 CONSTANT SCR-SEG
DECIMAL

\ define tune in musical note/duration pair
CREATE TUNE
C , 6 , E , 6 , G , 3 , G , 3 , G , 4 , E , 4 , G , 6 , 
C , 6 , E , 6 , D , 3 , D , 3 , G , 4 , G , 4 , <G , 6 , 
E , 6 , D , 6 , C , 3 , C , 3 , <A , 4 , <A , 4 , C , 6 , 0 , 20 ,

\ 18 ticks equals one second
18 CONSTANT TICKS/SECOND

0 CONSTANT TMUSIC#              \ task# for music playing task
1 CONSTANT TTIME#               \ task# for time-of-day display task
2 CONSTANT TLOOP#               \ task# for runaway task

0 USER POS                      \ current position in tune

\ enable timer-driven tone-generation
: SPEAKER-ON ( -- ) CONTROL-PORT PC@ 3 OR CONTROL-PORT PC! ;

\ disable timer-driven tone-generation
: SPEAKER-OFF ( -- ) CONTROL-PORT PC@ MASK AND CONTROL-PORT PC! ;

\ delay for n signals
: DELAY ( n -- ) DUP FOR ME WAIT 1 - NEXT DROP ;

\ music playing task logic
: TMUSIC ( -- )
	SETUP TIMER-PORT PC! SPEAKER-ON \ setup timer and enable speaker
	0 POS ! ME WAIT BEGIN           \ wait for first signal to start
	SPEAKER-OFF                     \ disable speaker
	SPEAKER-ON                      \ and enable it to give a brief pause
	TUNE POS @ + DUP DUP C@ COUNT-PORT PC!  \ output frequency divisor to
	1 + C@ COUNT-PORT PC!           \ timer count register
	2 + @ DELAY                     \ delay specified duration
	4 POS +!                        \ advance tune pointer
	TUNE POS @ + 
	@ 0 = IF 0 POS ! 
	SPEAKER-OFF 20 DELAY THEN       \ pasue a while song finishes
	AGAIN ;                         \ replay

\ replace standard word to quit: need to remove ISR and disable speaker
: BYE ( -- ) SPEAKER-OFF TIMER-INT INT-REMOVE BYE ;

\ declare variable for keeping the time-of-day
VARIABLE XCOUNT
TICKS/SECOND XCOUNT !
VARIABLE SECOND
VARIABLE MINUTE
VARIABLE HOUR

\ a semaphore is used for safe-guarding time-of-day variables access
\ note that all words accessing them do a wait in the beginning
\ and a signal at the end to provide mutial exclusive access
20 CONSTANT TODSEM
TODSEM SIGNAL

\ set time-of-day
: SETTOD ( n n n -- ) TODSEM WAIT SECOND ! MINUTE ! HOUR ! TODSEM SIGNAL ;

\ advance time-of-day clock by one second
: SECOND> ( -- ) TODSEM WAIT 1 SECOND @ + DUP 60 < IF
	SECOND ! ELSE 60 SWAP - SECOND !
	1 MINUTE @ + DUP 60 < IF
	MINUTE ! ELSE 60 SWAP - MINUTE !
	1 HOUR @ + DUP 24 < IF
	HOUR ! ELSE 24 SWAP - HOUR !
	THEN THEN THEN TODSEM SIGNAL ;

\ convert number to two ASCII characters on the stack
\ cannot use words like <#, #, #>, etc. because they allocate
\ memory and may interfere with the interpreter task #14
: DECODE ( n -- n n ) 10 EXTRACT SWAP 10 EXTRACT SWAP DROP ;

\ write ASCII characters to memory location until 0 is reached
: SCRWRITE  ( 0 n ... n seg off -- ) BEGIN 
	2 + ROT DUP WHILE 
	2 PICK 2 PICK MC! REPEAT 2DROP DROP ;

HEX

\ display time-of-day HH:MM:SS near top right hand corner of screen
: TODDISPLAY ( -- ) TODSEM WAIT
	0 SECOND @ DECODE 3A 
	MINUTE @ DECODE 3A 
	HOUR @ DECODE
	SCR-SEG 80 SCRWRITE
	TODSEM SIGNAL ;

DECIMAL

\ time-of-day display task logic
: TTIME ( -- ) 0 0 0 SETTOD BEGIN ME WAIT       \ set time to 00:00:00
	XCOUNT @ 1 - DUP XCOUNT ! 0 = IF        \ increment time every second
	SECOND> TICKS/SECOND XCOUNT ! THEN 
	TODDISPLAY AGAIN ;                      \ display time

\ runaway task logic
: TLOOP BEGIN AGAIN ;                           \ loop


HEX 

\ define interrupt service routine
ISR: CLOCKISR TMUSIC# SIGNAL                    \ signal tasks
	TTIME# SIGNAL IPREEMPT ISR;             \ multi-task

DECIMAL

\ install timer ISR
TIMER-INT INT-INSTALL CLOCKISR

\ create and start all tasks
CREATE STACKAREA
TMUSIC# THREAD TMUSIC
TTIME# THREAD TTIME
TLOOP# THREAD TLOOP

