;****************************************************************************
;                                                                           *
; RECORD								    *
;								  	    *
; Copyright (c) 1995 Mark Russinovich and Bryce Cogswell		    *
;                                                                           *
; You can modify and use this code in whatever manner you wish.             *
;									    *
;****************************************************************************
;									    *
; RECORD: Exports APIs that allow win32 programs to record and playback     *
; the mouse and keyboard.                                                   *
;                                                                           *
;****************************************************************************

;===========================================================================
	page	,132
	title	record - record VxD
	name	record.vxd
;===========================================================================
;
;   Module:
;	Contains everything
;
;===========================================================================
;
;   Functional Description: - After we start up we wait for a command from
;                             a win32 app. We have two modes that can be
;                             enabled: record and playback. The win32 app
;                             in both cases passes us a buffer
;                             either record into or playback out of. 
;			
;============================================================================

;============================================================================
;				I N C L U D E S
;============================================================================
.386p
	.xlist
	include	vmm.inc
	include vwin32.inc
	include vkd.inc
	include vmd.inc
	include debug.inc
	include record.inc
.list

;============================================================================
;	 	   	S T R U C T U R E S
;============================================================================

; record structure
REC_common		struc
rectype			db	?		; either keyboard or mouse
timestamp		dd	?		; system time
REC_common		ends

REC_kbd_entry		struc
common			db	size REC_common DUP (?)
scancode		db	?		; keyboard scancode
REC_kbd_entry		ends

REC_mouse_entry		struc
common			db	size REC_common DUP (?)
deltax			dd	?		; mouse delta x
deltay			dd	?		; mouse delta y
button			db	?		; button status
REC_mouse_entry		ends


;============================================================================
;		   I N I T I A L I Z A T I O N  D A T A
;============================================================================

VXD_IDATA_SEG


VXD_IDATA_ENDS

;============================================================================
; 			  P U B L I C   D A T A
;============================================================================

VXD_LOCKED_DATA_SEG

; Recording on
Record_On		db	0
Playback_On		db	0

; System time value address
System_Time		dd	0

; Previous routines on hook chain
Keyboard_Proc		dd	0
Mouse_Proc              dd	0

; Buffers for communicating with Devmon Windows program

; default 8K buffer (good for at about 15 seconds of max mouse movement)
BUFFER_SIZE		equ	8192	
Buffer			db	BUFFER_SIZE dup (?)
Buffer_ptr		dd	0
Window			dd	0
lpo			dd	0

; Jump table for commands initiated by Devmon Windows program
Service_Table	label	dword
	dd	offset32	closehandle
	dd	offset32	getversion
	dd	offset32	recordEvents
	dd	offset32	recordOff
	dd	offset32	playEvents
Service_Table_Size	EQU	($ - Service_Table) / 4

VXD_LOCKED_DATA_ENDS


;============================================================================
;	           D E V I C E   D E C L A R A T I O N
;============================================================================

VXD_LOCKED_CODE_SEG

; use the drive registration packet as initialization data

DECLARE_VIRTUAL_DEVICE RECORD,	\
	REC_MAJOR_VERSION, 	\
	REC_MINOR_VERSION,	\
	Record_Control,,	\
	UNDEFINED_INIT_ORDER


;============================================================================
;			    M A I N   C O D E
;============================================================================

;============================================================================
;
; Record_Control - Device control procedure for the VxD. Dispatches all
;                  Windows VxD messages.
;
; Exit:	If carry clear then
;	    Successful
;	else
;	    Control call failed
;
; Destroys: EAX, EBX, ECX, EDX, ESI, EDI, Flags
;
;============================================================================

public Record_Control
Record_Control PROC NEAR

	Control_Dispatch SYS_DYNAMIC_DEVICE_INIT, REC_Device_Init
	Control_Dispatch SYS_DYNAMIC_DEVICE_EXIT, REC_Device_Exit
	Control_Dispatch W32_DEVICEIOCONTROL, REC_ioctl
	clc
	ret

Record_Control ENDP

VXD_LOCKED_CODE_ENDS

;==============================================================================
;		 	 P A G E A B L E   C O D E
;==============================================================================

VXD_PAGEABLE_CODE_SEG

;============================================================================
;
; Record_Keyboard - Records keyboard input into the input buffer.
;
; Entry: CL - Scancode of key pressed.
;
;============================================================================

Public Record_Keyboard
BeginProc Record_Keyboard, Hook_Proc Keyboard_Proc

	push	eax
	push	ebx

	; record the input

	mov	ebx, Buffer_ptr
	cmp	ebx, offset32 Buffer + BUFFER_SIZE - sizeof REC_mouse_entry
	jae	rk_done
	mov	[ebx].rectype, KEYBDTYPE
	mov	[ebx].scancode, cl		; scancode is in cl
	push	eax
	mov	eax, System_Time
	mov     eax, [eax]			; get the timestamp
	mov	[ebx].timestamp, eax
	pop	eax
	add	ebx, sizeof REC_kbd_entry	; move forward in buffer
	mov	Buffer_ptr, ebx
	mov	[ebx].rectype, INVALIDTYPE	; mark end of macro

	; call the previous hooker
rk_done:

	call	Keyboard_Proc
	pop	ebx
	pop	eax
	clc					; let the key through
	ret

EndProc Record_Keyboard


;============================================================================
;
; Record_Mouse - Records mouse input into the input buffer.
;
; Entry: esi - delta x
;        edi - delta y
;         al - button status
;        edx - pointer to mouse instance struct
;
;
;============================================================================

Public Record_Mouse
BeginProc Record_Mouse, Hook_Proc Mouse_Proc

	push	eax
	push	ebx

	; record the input
	
	mov	ebx, Buffer_ptr
	cmp	ebx, offset32 Buffer + BUFFER_SIZE - sizeof REC_mouse_entry
	jae	rm_done
	mov	[ebx].rectype, MOUSETYPE	; save parameters
	mov	[ebx].deltax, esi
	mov	[ebx].deltay, edi
	mov	[ebx].button, al
	push	eax
	mov	eax, System_Time		; get timestamp
	mov     eax, [eax]
	mov	[ebx].timestamp, eax
	pop	eax	
	add	ebx, sizeof REC_mouse_entry	; move forward in buffer
	mov	Buffer_ptr, ebx
	mov	[ebx].rectype, INVALIDTYPE	; mark end of macro

	; call the previous service
rm_done:
	call	Mouse_Proc			; chain
	pop	ebx
	pop	eax
	ret

EndProc Record_Mouse


;============================================================================
;
; Record_Start - Hook the mouse and keyboard input routines with the
;  		 recording functions.
;
; Entry: esi - DIOC block
;
; Exit: Carry - set indicates success
;
;============================================================================

Public Record_Start
BeginProc Record_Start

	push	eax
	push	esi

	; set record on flag 

	mov	al, 1
	mov	Record_On, al

	; get handle of calling window

	mov	eax, [esi].lpvInBuffer		; pointer to window
	mov	eax, [eax]			; the window
	mov	Window, eax			; save it

	; reset the buffer pointer

	mov	Buffer_ptr, offset32 Buffer

	; first hook the keyboard input

	GetVxDServiceOrdinal eax, VKD_Filter_Keyboard_Input
	mov	esi, offset32 Record_Keyboard
	VMMCall	Hook_Device_Service
	jnc	installmouse

	; error on keyboard install

	pop	esi
	pop	eax
	ret		

installmouse:

	; hook the mouse input routine

	GetVxDServiceOrdinal eax, VMD_Post_Pointer_Message
	mov	esi, offset32 Record_Mouse
	VMMCall	Hook_Device_Service
	jc	error
	pop	esi
	pop	eax
	ret					; everything went fine!

error:
	; have to unhook the keyboard

	GetVxDServiceOrdinal eax, VKD_Filter_Keyboard_Input
	mov	esi, offset32 Record_Keyboard
	VMMCall	Unhook_Device_Service	
	pop	esi
	pop	eax
	clc
	ret

EndProc Record_Start


;============================================================================
; 
; Record_Stop - Unhook the mouse and keyboard routines.
;
;
;============================================================================

Public Record_Stop
BeginProc Record_Stop

	push	eax
	push	ecx
	push	edi
	push	esi
	
	; unset recording flag

	mov	al, 0
	mov	Record_On, al

	; return buffer and length of buffer

	cld
	mov	ecx, Buffer_ptr			; bytes written
	sub	ecx, (offset32 Buffer) - 1	; buffer length + eof mark
	mov	edi, [esi].lpcbBytesReturned	; pointer to length variable
	mov	[edi], ecx			; pass length back
	mov	edi, [esi].lpvOutBuffer		; pointer to output buffer
	mov	esi, offset32 Buffer
	rep movsb				; copy data out

	; unhook keyboard

	GetVxDServiceOrdinal eax, VKD_Filter_Keyboard_Input
	mov	esi, offset32 Record_Keyboard
	VMMCall	Unhook_Device_Service	

	; unhook mouse

	GetVxDServiceOrdinal eax, VMD_Post_Pointer_Message
	mov	esi, offset32 Record_Mouse
	VMMCall	Unhook_Device_Service		

	pop	esi
	pop	edi
	pop	ecx
	pop	eax
	ret

EndProc Record_Stop


;============================================================================
;
; Playback_Entry - Plays back the current record and then sets up the timeout
; 		   for the next one.
;
;============================================================================

Public Playback_Entry
BeginProc Playback_Entry 

	push	eax
	push	ebx
	push	ecx
	push	esi
	push	edi

	mov	ebx, Buffer_ptr
	mov	al, [ebx].rectype
 	cmp	al, INVALIDTYPE
	jz	playend
	cmp	al, KEYBDTYPE
	jz	playkeybd

	; play mouse entry

	mov	esi, [ebx].deltax
	mov	edi, [ebx].deltay
	mov	al,  [ebx].button
	VxDCall	VMD_Post_Pointer_Message
	mov	eax,  sizeof REC_mouse_entry
	jmp	getnext

playkeybd:						; play keyboard entry

	; send it through filter key so other hookers can see it
	
	mov	cl, [ebx].scancode
	VxDCall	VKD_Filter_Keyboard_Input		; save new scancode in
	mov	[ebx].scancode, cl			;   case it changed

	; send it to the system to simulate the input

	mov	ecx, 1
	lea	esi, [ebx].scancode
	VxDCall	VKD_Force_Keys
	mov	eax,  sizeof REC_kbd_entry

	; set-up timeout to next entry
getnext:
	mov	ecx, [ebx].timestamp			; save current time
	add	ebx, eax				; move to next record
	mov	al,  [ebx].rectype			; check if EOF
	cmp	al, INVALIDTYPE				;   ...
	jz	playlast				;   y: we're done
	mov	eax, [ebx].timestamp			; get time of next rec
	sub	eax, ecx				; compute delta time
	mov     esi, offset32 Playback_Entry		; set global timeout
	VMMcall Set_Global_Time_Out			;   ...
	mov	Buffer_ptr, ebx				; update buffer ptr
	jmp	playend

playlast:
	mov	Playback_On, 0			        ; playback done

	; Notify the window application that playback is complete.

	mov	ebx, lpo
	VxDcall	VWIN32_DIOCCompletionRoutine		; unblock win32 thread

playend:

	pop	edi
	pop	esi
	pop	ecx
	pop	ebx
	pop	eax
	ret

EndProc Playback_Entry

;============================================================================
;
; Playback_Start - Initiates the plaback of whats in the buffer.
;
;============================================================================

Public Playback_Start
BeginProc Playback_Start

	; if recording on, just return

	mov	al, Record_On
	cmp	al, 0
	jz	go
	ret

go:
	push	ecx
	push	edi
	push	esi

	; turn on playback

	mov	Playback_On, 1			; set flag indicating playback

	; setup for async completion

	mov	edi, [esi].lpoOverlapped	; get the overlapped struct
	mov	edi, [edi].O_Internal		; get handle for notify 
	mov	lpo, edi
	
	; get replay buffer 

	cld
	mov	ecx, [esi].cbInBuffer		; length of buffer
	mov	esi, [esi].lpvInBuffer		; pointer to buffer
	mov	edi, offset32 Buffer
	mov	Buffer_Ptr, edi
	rep movsb
	pop	esi
	pop	edi
	pop	ecx

	; make initial playback call

	jmp	Playback_Entry

EndProc Playback_Start


;============================================================================
;
; REC_ioctl - Respond to IOcontrol messages sent by Win32 program.
;
; Entry: esi -> DIOC block
; 
; Exit:
;
;============================================================================

Public REC_ioctl
BeginProc REC_ioctl

	push	ecx

	mov	ecx,[esi].dwIoControlCode	; get ioctl code
	inc	ecx				; base is -1
	cmp	ecx, Service_Table_Size		; out of bounds ?
	jae	RECioctl_fail			; y: bad code, exit
	jmp	Service_Table[4*ecx]		; index into table

closehandle:
getversion:
	; Nothing to do for these
	jmp	RECioctl_success		; exit successfully

recordEvents:
	call	Record_Start
	jmp	RECioctl_success		; done

recordOff:
	call	Record_Stop
	jmp	RECioctl_success		; done

playEvents:
	call	Playback_Start
	jmp	RECioctl_success		; done

RECioctl_success:
	pop	ecx
	xor	eax, eax			; return zero = success
	clc
	ret

RECioctl_fail:
	pop	ecx
	mov	eax, 50				; return 50 = record error
	stc
	ret

EndProc	REC_ioctl


;============================================================================
;
; REC_Device_Exit - Cleans up any hooks that are still installed before
;		    exiting.
;
;============================================================================

Public REC_Device_Exit
BeginProc REC_Device_Exit

	push	eax
	push	esi

	mov	al, Record_On
	cmp	al, 1
	jnz	exit

	; unhook keyboard

	GetVxDServiceOrdinal eax, VKD_Filter_Keyboard_Input
	mov	esi, offset32 Record_Keyboard
	VMMCall	Unhook_Device_Service	

	; unhook mouse

	GetVxDServiceOrdinal eax, VMD_Post_Pointer_Message
	mov	esi, offset32 Record_Mouse
	VMMCall	Unhook_Device_Service		

exit:
	pop	esi
	pop	eax
	clc
	ret

EndProc REC_Device_Exit

VXD_PAGEABLE_CODE_ENDS

;============================================================================
;	   D E V I C E   I N I T I A L I Z A T I O N   C O D E
;============================================================================

VXD_ICODE_SEG

;============================================================================
;									
; REC_Device_Init - RECord Initialization 	
;									
;									
; Entry: ebx -> System VM handle (not used)
;        edx -> Reference data from real mode init portion
;
; Exit: If successful then
;           Carry flag is clear
;       else
;           Carry flag is set to indicate an error -- Device not initialized
;
;============================================================================

BeginProc REC_Device_Init

	VMMCall Get_System_Time_Address
	mov     System_Time, eax
	clc
	ret

EndProc REC_Device_Init

VXD_ICODE_ENDS

end
