;****************************************************************************
;* Program Minesweeper                                                      *
;* ------------------------------------------------------------------------ *
;* This program is a graphics game for MCGA/CGA/TANDY systems with a mouse. *
;* The game is a clone of the windows game of the save name.                *
;*                                                                          *
;* Components: MINESWP.ASM, MINESWP.INC, MINESWP.EQU                        *
;* Executable: MINESWP.COM                                                  *
;*                                                                          *
;****************************************************************************

IDEAL

INCLUDE "MINESWP.EQU"

SEGMENT MAIN
ASSUME CS:MAIN,DS:MAIN,ES:MAIN,SS:MAIN

ORG 100h

BEGIN:
    JMP     NEAR START

;****************************************************************************
;* Variables                                                                *
;****************************************************************************

AOFI_TAG            DB  13,10
                    DB  "Mine Sweeper V1.0 ",13,10
                    DB  "Artists of the Imagination, 1995 ",13,10
                    DB  26

;Minefield Variables - These are listed first because they can be edited
;                       by the program to change the default values
MINEFIELD_COLS      DB  DEFAULT_COLUMNS
MINEFIELD_ROWS      DB  DEFAULT_ROWS
NUM_FLAGS           DB  DEFAULT_FLAGS
MINEFIELD_SIZE      DW  (DEFAULT_COLUMNS * DEFAULT_ROWS)
CURRENT_GAME        DW  DEFAULT_GAME
CURRENT_HI_SCORE    DW  OFFSET HI_SCORE1

HI_SCORE1           DW  0FFFFh
HI_SCORE2           DW  0FFFFh
HI_SCORE3           DW  0FFFFh
HI_SCORE4           DW  0FFFFh

LABEL END_DEFAULTS BYTE

;Game types
LABEL               GAME_TYPES BYTE
                    DB 16,8,20
                    DW (16 * 8)
                    DW OFFSET HI_SCORE1

                    DB 16,16,40
                    DW (16 * 16)
                    DW OFFSET HI_SCORE2

                    DB 30,16,99
                    DW (30 * 16)
                    DW OFFSET HI_SCORE3

                    DB 36,20,150
                    DW (36 * 20)
                    DW OFFSET HI_SCORE4
LABEL               END_GAME_TYPES BYTE

;Game Status
FLAGS_LEFT          DB DEFAULT_FLAGS
                    DB 0
SQUARES_LEFT        DW (DEFAULT_COLUMNS * DEFAULT_ROWS)
TIME                DW 0

;Random Number Variables
RND_PTR             DW OFFSET RND_NUMBERS
RND_LEFT            DW 0

;Coordinates
CORNER_X            DW 0
CORNER_Y            DW 0
MINEFIELD_X         DW 0
MINEFIELD_Y         DW 0
FLAGSLEFT_X         DW 0
FLAGSLEFT_Y         DW 0
TIME_X              DW 0
TIME_Y              DW 0
FACE_X              DW 0
FACE_Y              DW 0
TEXT_MINE_X         DW 0
TEXT_MINE_Y         DW 0
TEXT_SWEEPER_X      DW 0
TEXT_SWEEPER_Y      DW 0

;Flags
GAME_DONE           DB FALSE
MOUSE_EVENT         DB FALSE
TIMER_ON            DB FALSE
DISPLAY_MSTEXT      DB FALSE
CURSOR_STATUS       DB OFF
DIRECTION           DW FORWARD

;Math
MAX_DIVISOR         DW 0

;Timer
TIMER               DW 0

;Mouse information
AREA_CLICKED        DB IRRELEVANT
MOUSE_COUNTDOWN     DB 0
MOUSE_CLICK_STATUS  DB 0
MCLICK_X            DW 0
MCLICK_Y            DW 0
MOUSE_X             DW (SCREEN_WIDTH / 2)
MOUSE_Y             DW (SCREEN_HEIGHT / 2)

;Save info
RANDOMIZATIONS      DW 0
RANDOM_VALUE        DW 0
RANDOM_VALUE_BL     DW 0

OLDINT1CH           DD 0
LAST_MCOUNT_X       DW 0
LAST_MCOUNT_Y       DW 0
SAVE_BUFFER_PTR     DW 0
CURSOR_SAVE_BUFFER  DB ((MC_WIDTH+5)*MC_HEIGHT) DUP (0)

;Windows
WINDOW_WIDTH        DW 0
WINDOW_HEIGHT       DW 0
MINEFIELD_WIDTH     DW 0
MINEFIELD_HEIGHT    DW 0

;Display Driver
SET_MODE            DW OFFSET MCGA_SET_MODE
SET_BACKGROUND      DW OFFSET MCGA_SET_BACKGROUND
PLACE_SPRITE        DW OFFSET MCGA_SPRITE
PAINT_HLINE         DW OFFSET MCGA_HLINE
PAINT_VLINE         DW OFFSET MCGA_VLINE
DISPLAY_MOUSE       DW OFFSET MCGA_CURSOR_PLACE
VIDEO_BUFFER        DW 0A000h
VB_SCRATCHPAD       DW 0
VB_SEGMENT          DW 0
SCRATCHPAD_SIZE     DW 64000

;Error Prompts
NO_MOUSE_EP     DB  "Mouse Driver not installed.",13,10
                DB  "Load Mouse Driver before running MineSweeper.",13,10,"$"
RM_ERROR_EP     DB  "Error in DOS: Modify Memory Allocation.",13,10
                DB  "Program must have 128k of contiguous memory.",13,10
                DB  "Program aborted.",13,10,"$"
DISPLAY_NS_EP   DB  "This program requires an MCGA/VGA or Tandy/PCjr "
                DB  "compatible display system.",13,10,"$"

LABEL   PSEUDORAND_TABLE BYTE
DB 147,113,76,100,2,83,80,190,98,250,176,233,15,6,156,155,62,228,185,111
DB 30,68,161,251,31,230,177,239,152,182,47,225,82,71,144,18,89,5,196,138
DB 117,248,220,41,45,194,81,50,222,60,213,137,40,14,91,206,204,157,36,93
DB 78,24,107,149,35,94,20,84,124,73,33,209,26,202,198,57,226,19,114,55,59
DB 133,158,210,126,229,212,65,38,56,234,192,9,253,172,242,148,112,99,245
DB 170,115,106,74,85,17,197,164,141,207,163,75,167,162,168,25,218,110,39
DB 105,195,42,165,120,243,231,187,215,77,214,13,10,221,88,180,79,150,53
DB 52,72,63,44,118,1,48,4,191,146,159,201,11,109,184,128,64,32,171,189,132
DB 16,122,129,153,28,29,216,199,169,244,142,104,151,90,219,70,235,125,174
DB 49,186,54,160,86,241,92,140,130,145,61,102,22,252,87,119,131,205,66,173
DB 97,116,203,223,224,69,123,43,96,7,108,188,127,237,95,27,3,227,200,217
DB 208,46,166,121,101,181,254,175,34,51,238,211,154,178,58,139,21,247,183
DB 240,67,179,37,232,136,143,193,236,23,8,103,134,255,246,249,12,135

INCLUDE "MINESWP.INC"

;****************************************************************************

;****************************************************************************
;* Start Procedure                                                          *
;* ------------------------------------------------------------------------ *
;* This procedure calls the necessary routines to execute the program       *
;*                                                                          *
;****************************************************************************

PROC    START

    CALL    CHECK_MOUSE
    JC      NO_MOUSE

    CALL    GET_DISPLAY_TYPE

    CALL    RELEASE_MEMORY
    JC      RM_ERROR

    CALL    DISPLAY_LOGO

    CALL    SET_MOUSE_HANDLERS

NEW_GAME:
    CALL    MINEFIELD_INIT

    CALL    MINE_PLACER

    CALL    GAME_INIT

    CALL    DISPLAY_GAME_SCREEN

    CALL    GAME_MAIN
    JZ      NEW_GAME

    CALL    RESTORE_HANDLERS

    MOV     AX,0003h
    INT     10h

    XOR     AL,AL
    JMP     SHORT   SP_END

NO_MOUSE:
    MOV     DX,OFFSET NO_MOUSE_EP
    MOV     AH,09h
    INT     21h

    MOV     AL,01
    JMP     SHORT   SP_END

DISPLAY_ERROR:
    MOV     DX,OFFSET DISPLAY_NS_EP
    MOV     AH,09h
    INT     21h

    MOV     AL,02
    JMP     SHORT   SP_END

RM_ERROR:
    MOV     DX,OFFSET RM_ERROR_EP
    MOV     AH,09h
    INT     21h

    MOV     AL,03

SP_END:
    MOV     AH,4Ch
    INT     21h

ENDP    START

;****************************************************************************

;****************************************************************************
; Makes a call to the MCGA/VGA BIOS to determine if the 256-color mode is
; available, and if it is, swaps the driver settings with the MCGA mode
; settings

PROC    GET_DISPLAY_TYPE

    MOV     AX,1A00h
    INT     10h

    CMP     AL,1Ah
    JNE     USE_TANDY

    MOV     [SET_MODE],OFFSET MCGA_SET_MODE
    MOV     [SET_BACKGROUND],OFFSET MCGA_SET_BACKGROUND
    MOV     [PLACE_SPRITE],OFFSET MCGA_SPRITE
    MOV     [PAINT_HLINE],OFFSET MCGA_HLINE
    MOV     [PAINT_VLINE],OFFSET MCGA_VLINE
    MOV     [DISPLAY_MOUSE],OFFSET MCGA_CURSOR_PLACE
    MOV     [VIDEO_BUFFER],0A000h
    MOV     [SCRATCHPAD_SIZE],(SCREEN_WIDTH * SCREEN_HEIGHT)
    JMP     SHORT END_GDT

USE_TANDY:
    MOV     [SET_MODE],OFFSET TANDY_SET_MODE
    MOV     [SET_BACKGROUND],OFFSET TANDY_SET_BACKGROUND
    MOV     [PLACE_SPRITE],OFFSET TANDY_SPRITE
    MOV     [PAINT_HLINE],OFFSET TANDY_HLINE
    MOV     [PAINT_VLINE],OFFSET TANDY_VLINE
    MOV     [DISPLAY_MOUSE],OFFSET TANDY_CURSOR_PLACE
    MOV     [VIDEO_BUFFER],0B800h
    MOV     [SCRATCHPAD_SIZE],8000h
    JMP     SHORT END_GDT

END_GDT:
    RET

ENDP    GET_DISPLAY_TYPE

;****************************************************************************

;****************************************************************************
; This procedure checks to see if a mouse driver is installed, and if one
; isn't, returns the carry flag set to abort the program. (Mouse required)

PROC    CHECK_MOUSE

    PUSH    AX
    PUSH    BX
    PUSH    DS

    XOR     AX,AX
    MOV     DS,AX
    MOV     BX,(33h*4)

    MOV     AX,[WORD PTR BX]
    OR      AX,AX
    JNZ     VECTOR_THERE

    MOV     AX,[WORD PTR BX+2]
    OR      AX,AX
    JZ      MOUSE_NOT_FOUND

VECTOR_THERE:
    XOR     AX,AX
    INT     33h

    OR      AX,AX
    JZ      MOUSE_NOT_FOUND

    CLC
    JMP     SHORT   END_CHECK_MOUSE

MOUSE_NOT_FOUND:
    STC

END_CHECK_MOUSE:
    POP     DS
    POP     BX
    POP     AX
    RET

ENDP    CHECK_MOUSE

;****************************************************************************

;****************************************************************************
; Simply calls the active display driver to set the screen mode, can be used
; as a hook procedure to display an introductory logo.

PROC    DISPLAY_LOGO

    CALL    [SET_MODE]

    RET

ENDP    DISPLAY_LOGO

;****************************************************************************

;****************************************************************************
; Makes the minefield all zeros in preparation for the mine place procedure
; and initializes the mine setting variables

PROC    MINEFIELD_INIT

    MOV     [RANDOM_VALUE],0
    MOV     [RND_LEFT],0

    CLD
    MOV     CX,[MINEFIELD_SIZE]
    MOV     DI,OFFSET MINEFIELD
    XOR     AL,AL
    REP     STOSB

    MOV     CL,[NUM_FLAGS]
    MOV     [FLAGS_LEFT],CL

    RET

ENDP    MINEFIELD_INIT

;****************************************************************************

;****************************************************************************
; This procedure uses the GET_RANDOM and PLACE_MINE procedures to set up a
; minefield with random mine locations.

PROC    MINE_PLACER

    CLD
    MOV     DI,OFFSET CHANCEFIELD
    MOV     CX,[MINEFIELD_SIZE]

CHANCE_PLACE_LOOP:
    CALL    GET_RANDOM
    STOSB
    LOOP    CHANCE_PLACE_LOOP

    MOV     DL,[NUM_FLAGS]
    MOV     DH,[NUM_FLAGS]
    MOV     [DIRECTION],REVERSE

MINE_PLACE_LOOP:
    MOV     SI,OFFSET CHANCEFIELD
    MOV     DI,OFFSET MINEFIELD
    MOV     CX,[MINEFIELD_SIZE]

    NEG     [DIRECTION]
    JNS     MP_INNER_LOOP

    ADD     SI,CX
    ADD     DI,CX
    DEC     SI
    DEC     DI

MP_INNER_LOOP:
    TEST    [BYTE PTR DI],10000000b
    JNZ     END_MP_INNER

    CMP     [BYTE PTR SI],DL
    JB      END_MP_INNER

    CALL    PLACE_MINE
    DEC     DH
    JZ      END_MINE_PLACE

END_MP_INNER:
    ADD     SI,[DIRECTION]
    ADD     DI,[DIRECTION]
    LOOP    MP_INNER_LOOP

    DEC     DL
    JMP     SHORT MINE_PLACE_LOOP

END_MINE_PLACE:
    RET

ENDP    MINE_PLACER

;****************************************************************************

;****************************************************************************
; Places a mine at a particular location in the minefield and updates the
; squares surrounding the mine to give the proper clues

PROC    PLACE_MINE

    PUSH    DX
    PUSH    SI

    OR      [BYTE PTR DI],10000000b

    MOV     SI,DI
    MOV     AX,DI
    SUB     AX,OFFSET MINEFIELD
    XOR     BH,BH
    MOV     BL,[MINEFIELD_COLS]
    XOR     DX,DX
    DIV     BX

    MOV     AH,[MINEFIELD_ROWS]
    DEC     AH
    OR      AL,AL
    DEC     BX
    JZ      PLACE_CLUE_LOOP

    SUB     SI,BX
    DEC     SI

PLACE_CLUE_LOOP:
    INC     [BYTE PTR SI]

    OR      DL,DL
    JE      SHORT   NOT_BEHIND

    INC     [BYTE PTR SI-1]

NOT_BEHIND:
    CMP     DL,BL
    JE      SHORT   NOT_INFRONT

    INC     [BYTE PTR SI+1]

NOT_INFRONT:
    CMP     SI,DI
    JA      PCL_DONE
    JB      PCL_CONTINUE

    CMP     AL,AH
    JE      PCL_DONE

PCL_CONTINUE:
    ADD     SI,BX
    INC     SI
    JMP     SHORT PLACE_CLUE_LOOP

PCL_DONE:
    POP     SI
    POP     DX
    RET

ENDP    PLACE_MINE

;****************************************************************************

;****************************************************************************
; This procedure returns one of the psuedo-random numbers that are stored
; in the random number buffer by ADD_RANDOM. The values of the random
; numbers vary from 0 to NUM_FLAGS.

PROC    GET_RANDOM

    PUSH    BX

    CMP     [RND_LEFT],0
    JA      GOT_SOME_RNDS

    CALL    ADD_RANDOM

GOT_SOME_RNDS:
    MOV     BX,[RND_PTR]
    MOV     AL,[BYTE PTR BX]
    XOR     AH,AH
    INC     [RND_PTR]
    DEC     [RND_LEFT]

    POP     BX
    RET

ENDP    GET_RANDOM

;****************************************************************************

;****************************************************************************
; This procedure generates pseudo-random numbers by accessing a randomly-
; seeded area of memory (using the BIOS time count) and then using the number
; found there as an index to the next pseudo-random number.  This procedure
; utilizes 2 different memory areas of differing lengths: 1) the pseudo-
; random number table at the beginning of this program 2) The first 32 KB
; of the memory segment this program is loaded into.

PROC    ADD_RANDOM

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    SI
    PUSH    DI

    MOV     [RND_LEFT],0
    MOV     [RND_PTR],OFFSET RND_NUMBERS

    MOV     BX,[RANDOM_VALUE]
    CMP     BX,2000h
    JA      AR_CONTINUE

    PUSH    DS
    XOR     AX,AX
    MOV     DS,AX
    MOV     AX,[WORD PTR DS:046Ch]
    POP     DS

    MOV     CX,16

REV_LOOP:
    RCL     AX,1
    RCR     BX,1
    LOOP    REV_LOOP

AR_CONTINUE:
    MOV     CL,8
    MOV     CH,RND_BUFFER_SIZE
    MOV     DI,OFFSET RND_NUMBERS
    MOV     SI,OFFSET PSEUDORAND_TABLE
    MOV     AL,[NUM_FLAGS]
    INC     AL
    XOR     AH,AH
    MOV     [MAX_DIVISOR],AX
    MOV     AX,BX

AR_LOOP:
    XOR     DX,DX
    DIV     [WORD PTR MAX_DIVISOR]

    MOV     AL,DL
    STOSB
    INC     [RND_LEFT]

    MOV     AX,BX
    AND     BX,00FFh
    MOV     AL,[BYTE PTR SI+BX]

    MOV     BX,AX
    SHR     BX,CL
    MOV     AH,[BYTE PTR SI+BX]
    MOV     BX,AX
    AND     BX,7FFFh

    ADD     AX,[WORD PTR BX]
    ADD     AX,[WORD PTR DI-1]

    MOV     BX,AX

    DEC     CH
    JNZ     AR_LOOP

    MOV     [RANDOM_VALUE],BX

    POP     DI
    POP     SI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    ADD_RANDOM

;****************************************************************************

;****************************************************************************
; Uses the active display driver to paint a simple rectangle of a specified
; height and width at a specified start coordinate (upper left corner)

; Calling parameters:
; AL = palette value for rectangle
; BX = vertical coordinate (Y)
; CX = horizontal coordinate (X)
; DX = width of rectangle in pixels
; DI = height of rectangle in raster lines

PROC    PAINT_RECTANGLE

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    DI

REC_LOOP:
    CALL    [PAINT_HLINE]
    JC      REC_OFF_SCREEN

    INC     BX
    DEC     DI
    JNZ     REC_LOOP

END_RECLOOP:
    CLC
    JMP     SHORT END_PAINT_REC

REC_OFF_SCREEN:
    STC

END_PAINT_REC:
    POP     DI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    PAINT_RECTANGLE

;****************************************************************************

;****************************************************************************
; Uses the active display driver and the PAINT_RECTANGLE routine to make a
; 3-dimensional looking square on the screen, using the HIGHLIGHT and SHADOW
; equates

; Calling parameters:
; AL = palette value for window "background"
; AH = if 1, indicates that the highlight and shadows should be inverted
; BX = vertical coordinate (Y)
; CX = horizontal coordinate (X)
; DX = width of window in pixels
; DI = height of window in raster lines


PROC    PAINT_WINDOW

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    DI
    PUSH    SI

    CALL    PAINT_RECTANGLE
    JC      WINDOW_OFF_SCREEN

    MOV     SI,DX

    MOV     AL,HIGHLIGHT
    OR      AH,AH
    JZ      NO_ALTER0

    MOV     AL,SHADOW
    DEC     DX

NO_ALTER0:
    CALL    [PAINT_HLINE]
    MOV     DX,SI

    MOV     DX,DI
    OR      AH,AH
    JZ      NO_ALTER1

    DEC     DX

NO_ALTER1:
    CALL    [PAINT_VLINE]
    MOV     DX,SI

    MOV     AL,HIGHLIGHT
    PUSH    BX
    PUSH    CX

    ADD     BX,DI
    DEC     BX
    OR      AH,AH
    JNZ     NO_ALTER2

    MOV     AL,SHADOW
    DEC     DX
    INC     CX

NO_ALTER2:
    CALL    [PAINT_HLINE]
    MOV     DX,SI

    POP     CX
    POP     BX

    ADD     CX,DX
    DEC     CX
    MOV     DX,DI
    OR      AH,AH
    JNZ     NO_ALTER3

    DEC     DX
    INC     BX

NO_ALTER3:
    CALL    [PAINT_VLINE]
    CLC
    JMP     SHORT END_PAINT_WINDOW

WINDOW_OFF_SCREEN:
    STC

END_PAINT_WINDOW:
    POP     SI
    POP     DI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    PAINT_WINDOW

;****************************************************************************

;****************************************************************************
; Makes sure that sufficient memory is available for use by modifying the
; DOS memory block originally allocated.

PROC    RELEASE_MEMORY

    PUSH    AX
    PUSH    BX

    MOV     AH,4Ah
    MOV     BX,2000h
    INT     21h
    JC      END_RM

    MOV     AX,ES
    ADD     AX,1000h
    MOV     [VB_SCRATCHPAD],AX
    CLC

END_RM:
    POP     BX
    POP     AX
    RET

ENDP    RELEASE_MEMORY

;****************************************************************************

;****************************************************************************
; Creates the display of the minefield window in the segment of memory
; reserved for this purpose using the coordinates obtained from the
; GET_COORDINATES routine

PROC    DISPLAY_GAME_SCREEN

    CALL    GET_COORDINATES

    MOV     AX,[VB_SCRATCHPAD]
    MOV     [VB_SEGMENT],AX

    MOV     AL,BACKGROUND_COLOR
    CALL    [SET_BACKGROUND]

    XOR     AH,AH
    MOV     AL,WINDOW_FACE_COLOR
    MOV     BX,[CORNER_Y]
    MOV     CX,[CORNER_X]
    MOV     DX,[WINDOW_WIDTH]
    MOV     DI,[WINDOW_HEIGHT]
    CALL    PAINT_WINDOW
    JNC     DGS_OK_1

    JMP     NEAR DGS_ERROR

DGS_OK_1:
    MOV     AH,1
    MOV     AL,MINEFIELD_MAIN
    MOV     BX,[MINEFIELD_Y]
    DEC     BX
    DEC     BX
    MOV     CX,[MINEFIELD_X]
    DEC     CX
    DEC     CX
    MOV     DX,[MINEFIELD_WIDTH]
    ADD     DX,3
    MOV     DI,[MINEFIELD_HEIGHT]
    ADD     DI,3
    CALL    PAINT_WINDOW
    JNC     DGS_OK_2

    JMP     NEAR DGS_ERROR

DGS_OK_2:
    MOV     AL,STATS_BACKGROUND
    MOV     BX,[FLAGSLEFT_Y]
    DEC     BX
    MOV     CX,[FLAGSLEFT_X]
    DEC     CX
    MOV     DX,STATS_LENGTH
    INC     DX
    INC     DX
    MOV     DI,STATS_WIDTH
    INC     DI
    INC     DI
    CALL    PAINT_WINDOW
    JNC     DGS_OK_3

    JMP     NEAR DGS_ERROR

DGS_OK_3:
    MOV     BX,[TIME_Y]
    DEC     BX
    MOV     CX,[TIME_X]
    DEC     CX
    CALL    PAINT_WINDOW
    JNC     DGS_OK_4

    JMP     NEAR DGS_ERROR

DGS_OK_4:
    MOV     SI,OFFSET SMILEY_FACE
    MOV     BX,[FACE_Y]
    MOV     CX,[FACE_X]
    CALL    [PLACE_SPRITE]
    JNC     DGS_OK_5

    JMP     NEAR DGS_ERROR

DGS_OK_5:
    CMP     [DISPLAY_MSTEXT],TRUE
    JNE     NO_MSTEXT

    MOV     SI,OFFSET TEXT_MINE
    MOV     BX,[TEXT_MINE_Y]
    MOV     CX,[TEXT_MINE_X]
    CALL    [PLACE_SPRITE]
    JNC     DGS_OK_6

    JMP     NEAR DGS_ERROR

DGS_OK_6:
    MOV     SI,OFFSET TEXT_SWEEPER
    MOV     BX,[TEXT_SWEEPER_Y]
    MOV     CX,[TEXT_SWEEPER_X]
    CALL    [PLACE_SPRITE]
    JNC     NO_MSTEXT

    JMP     NEAR DGS_ERROR

NO_MSTEXT:
    MOV     SI,OFFSET UNFLAGGED_SQUARE
    MOV     DX,[MINEFIELD_SIZE]
    MOV     BX,[MINEFIELD_Y]

PLACE_LOOP0:
    MOV     CX,[MINEFIELD_X]

PLACE_LOOP1:
    CALL    [PLACE_SPRITE]
    JC      DGS_ERROR

    ADD     CX,8
    MOV     AX,CX
    SUB     AX,[MINEFIELD_X]
    CMP     AX,[MINEFIELD_WIDTH]
    JB      PLACE_LOOP1

    ADD     BX,8
    MOV     AX,BX
    SUB     AX,[MINEFIELD_Y]
    CMP     AX,[MINEFIELD_HEIGHT]
    JB      PLACE_LOOP0

    PUSH    ES
    PUSH    DS

    MOV     BX,[VIDEO_BUFFER]
    MOV     ES,BX
    MOV     [VB_SEGMENT],BX

    MOV     CX,[SCRATCHPAD_SIZE]
    MOV     AX,[VB_SCRATCHPAD]
    MOV     DS,AX

    XOR     SI,SI
    XOR     DI,DI

    REP     MOVSW

    POP     DS
    POP     ES

    CALL    WRITE_STATS

    CLC
    JMP     SHORT END_DGS

DGS_ERROR:
    STC

END_DGS:
    RET

ENDP    DISPLAY_GAME_SCREEN

;****************************************************************************

;****************************************************************************
; Writes the number of flags left and game time to the screen using the
; current display driver

PROC    WRITE_STATS

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    SI
    PUSH    DI

    MOV     AX,OFFSET WROTE_FLAGS_LEFT
    PUSH    AX
    MOV     AX,[WORD PTR FLAGS_LEFT]
    MOV     BX,[FLAGSLEFT_Y]
    MOV     CX,[FLAGSLEFT_X]

WS_LOOP:
    MOV     SI,10
    MOV     DI,4
    XOR     DX,DX

HEX_2_DEC:
    DIV     SI
    PUSH    DX

    XOR     DX,DX
    DEC     DI
    JNZ     HEX_2_DEC

    JMP     NEAR DISPLAY_STAT

WROTE_FLAGS_LEFT:
    MOV     AX,OFFSET WROTE_TIME
    PUSH    AX
    MOV     AX,[TIME]
    MOV     BX,[TIME_Y]
    MOV     CX,[TIME_X]
    JMP     SHORT WS_LOOP

WROTE_TIME:
    POP     DI
    POP     SI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    WRITE_STATS

;****************************************************************************

;****************************************************************************
; Displays a single game stat on the screen using the DIGIT_X bitmaps and
; the current display driver

PROC    DISPLAY_STAT

    MOV     DI,STATS_DIGITS

DS_LOOP:
    MOV     AX,((8*11)+4)
    POP     SI
    MUL     SI
    MOV     SI,OFFSET DIGIT_0
    ADD     SI,AX

    CALL    [PLACE_SPRITE]

    ADD     CX,8
    DEC     DI
    JNZ     DS_LOOP

    RET

ENDP    DISPLAY_STAT

;****************************************************************************

;****************************************************************************
; Calculates the coordinates of windows and bitmaps by using the number of
; minefield columns and rows

PROC    GET_COORDINATES

    MOV     AL,[MINEFIELD_COLS]
    XOR     AH,AH

    MOV     CL,3
    SHL     AX,CL
    MOV     [MINEFIELD_WIDTH],AX
    ADD     AX,(((SIDESTRIP_WIDTH)*2)+2+3)
    MOV     [WINDOW_WIDTH],AX
    MOV     BX,SCREEN_WIDTH
    SUB     BX,AX
    SHR     BX,1
    MOV     [CORNER_X],BX

    MOV     AL,[MINEFIELD_ROWS]
    XOR     AH,AH
    SHL     AX,CL
    MOV     [MINEFIELD_HEIGHT],AX
    ADD     AX,(SIDESTRIP_WIDTH+TOP_WIDTH+2+3)
    MOV     [WINDOW_HEIGHT],AX
    MOV     DX,SCREEN_HEIGHT
    SUB     DX,AX
    SHR     DX,1
    MOV     [CORNER_Y],DX

    PUSH    BX
    ADD     BX,(SIDESTRIP_WIDTH+1)
    MOV     [FLAGSLEFT_X],BX
    ADD     DX,(((TOP_WIDTH-STATS_WIDTH)/2)+1+1)
    MOV     [FLAGSLEFT_Y],DX
    MOV     [TIME_Y],DX
    POP     BX

    ADD     BX,[WINDOW_WIDTH]
    SUB     BX,(SIDESTRIP_WIDTH+STATS_LENGTH+2)
    MOV     [TIME_X],BX

    MOV     AX,[FLAGSLEFT_X]
    INC     AX
    MOV     [MINEFIELD_X],AX
    MOV     AX,[CORNER_Y]
    ADD     AX,(TOP_WIDTH+3)
    MOV     [MINEFIELD_Y],AX

    SUB     BX,[FLAGSLEFT_X]
    SUB     BX,(STATS_LENGTH+2)
    CMP     BX,(FACE_WIDTH+TEXT_WIDTH+12)
    JB      NO_TEXT

    MOV     [DISPLAY_MSTEXT],TRUE
    MOV     DX,(FACE_WIDTH+TEXT_WIDTH+12)
    SUB     BX,DX
    SHR     BX,1
    MOV     AX,[FLAGSLEFT_X]
    ADD     AX,(STATS_LENGTH+1+3)
    ADD     AX,BX
    MOV     [TEXT_MINE_X],AX
    ADD     AX,(TEXT_MINE_LENGTH+3)
    MOV     [FACE_X],AX
    ADD     AX,(FACE_WIDTH+3)
    MOV     [TEXT_SWEEPER_X],AX
    JMP     SHORT   GET_Y_COORDS

NO_TEXT:
    MOV     [DISPLAY_MSTEXT],FALSE
    SHR     BX,1
    MOV     AX,[TIME_X]
    SUB     AX,BX
    SUB     AX,((FACE_WIDTH/2)+1+1)
    MOV     [FACE_X],AX

GET_Y_COORDS:
    MOV     AX,[CORNER_Y]
    ADD     AX,(((TOP_WIDTH-FACE_HEIGHT)/2)+1+1)
    MOV     [TEXT_MINE_Y],AX
    MOV     [FACE_Y],AX
    MOV     [TEXT_SWEEPER_Y],AX

    RET

ENDP    GET_COORDINATES

;****************************************************************************

;****************************************************************************
; Calls the BIOS to beep the speaker

PROC    BEEP

    PUSH    AX
    PUSH    BX

    MOV     AX,0E07h
    XOR     BX,BX
    INT     10h

    POP     BX
    POP     AX
    RET

ENDP    BEEP

;****************************************************************************

;****************************************************************************
; Initializes variables in prepartion of beginning a new game.

PROC    GAME_INIT

    MOV     [GAME_DONE],FALSE
    MOV     [MOUSE_EVENT],FALSE
    MOV     [MOUSE_CLICK_STATUS],0

    MOV     [TIME],0
    MOV     [TIMER],0
    MOV     [TIMER_ON],FALSE
    MOV     AL,[NUM_FLAGS]
    MOV     [FLAGS_LEFT],AL

    MOV     AL,[MINEFIELD_ROWS]
    MOV     BL,[MINEFIELD_COLS]
    MUL     BL

    MOV     [SQUARES_LEFT],AX

    RET

ENDP    GAME_INIT

;****************************************************************************

;****************************************************************************
; This is the main active procedure during game play, processing game events.
; This procedure polls the input handlers (the BIOS keyboard handler and the
; mouse handler set by this program) to see if an "event" has occured, and
; if one has, calls the routines necessary to handle the "event."  This
; procedure also updates the mouse cursor and game time during its free time.

PROC    GAME_MAIN

    MOV     AX,[TIMER]
    MOV     BX,1000
    MUL     BX

    MOV     BX,18204
    DIV     BX

    CMP     AX,[TIME]
    JE      NO_UPDATE

    MOV     [TIME],AX
    CALL    WRITE_STATS

NO_UPDATE:
    CMP     [GAME_DONE],FALSE
    JNE     END_GMAIN

    CMP     [CURSOR_STATUS],ON
    JE      NO_NEED_TO_SM

    CALL    [DISPLAY_MOUSE]

NO_NEED_TO_SM:
    CMP     [MOUSE_EVENT],TRUE
    JNE     CHECK_KEYBOARD

    MOV     AX,[MCLICK_X]
    MOV     BX,[MCLICK_Y]
    CALL    CLICK_TO_SQUARE

    CMP     [AREA_CLICKED],FACE_AREA
    JNE     CHECK_MINESWEEP

    CALL    FACE_CLICK
    JMP     SHORT CLEAR_MOUSE_EVENT

CHECK_MINESWEEP:
    CMP     [AREA_CLICKED],MINEFIELD_AREA
    JNE     CLEAR_MOUSE_EVENT

    CALL    SQUARE_CLICKED

CLEAR_MOUSE_EVENT:
    MOV     [MOUSE_EVENT],FALSE
    MOV     [MOUSE_CLICK_STATUS],0

CHECK_KEYBOARD:
    MOV     AH,01h
    INT     16h
    JZ      GAME_MAIN

    XOR     AH,AH
    INT     16h

    CMP     AX,011Bh
    JNE     GAME_MAIN

    MOV     [MOUSE_CLICK_STATUS],1000b
    CALL    FACE_CLICK
    MOV     [MOUSE_CLICK_STATUS],0
    JMP     SHORT GAME_MAIN

END_GMAIN:
    CMP     [GAME_DONE],-2
    JE      GM_RETURN

    CMP     [GAME_DONE],-1
    JE      CLEAR_ZERO_FLAG

    CALL    GAME_FINALLY

GM_END_LOOP:
    CMP     [CURSOR_STATUS],ON
    JE      DONT_REDISPLAY

    CALL    [DISPLAY_MOUSE]

DONT_REDISPLAY:
    CMP     [MOUSE_EVENT],TRUE
    JNE     GM_GET_KEY

    MOV     AX,[MCLICK_X]
    MOV     BX,[MCLICK_Y]
    CALL    CLICK_TO_SQUARE

    CMP     [AREA_CLICKED],FACE_AREA
    JNE     GM_CLEAR_MOUSE

    CALL    FACE_CLICK

    CMP     [GAME_DONE],-2
    JE      GM_RETURN

    CMP     [GAME_DONE],-1
    JE      CLEAR_ZERO_FLAG

GM_CLEAR_MOUSE:
    MOV     [MOUSE_EVENT],FALSE
    MOV     [MOUSE_CLICK_STATUS],0

GM_GET_KEY:
    MOV     AH,01h
    INT     16h
    JZ      GM_END_LOOP

    CALL    FACE_CLICK

    XOR     AH,AH
    INT     16h

    CMP     AX,011Bh
    JE      CLEAR_ZERO_FLAG

    XOR     AX,AX
    JMP     SHORT GM_RETURN

CLEAR_ZERO_FLAG:
    MOV     AX,0001h
    CMP     AH,AL

GM_RETURN:
    PUSHF
    CMP     [CURSOR_STATUS],ON
    JNE     GM_CONTINUE

    CALL    MOUSE_REMOVE_CURSOR

GM_CONTINUE:
    POPF
    RET

ENDP    GAME_MAIN

;****************************************************************************

;****************************************************************************
; Displays unflagged or incorrectly flagged mines in the event of a loss, or
; beeps and shows the smiley face with shades in the event of a win

PROC    GAME_FINALLY

    MOV     [TIMER_ON],FALSE

    CMP     [CURSOR_STATUS],ON
    JNE     GFIN_CONTINUE

    CALL    MOUSE_REMOVE_CURSOR

GFIN_CONTINUE:
    CMP     [GAME_DONE],TRUE
    JA      WON_IT

    MOV     BX,[MINEFIELD_Y]
    MOV     DI,OFFSET MINEFIELD
    MOV     DH,[MINEFIELD_ROWS]

GF_OUTER:
    MOV     CX,[MINEFIELD_X]
    MOV     DL,[MINEFIELD_COLS]

GF_INNER:
    MOV     AL,[BYTE PTR DI]
    TEST    AL,10000000b
    JZ      ISNT_MINE

    TEST    AL,01110000b
    JNZ     END_GFI

    MOV     SI,OFFSET UNFLAGGED_MINE
    JMP     SHORT SHOWIT_NOW

ISNT_MINE:
    TEST    AL,01100000b
    JZ      END_GFI

    MOV     SI,OFFSET FLAGGED_INCORRECT

SHOWIT_NOW:
    CALL    [PLACE_SPRITE]

END_GFI:
    ADD     CX,8
    INC     DI
    DEC     DL
    JNZ     GF_INNER

    ADD     BX,8
    DEC     DH
    JNZ     GF_OUTER

    MOV     SI,OFFSET FROWNEY_FACE

END_SPRITE:
    MOV     BX,[FACE_Y]
    MOV     CX,[FACE_X]
    CALL    [PLACE_SPRITE]
    JMP     SHORT END_FINALLY

WON_IT:
    MOV     BX,[CURRENT_HI_SCORE]
    MOV     CX,[TIME]
    CMP     CX,[WORD PTR BX]
    JAE     DONT_SAVE_SCORE

    CALL    BEEP
    CALL    BEEP
    CALL    BEEP

    MOV     [WORD PTR BX],CX
    CALL    WRITE_CURRENT_PARMS

DONT_SAVE_SCORE:
    MOV     CX,[WORD PTR BX]
    MOV     [WORD PTR FLAGS_LEFT],CX
    CALL    WRITE_STATS

    MOV     [WORD PTR FLAGS_LEFT],0

    CALL    BEEP
    CALL    BEEP
    CALL    BEEP
    MOV     SI,OFFSET HAPPY_FACE
    JMP     SHORT END_SPRITE

END_FINALLY:
    RET

ENDP    GAME_FINALLY

;****************************************************************************

;****************************************************************************
; Calculates the exact square on the game screen that was clicked when a
; mouse button event occurs and calls the appropriate routine.

PROC    CLICK_TO_SQUARE

    MOV     [AREA_CLICKED],IRRELEVANT

    CMP     AX,[MINEFIELD_X]
    JB      NOT_ON_MINEFIELD

    CMP     BX,[MINEFIELD_Y]
    JB      NOT_ON_MINEFIELD

    MOV     CX,AX
    MOV     DX,BX

    SUB     CX,[MINEFIELD_X]
    SUB     DX,[MINEFIELD_Y]

    CMP     CX,[MINEFIELD_WIDTH]
    JAE     NOT_ON_MINEFIELD

    CMP     DX,[MINEFIELD_HEIGHT]
    JAE     NOT_ON_MINEFIELD

ON_MINEFIELD:
    MOV     AX,CX
    MOV     BX,DX
    MOV     CL,3

    SHR     AX,CL
    SHR     BX,CL

    MOV     [AREA_CLICKED],MINEFIELD_AREA
    JMP     NEAR NOT_ON_FACE

NOT_ON_MINEFIELD:
    CMP     AX,[FACE_X]
    JBE     NOT_ON_FACE

    CMP     BX,[FACE_Y]
    JBE     NOT_ON_FACE

    MOV     CX,AX
    MOV     DX,BX

    SUB     CX,[FACE_X]
    SUB     DX,[FACE_Y]
    DEC     CX
    DEC     DX

    CMP     CX,(FACE_WIDTH - 2)
    JAE     NOT_ON_FACE

    CMP     DX,(FACE_HEIGHT - 2)
    JAE     NOT_ON_FACE

ON_FACE:
    MOV     [AREA_CLICKED],FACE_AREA

NOT_ON_FACE:
    RET

ENDP    CLICK_TO_SQUARE

;****************************************************************************

;****************************************************************************
; Processes a mouse event where the smiley face at the top of the minefield
; window is clicked (or a keyboard event that generates the same result).

PROC    FACE_CLICK

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    SI

    MOV     [GAME_DONE],0

    MOV     [TIMER],0
    MOV     [TIMER_ON],TRUE

    CMP     [CURSOR_STATUS],ON
    JNE     FCLICK_CONTINUE

    CALL    MOUSE_REMOVE_CURSOR

FCLICK_CONTINUE:
    MOV     SI,OFFSET CLICKED_FACE
    MOV     BX,[FACE_Y]
    MOV     CX,[FACE_X]
    CALL    [PLACE_SPRITE]

    MOV     AL,[MOUSE_CLICK_STATUS]
    OR      AL,AL
    JZ      END_FCLICK

    CMP     AL,1010b
    JE      FACE_DOUBLE_CLICK

    CMP     AL,1000b
    JE      FACE_RIGHT_CLICK

    DEC     [GAME_DONE]

FACE_RIGHT_CLICK:
    RCR     AL,1
    RCR     AL,1
    ADC     AL,0
    DEC     [GAME_DONE]
    JMP     SHORT END_FCLICK

FACE_DOUBLE_CLICK:
    CALL    INCREMENT_GAME

    RCR     AL,1
    RCR     AL,1
    ADC     AL,0
    SUB     [GAME_DONE],2

END_FCLICK:
    CMP     [TIMER],MIN_FCLICK_WAIT
    JB      END_FCLICK

FCLICK_LOOP:
    PUSH    BX
    PUSH    CX
    PUSH    AX

    MOV     AX,0003h
    INT     33h

    POP     AX
    TEST    AL,BL
    POP     CX
    POP     BX
    JNZ     FCLICK_LOOP

    MOV     SI,OFFSET SMILEY_FACE
    CALL    [PLACE_SPRITE]

    MOV     [TIMER],0
    MOV     [TIMER_ON],FALSE

    POP     SI
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    FACE_CLICK

;****************************************************************************

;****************************************************************************
; Changes the size of the minefield due to a double button mouse click on
; the smiley face and saves the new size to disk

PROC    INCREMENT_GAME

    PUSH    AX
    PUSH    BX

    MOV     BX,[CURRENT_GAME]
    ADD     BX,7
    CMP     BX,OFFSET END_GAME_TYPES
    JB      GT_OK

    MOV     BX,OFFSET GAME_TYPES

GT_OK:
    MOV     AX,[WORD PTR BX]
    MOV     [MINEFIELD_COLS],AL
    MOV     [MINEFIELD_ROWS],AH

    MOV     AL,[BYTE PTR BX+2]
    MOV     [NUM_FLAGS],AL

    MOV     AX,[WORD PTR BX+3]
    MOV     [MINEFIELD_SIZE],AX

    MOV     AX,[WORD PTR BX+5]
    MOV     [CURRENT_HI_SCORE],AX

    MOV     [CURRENT_GAME],BX

    CALL    WRITE_CURRENT_PARMS

    POP     BX
    POP     AX
    RET

ENDP    INCREMENT_GAME

;****************************************************************************

;****************************************************************************
; Processes a mouse event where a particular square on the minefield is
; clicked

; Calling parameters:
; BX = minefield row of square clicked
; CX = minefield column of square clicked

PROC    SQUARE_CLICKED

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    SI
    PUSH    DI

    MOV     [TIMER_ON],TRUE

    AND     [MOUSE_CLICK_STATUS],1010b

    CMP     [MOUSE_CLICK_STATUS],0
    JNE     MC_OK

    JMP     NEAR DO_NOTHING

MC_OK:
    MOV     DI,OFFSET MINEFIELD
    MOV     CX,AX
    MOV     AL,[MINEFIELD_COLS]
    XOR     AH,AH
    MUL     BX
    ADD     DI,AX
    ADD     DI,CX

    MOV     AL,[BYTE PTR DI]

    CMP     [MOUSE_CLICK_STATUS],1010b
    JE      DOUBLE_CLICK

    CMP     [MOUSE_CLICK_STATUS],1000b
    JE      RIGHT_CLICK

LEFT_CLICK:
    TEST    AL,01110000b
    JZ      NOT_CLEARED_YET

    JMP     NEAR DO_NOTHING

NOT_CLEARED_YET:
    OR      [BYTE PTR DI],00010000b

    TEST    AL,10000000b
    JZ      NOT_MINE

    MOV     [GAME_DONE],TRUE
    MOV     SI,OFFSET STRUCK_MINE
    JMP     NEAR DISPSPRITE

NOT_MINE:
    DEC     [SQUARES_LEFT]
    PUSH    BX
    MOV     BL,[NUM_FLAGS]
    XOR     BH,BH
    CMP     BX,[SQUARES_LEFT]
    POP     BX
    JB      NOT_OVER_YET

    MOV     [GAME_DONE],TRUE
    INC     [GAME_DONE]

NOT_OVER_YET:
    AND     AL,0Fh
    OR      AL,AL
    JNZ     IS_NOT_ZERO

    CALL    CLEAR_ALL_ADJACENT

IS_NOT_ZERO:
    MOV     SI,OFFSET CLEARED_SQUARE
    XOR     AH,AH
    AND     AL,0Fh

    PUSH    CX
    PUSH    AX

    MOV     CL,6
    SHL     AX,CL
    ADD     SI,AX

    POP     AX
    POP     CX

    SHL     AX,1
    SHL     AX,1
    ADD     SI,AX

    JMP     SHORT DISPSPRITE

DOUBLE_CLICK:
    TEST    AL,00010000b
    JZ      DO_NOTHING

    AND     AL,0Fh
    JZ      DO_NOTHING

    MOV     [MOUSE_CLICK_STATUS],00000010b
    CALL    QUICK_CLEAR

    JMP     SHORT DO_NOTHING

RIGHT_CLICK:
    TEST    AL,00010000b
    JNZ     DO_NOTHING

    TEST    AL,00100000b
    JNZ     IS_QUESTION

    TEST    AL,01000000b
    JNZ     IS_FLAGGED

    CMP     [FLAGS_LEFT],0
    JE      DO_NOTHING

    DEC     [FLAGS_LEFT]
    CALL    WRITE_STATS
    OR      [BYTE PTR DI],01000000b
    MOV     SI,OFFSET FLAGGED_SQUARE
    JMP     SHORT DISPSPRITE

IS_QUESTION:
    XOR     [BYTE PTR DI],00100000b
    MOV     SI,OFFSET UNFLAGGED_SQUARE
    JMP     SHORT DISPSPRITE

IS_FLAGGED:
    XOR     [BYTE PTR DI],01100000b
    INC     [FLAGS_LEFT]
    CALL    WRITE_STATS
    MOV     SI,OFFSET QUESTIONED_SQUARE

DISPSPRITE:
    MOV     AX,CX
    MOV     CL,3
    SHL     AX,CL
    SHL     BX,CL

    ADD     AX,[MINEFIELD_X]
    ADD     BX,[MINEFIELD_Y]
    MOV     CX,AX

    CMP     [CURSOR_STATUS],ON
    JNE     PUTSPRITE

    CALL    MOUSE_REMOVE_CURSOR

PUTSPRITE:
    CALL    [PLACE_SPRITE]

    CMP     [CURSOR_STATUS],ON
    JE      DO_NOTHING

    CALL    [DISPLAY_MOUSE]

DO_NOTHING:
    POP     DI
    POP     SI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    SQUARE_CLICKED

;****************************************************************************

;****************************************************************************
; Processes a double button click to an already uncovered area of the
; minefield.  This procedure speeds up game play immensely.

PROC    QUICK_CLEAR

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    SI
    PUSH    DI
    PUSH    BP

    MOV     DX,BX
    MOV     BP,CX
    MOV     AH,[MINEFIELD_COLS]
    MOV     BL,AH
    XOR     BH,BH
    DEC     AH
    MOV     CH,[MINEFIELD_ROWS]
    DEC     CH

    OR      CL,CL
    JZ      NO_ADD_NW

    OR      DL,DL
    JZ      NO_ADD_NW

    PUSH    DI
    SUB     DI,BX
    TEST    [BYTE PTR DI-1],01000000b
    POP     DI
    JZ      NO_ADD_NW

    DEC     AL

NO_ADD_NW:
    OR      DL,DL
    JZ      NO_ADD_NORTH

    PUSH    DI
    SUB     DI,BX
    TEST    [BYTE PTR DI],01000000b
    POP     DI
    JZ      NO_ADD_NORTH

    DEC     AL

NO_ADD_NORTH:
    OR      DL,DL
    JZ      NO_ADD_NE

    CMP     CL,AH
    JE      NO_ADD_NE

    PUSH    DI
    SUB     DI,BX
    TEST    [BYTE PTR DI+1],01000000b
    POP     DI
    JZ      NO_ADD_NE

    DEC     AL

NO_ADD_NE:
    CMP     CL,AH
    JE      NO_ADD_EAST

    TEST    [BYTE PTR DI+1],01000000b
    JZ      NO_ADD_EAST

    DEC     AL

NO_ADD_EAST:
    CMP     DL,CH
    JE      NO_ADD_SE

    CMP     CL,AH
    JE      NO_ADD_SE

    TEST    [BYTE PTR DI+BX+1],01000000b
    JZ      NO_ADD_SE

    DEC     AL

NO_ADD_SE:
    CMP     DL,CH
    JE      NO_ADD_SOUTH

    TEST    [BYTE PTR DI+BX],01000000b
    JZ      NO_ADD_SOUTH

    DEC     AL

NO_ADD_SOUTH:
    CMP     DL,CH
    JE      NO_ADD_SW

    OR      CL,CL
    JZ      NO_ADD_SW

    TEST    [BYTE PTR DI+BX-1],01000000b
    JZ      NO_ADD_SW

    DEC     AL

NO_ADD_SW:
    OR      CL,CL
    JZ      NO_ADD_WEST

    TEST    [BYTE PTR DI-1],01000000b
    JZ      NO_ADD_WEST

    DEC     AL

NO_ADD_WEST:
    OR      AL,AL
    JZ      QC_OK

    CALL    BEEP
    JMP     SHORT END_QC

QC_OK:
    MOV     BX,DX
    MOV     CX,BP

    CALL    CLEAR_ALL_ADJACENT

END_QC:
    POP     BP
    POP     DI
    POP     SI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    QUICK_CLEAR

;****************************************************************************

;****************************************************************************
; Simulates a click to all adjacent squares in the event that a square is
; uncovered that has no adjacent mines or when the quick-clear button click
; is used.

PROC    CLEAR_ALL_ADJACENT

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    SI
    PUSH    DI

    MOV     AX,CX
    MOV     SI,BX

    MOV     DL,[MINEFIELD_ROWS]
    DEC     DL
    XOR     DH,DH
    MOV     DI,DX
    MOV     DL,[MINEFIELD_COLS]
    DEC     DL

    DEC     AL
    DEC     BL

    OR      CL,CL
    JZ      NO_NW

    OR      SI,SI
    JZ      NO_NW

    CALL    SQUARE_CLICKED

NO_NW:
    INC     AL

    OR      SI,SI
    JZ      NO_NORTH

    CALL    SQUARE_CLICKED

NO_NORTH:
    INC     AL

    CMP     CL,DL
    JE      NO_NE

    OR      SI,SI
    JZ      NO_NE

    CALL    SQUARE_CLICKED

NO_NE:
    INC     BL

    CMP     CL,DL
    JE      NO_EAST

    CALL    SQUARE_CLICKED

NO_EAST:
    INC     BL

    CMP     CL,DL
    JE      NO_SE

    CMP     SI,DI
    JE      NO_SE

    CALL    SQUARE_CLICKED

NO_SE:
    DEC     AL

    CMP     SI,DI
    JE      NO_SOUTH

    CALL    SQUARE_CLICKED

NO_SOUTH:
    DEC     AL

    OR      CL,CL
    JZ      NO_SW

    CMP     SI,DI
    JE      NO_SW

    CALL    SQUARE_CLICKED

NO_SW:
    DEC     BL

    OR      CL,CL
    JZ      NO_WEST

    CALL    SQUARE_CLICKED

NO_WEST:
    POP     DI
    POP     SI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    CLEAR_ALL_ADJACENT

;****************************************************************************

;*****************************************************************************
; Saves the current game parameters and high scores to disk by finding the
; program path and name directly after the DOS environment block and updating
; the beginning of the file found.

PROC    WRITE_CURRENT_PARMS

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    SI

    MOV     SI,002Ch
    MOV     AX,[WORD PTR SI]

    PUSH    DS
    MOV     DS,AX
    XOR     SI,SI
    XOR     AX,AX

FIND_ENV_END_LOOP:
    CMP     AX,[WORD PTR SI]
    JE      FOUND_END
    INC     SI
    JMP     SHORT   FIND_ENV_END_LOOP

FOUND_END:
    ADD     SI,4
    MOV     DX,SI
    MOV     AX,3D02h
    INT     21h
    POP     DS
    JC      WCP_FILE_ERROR

    MOV     BX,AX
    MOV     AH,40h
    MOV     CX,(OFFSET END_DEFAULTS - 100h)
    MOV     DX,0100h
    INT     21h
    JC      WCP_FILE_ERROR

    MOV     AH,3Eh
    INT     21h
    JNC     NO_FILE_ERROR

WCP_FILE_ERROR:
    STC
    JMP     SHORT   END_WCP

NO_FILE_ERROR:
    CLC

END_WCP:
    POP     SI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    WRITE_CURRENT_PARMS

;*****************************************************************************

;****************************************************************************
; Restores the timer and mouse event interrupt handlers to their orignal
; values

PROC    RESTORE_HANDLERS

    PUSH    DS

    LDS     DX,[OLDINT1CH]
    MOV     AX,251Ch
    INT     21h

    POP     DS

    MOV     AX,000Ch
    XOR     CX,CX
    INT     33h

    RET

ENDP    RESTORE_HANDLERS

;****************************************************************************

;****************************************************************************
; Calls the mouse driver to set up user interrupts on mouse events and
; initializes the mouse status variables

PROC    SET_MOUSE_HANDLERS


    XOR     AX,AX
    INT     33h

    MOV     AX,0004h
    MOV     CX,[MOUSE_X]
    SHL     CX,1
    MOV     DX,[MOUSE_Y]
    INT     33h

    PUSH    ES

    MOV     AX,351Ch
    INT     21h

    MOV     [WORD PTR OLDINT1CH],BX
    MOV     BX,ES
    MOV     [WORD PTR OLDINT1CH+2],BX

    POP     ES

    MOV     AX,251Ch
    MOV     DX,OFFSET TIMER_HANDLER
    INT     21h

    MOV     AX,000Ch
    MOV     CX,1011b
    MOV     DX,OFFSET MOUSE_HANDLER
    INT     33h

    MOV     [MOUSE_EVENT],TRUE
    MOV     [MOUSE_CLICK_STATUS],0
    MOV     [MOUSE_COUNTDOWN],0

    RET

ENDP    SET_MOUSE_HANDLERS

;****************************************************************************

;****************************************************************************
; Handles an interrupt triggered by a mouse event: movement or button click.

PROC    MOUSE_HANDLER FAR

    PUSH    DS
    PUSH    CS
    POP     DS

    SHR     CX,1
    CMP     CX,[MOUSE_X]
    JNE     MOUSE_MOVEMENT

    CMP     DX,[MOUSE_Y]
    JE      CHECK_BUTTONS

MOUSE_MOVEMENT:
    MOV     [MOUSE_X],CX
    MOV     [MOUSE_Y],DX

    CMP     [CURSOR_STATUS],ON
    JNE     CHECK_BUTTONS

    CALL    MOUSE_REMOVE_CURSOR

CHECK_BUTTONS:
    AND     AL,1010b
    JZ      END_MOUSE_HANDLER

    CMP     [MOUSE_EVENT],TRUE
    JE      END_MOUSE_HANDLER

    CMP     [MOUSE_CLICK_STATUS],0
    JE      IS_NEW_CLICK

    OR      [MOUSE_CLICK_STATUS],AL

    JMP     SHORT END_MOUSE_HANDLER

IS_NEW_CLICK:
    OR      [MOUSE_CLICK_STATUS],AL
    MOV     [MCLICK_X],CX
    MOV     [MCLICK_Y],DX
    MOV     [MOUSE_COUNTDOWN],MOUSE_DCLICK_TIME

END_MOUSE_HANDLER:
    POP     DS
    RET

ENDP    MOUSE_HANDLER

;****************************************************************************

;****************************************************************************
; Uses the information saved by the cursor placement routine of the display
; driver to quickly erase the mouse cursor from the screen.  This procedure
; only works with contiguous/interlaced memory-mapped video, it will not work
; with screen modes that use bit-planes or video pages, so if a new video
; driver is added to this program this routine may need to be changed as
; well.  (This is one reason an EGA driver was not developed.)

PROC    MOUSE_REMOVE_CURSOR

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    SI
    PUSH    DI
    PUSH    DS
    PUSH    ES

    MOV     AX,CS
    MOV     DS,AX

    MOV     AX,[VIDEO_BUFFER]
    MOV     ES,AX

    MOV     SI,OFFSET CURSOR_SAVE_BUFFER
    MOV     BX,MC_HEIGHT

REMOVE_LOOP:
    LODSW
    MOV     DI,AX
    LODSW
    MOV     CX,AX

    REP     MOVSB

    DEC     BX
    JNZ     REMOVE_LOOP

    MOV     [CURSOR_STATUS],OFF

    POP     ES
    POP     DS
    POP     DI
    POP     SI
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    MOUSE_REMOVE_CURSOR

;****************************************************************************

;****************************************************************************
; Called 18.204 times per second to process timing information needed by
; the mouse event handler and the game timer

PROC    TIMER_HANDLER FAR

    CMP     [CS:MOUSE_EVENT],TRUE
    JE      END_TIMER

    CMP     [CS:MOUSE_COUNTDOWN],0
    JE      END_TIMER

    DEC     [CS:MOUSE_COUNTDOWN]
    JNZ     END_TIMER

    INC     [CS:MOUSE_EVENT]

END_TIMER:
    CMP     [CS:TIMER_ON],TRUE
    JNE     GO_NEXT_HANDLER

    INC     [CS:TIMER]

GO_NEXT_HANDLER:
    JMP     [CS:OLDINT1CH]

ENDP    TIMER_HANDLER

;****************************************************************************

;****************************************************************************
; TANDY 16-COLOR MODE DISPLAY DRIVER

;****************************************************************************
; Sets the computer to Tandy 16-color graphics mode and updates the BIOS
; data area to trick the mouse driver into thinking the current screen mode
; is the 640 x 200 2-color mode.

PROC    TANDY_SET_MODE

    PUSH    AX
    PUSH    BX
    PUSH    DS

    MOV     AX,0009h
    INT     10h

    MOV     AX,0040h
    MOV     DS,AX

    MOV     BX,0049h
    MOV     [BYTE PTR BX],06

    MOV     [BYTE PTR BX+1],80

    CLC
    POP     DS
    POP     BX
    POP     AX
    RET

ENDP    TANDY_SET_MODE

;****************************************************************************

;****************************************************************************
; Displays a bitmap at a particular area on the screen in Tandy 16-color mode

; Calling parameters:
; DS:SI = sprite (bitmap_height, bitmap_width, bitmap)
; BX    = vertical coordinate for placement (Y)
; CX    = horizontal coordinate for placement (X)

PROC    TANDY_SPRITE

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    SI
    PUSH    DI
    PUSH    BP
    PUSH    ES

    PUSH    [VB_SEGMENT]
    POP     ES

    CLD
    LODSW
    MOV     DI,AX
    ADD     AX,CX
    CMP     AX,TANDY_MAXX
    JNA     X_LOC_OK

    JMP     NEAR TS_OFF_SCREEN

X_LOC_OK:
    LODSW
    PUSH    AX
    ADD     AX,BX
    CMP     AX,TANDY_MAXY
    POP     AX
    JA      TS_OFF_SCREEN

    MOV     BP,AX
    MOV     AX,BX
    AND     AX,0011b
    ROR     AX,1
    ROR     AX,1
    ROR     AX,1
    XCHG    AX,BX
    SHR     AX,1
    SHR     AX,1
    MOV     DX,TANDY_RASTER_LINE
    MUL     DX
    XCHG    DI,BX
    ADD     DI,AX
    MOV     DL,CL
    AND     DL,1
    SHR     CX,1
    ADD     DI,CX

TANDY_SPRITE_LOOP:
    PUSH    DI
    PUSH    DX
    PUSH    BX

    MOV     CL,4

TS_LOOP2:
    OR      DL,DL
    JZ      IS_EVEN_COORDINATE

    LODSB
    AND     [BYTE PTR ES:DI],0F0h
    AND     AL,0Fh
    OR      [BYTE PTR ES:DI],AL

    XOR     DL,DL
    INC     DI
    DEC     BX
    JMP     SHORT END_TSLOOP2

IS_EVEN_COORDINATE:
    CMP     BX,1
    JE      LAST_PIXEL

    LODSW
    SHL     AL,CL
    AND     AH,0Fh
    OR      AL,AH
    STOSB

    DEC     BX
    DEC     BX

END_TSLOOP2:
    JNZ     TS_LOOP2

    JMP     SHORT ENDED_EVEN

LAST_PIXEL:
    LODSB
    AND     [BYTE PTR ES:DI],0Fh
    SHL     AL,CL
    OR      [BYTE PTR ES:DI],AL

ENDED_EVEN:
    POP     BX
    POP     DX
    POP     DI

    ADD     DI,2000h
    TEST    DI,8000h
    JZ      END_TSLOOP

    AND     DI,7FFFh
    ADD     DI,TANDY_RASTER_LINE

END_TSLOOP:
    DEC     BP
    JNZ     TANDY_SPRITE_LOOP

    CLC
    JMP     SHORT END_TANDY_SPRITE

TS_OFF_SCREEN:
    STC

END_TANDY_SPRITE:
    POP     ES
    POP     BP
    POP     DI
    POP     SI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    TANDY_SPRITE

;****************************************************************************

;****************************************************************************
; Draws a horizontal line of specified width and starting point in Tandy mode

; Calling parameters:
; AL    = palette value for line
; BX    = vertical coordinate for placement (Y)
; CX    = horizontal coordinate for placement (X)
; DX    = length

PROC    TANDY_HLINE

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    SI
    PUSH    DI
    PUSH    ES

    PUSH    [VB_SEGMENT]
    POP     ES

    MOV     DI,DX
    ADD     DI,CX
    CMP     DI,TANDY_MAXX
    JA      THL_OFF_SCREEN

    CMP     BX,TANDY_MAXY
    JA      THL_OFF_SCREEN

    MOV     SI,AX
    MOV     AX,BX
    AND     AX,0011b
    ROR     AX,1
    ROR     AX,1
    ROR     AX,1
    MOV     DI,AX
    SHR     BX,1
    SHR     BX,1
    MOV     AX,TANDY_RASTER_LINE

    PUSH    DX
    MUL     BX
    POP     DX

    ADD     DI,AX
    MOV     BH,CL
    AND     BH,1
    SHR     CX,1
    ADD     DI,CX
    MOV     AX,SI
    AND     AL,0Fh
    MOV     CL,4
    MOV     BL,AL
    SHL     BL,CL
    OR      BL,AL

THL_LOOP:
    OR      BH,BH
    JZ      EVEN_PIXEL

    AND     [BYTE PTR ES:DI],0F0h
    OR      [BYTE PTR ES:DI],AL

    XOR     BH,BH
    INC     DI
    DEC     DX
    JMP     SHORT END_THLLOOP

EVEN_PIXEL:
    CMP     DX,1
    JE      FINAL_PIXEL

    MOV     [BYTE PTR ES:DI],BL

    INC     DI
    DEC     DX
    DEC     DX

END_THLLOOP:
    JNZ     THL_LOOP

    JMP     SHORT HLINE_DONE

FINAL_PIXEL:
    SHL     AL,CL
    AND     [BYTE PTR ES:DI],0Fh
    OR      [BYTE PTR ES:DI],AL

HLINE_DONE:
    CLC
    JMP     SHORT END_TANDY_HLINE

THL_OFF_SCREEN:
    STC

END_TANDY_HLINE:
    POP     ES
    POP     DI
    POP     SI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    TANDY_HLINE

;****************************************************************************

;****************************************************************************
; Draws a vertical line of specified height and starting point in Tandy mode

; Calling parameters:
; AL    = palette value for line
; BX    = vertical coordinate for placement (Y)
; CX    = horizontal coordinate for placement (X)
; DX    = length

PROC    TANDY_VLINE

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    SI
    PUSH    DI
    PUSH    ES

    PUSH    [VB_SEGMENT]
    POP     ES

    MOV     DI,DX
    ADD     DI,BX
    CMP     DI,TANDY_MAXY
    JA      TVL_OFF_SCREEN

    CMP     CX,TANDY_MAXX
    JA      TVL_OFF_SCREEN

    MOV     SI,AX
    MOV     AX,BX
    AND     AX,0011b
    ROR     AX,1
    ROR     AX,1
    ROR     AX,1
    MOV     DI,AX
    SHR     BX,1
    SHR     BX,1
    MOV     AX,TANDY_RASTER_LINE

    PUSH    DX
    MUL     BX
    POP     DX

    ADD     DI,AX
    MOV     BH,CL
    SHR     CX,1
    ADD     DI,CX
    MOV     AX,SI
    AND     AL,0Fh
    MOV     BL,0F0h

    AND     BH,1
    JNZ     TVL_LOOP

    MOV     CL,4
    SHL     AL,CL
    MOV     BL,0Fh

TVL_LOOP:
    AND     [BYTE PTR ES:DI],BL
    OR      [BYTE PTR ES:DI],AL

    ADD     DI,2000h
    TEST    DI,8000h
    JZ      END_TVLLOOP

    AND     DI,7FFFh
    ADD     DI,TANDY_RASTER_LINE

END_TVLLOOP:
    DEC     DX
    JNZ     TVL_LOOP

    CLC
    JMP     SHORT END_TANDY_VLINE

TVL_OFF_SCREEN:
    STC

END_TANDY_VLINE:
    POP     ES
    POP     DI
    POP     SI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    TANDY_VLINE

;****************************************************************************

;****************************************************************************
; Draws the background pattern used in the game (a checkerboard pattern)
; using the palette value in AL.

PROC    TANDY_SET_BACKGROUND

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    DI
    PUSH    ES

    PUSH    [VB_SEGMENT]
    POP     ES

    CLD
    MOV     CL,4
    MOV     AH,AL
    AND     AX,0F0Fh
    MOV     BX,AX
    SHL     AX,CL
    MOV     DX,4
    XOR     DI,DI

TSB_LOOP:
    PUSH    DI

    MOV     CX,((TANDY_RASTER_LINE*50)/2)
    REP     STOSW

    POP     DI
    ADD     DI,2000h
    XCHG    AX,BX
    DEC     DX
    JNZ     TSB_LOOP

    POP     ES
    POP     DI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    TANDY_SET_BACKGROUND

;****************************************************************************

;****************************************************************************
; Places the mouse cursor on the screen in Tandy mode.  Uses MOUSE_X and
; MOUSE_Y to determine where the mouse should go.

PROC    TANDY_CURSOR_PLACE

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    SI
    PUSH    DI
    PUSH    DS
    PUSH    ES

    MOV     AX,CS
    MOV     DS,AX

    MOV     AX,[VIDEO_BUFFER]
    MOV     ES,AX

    MOV     AX,OFFSET CURSOR_SAVE_BUFFER
    MOV     [SAVE_BUFFER_PTR],AX

    MOV     BX,[MOUSE_Y]
    MOV     AX,BX

    MOV     CL,3
    AND     BX,0011b
    ROR     BX,CL

    DEC     CL
    SHR     AX,CL

    MOV     CL,7
    SHL     AX,CL
    MOV     DI,AX

    SHR     AX,1
    SHR     AX,1
    ADD     DI,AX

    ADD     DI,BX
    MOV     BX,[MOUSE_X]

    MOV     DL,BL
    AND     DL,1
    SHR     BX,1
    ADD     DI,BX
    MOV     SI,OFFSET MOUSE_CURSOR
    MOV     BX,MC_HEIGHT

TCP_LOOP_OUTER:
    PUSH    BX
    PUSH    DI

    MOV     BX,[SAVE_BUFFER_PTR]
    MOV     [WORD PTR BX],DI
    ADD     BX,2
    MOV     CX,MC_WIDTH
    OR      DL,DL
    JZ      STARTED_EVEN

    INC     CX

STARTED_EVEN:
    MOV     [WORD PTR BX],CX
    ADD     BX,2

SAVE_AREA_LOOP:
    MOV     AL,[BYTE PTR ES:DI]
    MOV     [BYTE PTR BX],AL
    INC     DI
    INC     BX
    LOOP    SAVE_AREA_LOOP

    MOV     [SAVE_BUFFER_PTR],BX
    POP     DI
    POP     BX

    MOV     CX,MC_WIDTH
    PUSH    DI

TCP_LOOP_INNER:
    LODSB

    XOR     DL,1
    JNZ     ON_WORD_BOUNDARY

    CMP     AL,MOUSE_TRANSLUCENT
    JE      NO_PLACEMENT

    AND     [BYTE PTR ES:DI],0F0h
    OR      [BYTE PTR ES:DI],AL

NO_PLACEMENT:
    INC     DI
    JMP     SHORT END_TCPINNER

ON_WORD_BOUNDARY:
    CMP     AL,MOUSE_TRANSLUCENT
    JE      END_TCPINNER

    PUSH    CX
    MOV     CL,4
    SHL     AL,CL
    POP     CX
    AND     [BYTE PTR ES:DI],0Fh
    OR      [BYTE PTR ES:DI],AL

END_TCPINNER:
    LOOP    TCP_LOOP_INNER

    POP     DI
    ADD     DI,2000h
    TEST    DI,8000h
    JZ      END_TCPOUTER

    AND     DI,7FFFh
    ADD     DI,TANDY_RASTER_LINE

END_TCPOUTER:
    DEC     BX
    JNZ     TCP_LOOP_OUTER

    MOV     [CURSOR_STATUS],ON

    POP     ES
    POP     DS
    POP     DI
    POP     SI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    TANDY_CURSOR_PLACE

;****************************************************************************

;****************************************************************************
; MCGA 256-COLOR DISPALY DRIVER

;****************************************************************************
; Sets the current display mode to MCGA 256-color mode

PROC    MCGA_SET_MODE

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX

    MOV     AX,0013h
    INT     10h

    CLC
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    MCGA_SET_MODE

;****************************************************************************

;****************************************************************************
; Places a bitmap on the screen in MCGA mode

; Calling parameters:
; DS:SI = sprite (bitmap_height, bitmap_width, bitmap)
; BX    = vertical coordinate for placement (Y)
; CX    = horizontal coordinate for placement (X)

PROC    MCGA_SPRITE

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    SI
    PUSH    DI
    PUSH    BP
    PUSH    ES

    MOV     AX,[VB_SEGMENT]
    MOV     ES,AX

    CLD
    LODSW
    MOV     DI,AX
    ADD     AX,CX
    CMP     AX,MCGA_MAXX
    JA      OFF_SCREEN

    LODSW
    PUSH    AX
    ADD     AX,BX
    CMP     AX,MCGA_MAXY
    POP     AX
    JA      OFF_SCREEN

    MOV     BP,AX
    MOV     AX,SCREEN_WIDTH
    XOR     DX,DX
    MUL     BX
    ADD     AX,CX
    XCHG    AX,DI
    MOV     CX,BP

MCGA_PLACE_LOOP:
    PUSH    DI
    PUSH    CX

    MOV     CX,AX
    REP     MOVSB

    POP     CX
    POP     DI
    ADD     DI,SCREEN_WIDTH
    LOOP    MCGA_PLACE_LOOP

    CLC
    JMP     SHORT END_MCGA_SPRITE

OFF_SCREEN:
    STC

END_MCGA_SPRITE:
    POP     ES
    POP     BP
    POP     DI
    POP     SI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    MCGA_SPRITE

;****************************************************************************

;****************************************************************************
; Draws a vertical line in MCGA mode

; Calling parameters:
; AL    = palette value for line
; BX    = vertical coordinate for placement (Y)
; CX    = horizontal coordinate for placement (X)
; DX    = length

PROC    MCGA_VLINE

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    DI
    PUSH    ES

    PUSH    [VB_SEGMENT]
    POP     ES

    MOV     DI,DX
    ADD     DI,BX
    CMP     DI,MCGA_MAXY
    JA      VLINE_OFF_SCREEN

    CMP     CX,MCGA_MAXX
    JA      VLINE_OFF_SCREEN

    MOV     DI,AX
    PUSH    DX
    XOR     DX,DX
    MOV     AX,SCREEN_WIDTH
    MUL     BX
    ADD     AX,CX
    POP     CX
    XCHG    AX,DI

VLINE_LOOP:
    STOSB
    DEC     DI
    ADD     DI,SCREEN_WIDTH
    LOOP    VLINE_LOOP

    CLC
    JMP     SHORT END_MCGA_VLINE

VLINE_OFF_SCREEN:
    STC

END_MCGA_VLINE:
    POP     ES
    POP     DI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    MCGA_VLINE

;****************************************************************************

;****************************************************************************
; Draws a horizontal line in MCGA mode

; Calling parameters:
; AL    = palette value for line
; BX    = vertical coordinate for placement (Y)
; CX    = horizontal coordinate for placement (X)
; DX    = length

PROC    MCGA_HLINE

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    DI
    PUSH    ES

    PUSH    [VB_SEGMENT]
    POP     ES

    MOV     DI,DX
    ADD     DI,CX
    CMP     DI,MCGA_MAXX
    JA      HLINE_OFF_SCREEN

    CMP     BX,MCGA_MAXY
    JA      HLINE_OFF_SCREEN

    MOV     DI,AX
    PUSH    DX
    XOR     DX,DX
    MOV     AX,SCREEN_WIDTH
    MUL     BX
    ADD     AX,CX
    POP     CX
    XCHG    AX,DI

    REP     STOSB

    CLC
    JMP     SHORT END_MCGA_HLINE

HLINE_OFF_SCREEN:
    STC

END_MCGA_HLINE:
    POP     ES
    POP     DI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    MCGA_HLINE

;****************************************************************************

;****************************************************************************
; Draws the background pattern used in the game (a checkerboard pattern)
; using the palette value in AL.  (MCGA mode)

PROC    MCGA_SET_BACKGROUND

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DI
    PUSH    ES

    MOV     BX,[VB_SEGMENT]
    MOV     ES,BX

    XOR     DI,DI
    XOR     AH,AH
    XCHG    AH,AL

    MOV     CX,SCREEN_WIDTH
    SHR     CX,1
    MOV     BX,SCREEN_HEIGHT

SB_LOOP:
    XCHG    AH,AL

    PUSH    CX
    REP     STOSW
    POP     CX

    DEC     BX
    JNZ     SB_LOOP

    POP     ES
    POP     DI
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    MCGA_SET_BACKGROUND

;****************************************************************************

;****************************************************************************
; Places the mouse cursor on the screen in MCGA mode.  Uses MOUSE_X and
; MOUSE_Y to determine where the mouse should go.

PROC    MCGA_CURSOR_PLACE

    PUSH    AX
    PUSH    BX
    PUSH    CX
    PUSH    DX
    PUSH    SI
    PUSH    DI
    PUSH    DS
    PUSH    ES

    MOV     AX,CS
    MOV     DS,AX

    MOV     AX,[VIDEO_BUFFER]
    MOV     ES,AX

    MOV     AX,OFFSET CURSOR_SAVE_BUFFER
    MOV     [SAVE_BUFFER_PTR],AX

    MOV     BX,[MOUSE_Y]

    MOV     CL,8
    SHL     BX,CL
    MOV     DI,BX

    SHR     BX,1
    SHR     BX,1
    ADD     DI,BX

    MOV     BX,[MOUSE_X]
    ADD     DI,BX
    MOV     SI,OFFSET MOUSE_CURSOR
    MOV     DX,MC_HEIGHT

MCP_LOOP_OUTER:
    PUSH    SI
    MOV     BX,DI

    MOV     SI,[SAVE_BUFFER_PTR]
    MOV     [WORD PTR SI],BX
    ADD     SI,2
    MOV     CX,MC_WIDTH
    MOV     [WORD PTR SI],CX
    ADD     SI,2

MCGA_SAVE_LOOP:
    MOV     AL,[BYTE PTR ES:BX]
    MOV     [BYTE PTR SI],AL
    INC     BX
    INC     SI
    LOOP    MCGA_SAVE_LOOP

    MOV     [SAVE_BUFFER_PTR],SI
    POP     SI

    MOV     CX,MC_WIDTH
    PUSH    DI

MCP_LOOP_INNER:
    LODSB

    CMP     AL,MOUSE_TRANSLUCENT
    JE      NOT_SHOWN

    MOV     [BYTE PTR ES:DI],AL

NOT_SHOWN:
    INC     DI
    LOOP    MCP_LOOP_INNER

    POP     DI
    ADD     DI,SCREEN_WIDTH

    DEC     DX
    JNZ     MCP_LOOP_OUTER

    MOV     [CURSOR_STATUS],ON

    POP     ES
    POP     DS
    POP     DI
    POP     SI
    POP     DX
    POP     CX
    POP     BX
    POP     AX
    RET

ENDP    MCGA_CURSOR_PLACE

;****************************************************************************

LABEL   RND_NUMBERS     BYTE

ENDS    MAIN

END     BEGIN




