$IF 0  '--------------------------------------------------------------

  SORT.BAS   --  demonstrates sorting data with MAXLIB For PB

  Written by Brian McLaughlin. Released to the public domain 10/1/95.

  This demo code shows how to write a shell sort for MAXLIB.  The
  shell sort is a fairly speedy, fairly simple sort routine that
  compares favorably with the well-known Quicksort.

  This demo also shows the relative speeds of sorting the exact same
  data, using the exact same algorithm, when using each of these
  resources:

    - an array in conventional memory
    - a file opened FOR BINARY in PowerBASIC
    - a file accessed in EMS, using MAXFiles
    - an array stored in EMS, using EMSArray
    - an array stored in XMS, using XMSArray


  I'm sure that some people will be disappointed that EMS and XMS
  don't even come close to matching the speed of conventional
  memory.  However, if you stop to think, in almost every case the
  real alternative to using EMS and XMS is using disk files, not
  conventional memory.  I'm equally sure that many people will be
  surprised at the speed advantage EMS has over XMS.

  The data consists of 1000 random long integers, so it represents a
  "worst case" for unsorted data.  Data that begins in a closer
  approximation to correct order will sort more rapidly.  Each sort
  routine is hard-coded for long integer data.  It is up to you to
  rewrite the routines if you need to sort other types of data.

$ENDIF '-----------------------------------------------------------

 $LIB ALL OFF
 $LINK ".\MAXLIB.PBL"       '<-- assumes current directory
 $INCLUDE ".\MAXLIB.BI"     '<-- assumes current directory

 DECLARE SUB ShellSort (LArray() AS LONG)
 DECLARE SUB EMSShellSort (EMSHandle%)
 DECLARE SUB XMSShellSort (XMSHandle%)
 DECLARE SUB MAXFileShellSort (FileName$)
 DECLARE SUB BINARYFileShellSort (FileName$)

'\\\\\\\\\\\\\\\\\\\\\\\\\\ START OF MAIN //////////////////////////////

 RANDOMIZE TIMER
 InitMAXFiles      ' <-- we'll use MAXFiles
 InitXMSArray      ' <-- we'll use XMSArray
 InitEMSArray      ' <-- we'll use EMSArray
 SetDiskFile 0     ' when this is non-zero MAXFiles uses disk storage

 FirstLong% = 1    'array lower bounds
 LastLong% = 1000  'array upper bounds

 FileName$ = "LONGINT.$$$"
  IF LEN(DIR$(FileName$)) THEN KillX FileName$
 FileName2$ =  "LONGINT2.$$$"
  IF LEN(DIR$(FileName2$)) THEN KillX FileName2$

 COLOR 7, 1
 CLS
 PRINT "Creating files and arrays that we'll be sorting..."
 DIM LongArray (FirstLong% TO LastLong%) AS LONG
 LongXMS% = DimXMS%(FirstLong%, LastLong%, 4)
   IF LongXMS% = 0 THEN PRINT "Not enough XMS memory found."
 LongEMS% = DimEMS%(FirstLong%, LastLong%, 4)
   IF LongEMS% = 0 THEN PRINT "Not enough EMS memory found."
 Handle% = OpenX%(FileName$)
 Handle2% = OpenX%(FileName2$)

 PRINT "Filling files and arrays with random data..."
 FOR X% = FirstLong% TO LastLong%
   RandLong& = CLNG(RND * X% * 1000)
   LongArray(X%) = RandLong&
   IF LongXMS% THEN InXMS LongXMS%, X%, RandLong&
   IF LongEMS% THEN InEMS LongEMS%, X%, RandLong&
   PutX Handle%, 4, RandLong&
   PutX Handle2%, 4, RandLong&
 NEXT X%
 PRINT
 CloseX Handle%
 CloseX Handle2%

'this section calls each sort and records its execution time

 PRINT "Sorting conventional array..."
 FalseStart! = TIMER
 DO
   ArrayStart! = TIMER
 LOOP UNTIL ArrayStart! > FalseStart!
 ShellSort LongArray&()
 ArrayDone! = TIMER

 IF LongEMS% THEN
   PRINT "Sorting EMS array..."
   FalseStart! = TIMER
   DO
     EMSStart! = TIMER
   LOOP UNTIL EMSStart! > FalseStart!
   EMSShellSort LongEMS%
   EMSDone! = TIMER
 END IF

 IF LongXMS% THEN
   PRINT "Sorting XMS array..."
   FalseStart! = TIMER
   DO
     XMSStart! = TIMER
   LOOP UNTIL XMSStart! > FalseStart!
   XMSShellSort LongXMS%
   XMSDone! = TIMER
 END IF

 PRINT "Sorting MAXFiles file..."
 FalseStart! = TIMER
 DO
   MAXStart! = TIMER
 LOOP UNTIL MAXStart! > FalseStart!
 MAXFileShellSort FileName$
 MAXDone! = TIMER

 PRINT "Sorting PB BINARY file..."
 FalseStart! = TIMER
 DO
   PBStart! = TIMER
 LOOP UNTIL PBStart! > FalseStart!
 BINARYFileShellSort FileName2$
 PBDone! = TIMER

 PRINT
 PRINT "Verifying files and arrays were sorted correctly..."
 Handle% = OpenX%(FileName$)
 Handle2% = OpenX%(FileName2$)

 FOR X% = FirstLong% TO LastLong%
   CheckNum% = 5
   GetX Handle%, 4, LongVar&
   GetX Handle2%, 4, LongVar2&
   LongVar3& = LongArray(X%)
   IF LongXMS% THEN
      OutXMS LongVar4&, LongXMS%, X%
   ELSE
      DECR CheckNum%
   END IF
   IF LongEMS% THEN
     OutEMS LongVar5&, LongEMS%, X%
   ELSE
     DECR CheckNum%
   END IF
   IF X% > FirstLong% THEN
     ElTotal& = (LongVar& + LongVar2& + LongVar3& + LongVar4& + LongVar5&)
     ElTotal& = ElTotal& / CheckNum%
     IF ElTotal& <> LongVar& THEN
       PRINT "Sort failed at element:"; X%
       END
     ELSEIF LastVal& > LongVar& THEN
       PRINT "Sort failed at element:"; X%
       END
     END IF
     LastVal& = LongVar&
   END IF
 NEXT X%
 PRINT "Sort succeeded..."
 CloseX Handle%
 CloseX Handle2%
 IF LEN(DIR$(FileName$)) THEN KillX FileName$    'kill data files
 IF LEN(DIR$(FileName2$)) THEN KillX FileName2$

'this section reports the elapsed times for each type of sort
 PRINT
 PRINT "Conventional array sort took: "; (ArrayDone! - ArrayStart!)
 IF LongEMS% THEN
   PRINT "EMS array sort:               "; (EMSDone! - EMSStart!)
 END IF
 IF LongXMS% THEN
   PRINT "XMS array sort:               "; (XMSDone! - XMSStart!)
 END IF
 PRINT "MAXFiles sort took:           "; (MAXDone! - MAXStart!)
 PRINT "PB BINARY file sort took:     "; (PBDone! - PBStart!)
 END

'\\\\\\\\\\\\\\\\\\\\\\\\\\ END OF MAIN ////////////////////////////////

'=============================================================================
 SUB MAXFileShellSort (FileName$)
'=============================================================================

 Handle% = OpenX%(FileName$)

 FirstRec% = 0
 LastRec% = CINT(SizeX&(Handle%) \ 4) - 1
 Total% = LastRec% - FirstRec% + 1

 IF Total% > 1 THEN      'account for one record or zero-length file
   Place1% = Total%

   DO WHILE Place1%
     Place1% = Place1% \ 2
     Place2% = Total% - Place1%

     FOR Count% = 1 TO Place2%
       Place3% = Count%
       DO
         Place4% = Place3% + Place1%

         FilePtr3& = CLNG(Place3% * 4) - 4
         SetLocX Handle%, FilePtr3&
         GetX Handle%, 4&, Long3&

         FilePtr4& = CLNG(Place4% * 4) - 4
         SetLocX Handle%, FilePtr4&
         GetX Handle%, 4&, Long4&

         IF Long3& <= Long4& THEN
           EXIT DO
         ELSE
           SetLocX Handle%, FilePtr3&
           PutX Handle%, 4&, Long4&

           SetLocX Handle%, FilePtr4&
           PutX Handle%, 4&, Long3&

           Place3% = Place3% - Place1%
         END IF
       LOOP WHILE (Place3% > 0)
     NEXT Count%
   LOOP
 END IF

 CloseX Handle%

END SUB


'=============================================================================
 SUB BINARYFileShellSort (FileName$)
'=============================================================================

 Handle% = 1
 OPEN FileName$ FOR BINARY AS Handle%

 FirstRec% = 0
 LastRec% = CINT(LOF(Handle%) \ 4) - 1
 Total% = LastRec% - FirstRec% + 1

 IF Total% > 1 THEN      'account for one record or zero-length file
   Place1% = Total%

   DO WHILE Place1%
     Place1% = Place1% \ 2
     Place2% = Total% - Place1%

     FOR Count% = 1 TO Place2%
       Place3% = Count%
       DO
         Place4% = Place3% + Place1%

         FilePtr3& = CLNG(Place3% * 4) - 4
         SEEK Handle%, FilePtr3&
         GET Handle%,, Long3&

         FilePtr4& = CLNG(Place4% * 4) - 4
         SEEK Handle%, FilePtr4&
         GET Handle%,, Long4&

         IF Long3& <= Long4& THEN
           EXIT DO
         ELSE
           SEEK Handle%, FilePtr3&
           PUT Handle%,, Long4&

           SEEK Handle%, FilePtr4&
           PUT Handle%,, Long3&

           Place3% = Place3% - Place1%
         END IF
       LOOP WHILE (Place3% > 0)
     NEXT Count%
   LOOP
 END IF

 CLOSE Handle%

END SUB


'=============================================================================
 SUB ShellSort (LongArray() AS LONG)
'=============================================================================

 Total% = UBOUND(LongArray) - LBOUND(LongArray) + 1

 IF Total% > 1 THEN      'account for one or no-element arrays
   Place1% = Total%

   DO WHILE Place1%
     Place1% = Place1% \ 2
     Place2% = Total% - Place1%

     FOR Count% = 1 TO Place2%
       Place3% = Count%
       DO
         Place4% = Place3% + Place1%
         IF LongArray(Place3%) <= LongArray(Place4%) THEN
           EXIT DO
         ELSE
           SWAP LongArray(Place3%), LongArray(Place4%)
           Place3% = Place3% - Place1%
         END IF
       LOOP WHILE (Place3% > 0)
     NEXT Count%
   LOOP
 END IF

END SUB

'=============================================================================
 SUB XMSShellSort (LongArray%)
'=============================================================================

 Total% = UBoundXMS%(LongArray%) - LBoundXMS%(LongArray%) + 1

 IF Total% > 1 THEN      'account for one or no-element arrays
   Place1% = Total%

   DO WHILE Place1%
     Place1% = Place1% \ 2
     Place2% = Total% - Place1%

     FOR Count% = 1 TO Place2%
       Place3% = Count%
       DO
         Place4% = Place3% + Place1%
         OutXMS Long3&, LongArray%, Place3%
         OutXMS Long4&, LongArray%, Place4%

         IF Long3& <= Long4& THEN
           EXIT DO
         ELSE
           InXMS LongArray%, Place3%, Long4&
           InXMS LongArray%, Place4%, Long3&
           Place3% = Place3% - Place1%
         END IF
       LOOP WHILE (Place3% > 0)
     NEXT Count%
   LOOP
 END IF

END SUB

'=============================================================================
 SUB EMSShellSort (LongArray%)
'=============================================================================

 Total% = UBoundEMS%(LongArray%) - LBoundEMS%(LongArray%) + 1

 IF Total% > 1 THEN      'account for one or no-element arrays
   Place1% = Total%

   DO WHILE Place1%
     Place1% = Place1% \ 2
     Place2% = Total% - Place1%

     FOR Count% = 1 TO Place2%
       Place3% = Count%
       DO
         Place4% = Place3% + Place1%
         OutEMS Long3&, LongArray%, Place3%
         OutEMS Long4&, LongArray%, Place4%

         IF Long3& <= Long4& THEN
           EXIT DO
         ELSE
           InEMS LongArray%, Place3%, Long4&
           InEMS LongArray%, Place4%, Long3&
           Place3% = Place3% - Place1%
         END IF
       LOOP WHILE (Place3% > 0)
     NEXT Count%
   LOOP
 END IF

END SUB

