{===============================================================}
{= EmpObj Unit - Employee Business Object Implementation       =}
{===============================================================}

unit EmpObj;

interface

  uses
    Classes, DB, DBTables, BizObj;

  type
    {== TEmployeeObject Class Declaration ==}

    TEmployeeObject = class( TBusinessObject )
    private
      FEmpNo     : TIntegerField;
      FLastName  : TStringField;
      FFirstName : TStringField;
      FHireDate  : TDateTimeField;
      FSalary    : TFloatField;
      FMonthlySal: TFloatField;
      FTable     : TTable;
    protected
      function GetTable : TTable;
      procedure SetTable( ATable : TTable );
      procedure CreateFields; override;
    public
            { Specify properties to enforce read-only attribute }
      property EmpNo     : TIntegerField  read FEmpNo;
      property LastName  : TStringField   read FLastName;
      property FirstName : TStringField   read FFirstName;
      property HireDate  : TDateTimeField read FHireDate;
      property Salary    : TFloatField    read FSalary;
      property MonthlySal: TFloatField    read FMonthlySal;
                          { Methods implementing Business Rules }
      function FullName : string;
      function LengthOfEmploy : Real;
    published
      property Table : TTable read GetTable write SetTable;
    end;


  procedure Register;


implementation

  uses
    SysUtils;

  {=============================}
  {== TEmployeeObject Methods ==}
  {=============================}

  {= GetTable                                                  =}
  {=   For each business table (or query) that you encapsulate =}
  {=   in the business object, the GetTable "read" access      =}
  {=   method must call CheckDataSet before returning the      =}
  {=   internal table (or query) object.                       =}
  {=                                                           =}
  {= NOTE:  If you need to create a new table, then be sure to =}
  {=        call SetupFields().  This ensures the field        =}
  {=        objects get created.                               =}

  function TEmployeeObject.GetTable : TTable;
  begin
    if not Assigned( FTable ) and
       not ( csDesigning in ComponentState ) then
    begin
      FTable := TTable.Create( Self );
      FTable.DatabaseName := 'DBDEMOS';
      FTable.TableName := 'EMPLOYEE.DB';
      FTable.Name := 'TblEmployee';
      SetupFields( FTable );
    end;
      { Check to see if FTable actually points to a real table =}
    CheckDataSet( TDataSet( FTable ) );
    Result := FTable;
  end; {= TEmployeeObject.GetTable =}


  {= SetTable                                                  =}
  {=   For each business table (or query) that you encapsulate =}
  {=   in the business object, the SetTable "read" access      =}
  {=   method must call SetupFields after setting the internal =}
  {=   table (or query) object to the new object.  This        =}
  {=   ensures that the Field Objects get created.             =}

  procedure TEmployeeObject.SetTable( ATable : TTable );
  begin
    if ( FTable <> ATable ) then
    begin
      FTable := ATable;
      SetupFields( FTable );
    end;
  end; {= TEmployeeObject.SetTable =}


  {= CreateFields                                              =}
  {=   Override this method to create the fields necessary to  =}
  {=   support the business object.                            =}

  procedure TEmployeeObject.CreateFields;
  begin
    FEmpNo     := CreateField( 'EmpNo', FTable ) as TIntegerField;

    FLastName  := CreateField( 'LastName', FTable ) as TStringField;
    FLastName.DisplayLabel := 'Last Name';

    FFirstName := CreateField( 'FirstName', FTable) as TStringField;
    FFirstName.DisplayLabel := 'First Name';

    FHireDate := CreateField( 'HireDate',FTable ) as TDateTimeField;
    FHireDate.DisplayLabel := 'Hire Date';
    FHireDate.EditMask := '!99/99/00;1;_';

    FSalary := CreateField( 'Salary', FTable ) as TFloatField;
    FSalary.Currency := True;

    FMonthlySal := CreateCalcField( 'MonthlySal', ftFloat,
                                    0, FTable ) as TFloatField;
    FMonthlySal.Currency := True;

  end; {= TEmployeeObject.CreateFields =}


  function TEmployeeObject.FullName : string;
  begin
    Result := FirstName.Value + ' ' + LastName.Value;
  end; {= TEmployeeObject.FullName =}


  function TEmployeeObject.LengthOfEmploy : Real;
  begin
    Result := Trunc( Now - HireDate.Value ) / 365;
  end; {= TEmployeeObject.LengthOfEmploy =}


  {========================}
  {== Register Procedure ==}
  {========================}

  procedure Register;
  begin
    RegisterComponents( 'Business Objects', [ TEmployeeObject ] );
  end;


end.
