{===============================================================}
{= BizObj Unit - Base Business Object Implementation           =}
{===============================================================}

unit BizObj;

interface

  uses
    Classes, DB, DBTables;

  type
    {== TBusinessObject Class Declaration ==}

    TBusinessObject = class( TComponent )
    protected
      procedure SetupFields( DataSet : TDataSet ); virtual;
      procedure CreateFields; virtual; abstract;
      function CreateField( FieldName : string;
                            DataSet : TDataSet ) : TField;
      function CreateCalcField( FieldName : string;
                                FldType : TFieldType;
                                Size : Word;
                                DataSet : TDataSet ) : TField;
      procedure CheckDataSet( var DataSet : TDataSet ); virtual;
    end;


implementation

  uses
    DBConsts;

  {=============================}
  {== TBusinessObject Methods ==}
  {=============================}

  {= SetupFields                                               =}
  {=   This method is responsible for making sure the          =}
  {=   appropriate functions are called when the field objects =}
  {=   are created. More precisely, there is pre- and post-    =}
  {=   operations that must be performed.                      =}
  {=                                                           =}
  {= Pre-   DataSet must be closed                             =}
  {=        DataSet's FieldDefs must be updated                =}
  {= Post-  Re-establish original dataset state                =}
  {=        Send DataEvent message to DataSet.Designer         =}

  procedure TBusinessObject.SetupFields( DataSet : TDataSet );
  var
    Fld         : TField;
    ActiveState : Boolean;
  begin
    if not Assigned( DataSet ) then
      Exit;
                  { DataSet must be closed to create new fields }
    ActiveState := DataSet.Active;         { Save current state }
    if ActiveState then
      DataSet.Active := False;
            { Must call Update to have access to all Field Defs }
    DataSet.FieldDefs.Update;

    CreateFields;    { Call to Descendant's CreateFields method }

    DataSet.Active := ActiveState;

    { If the developer has not yet invoked the Field Editor,    }
    { then the DataSet Designer does not exist. If the designer }
    { doesn't exist, it is not necessary to call DataEvent.     }
    { Besides if you do, it will cause a GPF                    }

    if DataSet.Designer <> nil then
      DataSet.Designer.DataEvent( deFieldListChange, 0 );
  end; {= TBusinessObject.SetupFields =}


  {= CreateField                                               =}
  {=   This method simplifies the process of creating fields.  =}
  {=   First, the DataSet is searched for a matching FieldName.=}
  {=   If one exists, then that one is used.  If not, then a   =}
  {=   new field is created from the FieldDef object           =}
  {=   corresponding to the desired field name.                =}

  function TBusinessObject.CreateField( FieldName: string;
                                        DataSet: TDataSet ): TField;
  begin
    with DataSet do
    begin
                  { First, try to Find an Existing Field Object }
      Result := FindField( FieldName );
      if Result = nil then
      begin
         { Have the FieldDef object create its own Field Object }
        if Owner <> nil then
          Result := FieldDefs.Find(FieldName).CreateField(Owner)
        else
          Result := FieldDefs.Find(FieldName).CreateField(DataSet);

        { Give the new Field Object a generic Name so that it   }
        { appears in the Object Inspector                       }
        Result.Name := Name + FieldName;
      end;
    end; { with }
  end; {= TBusinessObject.CreateField =}


  function TBusinessObject.CreateCalcField( FieldName : string;
    FldType : TFieldType; Size : Word;
    DataSet : TDataSet ) : TField;
  const
    FieldClasses: array[ TFieldType ] of TFieldClass = (
      nil,                { ftUnknown }
      TStringField,       { ftString }
      TSmallintField,     { ftSmallint }
      TIntegerField,      { ftInteger }
      TWordField,         { ftWord }
      TBooleanField,      { ftBoolean }
      TFloatField,        { ftFloat }
      TCurrencyField,     { ftCurrency }
      TBCDField,          { ftBCD }
      TDateField,         { ftDate }
      TTimeField,         { ftTime }
      TDateTimeField,     { ftDateTime }
      TBytesField,        { ftBytes }
      TVarBytesField,     { ftVarBytes }
      TBlobField,         { ftBlob }
      TMemoField,         { ftMemo }
      TGraphicField );    { ftGraphic }
  var
    FieldClass : TFieldClass;
  begin
    Result := DataSet.FindField( FieldName );
    if Result = nil then
    begin
      FieldClass := FieldClasses[ FldType ];
      if FieldClass = nil then
        DBErrorFmt( SUnknownFieldType, [ FieldName ] );
      Result := FieldClass.Create( Owner );
      try
        Result.FieldName := FieldName;
        Result.Size := Size;
        Result.Calculated := True;
        Result.DataSet := DataSet;
        Result.Name := DataSet.Name + FieldName;
      except
        Result.Free;
        raise;
      end;
    end;
  end; {= TBusinessObject.CreateCalcField =}


  {= CheckDataSet                                              =}
  {=   This method first checks to make sure we are in Design  =}
  {=   Mode. If so, then the business object tests to make     =}
  {=   sure the embedded dataset actually exists on the form.  =}

  procedure TBusinessObject.CheckDataSet( var DataSet : TDataSet );
  begin
    if csDesigning in ComponentState then
    begin
      if Assigned( DataSet ) and
         ( Owner.FindComponent( DataSet.Name ) = nil ) then
        DataSet := nil;
    end;
  end; {= TBusinessObject.CheckDataSet =}

end.
