//---------------------------------------------------------------------------
//
//	File:			LOGPAL.CPP
//	Programmer: 	Bob Provencher
//	Create Date:	11/9/94
//	Last Mod Date:	3/5/95
//	Description:	LogicalPalette class implementation
//
//	Copyright (c) 1994, 1995. Aesir Software, Inc.
//	All Rights Reserved.
//
//---------------------------------------------------------------------------

#include "precomp.h"
#pragma hdrstop

#include "logpal.h"

#ifndef __GEN_H__
#include "gen.h"
#endif

#ifndef __DIBITMAP_H__
#include "dibitmap.h"
#endif

PALETTEENTRY LogPalette::pal = { 0, 0, 0, 0 };

LogPalette::LogPalette( const RGBQUAD& color, UINT num, UINT flags ) :
	lpLogPalette( 0 )
{
	if ( ResizeTo( num ) )
	{
		for ( int i = 0; i < num; i++ )
			Copy( Color( i ), color );
		SetEntryFlags( 0, num, flags );
	}
}

BOOL LogPalette::ResizeTo( WORD wNewSize )
{
	DWORD dwNewBytes = NumBytes( wNewSize );
	DWORD dwBytes = NumBytes();
	LOGPALETTE FAR* lpNewLogPalette = 0;
	if ( dwNewBytes )
	{
		lpNewLogPalette = (LOGPALETTE FAR*)new LPBYTE[ (size_t)dwNewBytes ];
		if ( !lpNewLogPalette )
			return FALSE;
		_fmemset( lpNewLogPalette, 0, (size_t)dwNewBytes );
	}
	if ( lpNewLogPalette )
	{
		if ( lpLogPalette )
		{
			_fmemcpy( lpNewLogPalette, lpLogPalette,
					  (size_t)min( dwBytes, dwNewBytes ) );
		}
		else
			lpNewLogPalette->palVersion = 0x0300;
		lpNewLogPalette->palNumEntries = wNewSize;
	}
	delete [] (LPBYTE)lpLogPalette;
	lpLogPalette = lpNewLogPalette;
	return TRUE;
}

BOOL LogPalette::GetSystemEntries( UINT uStart, UINT uEntries )
{
	if ( IsCreated() && uStart + uEntries <= Size() )
	{
		HDC hDC = GetDC( 0 );
		if ( hDC )
		{
			UINT u = GetSystemPaletteEntries( hDC, uStart, uEntries,
											  lpLogPalette->palPalEntry +
											  	uStart );
			ReleaseDC( 0, hDC );
			return u == uEntries;
		}
	}
	return FALSE;
}

BOOL LogPalette::CopyColors( const DIBitmap& dib )
{
	BOOL fReturn = FALSE;
	const BitmapInfo& bi( dib.Info() );
	UINT colors = bi.Colors();
	if ( ResizeTo( (WORD)colors ) )
	{
		for ( int i = 0; i < colors; i++ )
			Copy( (*this)[ i ], bi[ i ] );
		fReturn = TRUE;
	}
	return fReturn;
}

BOOL LogPalette::SetEntryFlags( UINT wStart, UINT wEntries, BYTE peFlags )
{
	if ( Palette() && wStart + wEntries <= Size() )
	{
		for ( int i = wStart; i < wStart + wEntries; i++ )
			(*this)[ i ].peFlags = peFlags;
		return TRUE;
	}
	else
		return FALSE;
}

BOOL LogPalette::EasyStepTo( const LogPalette& pal )
{
	BOOL fReturn = TRUE;
	if ( IsCreated() && pal.IsCreated() )
	{
		for ( int i = 0; i < Size(); i++ )
		{
			PALETTEENTRY& entry = Color( i );
			PALETTEENTRY  tgt   = pal.Color( i );
			fReturn = StepTo( entry.peRed,   tgt.peRed )   && fReturn;
			fReturn = StepTo( entry.peGreen, tgt.peGreen ) && fReturn;
			fReturn = StepTo( entry.peBlue,  tgt.peBlue )  && fReturn;
			entry.peFlags = tgt.peFlags;
		}
	}
    return fReturn;
}

void LogPalette::EvenStepTo( BYTE start, BYTE& tgt, float step, int stepno )
{
	tgt = (float)start + ( step * (float)stepno );
}

void LogPalette::EvenStepTo( const PALETTEENTRY& start, PALETTEENTRY& tgt,
							 const PaletteStep& step, int stepno )
{
	EvenStepTo( start.peRed,   tgt.peRed,   step.stRed,   stepno );
	EvenStepTo( start.peGreen, tgt.peGreen, step.stGreen, stepno );
	EvenStepTo( start.peBlue,  tgt.peBlue,  step.stBlue,  stepno );
}

void LogPalette::EvenStepTo( const LogPalette& start, PaletteStep* step, 
							 int stepno )
{
	for ( int i = 0; i < Size(); i++ )
		EvenStepTo( start.Color( i ), Color( i ), step[ i ], stepno );
}

void LogPalette::Dump( dbugstream& debug ) const
{
	debug << "Start LogPalette"				   << crlf
		  << " this: "         << (void*)this  << crlf
		  << " lpLogPalette: " << lpLogPalette << crlf;
	if ( IsCreated() )
		Dump( debug, *lpLogPalette );
	debug << "End LogPalette"				   << crlf;
}

void LogPalette::Dump( dbugstream& debug, const LOGPALETTE& pal ) const
{
	debug << " Start LOGPALETTE"                      << crlf
		  << "  palVersion:    " << pal.palVersion 	  << crlf
		  << "  palNumEntries: " << pal.palNumEntries << crlf;
	DWORD c = NumColors( pal );
	if ( c > 0 )
	{
		debug << "  Colors:" << crlf;
		for ( int i = 0; i < c; i++ )
		{
			debug << "   palPalEntry[ " << (BYTE)i << " ]: ";
			Dump( debug, Color( pal, i ) );
		}
	}
	debug << " End LOGPALETTE"                        << crlf;
}

void LogPalette::Dump( dbugstream& debug, const PALETTEENTRY& pal ) const
{
	debug << " peRed: "   << pal.peRed
		  << " peGreen: " << pal.peGreen
		  << " peBlue: "  << pal.peBlue
		  << " peFlags: " << pal.peFlags << crlf;
}

BOOL LogPalette::Negate()
{
	BOOL fReturn = FALSE;
	if ( IsCreated() )
	{
		for ( int i = 0; i < Size(); i++ )
		{
			PALETTEENTRY& pal = Color( i );
			pal.peRed   = ~pal.peRed;
			pal.peGreen = ~pal.peGreen;
			pal.peBlue  = ~pal.peBlue;
		}
		fReturn = TRUE;
	}
	return fReturn;
}

LogPalette LogPalette::operator ~() const
{
	LogPalette pal( *this );
	pal.Negate();
	return pal;
}

void LogPalette::ComputeFadeStep( BYTE a, BYTE b, float& step, BYTE steps )
{
	step = ( (float)b - (float)a ) / (float)steps;
}

void LogPalette::ComputeFadeStep( const PALETTEENTRY& start, 
							      const PALETTEENTRY& end, 
							      PaletteStep& step, BYTE steps )
{
	ComputeFadeStep( start.peRed,   end.peRed,   step.stRed,   steps );
	ComputeFadeStep( start.peGreen, end.peGreen, step.stGreen, steps );
	ComputeFadeStep( start.peBlue,  end.peBlue,  step.stBlue,  steps );
}

void LogPalette::ComputeFadeSteps( const LogPalette& target, 
								   PaletteStep* step, 
								   BYTE steps )
{
	for ( int i = 0; i < Size(); i++ )
		ComputeFadeStep( Color( i ), target.Color( i ), step[ i ], steps );
}

