// Win32 Zen timer; uses QueryPerformanceCounter() to get
// 838 ns timer resolution. Will not run on Win 3.1, because
// QueryPerformanceCounter() is a Win32-only API.
//
// Call WZTimerOn() to start timing, WZTimerOff() to stop
// timing, and WZTimerReport() to display the results in
// a message box. Set the parameter to WZTimerOn() to TRUE
// in order to boost priority so that the process being
// timed gets as much CPU time as possible; note that this
// causes everything else in the system, including the
// mouse, to stop. WZTimerOff() lowers the priority again
// if WZTimerOn() boosted it.
//
// Maximum timing interval is (2**32)-1 ticks, and maximum
// timer resolution is (2**32)-1 ticks per second. No
// current timers run anywhere near that fast, and (2**32)-1
// ticks on a PC is about an hour, so this isn't generally
// a problem.
//
// Ideally, this routine should attempt to compensate for
// overhead in calling the timing functions by calling them
// again back to back and seeing how long the null interval
// is, then subtracting this from the measured interval.
// However, the overhead is small, such compensation is
// imperfect, and most intervals timed are long enough that
// the overhead is irrelevant; at least, I haven't found it
// to be necessary. Feel free to add it if you'd like.
//
// Tested with VC++ 2.0 running on Windows NT 3.5.

#include <windows.h>
#include <stdio.h>
#include <mmsystem.h>

static LARGE_INTEGER    StartTime;
static LARGE_INTEGER    StopTime;
static INT              PriorityBoost;
static DWORD            PriorityClass;
static INT              ThreadPriority;

// Called to start a timing interval.
VOID WZTimerOn(INT BoostPriority)
{
    // Boost the priority to realtime if requested
    if (BoostPriority)
    {
        PriorityClass = GetPriorityClass(GetCurrentProcess());
        ThreadPriority = GetThreadPriority(GetCurrentThread());
        SetPriorityClass(GetCurrentProcess(),
                         REALTIME_PRIORITY_CLASS);
        SetThreadPriority(GetCurrentThread(),
                          THREAD_PRIORITY_TIME_CRITICAL);
        PriorityBoost = 1;
    }
    else
    {
        PriorityBoost = 0;
    }

    QueryPerformanceCounter(&StartTime);
}

// Called to end a timing interval.
VOID WZTimerOff()
{
    QueryPerformanceCounter(&StopTime);

    // Restore original priority if we boosted it
    if (PriorityBoost)
    {
        SetPriorityClass(GetCurrentProcess(),PriorityClass);
        SetThreadPriority(GetCurrentThread(),ThreadPriority);
    }
}

// Pops up a message box with the timing results
VOID WZTimerReport(HWND hwnd)
{
    CHAR            ReportString[80];
    BOOL            TimerPresent;
    LARGE_INTEGER   TimerFreq;
    double          ElapsedTicks, TickTime;

    TimerPresent = QueryPerformanceFrequency(&TimerFreq);

    if (!TimerPresent)
    {
        MessageBox(hwnd, "QueryPerformanceCounter not supported",
                   "Timing Results", MB_OK);       
    }
    else
    {
        ElapsedTicks = (double)
                (*(ULONG *)&StopTime - *(ULONG *)&StartTime);
        TickTime = 1000000.0 / (double)*(ULONG *)&TimerFreq;
        sprintf(ReportString,
                "Timed interval = %7.0f microseconds",
                ElapsedTicks * TickTime);
        MessageBox(hwnd, ReportString, "Timing Results", MB_OK);
    }
}
